(*
 * Coq code for "An Abstract, Approximation-Based Approach to
 *                               Embedded Code Pointers and Partial-Correctness"
 *
 * This file contains code adapted from
 *                  "Certified Assembly Programming with Embedded Code Pointers"
 *
 * Set level mapping library
 *
 * (for Coq version 8)
 *)

Section Map.

Variable A B : Set.

Set Implicit Arguments.
Unset Strict Implicit.

Definition Map := A -> option B.


Definition lookup (M : Map) a b := M a = Some b.

Definition in_dom (M : Map) a := exists b : _, M a = Some b.

Definition not_in_dom (M : Map) a := M a = None.

Definition empty_map : Map := fun _ => None.

Definition merge (M1 M2 : Map) a :=
  match M1 a with
  | None => M2 a
  | _ => M1 a
  end.

Definition disjoint (M1 M2 : Map) :=
  forall a : A,
  match M1 a, M2 a with
  | Some _, Some _ => False
  | _, _ => True
  end.

Definition subseteq (M1 M2 : Map) :=
  forall a, match M1 a with
            | None => True
            | Some _ => M1 a = M2 a
            end.

Definition meq (M1 M2 : Map) := forall a, M1 a = M2 a.

Lemma mergelookup :
 forall (M1 M2 : Map) a b,
 lookup (merge M1 M2) a b -> lookup M1 a b \/ lookup M2 a b.
unfold lookup, merge in |- *.
intros.
destruct (M1 a).
 left.
   auto.
 right.
   auto.
Qed.

Lemma lookup_meq_lookup :
  forall M1 M2 a b, lookup M1 a b -> meq M1 M2 -> lookup M2 a b.
unfold lookup, meq. intros. rewrite <- (H0 a). auto.
Qed.

Lemma lookup_disj_merge_lookup_1 :
  forall M1 M2 a b, lookup M1 a b -> disjoint M1 M2 -> lookup (merge M1 M2) a b.
unfold lookup, disjoint, merge. intros. generalize (H0 a). clear H0. intros.
destruct (M1 a). auto. inversion H.
Qed.

Lemma lookup_disj_merge_lookup_2 :
  forall M1 M2 a b, lookup M2 a b -> disjoint M1 M2 -> lookup (merge M1 M2) a b.
unfold lookup, disjoint, merge. intros. generalize (H0 a). clear H0. intros.
destruct (M1 a). rewrite H in H0. tauto. auto.
Qed.

Lemma indom_meq_indom :
  forall M1 M2 a, in_dom M1 a -> meq M1 M2 -> in_dom M2 a.
unfold in_dom, meq. intros. rewrite <- (H0 a). auto.
Qed.

Lemma indom_disj_merge_indom_1 :
  forall M1 M2 a, in_dom M1 a -> disjoint M1 M2 -> in_dom (merge M1 M2) a.
unfold in_dom, disjoint, merge. intros. generalize (H0 a). clear H0. intros.
destruct (M1 a). auto. inversion H. inversion H1.
Qed.

Lemma indom_disj_merge_indom_2 :
  forall M1 M2 a, in_dom M2 a -> disjoint M1 M2 -> in_dom (merge M1 M2) a.
unfold in_dom, disjoint, merge. intros. generalize (H0 a). clear H0. intros.
destruct (M1 a). destruct H. rewrite H in H0. tauto. auto.
Qed.

Lemma indom_disj_notindom :
  forall M1 M2 a, in_dom M1 a -> disjoint M1 M2 -> not_in_dom M2 a.
unfold in_dom, disjoint, not_in_dom. intros. destruct H. generalize (H0 a).
rewrite H. destruct (M2 a). tauto. auto.
Qed.

Lemma disj_sym : forall M1 M2, disjoint M1 M2 -> disjoint M2 M1.
intros. unfold disjoint. intros. generalize (H a). clear H.
intros. destruct (M1 a). auto. destruct (M2 a). auto. auto.
Qed.

Lemma meq_refl : forall M, meq M M.
unfold meq. auto.
Qed.

Lemma meq_sym : forall M1 M2, meq M1 M2 -> meq M2 M1.
unfold meq. auto.
Qed.

Lemma meq_trans : forall M1 M2 M3, meq M1 M2 -> meq M2 M3 -> meq M1 M3.
unfold meq. intros. rewrite (H a). auto.
Qed.

Lemma meq_merge_meq :
  forall M1 M2 M3 M4, meq M1 M2 -> meq M3 M4 -> meq (merge M1 M3) (merge M2 M4).
unfold meq, merge. intros. rewrite (H a). rewrite (H0 a). auto.
Qed.

Lemma disj_merge_sym :
  forall M1 M2, disjoint M1 M2 -> meq (merge M1 M2) (merge M2 M1).
unfold meq, merge, disjoint. intros. generalize (H a). clear H. intros.
destruct (M1 a). destruct (M2 a). tauto. auto.
destruct (M2 a). auto. auto.
Qed.

Lemma disj_meq_disj :
  forall M1 M2 M3, disjoint M1 M2 -> meq M2 M3 -> disjoint M1 M3.
unfold disjoint, meq. intros. generalize (H a). clear H. intros.
rewrite <- (H0 a). auto.
Qed.

Lemma disj_merge_disj :
  forall M1 M2 M3, disjoint M1 M3 -> disjoint M2 M3 -> disjoint (merge M1 M2) M3.
unfold disjoint. intros. generalize (H a) (H0 a). clear H H0. intros.
unfold merge. destruct (M1 a). auto. auto.
Qed.

Lemma merge_assoc_L :
  forall M1 M2 M3, meq (merge (merge M1 M2) M3) (merge M1 (merge M2 M3)).
unfold merge, meq. intros. destruct (M1 a). auto. auto.
Qed.

Lemma merge_assoc_R :
  forall M1 M2 M3, meq (merge M1 (merge M2 M3)) (merge (merge M1 M2) M3).
intros. apply meq_sym. apply merge_assoc_L.
Qed.

Lemma merge_disj_disj_1 :
  forall M1 M2 M3, disjoint M1 (merge M2 M3) -> disjoint M1 M2.
unfold disjoint, merge. intros. generalize (H a). clear H. intros.
destruct (M1 a). destruct (M2 a). auto. auto. auto.
Qed.

Lemma merge_disj_disj_2 :
  forall M1 M2 M3, disjoint M1 (merge M2 M3) -> disjoint M1 M3.
unfold disjoint, merge. intros. generalize (H a). clear H. intros.
destruct (M1 a). destruct (M2 a). tauto. auto. auto.
Qed.

End Map.