(*
 * Coq code for "An Abstract, Approximation-Based Approach to
 *                               Embedded Code Pointers and Partial-Correctness"
 *
 * This file contains code adapted from
 *                  "Certified Assembly Programming with Embedded Code Pointers"
 *
 * Type level mapping library
 *
 * (for Coq version 8)
 *)

Section Mapt.

Inductive option (A : Type) : Type :=
  | Some : A -> option A
  | None : option A.

Variable A B : Type.

Set Implicit Arguments.
Unset Strict Implicit.

Definition Map := A -> option B.

Definition in_dom (M : Map) a := exists b : _, M a = Some _ b.

Definition not_in_dom (M : Map) a := M a = None _.

Definition empty_map : Map := fun _ => None _.

Definition lookup (M : Map) a b := M a = Some _ b.

Definition merge (M1 M2 : Map) a :=
  match M1 a with
  | None => M2 a
  | _ => M1 a
  end.

Definition disjoint (M1 M2 : Map) :=
  forall a : A,
  match M1 a, M2 a with
  | Some _, Some _ => False
  | _, _ => True
  end.

Definition subseteq (M1 M2 : Map) :=
  forall a, match M1 a with
            | None => True
            | Some _ => M1 a = M2 a
            end.

Definition meq (M1 M2 : Map) := forall a, M1 a = M2 a.

Lemma mergelookup :
 forall (M1 M2 : Map) a b,
 lookup (merge M1 M2) a b -> lookup M1 a b \/ lookup M2 a b.
unfold lookup, merge in |- *.
intros.
destruct (M1 a).
 left.
   auto.
 right.
   auto.
Qed.

Lemma disj_sym : forall M1 M2, disjoint M1 M2 -> disjoint M2 M1.
intros. unfold disjoint. intros. generalize (H a). clear H.
intros. destruct (M1 a). auto. destruct (M2 a). auto. auto.
Qed.

Lemma meq_trans : forall M1 M2 M3, meq M1 M2 -> meq M2 M3 -> meq M1 M3.
unfold meq. intros. rewrite (H a). auto.
Qed.

Lemma disj_merge_sym :
  forall M1 M2, disjoint M1 M2 -> meq (merge M1 M2) (merge M2 M1).
unfold meq, merge, disjoint. intros. generalize (H a). clear H. intros.
destruct (M1 a). destruct (M2 a). tauto. auto.
destruct (M2 a). auto. auto.
Qed.

Lemma disj_merge_disj :
  forall M1 M2 M3, disjoint M1 M3 -> disjoint M2 M3 -> disjoint (merge M1 M2) M3.
unfold disjoint. intros. generalize (H a) (H0 a). clear H H0. intros.
unfold merge. destruct (M1 a). auto. auto.
Qed.

Lemma merge_disj_disj_1 :
  forall M1 M2 M3, disjoint M1 (merge M2 M3) -> disjoint M1 M2.
unfold disjoint, merge. intros. generalize (H a). clear H. intros.
destruct (M1 a). destruct (M2 a). auto. auto. auto.
Qed.

Lemma merge_disj_disj_2 :
  forall M1 M2 M3, disjoint M1 (merge M2 M3) -> disjoint M1 M3.
unfold disjoint, merge. intros. generalize (H a). clear H. intros.
destruct (M1 a). destruct (M2 a). tauto. auto. auto.
Qed.

Lemma lookup_subseteq_lookup :
  forall M1 M2 a b, subseteq M1 M2 -> lookup M1 a b -> lookup M2 a b.
unfold subseteq, lookup; intros; generalize (H a); intro; rewrite H0 in H1; auto.
Qed.

End Mapt.