(*
 * Coq code for "An Abstract, Approximation-Based Approach to
 *                               Embedded Code Pointers and Partial-Correctness"
 *
 * This file contains code adapted from
 *                  "Certified Assembly Programming with Embedded Code Pointers"
 *
 * natural number utilities
 *
 * (for Coq version 8)
 *)


(***************************************************************************)
(* Utilities for manipulating natural numbers *)
(***************************************************************************)

Require Import Div2.
Require Import EqNat.
Require Import Lt.
Require Import Le.



(***************************************************************************)

Fixpoint bgt_nat (n m : nat) {struct n} : bool :=
  match n, m with
  | O, O => false
  | O, S _ => false
  | S _, O => true
  | S n1, S m1 => bgt_nat n1 m1
  end.

Fixpoint blt_nat (n m : nat) {struct n} : bool :=
  match n, m with
  | O, O => false
  | O, S _ => true
  | S _, O => false
  | S n1, S m1 => blt_nat n1 m1
  end.

Fixpoint ble_nat (n m : nat) {struct n} : bool :=
  match n, m with
  | O, O => true
  | O, S _ => true
  | S _, O => false
  | S n1, S m1 => ble_nat n1 m1
  end.

Lemma ble_le_true : forall a b : nat, a <= b -> ble_nat a b = true.
intro.
induction  a as [| a Hreca].
 intros.
   simpl in |- *.
   destruct b as [| n].
  auto.
  auto.
 intros.
   destruct H.
  simpl in |- *.
    apply Hreca.
    auto.
  simpl in |- *.
    apply Hreca.
    auto.
    apply le_Sn_le.
    auto.
Qed.

Lemma ble_gt_false : forall a b : nat, a > b -> ble_nat a b = false.
unfold gt in |- *.
simple induction a.
 intros.
   inversion H.
 intros.
   destruct b as [| n0].
  auto.
  simpl in |- *.
    apply H.
    auto.
    apply lt_S_n.
    auto.
Qed.

Lemma blt_S_false : forall a : nat, blt_nat (S a) a = false.
Proof.
simple induction a.
unfold blt_nat in |- *.
auto.
intros.
unfold blt_nat in |- *.
auto.
Qed.

Lemma blt_S_eq :
 forall (a b : nat) (t : bool), blt_nat a b = t -> blt_nat (S a) (S b) = t.
Proof.
intros.
unfold blt_nat in |- *.
fold blt_nat in |- *.
auto.
Qed.

Lemma blt_lt_true : forall a b : nat, a < b -> blt_nat a b = true.
Proof.
simple induction a.
intros.
inversion H.
unfold blt_nat in |- *.
auto.
unfold blt_nat in |- *.
auto.
intros.
inversion H0.
apply blt_S_eq.
apply H.
apply lt_n_Sn.
apply blt_S_eq.
apply H.
rewrite <- H2 in H0.
apply lt_S_n.
auto.
Qed.


Lemma blt_Sm_false :
 forall a b : nat, blt_nat a b = false -> blt_nat (S a) b = false.
Proof.
simple induction a.
simple induction b.
intros.
unfold blt_nat in |- *.
auto.

intros.
unfold blt_nat in H0.
absurd (true = false).
discriminate.

auto.

simple induction b.
intros.
unfold blt_nat in |- *.
auto.

intros.
intros.
unfold blt_nat in H1.
fold blt_nat in H1.
cut (blt_nat (S n) n0 = false).
intros.
apply blt_S_eq.
auto.

apply (H n0).
auto.
Qed.

Lemma blt_false : forall a b : nat, a < b -> blt_nat b a = false.
Proof.
intros.
induction  H as [| m H HrecH].
apply blt_S_false.
apply blt_Sm_false.
auto.
Qed.


Lemma blt_eq_false : forall a : nat, blt_nat a a = false.
Proof.
simple induction a.
unfold blt_nat in |- *.
auto.
intros.
unfold blt_nat in |- *.
auto.
Qed.


(*Lemma le_blt_false : (a,b:nat) (le a b) -> (blt_nat b a)=false.
Proof.
Intros.
Induction H.
Apply blt_eq_false.
Apply blt_false.
Unfold lt.
Apply le_n_S.
Auto.
Save.
*)



(***************************************************************************)
Lemma beq_eq_true : forall a b : nat, a = b -> true = beq_nat a b.
simple induction a.
 simple induction b.
  auto.
  intros.
    inversion H0.
 simple induction b.
  intros.
    inversion H0.
  intros.
    simpl in |- *.
    auto.
Qed.

Lemma beq_neq_false : forall a b : nat, a <> b -> false = beq_nat a b.
Proof.
simple induction a.
simple induction b.
intros.
absurd (0 <> 0).
auto.

auto.

intros.
unfold beq_nat in |- *.
auto.

intros.
unfold beq_nat in |- *.
auto.
induction  b as [| b Hrecb].
auto.

unfold beq_nat in |- *.
fold beq_nat in |- *.
unfold beq_nat in |- *.
auto.
Qed.


Lemma eq_neq_Sl : forall a b : nat, a = b -> S a <> b.
Proof.
simple induction a.
simple induction b.
intros.
auto.

intros.
absurd (0 = S n).
discriminate.

auto.

simple induction b.
intros.
absurd (S n = 0).
discriminate.

auto.

intros.
apply not_eq_S.
auto.
Qed.


(***************************************************************************)
(* definitions of numbers *)
Definition N0 := 0.
Definition N1 := S N0.
Definition N2 := S N1.
Definition N3 := S N2.
Definition N4 := S N3.
Definition N5 := S N4.
Definition N6 := S N5.
Definition N7 := S N6.
Definition N8 := S N7.
Definition N9 := S N8.
Definition N10 := S N9.
Definition N11 := S N10.
Definition N12 := S N11.
Definition N13 := S N12.
Definition N14 := S N13.
Definition N15 := S N14.
Definition N16 := S N15.
Definition N17 := S N16.
Definition N18 := S N17.
Definition N19 := S N18.
Definition N20 := S N19.
Definition N21 := S N20.
Definition N22 := S N21.
Definition N23 := S N22.
Definition N24 := S N23.
Definition N25 := S N24.
Definition N26 := S N25.
Definition N27 := S N26.
Definition N28 := S N27.
Definition N29 := S N28.
Definition N30 := S N29.
Definition N31 := S N30.
Definition N32 := S N31.

(***************************************************************************)
(* utilities for decoding *)

(* to get  n % d  call this as (modaux n d d O), the second argument is
   decremented in recursive calls, the fourth is incremented *)
Fixpoint modaux (n c d r : nat) {struct n} : nat :=
  match n with
  | O => r
  | S p =>
      match c with
      | O => 0
      | S O => modaux p d d 0
      | S c' => modaux p c' d (S r)
      end
  end.

Definition mod_ (n d : nat) : nat := modaux n d d 0.

Definition div4 (n : nat) := div2 (div2 n).
Definition div8 (n : nat) := div4 (div2 n).
Definition div16 (n : nat) := div2 (div8 n).
Definition div32 (n : nat) := div2 (div16 n).


Definition mult4 (n : nat) := N4 * n.
Definition mult8 (n : nat) := N8 * n.
Definition mult16 (n : nat) := N16 * n.
Definition mult32 (n : nat) := N32 * n.


(***************************************************************************)

Theorem eqorneq_nat : forall a a' : nat, a = a' \/ a <> a'.
Proof.
intros.
compare a a'.
auto.
auto.
Qed.


Theorem le_or_gt : forall m n : nat, n <= m \/ n > m.
Proof.
double induction m n.
left.
auto.
intros.
right.
red in |- *.
red in |- *.
apply le_n_S.
apply le_O_n.
intros.
left.
apply le_O_n.
intros.
cut (n <= n1 \/ n > n1).
intros.
inversion_clear H1.
left.
apply le_n_S.
auto.
right.
unfold gt in |- *.
unfold gt in H2.
apply lt_n_S.
auto.
apply H0.
Qed.


Lemma lt_neq : forall m n : nat, n < m -> n <> m.
Proof.
simple induction m.
intros.
inversion H.
intros.
induction  n0 as [| n0 Hrecn0].
auto.
cut (n0 < n).
intros.
cut (n0 <> n).
intros.
auto.
apply H.
auto.
apply lt_S_n.
auto.
Qed.


Lemma lt_S_neq : forall n x : nat, n < S x -> x <> n -> n < x.
Proof.
double induction n x.
intros.
absurd (0 <> 0).
auto.
auto.
intros.
apply lt_O_Sn.
intros.
unfold lt in H0.
cut (S n0 <= 0).
intros.
absurd (S n0 <= 0).
apply le_Sn_O.
auto.
apply le_S_n.
auto.
intros.
apply lt_n_S.
cut (n1 = n0 \/ n1 <> n0).
intros.
inversion_clear H3.
absurd (S n1 <> S n1).
auto.
rewrite <- H4 in H2.
auto.
cut (n1 < S n0).
intros.
apply (H0 n0 H3).
auto.
apply lt_S_n.
auto.
apply eqorneq_nat.
Qed.


Lemma lt_S_eq_or_lt : forall n x : nat, n < S x -> x = n \/ n < x.
Proof.
intros.
cut (x = n \/ x <> n).
intros.
inversion_clear H0.
left; auto.
right; apply lt_S_neq.
auto.
auto.
apply eqorneq_nat.
Qed.


Lemma plus_n_O_n : forall n : nat, n + 0 = n.
Proof.
simple induction n.
unfold plus in |- *.
auto.
intros.
unfold plus in |- *.
fold plus in |- *.
rewrite H.
auto.
Qed.


Lemma plus_S_neq : forall n m : nat, n <> n + S m.
Proof.
simple induction n.
intros.
unfold plus in |- *.
auto.
intros.
unfold plus in |- *.
fold plus in |- *.
auto.
Qed.

(***************************************************************************)