(*
 * Coq code for "An Abstract, Approximation-Based Approach to
 *                               Embedded Code Pointers and Partial-Correctness"
 *
 * The Abstract, Approximation-Based Verification System
 *
 * (for Coq version 8)
 *)

Require Import natutil tm Mapt.

(* abstract assertion language and inference rules for user *)

Definition X :=
  {Spec : _ & ((Label -> (State -> Spec -> Prop) -> Spec -> Prop) * Spec) %type}.

Definition PropX     := X -> Prop.

Definition Assertion := State -> PropX.

Definition CdHpSpec  := Map Label Assertion.

Notation "a ==> b" := (forall s x, (a : Assertion) s x -> (b : Assertion) s x)
                      (at level 70, right associativity).

Definition cptr f (a : Assertion) x :=
  match x with existT Spec (pair cptr Si) =>
    cptr f (fun S Si => a S (existT _ Spec (pair cptr Si))) Si
  end.

Definition codeptr f a x := exists a', cptr f a' x /\ a ==> a'.

Inductive WFiseq : CdHpSpec -> Assertion -> InstrSeq -> Prop :=
  | wfiseq : forall Si a c I a',
               a ==> (fun s x => exists s', Next c s s' /\ a' s' x) ->
               WFiseq Si a' I                          -> WFiseq Si a (iseq c I)
  | wfbgt  : forall Si a rs rt f I (a' a'' : Assertion),
               (forall s x, (_R s rs <= _R s rt /\ a s x) -> a'' s x) ->
               (forall s x, (_R s rs >  _R s rt /\ a s x) -> a'  s x) ->
               lookup Si f a' -> WFiseq Si a'' I  -> WFiseq Si a (bgt rs rt f I)
  | wfbgti : forall Si a rs w f I (a' a'' : Assertion),
               (forall s x, (_R s rs <= w /\ a s x) -> a'' s x) ->
               (forall s x, (_R s rs >  w /\ a s x) -> a'  s x) ->
               lookup Si f a' -> WFiseq Si a'' I  -> WFiseq Si a (bgti rs w f I)
  | wfjd   : forall Si a f a',
               lookup Si f a' -> a ==> a'                  -> WFiseq Si a (jd f)
  | wfjmp  : forall Si a r,
               a ==> (fun s x => exists a', cptr (_R s r) a' x /\ a' s x)
                                                          -> WFiseq Si a (jmp r)
  | wfecp  : forall Si a f a' a'' I,
               WFiseq Si a' I -> lookup Si f a'' ->
               (fun s x => cptr f a'' x /\ a s x) ==> a'        -> WFiseq Si a I.

Definition ok A (a : A -> PropX) :=
  forall x y, (forall f a, cptr f a x -> cptr f a y) -> forall s, a s x -> a s y.
Implicit Arguments ok [A].

Definition WFcode Si C (Si' : CdHpSpec) :=
  forall f a, lookup Si' f a ->
    ok a /\ exists I, Map.lookup C f I /\ WFiseq Si a I.

Definition WFprog Si a P :=
  match P with pair C (pair s i) =>
    WFcode Si C Si /\ WFiseq Si a i /\
                   forall x, (forall f a, lookup Si f a -> cptr f a x) -> a s x
  end.

Lemma InstrSeqWeakening :
  forall Si a I, WFiseq Si a I ->
    forall Si' a', subseteq Si Si' -> a' ==> a -> WFiseq Si' a' I.
induction 1; intros.
 econstructor; eauto; firstorder.
 econstructor; eauto; firstorder.
  apply (H0 _ _ (conj H5 (H4 _ _ H6))). eapply lookup_subseteq_lookup; eauto.
 econstructor; eauto; firstorder.
  apply (H0 _ _ (conj H5 (H4 _ _ H6))). eapply lookup_subseteq_lookup; eauto.
 econstructor. eapply lookup_subseteq_lookup; eauto. firstorder.
 econstructor; eauto; firstorder.
 econstructor; eauto; firstorder.
  eapply lookup_subseteq_lookup; eauto. firstorder.
Qed.

Lemma CodeHeapLinking :
  forall Si1 Si2 Si'1 Si'2 C1 C2,
    WFcode Si1 C1 Si'1 -> WFcode Si2 C2 Si'2 -> Map.disjoint C1 C2 ->
    subseteq Si1 (merge Si1 Si2) -> subseteq Si2 (merge Si1 Si2)
                  -> WFcode (merge Si1  Si2) (Map.merge C1 C2) (merge Si'1 Si'2).
unfold subseteq, merge, Map.merge, WFcode, lookup, Map.lookup; intros.
generalize (H f a) (H0 f a); clear H H0; intros.
destruct (Si'1 f).
 destruct H; auto; split; auto.
   destruct H5; exists x; destruct H5; rewrite H5; split; auto.
   apply InstrSeqWeakening with Si1 a; auto.
destruct (Si'2 f).
 destruct H0; auto; split; auto.
   destruct H5; exists x; destruct H5; rewrite H5; split; auto.
    unfold Map.disjoint in *; generalize (H1 f); destruct (C1 f); auto.
      rewrite H5; tauto.
   apply InstrSeqWeakening with Si2 a; auto.
inversion H4.
Qed.

Lemma cptr_mono_psv : forall x x' : X,
  match x, x' with existT Spec (pair cptr Si), existT Spec' (pair cptr' Si') =>
    (forall l a, cptr l (a Spec cptr) Si -> cptr' l (a Spec' cptr') Si')
  end                                   -> forall l a, cptr l a x -> cptr l a x'.
destruct x; destruct p; destruct x'; destruct p; simpl; intros.
apply H with (a := fun Spec cptr S Si => a S (existT _ Spec (pairT cptr Si))).
auto.
Qed.

(* concrete approximation-based proof of soundness and partial-correctness *)

Fixpoint nSpec n : Type :=
  match n with
  | O   => unit
  | S n => prod (nSpec n) (Label -> option (State -> nSpec n -> Prop))
  end.

Definition iSpec  := sigT nSpec.

Definition iAssertion  := State -> iSpec -> Prop.

Definition iCdHpSpec := Label -> option iAssertion.

Fixpoint nspec n (Si : iCdHpSpec) : nSpec n :=
  match n return nSpec n with
    O   => tt
  | S _ => pair (nspec _ Si)
                (fun l => match Si l with
                            None   => None _
                          | Some a => Some _ (fun S Si => a S (existT _ _ Si))
                          end)
  end.

Definition iok (a : iAssertion) :=
  forall s Si n,
    a s (existT _ _ (nspec (S n) Si)) -> a s (existT _ _ (nspec n Si)).

Definition icptr l a (SI : iSpec) : Prop :=
  match SI with existT n Si =>
    iok a /\ exists si, nspec n si = Si /\ lookup si l a
  end.

Definition trans (a : Assertion) :=
  fun S Si => a S (existT _ _ (pair icptr Si)).

Definition Trans (Si : CdHpSpec) l :=
  match Si l with
    None   => None _
  | Some a => Some _ (trans a)
  end.

Definition inspec n Si : iSpec := existT _ _ (nspec n (Trans Si)).

Definition WFprogram Si a P :=
  match P with pair C (pair s i) =>
    WFcode Si C Si /\ WFiseq Si a i /\
    forall n, a s (existT _ _ (pair icptr (inspec n Si)))
  end.

Lemma ok_iok : forall a, ok a -> iok (trans a).
unfold ok, iok, trans; intros; generalize H0; apply H; intros f a0.
apply cptr_mono_psv; simpl; intros; destruct H1; split; auto; destruct H2.
exists x; destruct H2; split; auto; inversion H2; auto.
Qed.

Lemma CodePointerLookup :
  forall Si l a, icptr l a (inspec (S O) Si) -> exists a, lookup Si l a.
intros; destruct H; unfold lookup, Trans in *; destruct H0; destruct H0.
  inversion H0; clear H0; generalize (f_equal (fun f => f l) H3); clear H3.
  intro; rewrite H1 in H0; destruct (Si l).
 eauto.
 inversion H0.
Qed.

Lemma LookupCodePointer :
  forall Si l a, lookup Si l a -> ok a ->
    forall n, icptr l (trans a) (inspec n Si).
unfold trans; intros; split.
 apply (ok_iok _ H0).
 exists (Trans Si); split; auto.
  unfold lookup in *; unfold Trans; rewrite H; auto.
Qed.

Lemma Invariant :
  forall Si a P, WFprogram Si a P ->
    exists a', exists P', STEP P P' /\ WFprogram Si a' P'.
 intros; destruct P as [c]; destruct p; destruct H; destruct H0; induction H0.
  exists a'; destruct (H0 _ _ (H1 O)); destruct H3; exists (c, (x, I)); split.
   apply stp_iseq; auto.
   split; auto.
   split; auto.
   intro; destruct (H0 _ _ (H1 n)); destruct H5;
     rewrite (Determinstic _ _ _ _ H3 H5); auto.
  destruct s; destruct (le_or_gt (r rt) (r rs)).
   exists a''; exists (c, (h, r, I)); split.
    apply stp_bgt'; auto.
    split; auto; split; auto.
   exists a'; destruct (H _ _ H3).
    destruct H7; destruct H7; exists (c, (h, r, x)); split.
    apply stp_bgt; auto.
    split; auto; split; auto.
  destruct s; destruct (le_or_gt w (r rs)).
   exists a''; exists (c, (h, r, I)); split.
    apply stp_bgti'; auto.
    split; auto; split; auto.
   exists a'; destruct (H _ _ H3).
    destruct H7; destruct H7; exists (c, (h, r, x)); split.
    apply stp_bgti; auto.
    split; auto; split; auto.
  exists a'; destruct (H _ _ H0); destruct H4; destruct H4.
   exists (c, (s, x)); split.
   apply stp_jd; auto.
   firstorder; apply ((existT _ iSpec (pair icptr (inspec O Si))):X).
  destruct s; destruct (H0 _ _ (H1 (S O))); destruct H2.
    destruct (CodePointerLookup _ _ _ H2); clear x H2 H3; exists x0.
    destruct (H _ _ H4); destruct H3; destruct H3.
      exists (c, (h, r0, x)); split.
     apply stp_jmp; auto.
     split; auto; split; auto.
     intros; destruct (H0 _ _ (H1 (S n))); destruct H6; destruct H6.
       destruct H8; destruct H8; unfold lookup in *; unfold Trans in H8.
       generalize (f_equal (fun f => match f with pair _ si => si (r0 r) end) H8).
       clear H8; intro; simpl in H8.
         rewrite H4 in H8; rewrite H9 in H8; inversion H8.
       unfold trans, inspec in *.
       generalize (H6 _ _ _ H7).
       generalize (f_equal (fun f => f (h, r0) (nspec n (Trans Si))) H11).
       intros; rewrite H10 in H12; auto.
  apply IHWFiseq; auto; intros; apply H3; split; auto; unfold cptr.
   apply LookupCodePointer with (a:=a''); auto.
   destruct (H _ _ H2); auto.
Qed.

Lemma WFprogWFprogram : forall Si a P, WFprog Si a P -> WFprogram Si a P.
intros; destruct P; destruct p; destruct H; split; auto; destruct H0; split; auto.
intros; apply H1; intros; simpl; split.
 destruct (H _ _ H2); unfold ok, iok in *; clear H1 H4.
   intro; intro; intro; apply H3; firstorder.
   exists x; split; auto. simpl in H4; inversion H4; auto.
  exists (Trans Si); split; auto; unfold lookup, Trans in *; rewrite H2; auto.
Qed.

Definition PCorrect (Si : CdHpSpec) (P : Program) :=
  match P with (_, (s, i)) => match i with
  | jd f          => exists a, lookup Si f a /\ exists x, a s x
  | jmp r         => exists a, lookup Si (_R s r) a /\ exists x, a s x
  | bgt rs rt f _ => _R s rs > _R s rt ->
                       exists a, lookup Si f a /\ exists x, a s x
  | bgti rs i f _ => _R s rs > i ->
                       exists a, lookup Si f a /\ exists x, a s x
  | _ => True
  end end.

Lemma WFprogramPartialCorrectness :
  forall Si a P, WFprogram Si a P -> PCorrect Si P.
intros; destruct P as [C]; destruct p as [S I].
  destruct H; destruct H0; induction H0; simpl; auto; intros.
 exists a'; split; auto. exists ((existT _ iSpec (icptr, inspec O Si)):X); auto.
 exists a'; split; auto. exists ((existT _ iSpec (icptr, inspec O Si)):X); auto.
 exists a'; split; auto. exists ((existT _ iSpec (icptr, inspec O Si)):X); auto.
 destruct (H0 _ _ (H1 1)); destruct H2; destruct (CodePointerLookup _ _ _ H2).
   exists x0; destruct S; split; auto; simpl in H2.
   exists ((existT _ iSpec (icptr, inspec O Si)):X); auto.
   destruct H2; generalize (H2 _ _ _ H3); destruct H5; destruct H5; inversion H5.
   unfold lookup in *; generalize (f_equal (fun f => f (r0 r)) H8).
   rewrite H6; unfold Trans at 1; rewrite H4; unfold trans; inversion 1.
   rewrite (f_equal (fun f => f (h, r0) (nspec 0 (Trans Si))) H10); auto.
 apply IHWFiseq; auto; intros; apply H3; split; auto; unfold cptr.
   apply LookupCodePointer with (a:=a''); auto.
   destruct (H _ _ H2); auto.
Qed.

Fixpoint STEPn n P P' : Prop :=
  match n with
  | O   => P = P'
  | S n => exists P'', STEP P P'' /\ STEPn n P'' P'
  end.

Theorem PartialCorrectness :
  forall Si a P, WFprog Si a P ->
    forall n, exists P', STEPn n P P' /\ PCorrect Si P'.
intros; generalize Si a P (WFprogWFprogram _ _ _ H); clear Si a P H.
  induction n; simpl; intros.
 exists P; split; auto; apply WFprogramPartialCorrectness with a; auto.
 destruct (Invariant _ _ _ H); destruct H0; destruct H0; destruct (IHn _ _ _ H1).
   destruct H2; eauto.
Qed.

Theorem Soundness :
  forall Si a P, WFprog Si a P -> forall n, exists P', STEPn n P P'.
intros; destruct (PartialCorrectness _ _ _ H n); exists x; tauto.
Qed.