(*
 * Coq code for "An Abstract, Approximation-Based Approach to
 *                               Embedded Code Pointers and Partial-Correctness"
 *
 * The Abstract, Approximation-Based Verification System with Dynamic Code Generation
 *
 * (for Coq version 8)
 *)

Require Import natutil tm1 Mapt.

(* abstract assertion language and inference rules for user *)

Definition X :=
  {Spec : _ & ((Label -> (State -> Spec -> Prop) -> Spec -> Prop) * Spec) %type}.

Definition PropX     := X -> Prop.

Definition Assertion := State -> PropX.

Definition CdHpSpec  := Map Label Assertion.

Notation "a ==> b" := (forall s x, (a : Assertion) s x -> (b : Assertion) s x)
                      (at level 70, right associativity).

Definition cptr f (a : Assertion) x :=
  match x with existT Spec (pair cptr Si) =>
    cptr f (fun S Si => a S (existT _ Spec (pair cptr Si))) Si
  end.

Definition codeptr f a x := exists a', cptr f a' x /\ a ==> a'.

Definition ok A (a : A -> PropX) :=
  forall x y, (forall f a, cptr f a x -> cptr f a y) -> forall s, a s x -> a s y.
Implicit Arguments ok [A].

Inductive WFiseq : CdHpSpec -> Assertion -> InstrSeq -> Prop :=
  | wfiseq : forall Si a c I a',
               a ==> (fun s x => exists s', Next c s s' /\ a' s' x) ->
               WFiseq Si a' I                          -> WFiseq Si a (iseq c I)
  | wfbgt  : forall Si a rs rt f I (a' a'' : Assertion),
               (forall s x, (_R s rs <= _R s rt /\ a s x) -> a'' s x) ->
               (forall s x, (_R s rs >  _R s rt /\ a s x) -> a'  s x) ->
               lookup Si f a' -> WFiseq Si a'' I  -> WFiseq Si a (bgt rs rt f I)
  | wfbgti : forall Si a rs w f I (a' a'' : Assertion),
               (forall s x, (_R s rs <= w /\ a s x) -> a'' s x) ->
               (forall s x, (_R s rs >  w /\ a s x) -> a'  s x) ->
               lookup Si f a' -> WFiseq Si a'' I  -> WFiseq Si a (bgti rs w f I)
  | wfjd   : forall Si a f a',
               lookup Si f a' -> a ==> a'                  -> WFiseq Si a (jd f)
  | wfjmp  : forall Si a r,
               a ==> (fun s x => exists a', cptr (_R s r) a' x /\ a' s x)
                                                          -> WFiseq Si a (jmp r)
  | wfecp  : forall Si a f a' a'' I,
               WFiseq Si a' I -> lookup Si f a'' ->
               (fun s x => cptr f a'' x /\ a s x) ==> a'        -> WFiseq Si a I
  | wfloadcode : forall Si a rt I I' a' a'',
                   ok a' -> WFiseq Si a' I -> WFiseq Si a'' I' ->
                   (fun s x => cptr (_R s rt) a' x
                            /\ exists w, a (_H s, uR (_R s) rt w) x) ==> a''
                                               -> WFiseq Si a (loadcode rt I I').

Definition WFcode Si C (Si' : CdHpSpec) :=
  forall f a, lookup Si' f a ->
    ok a /\ exists I, Map.lookup C f I /\ WFiseq Si a I.

Definition WFprog Si a P :=
  match P with pair C (pair s i) =>
    exists Six, subseteq Si Six /\ WFcode Six C Six /\ WFiseq Six a i /\
                   forall x, (forall f a, lookup Six f a -> cptr f a x) -> a s x
  end.

Lemma InstrSeqWeakening :
  forall Si a I, WFiseq Si a I ->
    forall Si' a', subseteq Si Si' -> a' ==> a -> WFiseq Si' a' I.
induction 1; intros.
 econstructor; eauto; firstorder.
 econstructor; eauto; firstorder.
  apply (H0 _ _ (conj H5 (H4 _ _ H6))). eapply lookup_subseteq_lookup; eauto.
 econstructor; eauto; firstorder.
  apply (H0 _ _ (conj H5 (H4 _ _ H6))). eapply lookup_subseteq_lookup; eauto.
 econstructor. eapply lookup_subseteq_lookup; eauto. firstorder.
 econstructor; eauto; firstorder.
 econstructor; eauto; firstorder.
  eapply lookup_subseteq_lookup; eauto. firstorder.
 apply wfloadcode with a' a''; auto.
  intros; apply H2; destruct H5; split; auto; destruct H6; exists x0; auto.
Qed.

Lemma CodeHeapLinking :
  forall Si1 Si2 Si'1 Si'2 C1 C2,
    WFcode Si1 C1 Si'1 -> WFcode Si2 C2 Si'2 -> Map.disjoint C1 C2 ->
    subseteq Si1 (merge Si1 Si2) -> subseteq Si2 (merge Si1 Si2)
                  -> WFcode (merge Si1  Si2) (Map.merge C1 C2) (merge Si'1 Si'2).
unfold subseteq, merge, Map.merge, WFcode, lookup, Map.lookup; intros.
generalize (H f a) (H0 f a); clear H H0; intros.
destruct (Si'1 f).
 destruct H; auto; split; auto.
   destruct H5; exists x; destruct H5; rewrite H5; split; auto.
   apply InstrSeqWeakening with Si1 a; auto.
destruct (Si'2 f).
 destruct H0; auto; split; auto.
   destruct H5; exists x; destruct H5; rewrite H5; split; auto.
    unfold Map.disjoint in *; generalize (H1 f); destruct (C1 f); auto.
      rewrite H5; tauto.
   apply InstrSeqWeakening with Si2 a; auto.
inversion H4.
Qed.

Lemma cptr_mono_psv : forall x x' : X,
  match x, x' with existT Spec (pair cptr Si), existT Spec' (pair cptr' Si') =>
    (forall l a, cptr l (a Spec cptr) Si -> cptr' l (a Spec' cptr') Si')
  end                                   -> forall l a, cptr l a x -> cptr l a x'.
destruct x; destruct p; destruct x'; destruct p; simpl; intros.
apply H with (a := fun Spec cptr S Si => a S (existT _ Spec (pairT cptr Si))).
auto.
Qed.

(* concrete approximation-based proof of soundness and partial-correctness *)

(* Extensional equality on functions, supposed to be a "safe" extention of Coq,
   i.e., does not violate the consistency of Coq. See Coq Faq for details. *)
Axiom ext_eq : forall A B (f g : A -> B), (forall x, f x = g x) -> f = g.

(* Assumption about code heap being finite at any given time.
   There are two ways to remove the axiom, either by changing the encoding
   of mapping into a finite one, or directly include the following condition
   in the top-level global invariant rule.
*)
Axiom FiniteCodeHeap : forall C : CodeHeap, exists f,
                      Map.in_dom C f /\ forall f', f' > f -> Map.not_in_dom C f'.

Fixpoint nSpec n : Type :=
  match n with
  | O   => unit
  | S n => prod (nSpec n) (Label -> option (State -> nSpec n -> Prop))
  end.

Definition iSpec  := sigT nSpec.

Definition iAssertion  := State -> iSpec -> Prop.

Definition iCdHpSpec := Label -> option iAssertion.

Fixpoint nspec n (Si : iCdHpSpec) : nSpec n :=
  match n return nSpec n with
    O   => tt
  | S _ => pair (nspec _ Si)
                (fun l => match Si l with
                            None   => None _
                          | Some a => Some _ (fun S Si => a S (existT _ _ Si))
                          end)
  end.

Definition iok (a : iAssertion) :=
  forall s Si n,
    a s (existT _ _ (nspec (S n) Si)) -> a s (existT _ _ (nspec n Si)).


Definition icptr l a (SI : iSpec) : Prop :=
  match SI with existT n Si =>
    iok a /\ exists si, nspec n si = Si /\ lookup si l a
  end.

Definition trans (a : Assertion) :=
  fun S Si => a S (existT _ _ (pair icptr Si)).

Definition Trans (Si : CdHpSpec) l :=
  match Si l with
    None   => None _
  | Some a => Some _ (trans a)
  end.

Definition inspec n Si : iSpec := existT _ _ (nspec n (Trans Si)).

Definition WFprogram Si a P :=
  match P with pair C (pair s i) =>
    exists Six, subseteq Si Six /\ WFcode Six C Six /\ WFiseq Six a i /\
    forall Si' n, subseteq Six Si' -> a s (existT _ _ (pair icptr (inspec n Si')))
  end.

Lemma ok_iok : forall a, ok a -> iok (trans a).
unfold ok, iok, trans; intros; generalize H0; apply H; intros f a0.
apply cptr_mono_psv; simpl; intros; destruct H1; split; auto; destruct H2.
exists x; destruct H2; split; auto; inversion H2; auto.
Qed.

Lemma CodePointerLookup :
  forall Si l a, icptr l a (inspec (S O) Si) -> exists a, lookup Si l a.
intros; destruct H; unfold lookup, Trans in *; destruct H0; destruct H0.
  inversion H0; clear H0; generalize (f_equal (fun f => f l) H3); clear H3.
  intro; rewrite H1 in H0; destruct (Si l).
 eauto.
 inversion H0.
Qed.

Lemma LookupCodePointer :
  forall Si l a, lookup Si l a -> ok a ->
    forall n, icptr l (trans a) (inspec n Si).
unfold trans; intros; split.
 apply (ok_iok _ H0).
 exists (Trans Si); split; auto.
  unfold lookup in *; unfold Trans; rewrite H; auto.
Qed.

Lemma subseteq_refl : forall Si : CdHpSpec, subseteq Si Si.
  unfold subseteq; intros; destruct (Si a); auto.
Qed.

Lemma Invariant :
  forall Si a P, WFprogram Si a P ->
    exists a', exists P', STEP P P' /\ WFprogram Si a' P'.
intros; destruct P as [c]; destruct p; destruct H; destruct H as [HH]; destruct H.
 destruct H0; induction H0.
  exists a'; destruct (H0 _ _ (H1 Si0 O (subseteq_refl _))); destruct H3.
    exists (c, (x, I)); split.
   apply stp_iseq; auto.
   exists Si0; split; auto.
   split; auto.
   split; auto.
   intros; destruct (H0 _ _ (H1 Si' n H5)); destruct H6.
     rewrite (Determinstic _ _ _ _ H3 H6); auto.
  destruct s; destruct (le_or_gt (r rt) (r rs)).
   exists a''; exists (c, (h, r, I)); split.
    apply stp_bgt'; auto.
    exists Si0; split; auto.
    split; auto; split; auto.
   exists a'; destruct (H _ _ H3).
    destruct H7; destruct H7; exists (c, (h, r, x)); split.
    apply stp_bgt; auto.
    exists Si0; split; auto.
    split; auto; split; auto.
  destruct s; destruct (le_or_gt w (r rs)).
   exists a''; exists (c, (h, r, I)); split.
    apply stp_bgti'; auto.
    exists Si0; split; auto.
    split; auto; split; auto.
   exists a'; destruct (H _ _ H3).
    destruct H7; destruct H7; exists (c, (h, r, x)); split.
    apply stp_bgti; auto.
    exists Si0; split; auto.
    split; auto; split; auto.
  exists a'; destruct (H _ _ H0); destruct H4; destruct H4.
   exists (c, (s, x)); split.
   apply stp_jd; auto.
   exists Si0; split; auto.
  destruct s; destruct (H0 _ _ (H1 Si0 (S O) (subseteq_refl _))); destruct H2.
    destruct (CodePointerLookup _ _ _ H2); clear x H2 H3; exists x0.
    destruct (H _ _ H4); destruct H3; destruct H3.
      exists (c, (h, r0, x)); split.
     apply stp_jmp; auto.
     exists Si0; split; auto.
     split; auto; split; auto.
     intros; destruct (H0 _ _ (H1 Si' (S n) H6)); destruct H7; destruct H7.
       destruct H9; destruct H9; unfold lookup in *; unfold Trans in H9.
       generalize (f_equal (fun f => match f with pair _ si => si (r0 r) end) H9).
       clear H9; intro; simpl in H9.
         rewrite (lookup_subseteq_lookup H6 H4) in H9.
         rewrite H10 in H9; inversion H9.
       unfold trans, inspec in *.
       generalize (H7 _ _ _ H8).
       generalize (f_equal (fun f => f (h, r0) (nspec n (Trans Si'))) H12).
       intros; rewrite H11 in H13; auto.
  apply IHWFiseq; auto; intros; apply H3; split; auto; unfold cptr.
   apply LookupCodePointer with (a:=a'').
   apply (lookup_subseteq_lookup H4 H2).
   destruct (H _ _ H2); auto.
  exists a''; destruct (FiniteCodeHeap c); destruct H3; destruct s.
   clear IHWFiseq1; clear IHWFiseq2.
   exists (uC c (S x) I, ((h, uR r rt (S x)), I')); split.
    apply stp_loadcode; auto.
    cut (subseteq Si0 (fun f => if EqNat.beq_nat (S x) f then Some _ a' else Si0 f)).
     intros; exists (fun f => if EqNat.beq_nat (S x) f then Some _ a' else Si0 f).
     split; auto.
      unfold subseteq in *; intros; generalize (HH a0) (H5 a0).
        destruct (Si0 a0); destruct (Si a0); auto; inversion 1; auto.
      split.
       unfold WFcode; intros; unfold lookup in H6; unfold uC, Map.lookup.
         destruct (EqNat.beq_nat (S x) f).
        inversion H6; rewrite <- H8; split; auto; exists I; split; auto.
          apply InstrSeqWeakening with Si0 a'; auto.
        destruct (H _ _ H6); split; auto.
          destruct H8; exists x0; destruct H8; split; auto.
          apply InstrSeqWeakening with Si0 a0; auto.
       split.
        apply InstrSeqWeakening with Si0 a''; auto.
        intros; apply H2; split.
         unfold cptr; generalize LookupCodePointer; unfold trans; intro.
           apply H7; auto; eapply lookup_subseteq_lookup.
          apply H6.
          unfold lookup, uR; rewrite beq_req_true; auto.
            rewrite <- beq_eq_true; auto.
         exists (r rt); cut (uR (uR r rt (S x)) rt (r rt) = r); intros.
          rewrite H7; apply H1.
            unfold subseteq in *; intros; generalize (H5 a0) (H6 a0).
            destruct (EqNat.beq_nat (S x) a0).
           destruct (Si0 a0); inversion 1; auto.
           destruct (Si a0); auto; inversion 1; auto.
          unfold uR; apply ext_eq; intro; compare rt x0; intros.
           rewrite beq_req_true; auto; rewrite e; auto.
           rewrite beq_rneq_false; auto.
    unfold subseteq; intro; compare a0 (S x); intro.
     rewrite e; cut (Si0 (S x) = None _); intros.
      rewrite H5; auto.
      generalize (fun a => H (S x) a); unfold lookup.
        destruct (Si0 (S x)); auto.
        intros; destruct (H5 a1); auto; destruct H7; destruct H7.
        cut (S x > x); auto; intro; generalize (H4 _ H9); unfold Map.lookup in H7.
        unfold Map.not_in_dom; rewrite H7; inversion 1.
     destruct (Si0 a0); auto; rewrite <- beq_neq_false; auto.
Qed.

Lemma WFprogWFprogram : forall Si a P, WFprog Si a P -> WFprogram Si a P.
intros; destruct P; destruct p; destruct H; exists x; destruct H as [HH]; split; auto.
destruct H; split; auto; destruct H0; split; auto.
intros; apply H1; intros; simpl; split.
 destruct (H _ _ H3); unfold ok, iok in *.
   intro; intro; intro; apply H4; firstorder.
   exists x0; split; auto. simpl in H7; inversion H7; auto.
  exists (Trans Si'); split; auto.
   unfold lookup, Trans in *; rewrite (lookup_subseteq_lookup H2 H3); auto.
Qed.

Definition PCorrect (Si : CdHpSpec) (P : Program) :=
  match P with (_, (s, i)) => match i with
  | jd f          => forall a, lookup Si f a -> exists x, a s x
  | jmp r         => forall a, lookup Si (_R s r) a -> exists x, a s x
  | bgt rs rt f _ => _R s rs > _R s rt ->
                       forall a, lookup Si f a -> exists x, a s x
  | bgti rs i f _ => _R s rs > i ->
                       forall a, lookup Si f a -> exists x, a s x
  | _ => True
  end end.

Lemma WFprogramPartialCorrectness :
  forall Si a P, WFprogram Si a P -> PCorrect Si P.
intros; destruct P as [C]; destruct p as [S I]; destruct H; destruct H as [HH].
   destruct H; destruct H0; induction H0; simpl; auto; intros.
 generalize (HH f); unfold lookup in *; rewrite H3; rewrite H6; inversion 1.
   exists ((existT _ iSpec (icptr, inspec O Si0)):X); auto.
   apply H2; split; auto; apply H1; apply subseteq_refl.
 generalize (HH f); unfold lookup in *; rewrite H3; rewrite H6; inversion 1.
   exists ((existT _ iSpec (icptr, inspec O Si0)):X); auto.
   apply H2; split; auto; apply H1; apply subseteq_refl.
 generalize (HH f); unfold lookup in *; rewrite H0; rewrite H3; inversion 1.
   exists ((existT _ iSpec (icptr, inspec O Si0)):X); auto.
   apply H2; apply H1; apply subseteq_refl.
 destruct (H0 _ _ (H1 _ 1 (subseteq_refl Si0))); destruct H3.
    destruct (CodePointerLookup _ _ _ H3).
   generalize (HH (_R S r)); unfold lookup in *; rewrite H2; rewrite H5; inversion 1.
   exists ((existT _ iSpec (icptr, inspec O Si0)):X); auto.
   destruct H3; generalize (H3 _ _ _ H4); destruct H7; destruct H7; inversion H7.
   destruct S; unfold lookup in *; generalize (f_equal (fun f => f (r0 r)) H11).
   rewrite H9; unfold Trans at 1; unfold trans.
    generalize (HH (r0 r)); unfold lookup in *; rewrite H2; intro.
    rewrite <- H10; inversion 1.
   rewrite (f_equal (fun f => f (h, r0) (nspec 0 (Trans Si0))) H14); auto.
   rewrite H8; auto.
 apply IHWFiseq; auto; intros; apply H3; split; auto; unfold cptr.
   apply LookupCodePointer with (a:=a''); auto.
     apply lookup_subseteq_lookup with Si0; auto.
   destruct (H _ _ H2); auto.
Qed.

Fixpoint STEPn n P P' : Prop :=
  match n with
  | O   => P = P'
  | S n => exists P'', STEP P P'' /\ STEPn n P'' P'
  end.

Theorem PartialCorrectness :
  forall Si a P, WFprog Si a P ->
    forall n, exists P', STEPn n P P' /\ PCorrect Si P'.
intros; generalize Si a P (WFprogWFprogram _ _ _ H); clear Si a P H.
  induction n; simpl; intros.
 exists P; split; auto; apply WFprogramPartialCorrectness with a; auto.
 destruct (Invariant _ _ _ H); destruct H0; destruct H0; destruct (IHn _ _ _ H1).
   destruct H2; eauto.
Qed.

Theorem Soundness :
  forall Si a P, WFprog Si a P -> forall n, exists P', STEPn n P P'.
intros; destruct (PartialCorrectness _ _ _ H n); exists x; tauto.
Qed.