(*
 * Coq code for "An Abstract, Approximation-Based Approach to
 *                               Embedded Code Pointers and Partial-Correctness"
 *
 * This file contains code adapted from
 *                  "Certified Assembly Programming with Embedded Code Pointers"
 *
 * Target Machine (TM)
 *
 * (for Coq version 8)
 *)

Require Import EqNat.
Require Import List.
Require Import Map.
Require Import natutil.

(********************************* Syntax *************************************)
Definition Word := nat.

Definition Label := Word.

Inductive Register : Set :=
  | r0 : _
  | r1 : _
  | r2 : _
  | r3 : _
  | r4 : _
  | r5 : _
  | r6 : _
  | r7 : _
  | r8 : _
  | r9 : _
  | r10 : _
  | r11 : _
  | r12 : _
  | r13 : _
  | r14 : _
  | r15 : _
  | r16 : _
  | r17 : _
  | r18 : _
  | r19 : _
  | r20 : _
  | r21 : _
  | r22 : _
  | r23 : _
  | r24 : _
  | r25 : _
  | r26 : _
  | r27 : _
  | r28 : _
  | r29 : _
  | r30 : _
  | r31 : _.

Inductive Command : Set :=
  | add : Register -> Register -> Register -> Command
  | addi : Register -> Register -> Word -> Command
  | mov : Register -> Register -> Command
  | movi : Register -> Word -> Command
  | ld : Register -> Register -> Word -> Command
  | st : Register -> Word -> Register -> Command
  | alloc : Register -> Word -> Command
  | free : Register -> Word -> Command.

Inductive InstrSeq : Set :=
  | iseq : Command -> InstrSeq -> InstrSeq
  | bgt : Register -> Register -> Word -> InstrSeq -> InstrSeq
  | bgti : Register -> Word -> Word -> InstrSeq -> InstrSeq
  | jd : Word -> InstrSeq
  | jmp : Register -> InstrSeq.

Definition CodeHeap := Map Label InstrSeq.

Definition Heap := Map Label Word.

Definition RegFile := Register -> Word.

Definition State := (Heap * RegFile)%type.

Definition Program := (CodeHeap * (State * InstrSeq))%type.

Notation _H := (fun S : State => let (H, _) := S in H) (only parsing).

Notation _R := (fun S : State => let (_, R) := S in R) (only parsing).

Notation _C := (fun P : Program => let (C, p) := P in let (_, _) := p in C)
  (only parsing).

Notation _S := (fun P : Program => let (_, p) := P in let (s, _) := p in s)
  (only parsing).

Notation _I := (fun P : Program => let (_, p) := P in let (_, i) := p in i)
  (only parsing).

Definition beq_reg (r r' : Register) :=
  match r, r' with
  | r0, r0 => true
  | r1, r1 => true
  | r2, r2 => true
  | r3, r3 => true
  | r4, r4 => true
  | r5, r5 => true
  | r6, r6 => true
  | r7, r7 => true
  | r8, r8 => true
  | r9, r9 => true
  | r10, r10 => true
  | r11, r11 => true
  | r12, r12 => true
  | r13, r13 => true
  | r14, r14 => true
  | r15, r15 => true
  | r16, r16 => true
  | r17, r17 => true
  | r18, r18 => true
  | r19, r19 => true
  | r20, r20 => true
  | r21, r21 => true
  | r22, r22 => true
  | r23, r23 => true
  | r24, r24 => true
  | r25, r25 => true
  | r26, r26 => true
  | r27, r27 => true
  | r28, r28 => true
  | r29, r29 => true
  | r30, r30 => true
  | r31, r31 => true
  | _, _ => false
  end.

(**************************** Dynamic Semantics *******************************)
Definition uR (R : RegFile) r n r' := if beq_reg r r' then n else R r'.

Definition uH (H : Heap) l v l' := if beq_nat l l' then Some v else H l'.

Fixpoint uHn (H : Heap) (l : Label) (n : nat) {struct n} : Heap :=
  match n with
  | O => H
  | S n => uH (uHn H (S l) n) l l
  end.

Definition fH (H : Heap) l l' := if beq_nat l l' then None else H l'.

Fixpoint fHn (H : Heap) (l : Label) (n : nat) {struct n} : Heap :=
  match n with
  | O => H
  | S n => fH (fHn H (S l) n) l
  end.

Inductive Next : Command -> State -> State -> Prop :=
  | stp_add :
      forall rd rs rt H R,
      Next (add rd rs rt) (H, R) (H, uR R rd (R rs + R rt))
  | stp_addi :
      forall rd rs w H R, Next (addi rd rs w) (H, R) (H, uR R rd (R rs + w))
  | stp_mov : forall rd rs H R, Next (mov rd rs) (H, R) (H, uR R rd (R rs))
  | stp_movi : forall rd w H R, Next (movi rd w) (H, R) (H, uR R rd w)
  | stp_ld :
      forall rd rs w H R w',
      lookup H (R rs + w) w' -> Next (ld rd rs w) (H, R) (H, uR R rd w')
  | stp_st :
      forall rd w rs H R w',
      lookup H (R rd + w) w' ->
      Next (st rd w rs) (H, R) (uH H (R rd + w) (R rs), R)
  | stp_alloc :
      forall rd H R l n,
      in_dom H l ->
      (forall i, i > l -> not_in_dom H i) ->
      Next (alloc rd n) (H, R) (uHn H (l+1) n, uR R rd (l+1))
  | stp_free :
      forall rd H R n,
      (forall i, i < n -> in_dom H (R rd + i)) ->
      Next (free rd n) (H, R) (fHn H (R rd) n, R).

Inductive STEP : Program -> Program -> Prop :=
  | stp_iseq :
      forall C S S' c I, Next c S S' -> STEP (C, (S, iseq c I)) (C, (S', I))
  | stp_bgt :
      forall C S rs rt l I I',
      (fun S : State => let (_, R) := S in R) S rs >
      (fun S : State => let (_, R) := S in R) S rt ->
      lookup C l I' -> STEP (C, (S, bgt rs rt l I)) (C, (S, I'))
  | stp_bgt' :
      forall C S rs rt l I,
      (fun S : State => let (_, R) := S in R) S rs <=
      (fun S : State => let (_, R) := S in R) S rt ->
      STEP (C, (S, bgt rs rt l I)) (C, (S, I))
  | stp_bgti :
      forall C S rs w l I I',
      (fun S : State => let (_, R) := S in R) S rs > w ->
      lookup C l I' -> STEP (C, (S, bgti rs w l I)) (C, (S, I'))
  | stp_bgti' :
      forall C S rs w l I,
      (fun S : State => let (_, R) := S in R) S rs <= w ->
      STEP (C, (S, bgti rs w l I)) (C, (S, I))
  | stp_jd : forall C S l I, lookup C l I -> STEP (C, (S, jd l)) (C, (S, I))
  | stp_jmp :
      forall C S r I,
      lookup C ((fun S : State => let (_, R) := S in R) S r) I ->
      STEP (C, (S, jmp r)) (C, (S, I)).

Notation "P |--> Q" := (STEP P Q) (at level 150).

Lemma beq_req_true : forall r r', r = r' -> beq_reg r r' = true.
intros.
rewrite H.
destruct r'.
 auto. auto. auto. auto. auto. auto. auto. auto.
 auto. auto. auto. auto. auto. auto. auto. auto.
 auto. auto. auto. auto. auto. auto. auto. auto.
 auto. auto. auto. auto. auto. auto. auto. auto.
Qed.

Lemma beq_rneq_false : forall r r', r <> r' -> beq_reg r r' = false.
intros.
destruct r.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
 destruct r'. tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
                 tauto. tauto. tauto. tauto. tauto. tauto. tauto. tauto.
Qed.

Lemma Determinstic :
 forall S S' S'' c, Next c S S' -> Next c S S'' -> S' = S''.
intros.
destruct H.
 inversion H0.
   auto.
 inversion H0.
   auto.
 inversion H0.
   auto.
 inversion H0.
   auto.
 inversion H0.
   unfold lookup in H1, H9.
   rewrite H1 in H9.
   inversion H9.
   auto.
 inversion H0.
   auto.
 inversion H0.
   generalize (le_or_gt l l0).
   intros.
   destruct H11.
  unfold gt in H2, H10.
    unfold lt in H2, H10.
    destruct H11.
   auto.
   generalize (H10 (S m) (Le.le_n_S l0 m H11)).
     destruct H1.
     unfold not_in_dom in |- *.
     rewrite H1.
     intros.
     inversion H12.
  generalize (H2 l0 H11).
    destruct H9.
    unfold not_in_dom in |- *.
    rewrite H9.
    intros.
    inversion H12.
 inversion H0.
   auto.
Qed.