/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Giovanni Busonera													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	Debug_dp			 												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	Debug extension datapath.														*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	eBug Extension, HW WP									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module Debug_dp(

	// Input Ports
		//eMips System Signals
	input clk,							// System Clock
	input pclk,							// Pipeline synchronization signal
	input rst,							// System Reset
	
		// UART 
	input [7:0] Byte_Data,			// Data from UART
	
		// Control signals from Ext_Debug_control
	input sel_tisa_pc,
		
		// Debug Module Control signals
	input ld_inreg,					// Control Signal: load data on the Input Register
	input fw_shift,					// Control Signal: load bytes on the fullword register
	input fw_clr,						// Control Signal: clear fullword register
	input reg_we,						// Control Signal: eMips registers write enable 
	input num_shift,					// Control Signal: eMips num_byte register shift 
	input sel_addr,					// Control Signal: eMips mux_addr selector. Select between option field and num_byte reg
	input addr_init,					// Control Signal: eMips addr_counter init signal 
	input addr_count,					//	Control Signal: eMips add_counter count signal 
	input ld_mem_addr,				//	Control Signal: load signal for mem_addr register
	input [1:0] sel_out,				// Control Signal: select data that has been requested
	input [1:0] sel_byte,			// Control Signal: select full word bytes
	
		// Ext0 Buses - Signals
	input WE_IN,						// Peripheral Write enable Monitor
	input [31:0] from_Regs,			// Databus from eMIPS registers or GPR file
	input [31:0] from_PC,			// Databus from PC - 
	input [31:0] from_Mem,			// Databus from Memory subsystem
	input [31:0] MADDR_IN,			// Peripheral Address Bus Monitor
	
		// Input from ext_debug_control
	input [7:0] nAck,
	
		// Inputs from Debug_Control::wbpoints_fsm
	input ld_CR, 						
	input sh_wp_reg, 					
	input ld_wp_type, 
	input	set_wp_en, 
	input	rst_wp_en, 
	input	sel_wp, 
	input	sel_bp,
	
	// Output Ports
		//	RS232
	output [7:0] TxD_Data,			// Data to UART
		
		// Debug Module
	output fspecial,
	output RnW,
	output Regs_we,
	output end_mem,	
	output sel_m_byte,
	output [1:0] pclkedge_out,
	
	output [2:0] opcode,
	output [4:0] nReg,
	output [4:0] options,

		//	Ext0 Buses
	output [31:0] to_Regs,			// Databus to GPR file
	output [31:0] to_PC,
	output [31:0] to_Mem_Addr,    // To TISA Memory Subsystem
	output [31:0] to_Mem_Data,
	
		// Watchpoints Breakpoints output port
	output is_wp, 						// To Debug_Control::wbpoints_fsm
	output wp_en, 						// To Debug_Control::wbpoints_fsm
	output wp_hit, 					// To ext_debug_control::To ext_debug_control_fsm
	output bp_hit						// To ext_debug_control::To ext_debug_control_fsm
	
	);


	// Regs
	reg [1:0]  pclkedge;
	reg [7:0]  inReg;
	reg [7:0]  mux_out;
	reg [7:0]  mux_byte;
	reg [7:0]  mux_mem_byte;
	reg [15:0] mux_count;
	reg [31:0] fwReg;
	reg [31:0] PC_break;
	reg [31:0] mux_reg;
	reg [31:0] mux_pc;
	reg [31:0] mux_addr;
	
	
	reg[15:0] num_byte;
	reg[31:0] mem_addr;
	
	reg [15:0] addr_counter_value;
	
	// Nets
	wire sel_reg;
	wire PC_we;
	
	wire [1:0] sel_mem_byte;
	wire [2:0] fspecial_num;
//	wire [4:0] options;
	wire [7:0] Ack;
	wire [7:0] in_fwReg;
	wire [7:0] in_num_byte;
	wire [31:0] addr_inc;
	
// Datapath Description
	
// Signals 
		
		assign RnW				= ~inReg[0];
		assign fspecial		= inReg[7];
		assign opcode			= inReg[2:0];
		assign options			= inReg[7:3];
		assign nReg 	 		= inReg[6:2];
		assign fspecial_num 	= inReg[4:2];
		assign in_fwReg		= Byte_Data;
		assign in_num_byte	= Byte_Data;
		assign TxD_Data 		= mux_out; 
		assign to_Regs			= fwReg;
		assign to_PC			= PC_break;
		assign to_Mem_Data	= {24'b0, Byte_Data};
		assign pclkedge_out	= pclkedge;
		assign to_Mem_Addr   = mem_addr;
		assign sel_mem_byte  = mem_addr[1:0]-1'b1;
		
		
// Ack definition
		assign Ack = 8'hFF;
	//	assign nAck = 8'h00;
		
		
		
// PCLK Edge detector
		 always @(posedge clk)
			if (rst)
				pclkedge <= 2'b00;
			else
				pclkedge <= {pclkedge[0], pclk};
	
	
	
// Registers
			// inReg
		always @(posedge clk)
			if (ld_inreg)
				inReg <= Byte_Data;
				
			// fwReg
		always @(posedge clk)
			if (fw_clr)	
				fwReg <= 0;
			else if (fw_shift)
					fwReg	<= {fwReg[23:0], in_fwReg};  // Left shift: Most significant byte loaded first			
	
			// PC_break register
		 always @(posedge clk)
			if (((pclkedge == 2'b01)& sel_tisa_pc) || PC_we) 
					PC_break 	<= mux_pc;
	
			// num_byte
		 always @(posedge clk)
			if (num_shift)
				num_byte <= {num_byte[7:0],in_num_byte};
						
			// mem_addr
		 always @(posedge clk)
			if (ld_mem_addr)
				mem_addr <= mux_addr;
			
			
			
			
// Muxes
			//mux2_32 - sel_reg
		always @(sel_reg, from_Regs, PC_break)
		case (sel_reg)
			1'b0	:  mux_reg = from_Regs;
			1'b1	:  mux_reg = PC_break;
		endcase
		
			// Mux PC
      always @(sel_tisa_pc, fwReg, from_PC)		
		case (sel_tisa_pc)
			1'b0		:  mux_pc = fwReg;
		   1'b1		:  mux_pc = from_PC;
			default 	: mux_pc = fwReg;
		endcase
			
			//mux4_8 - sel_byte
		always @(sel_byte, mux_reg)
		case (sel_byte)
			2'b00		:	mux_byte = mux_reg[31:24];
			2'b01 	:	mux_byte = mux_reg[23:16];
			2'b10		:  mux_byte = mux_reg[15:8];
			2'b11		:  mux_byte = mux_reg[7:0];
			default 	: 	mux_byte = mux_reg[31:24];
		endcase 
		
			//mux4_8 - sel_mem_byte
		always @(sel_mem_byte, from_Mem)
		case (sel_mem_byte)
			2'b00		:	mux_mem_byte = from_Mem[31:24];  
			2'b01 	:	mux_mem_byte = from_Mem[23:16];
			2'b10		:  mux_mem_byte = from_Mem[15:8];
			2'b11		:  mux_mem_byte = from_Mem[7:0];
			default 	: 	mux_mem_byte = from_Mem[31:24];

			// Endianness issues !!
//			2'b00		:	mux_mem_byte = from_Mem[7:0];
//			2'b01 	:	mux_mem_byte = from_Mem[15:8];
//			2'b10		:  mux_mem_byte = from_Mem[23:16];
//			2'b11		:  mux_mem_byte = from_Mem[31:24];
//			default 	: 	mux_mem_byte = from_Mem[7:0];
		endcase 		
		
			//mux4_8 - sel_out
		always @(sel_out, mux_byte, mux_mem_byte, Ack, nAck)
		case (sel_out)
			2'b00		: 	mux_out = mux_byte;
			2'b01 	: 	mux_out = mux_mem_byte;
			2'b10		: 	mux_out = Ack;
			2'b11		: 	mux_out = nAck;
			default 	:	mux_out = mux_byte;
		endcase 
			
			//mux2_32 - mux_addr
		always @(sel_addr, addr_inc, fwReg)
		case (sel_addr)
			1'b0		:	mux_addr = fwReg;
			1'b1		:	mux_addr = addr_inc;
			default 	:	mux_addr = fwReg;
		endcase
		
			// mux2_16 - mux_count
		always @(sel_m_byte, num_byte, options)
		case (sel_m_byte)
			1'b0		:	mux_count = {11'b0,options};
			1'b1		:  mux_count = num_byte;
			default	: 	mux_count = {11'b0,options};
		endcase
			
	
// Adder
		assign addr_inc = mem_addr + 1;
			
			
			
// Decoders
			// sel_reg decoder
		decoder_sel_reg DECODER_SEL_REGS (
			 .fspecial(fspecial), 
			 .reg_we(reg_we), 
			 .fspecial_num(fspecial_num), 
			 .Regs_we(Regs_we), 
			 .PC_we(PC_we), 
			 .sel_reg(sel_reg)
			 );
			 
			// sel_m_byte decoder
		decoder_sel_m_byte DECODER_SEL_M_BYTE (
			.options(options),
			.sel_m_byte(sel_m_byte)
			);
	
	
	
// Addr_Counter
		always @(posedge clk)
			if (addr_init)
				addr_counter_value <= mux_count;
			else if (addr_count)
						addr_counter_value <= addr_counter_value - 1;
		
		assign end_mem = (addr_counter_value == 0);
		
		
// Hardware Watchpoint Breakpoint Support
		wbpoints_dp HW_WBPOINTS (
			 .clk(clk), 
			 .rst(rst), 
			 
			 .ld_CR(ld_CR), 
			 .sh_wp_reg(sh_wp_reg), 
			 .ld_wp_type(ld_wp_type), 
			 .set_wp_en(set_wp_en), 
			 .rst_wp_en(rst_wp_en), 
			 .sel_wp(sel_wp), 
			 .sel_bp(sel_bp), 
			 .WE_IN(WE_IN), 
			 .Byte_Data(Byte_Data), 
			 .MADDR_IN(MADDR_IN), 
			 .PC(from_PC), 
			 
			 .is_wp(is_wp), 
			 .wp_en(wp_en), 
			 .wp_hit(wp_hit), 
			 .bp_hit(bp_hit)
			 );

				
	endmodule
