/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Giovanni Busonera													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	ext_debug_control												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	This module manages the pipeline arbiter handshake if a break		*/
/*	instruction is in ID stage or if emips2gdb tries to connect to 	*/
/*	the extesion.																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	eBug Extension, HW WP									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module ext_debug_control(
	
// Input ports

	// Input from TISA
	input clk, 
	input rst, 

	// Input from Extension interface
	input EN,
	input GR,	
	input [31:0] INSTR, 

	// Input from debug module 
	input suspend,
	input [1:0] pclkedge,
	
	// Input from debug_module (HW Watchpoint, Breakpoint support located in Debug_dp)
	input wp_hit,
	input bp_hit,	
	
// Output ports
	
	// Ouputs to debug module 
	output break,			// Active high signal to the main fsm of the debug module
								// It notifies that a break is taken or the suspend signal is 
								// acknoledge and TISA is stalled. A debugging section can start
	output sel_tisa_pc,
	output suspend_ACK,
	output [7:0] nAck,
	
	// Output to Extension Interface
	output RI, 
	output ACK,
	output PCNEXT
);
	
		
wire is_branch;
wire is_break;


// FSM 

ext_debug_control_fsm EXT_DEBUG_CONTROL_FSM (
    .clk(clk), 
    .rst(rst), 
    .is_branch(is_branch), 
    .is_break(is_break),
	 .EN(EN), 
    .GR(GR),  
    .suspend(suspend), 
    .pclkedge(pclkedge),
	 .wp_hit(wp_hit),
	 .bp_hit(bp_hit),	
	 
    .break(break), 
    .sel_tisa_pc(sel_tisa_pc), 
    .suspend_ACK(suspend_ACK), 
    .RI(RI), 
    .ACK(ACK), 
    .PCNEXT(PCNEXT)
    );

// Branch Decoder

branch_break_detector BRANCH_BREAK_DETECTOR (
    .instr(INSTR), 
    .is_branch(is_branch), 
    .is_break(is_break)
    );

// nAck generator 
nAck_generator nACK_GEN (
    .break_code(INSTR[25:6]), 
    .is_break(is_break), 
    .nAck(nAck)
    );

endmodule
