/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Giovanni Busonera													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	ext_debug_control_fsm										*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	TISA pipeline arbiter handshake implementation.							*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	eBug Extension, HW WP									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module ext_debug_control_fsm(	

// Input ports

	// Input from TISA
	input clk, 
	input rst, 

	// Input from Branch decoder
	input is_branch,
	input is_break,
	
	// Input from Extension interface
	input EN,
	input GR,	 

	// Input from debug module 
	input suspend,
	input [1:0] pclkedge,

	// Input from debug_module (HW Watchpoint, Breakpoint support located in Debug_dp)
	input wp_hit,
	input bp_hit,	
	
// Output ports
	
	// Ouputs to debug module 
	output break,			// Active high signal to the main fsm of the debug module
								// It notifies that a break is taken or the suspend signal is 
								// acknoledge and TISA is stalled. A debugging section can start
	output sel_tisa_pc,
	output suspend_ACK,
	
	// Output to Extension Interface
	output RI, 
	output ACK,
	output PCNEXT
);
	
	parameter IDLE=0, RI_ASSERT=1, ACK_HIGH=2, STALLED=3, SEND_BREAK =4,
					WAIT_PCLK_NEGEDGE=5, SET_PC=6, WAIT_PCLK=7;
	
	reg [2:0] cs, ns; 	// Currentstate, Nextstate
	reg [5:0] moore_out; // Moore output vector
	
// break_recognized goes high if break instruction is recognized
	assign pclk_posedge = pclkedge ==2'b01; 
	assign pclk_negedge = pclkedge ==2'b10;
	
	assign {break, sel_tisa_pc, suspend_ACK, RI, ACK, PCNEXT} = moore_out;


// Current State Refresh
	always @(posedge clk)
		if (rst)
			cs <= IDLE;
		else
			cs <= ns;
			
			
// Next State Logic
	always @(cs, is_break, is_branch, pclk_posedge, pclk_negedge, suspend, EN, GR, wp_hit, bp_hit)
		case (cs)	
			IDLE 					:	if ((is_break) || wp_hit || bp_hit || (suspend & pclk_posedge & ~is_branch))	
											ns = RI_ASSERT;
										else
											ns = IDLE;
										
			RI_ASSERT			:	if (pclk_posedge && EN)
											ns = ACK_HIGH;
										else if (pclk_posedge && ~EN)
													ns = IDLE;
											  else
													ns = RI_ASSERT;
										
			ACK_HIGH				:	if (GR && suspend)
												ns = STALLED;
										else if (GR && ~suspend)
													ns = SEND_BREAK;
												else	
													ns = ACK_HIGH;
												
			STALLED				:	if (~suspend)
											ns = WAIT_PCLK_NEGEDGE;
										else
											ns = STALLED;
										
			SEND_BREAK			:	if (suspend)
											ns = STALLED;
										else
											ns = SEND_BREAK;
											
			WAIT_PCLK_NEGEDGE	: if (pclk_negedge)
											ns = SET_PC;
									  else	
											ns = WAIT_PCLK_NEGEDGE;
			
			SET_PC				:	if (pclk_posedge)
											ns = WAIT_PCLK;
										else
											ns = SET_PC;
			
			WAIT_PCLK			:	if (pclk_posedge)
											ns = IDLE;
										else
											ns = WAIT_PCLK;
			
			default 				:	ns = IDLE;
		endcase
		
		
// Output Logic
// 						     moore_out	= {break, sel_tisa_pc, suspend_ACK, RI, ACK, PCNEXT}
	always @(cs)
		case (cs)	
			IDLE 					: moore_out = 6'b0_1_0_1_0_0;	
			RI_ASSERT			: moore_out = 6'b0_1_0_0_0_0;
			ACK_HIGH				: moore_out = 6'b0_0_0_1_1_0;
			STALLED				: moore_out = 6'b0_0_1_1_1_0;
			SEND_BREAK			: moore_out = 6'b1_0_0_1_1_0; 
			WAIT_PCLK_NEGEDGE	: moore_out = 6'b0_0_0_1_1_0;
			SET_PC				: moore_out = 6'b0_1_0_1_1_1;
			WAIT_PCLK			: moore_out = 6'b0_1_0_1_0_0;
			default 				: moore_out = 6'b0_1_0_1_0_0;	
		endcase

endmodule
