/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Giovanni Busonera													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	main_fsm															*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	main fsm coordinate the control of the whole system iterfacing		*/
/*	ext_debug_control module, uart module, registers fsm and 			*/
/*	memory_fsm 																			*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	eBug Extension, HW WP									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

// Opcodes Definition
`define GETREG 	3'bx00
`define SETREG 	3'bx01
`define FETCH 		3'b010
`define STORE 		3'b011
`define INTERRUPT	3'b110
`define EXTENDED	3'b111	// Used for extended instructions

// Options Definition
`define SUSPEND	5'b00000
`define CONTINUE	5'b00001
`define WBPOINT	5'b00001


module main_fsm(

// Input Ports
input clk,
input rst,
input valid,						// Control signal: RS232 incoming Data
input busy, 
input RnW,
input reg_done,
input mem_done,
input suspend_ACK,				// Input from ext_debug_control module
input break,						// Input from ext_debug_control module
input [2:0] opcode,				// Control Signal: opcode field	
input [4:0] options,				// Options field: used to extend opcode field
input wp_done,

// Output Ports

output reg ld_inreg,				// Control Signal: load data on the Input Register
output reg reg_access,			// Control Signal: enables registers_fsm
output reg mem_access,			// Control Signal: enables mem_fsm
output suspend, 					// Output to ext_debug_control module
output TxD_Start_main,			// Control Signal: Start a UART transmission
output [1:0] sel_out,
output reg wp_op
);

parameter IDLE=0, BREAK=1, GET=2, SEND_ACK=3, SEND_nACK=4, WAIT_GDB=5, 
			 DECODE=6, SEND_nACK_2=7, REGACCESS=8, MEMACCESS=9, WAIT_nBUSY=10,
			 WBP_OP=11;

reg [3:0] cs, ns;
reg [3:0] out;

assign {TxD_Start_main, sel_out, suspend} = out; 


// Current State Refresh
always @(posedge clk)
	if (rst)
		cs <= IDLE;
	else
		cs <= ns;
		
// Next State Logic and Mealy outputs
always @(cs, valid, RnW, reg_done, mem_done, opcode, options, break, suspend_ACK, busy, wp_done)
begin
ld_inreg 	= 0;
reg_access 	= 0;
mem_access 	= 0;
wp_op = 0;

case (cs)
	IDLE 			:	if (valid)	
							begin
								ns = GET;
								ld_inreg =1;
							end
						else if (break)
									ns = BREAK;
							  else
									ns = IDLE;	
	
	BREAK			: 	ns = WAIT_GDB;
	
	GET 			:	if ((opcode == `INTERRUPT) && (options == `SUSPEND))
							ns = SEND_ACK;
						else
							ns = SEND_nACK;
	
	SEND_ACK		: 	if ((opcode == `INTERRUPT) && (options == `CONTINUE))
							ns = WAIT_nBUSY;
						else if (suspend_ACK)
									ns = WAIT_GDB;
								else 
									ns = SEND_ACK;
	
	SEND_nACK	:  ns = IDLE;
	
	WAIT_GDB		:	if (valid)	
							begin
								ns = DECODE;
								ld_inreg =1;
							end
						else
							ns = WAIT_GDB;
	
	DECODE		: 	casex (opcode)
							`GETREG 			: 	begin ns = REGACCESS; reg_access = 1; end
							`SETREG 		 	: 	begin ns = REGACCESS; reg_access = 1; end
							`FETCH		   : 	begin ns = MEMACCESS; mem_access = 1; end
							`STORE		   : 	begin ns = MEMACCESS; mem_access = 1; end
							`INTERRUPT  	:  if ((options == `SUSPEND) || (options == `CONTINUE))
														begin ns = SEND_ACK; end
													else 
														begin ns = SEND_nACK_2; end
							`EXTENDED		: 	case (options)
														`WBPOINT		:	begin ns = WBP_OP; wp_op = 1; end
														default 		:  begin ns = SEND_nACK_2; end
													endcase
														
							default  		: 	begin ns = SEND_nACK_2; end
						endcase
						
	SEND_nACK_2	:	ns = WAIT_GDB;
			
	REGACCESS	: 	if (reg_done)
							if (RnW) 
								ns = WAIT_GDB;
							else
								ns = SEND_ACK;
						else
							ns = REGACCESS;
						
	
	MEMACCESS	: 	if (mem_done)
							if (RnW)
								ns = WAIT_GDB;
							else
								ns = SEND_ACK;
						else
							ns = MEMACCESS;
							
	WBP_OP		:  if (wp_done)
							ns = SEND_ACK;
						else
							ns = WBP_OP;
	
	WAIT_nBUSY	:	if (~busy) 
							ns = IDLE;
						else	
							ns = WAIT_nBUSY;
	
	default 		: ns = IDLE;
endcase
end

// Output Logic
always @(cs)
case (cs)
// 				  out = {TxD_Start, sel_out, suspend, continue} 
	IDLE 			: 	out = 5'b0_xx_0;
	BREAK			: 	out = 5'b1_11_0;
	GET 			: 	out = 5'b0_xx_0;
	SEND_ACK		:	out = 5'b1_10_1;
	SEND_nACK	:	out = 5'b1_11_0;
	WAIT_GDB		:	out = 5'b0_xx_1;
	DECODE		:	out = 5'b0_xx_1;
	SEND_nACK_2	:  out = 5'b1_11_1;
	REGACCESS	:	out = 5'b0_00_1;
	MEMACCESS	:	out = 5'b0_01_1;
	WBP_OP 		:	out = 5'b0_11_1;
	WAIT_nBUSY  :  out = 5'b0_10_1;
	default 		: 	out = 5'b0_xx_0;
endcase

endmodule
