/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Giovanni Busonera													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	memory_fsm														*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	Implementation of memory subsystem protocol								*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	eBug Extension, HW WP									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module memory_fsm(

// Input ports 
	input clk,
	input rst,
	
	input mem_access,			// From main_fsm
	
	input valid,				// From UART and Valid Gen
	input busy,			
	
	input RnW,					// From Datapath
	input end_mem,
	input sel_m_byte,
	input [1:0] pclkedge,
	
	input MDATA_VLD,			// From TISA
	
	input end_count,			// From CTRL_COUNTER
	
// Ouptut Ports
	output mem_done,			// To main_fsm
	
	output TxD_Start_mem,	// To UART	
			
	output num_shift,			// To Datapath
	output sel_addr,
	output addr_init,
	output addr_count,
	output fw_shift_mem,
	output ld_mem_addr,
	
	output MOE,					// To TISA
	output mem_we,
	
	output count_mem			// To CTRL_COUNTER
);


	parameter IDLE=0, 			WAIT_HI_NUM=1, 	SET_HI_NUM=2, 			WAIT_LO_NUM=3, 
				 SET_LO_NUM=4, 	ADDR_CNT_INIT=5, 	GET_ADDR_COUNT=6, 	WAIT_VALID=7, 
				 SHIFT_FWREG=8,	SET_MEM_ADDR=9, 	WAIT_PCLK_POS=10, 	WAIT_DATA=11,
				 SET_MOE=12, 		SET_MWE=13, 	ADDR_UPDATE=14, SEND_BYTE=15, 
				 WAIT_nBUSY=16, WAIT_NEXT=17;
	
	reg [4:0] ns, cs;
	reg [10:0] out;
	
	
	assign pclk_posedge = (pclkedge == 2'b01);
	assign pclk_negedge = (pclkedge == 2'b10);
	
	assign {mem_done, TxD_Start_mem, num_shift, sel_addr, addr_init, addr_count,
				fw_shift_mem, MOE, mem_we, count_mem, ld_mem_addr} = out;
	
	
// Current State Refresh
	always @(posedge clk)
		if (rst)	
			cs <= IDLE;
		else
			cs <= ns;

// Next State Logic
	always @(cs, mem_access, valid, busy, RnW, end_mem, sel_m_byte, pclk_posedge, pclk_negedge, MDATA_VLD, end_count)
		case (cs)
			
			IDLE						:	if (mem_access)
												ns = (sel_m_byte) ? WAIT_HI_NUM : ADDR_CNT_INIT;
											else
												ns = IDLE;
			
			WAIT_HI_NUM				:	if (valid)	
												ns = SET_HI_NUM;
											else
												ns = WAIT_HI_NUM;
					
			SET_HI_NUM				:	ns = WAIT_LO_NUM;
			
			WAIT_LO_NUM				:	if (valid)	
												ns = SET_LO_NUM;
											else
												ns = WAIT_LO_NUM;
					
			SET_LO_NUM				:	ns = ADDR_CNT_INIT;
			
			ADDR_CNT_INIT			:	ns = GET_ADDR_COUNT;
			
			GET_ADDR_COUNT			:	ns = WAIT_VALID;
			
			WAIT_VALID				: 	if (valid)
												ns = SHIFT_FWREG;
											else
												ns = WAIT_VALID;
												
			SHIFT_FWREG				:	if (end_count)
												ns = SET_MEM_ADDR;
											else
												ns = GET_ADDR_COUNT;
											
			SET_MEM_ADDR			: 	if (~RnW && ~valid)
												ns = WAIT_DATA;
											else
												ns = WAIT_PCLK_POS;
			
			WAIT_PCLK_POS			:	if (pclk_posedge)
												ns = (RnW) ? SET_MOE : SET_MWE;
											else
												ns = WAIT_PCLK_POS;
			
			SET_MOE					: 	if (pclk_negedge)
												ns = ADDR_UPDATE;
											else
												ns = SET_MOE;

			SET_MWE					: 	if (pclk_negedge)
												ns = ADDR_UPDATE;
											else
												ns = SET_MWE;
															
			ADDR_UPDATE		:	ns = WAIT_NEXT;
			
			WAIT_NEXT    : if (RnW)
												ns = (MDATA_VLD) ? SEND_BYTE : WAIT_NEXT;
											else 
												ns = WAIT_DATA; 
			
			
			SEND_BYTE				: 	ns = WAIT_nBUSY;
			
			WAIT_nBUSY				:	if (busy)
												ns = WAIT_nBUSY;
											else
												ns = (end_mem) ? IDLE : WAIT_PCLK_POS;
												
			WAIT_DATA				:	if (end_mem)
												ns = IDLE;
											else if (valid)
														ns = WAIT_PCLK_POS;
													else
														ns	= WAIT_DATA;
														
			default 					: 	ns = IDLE;
			
		endcase
											
											
// Output Logic
//	{mem_done,TxD_Start_mem,num_shift __ sel_addr,addr_init,addr_count,fw_shift_mem __ MOE, mem_we,count_mem,ld_mem_addr}
	always @(cs)
		case (cs)
			IDLE						:	out = 11'b100_x000_0000;	
			
			WAIT_HI_NUM				:	out = 11'b000_x000_0000;		
			SET_HI_NUM				:	out = 11'b001_x000_0000;
			WAIT_LO_NUM				:	out = 11'b000_x000_0000;
			SET_LO_NUM				:	out = 11'b001_x000_0000;
			
			ADDR_CNT_INIT			:	out = 11'b000_x100_0000;
			GET_ADDR_COUNT			:	out = 11'b000_x000_0010;
			WAIT_VALID				: 	out = 11'b000_x000_0000;
												
			SHIFT_FWREG				:	out = 11'b000_x001_0000;							
			SET_MEM_ADDR			: 	out = 11'b000_0000_0001;
			WAIT_PCLK_POS			:	out = 11'b000_x000_0000;
			
			SET_MOE					: 	out = 11'b000_x000_1000;
			SET_MWE					: 	out = 11'b000_x000_0100;										
			ADDR_UPDATE		   :	out = 11'b000_1010_0001;
			WAIT_NEXT      :   out = 11'b000_x000_0000;
			
			SEND_BYTE				: 	out = 11'b010_x000_0000;
			WAIT_nBUSY				: 	out = 11'b000_x000_0000;
			WAIT_DATA				:	out = 11'b000_x000_0000;
														
			default 					: 	out = 11'b100_0000_0000;
		endcase

endmodule
