/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Giovanni Busonera													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	registers_fsm													*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	eBug Extension, HW WP									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module registers_fsm(

// Input Ports
input clk,
input rst,

input reg_access,					// Control Signal: is used by main_fsm to start a register IO operation
input busy,
input regrdy,
input fspecial,					// Control Signal: fspecial field
input valid,
input RnW,							// Control Signal: RnW field
input end_count,
input data_ready,
	
// Output Ports
			
output fw_shift,					// Control Signal: load bytes on the fullword register
output fw_clr,						// Control Signal: clear fullword register
output TxD_Start_reg,			// Control Signal: Start a UART transmission
output reg_we,
output reg_done,
output count_reg,
output init_latency,
output count_latency
);

parameter IDLE=0, WAIT_FOR=1, GET=2, BUSY=3, COUNT=4, VALID=5, SHIFT=6, CHECK_REGRDY=7, SET=8, DONE=9;

reg [3:0] cs, ns;
reg [7:0] moore_out;

assign {TxD_Start_reg, reg_done, fw_clr, fw_shift, reg_we, count_reg, init_latency, count_latency} = moore_out; 

// Current State Refresh
always @(posedge clk)
	if (rst)
		cs <= IDLE;
	else
		cs <= ns;
		
// Next State Logic
always @(cs, reg_access, RnW, busy, valid, end_count, data_ready, fspecial, regrdy)
case (cs)
	IDLE 				: if (reg_access)
								ns = (RnW) ? WAIT_FOR : COUNT;
							else
								ns = IDLE;
	WAIT_FOR			: if (data_ready)	
								ns = GET;
							else
								ns = WAIT_FOR;
					
	GET				: ns = BUSY;
	
	BUSY 				: if (end_count)
								ns = DONE;
							else if (~busy)
										ns = GET;
									else
										ns = BUSY;
	COUNT				: ns = VALID;
	
	VALID				: if (valid)	
								ns = SHIFT;
							else
								ns = VALID;
	
	SHIFT				: if (end_count && fspecial)
								ns = SET;
							else if (end_count && ~fspecial)
										ns = CHECK_REGRDY;
									else 
										ns = COUNT;
								
	CHECK_REGRDY	: if (regrdy)
								ns = SET;
							else
								ns = CHECK_REGRDY;
	
	SET				:	ns = DONE;
	
	DONE				: ns = IDLE;
				
	default 			: ns = IDLE;
	
endcase

// Moore Output Logic
always @(cs)
case (cs)
// moore_out = {TxD_Start_reg, reg_done, fw_clr, fw_shift, reg_we, count_reg, init_latency, count_latency} 
	IDLE 				: moore_out = 8'b0_0_0_0_0_0_1_0;
	WAIT_FOR			: moore_out = 8'b0_0_0_0_0_0_0_1;
	GET				: moore_out = 8'b1_0_0_0_0_1_0_0;
	BUSY				: moore_out = 8'b0_0_0_0_0_0_0_0;
	COUNT				: moore_out = 8'b0_0_0_0_0_1_0_0;
	VALID				: moore_out = 8'b0_0_0_0_0_0_0_0;
	SHIFT				: moore_out = 8'b0_0_0_1_0_0_0_0;
	CHECK_REGRDY	: moore_out = 8'b0_0_0_0_0_0_0_0;
	SET				: moore_out = 8'b0_0_0_0_1_0_0_0;
	DONE				: moore_out = 8'b0_1_0_0_0_0_0_0;
	default 			: moore_out = 8'b0_0_1_0_0_0_1_0;
endcase

endmodule
