/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	ext_op7c_instr_v2												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"
`define OP7C	6'b011111

/* This extension performs the following extended instruction:

	7fbf0018   op7c  ra,sp,18
	
	which corresponds to these standard MIPS instructions
	
	8fbf0010   lw    ra,10(sp)
   3e00008    jr    ra
   27bd0018   ddiu sp,sp,18

	It enters the main pipeline in the *** stage.
	@Note: This version has been ****.
*/
module ext_op7c_instr_v2(CLK_IN, PCLK_IN, RESET_IN, EN_IN, GR_IN, RI_OUT, PI_OUT, ACK_OUT, PC_IN, 
		NEXT_PC_OUT, INSTR_IN, REGRDY_IN, REGFULL_IN, REGEMPTY_IN,
		REG1_DATA_IN, REG2_DATA_IN, REG3_DATA_IN, REG4_DATA_IN,
		REG1_ADDR_OUT, REG2_ADDR_OUT, REG3_ADDR_OUT, REG4_ADDR_OUT,
		REG1_WRREQ_OUT, REG2_WRREQ_OUT, REG1_DATA_OUT, REG2_DATA_OUT,
		REMA_OUT, REWB_OUT, EXTNOP_EX_IN, EXTNOP_MA_IN,
		EXCEXT_IN, EXC_OUT, PCNEXT_OUT,
		MOE_OUT, MWE_OUT, MLOCK_OUT, MADDR_OUT, RNL_OUT, BLS_OUT, HLS_OUT, MEXC_OUT,
		MDATA_IN, MDATA_VLD_IN
		);
	input CLK_IN, PCLK_IN, RESET_IN, EN_IN, GR_IN, REGRDY_IN, REGFULL_IN, REGEMPTY_IN;
	input EXTNOP_EX_IN, EXTNOP_MA_IN;
	input EXCEXT_IN;
	input [31:0] PC_IN, INSTR_IN, REG1_DATA_IN, REG2_DATA_IN, REG3_DATA_IN, REG4_DATA_IN;
	input [31:0] MDATA_IN;
	input MDATA_VLD_IN;
	
	output ACK_OUT, RI_OUT, PI_OUT, REG1_WRREQ_OUT, REG2_WRREQ_OUT, REMA_OUT, REWB_OUT, EXC_OUT;
	output [4:0] REG1_ADDR_OUT, REG2_ADDR_OUT, REG3_ADDR_OUT, REG4_ADDR_OUT;
	output [31:0] NEXT_PC_OUT, REG1_DATA_OUT, REG2_DATA_OUT;
	output PCNEXT_OUT;
	output MOE_OUT, MWE_OUT, MLOCK_OUT, RNL_OUT, BLS_OUT, HLS_OUT, MEXC_OUT;
	output [31:0] MADDR_OUT;
		
	
	wire W_REGWRITE_ID, W_EX_LD_ID, W_REGWRITE_EX, W_EX_LD_EX, W_RESET_EX, W_ACK, W_EN_EX;
	wire W_DNE;
	wire [4:0] W_RS_ID, W_RT_ID, W_RS_EX, W_RT_EX;
	wire [4:0] W_WRREG_MA;		// Register number for register write request from the extension
	wire [4:0] W_RDREG1_ADDR;	// Register number for register read request from the extension
	wire [15:0] W_IMM_ID, W_IMM_EX;
	wire [31:0] W_PC_EX, W_PC_MA;
	wire [31:0] W_NEXT_PC_ID, W_NEXT_PC_EX;	// New PCs from ID and EX stages resp.
	wire [31:0] W_RESULT_EX, W_RESULT_MA, W_REG2DATA_MA;
	wire W_REGWRITE_MA, W_MEMTOREG_MA, W_MEMREAD_MA, W_MEMWRITE_MA;
	wire W_BLS_MA, W_HLS_MA, W_RNL_MA, W_MEXT_MA, W_LC_MA;
	
	wire [31:0] W_PC;
	
	// Added newly for this extension
	wire [4:0] W_RT_REG; 
	wire W_REG_WRITE;
	wire [31:0] W_REG_DATA;
	
	assign REG1_WRREQ_OUT = 1'b0;					// We are not writing directly to the register file, we are re-entering the main 
															// pipe and writing to the register from there, so, reg. write requests = 0
	
	// 
	assign REG2_WRREQ_OUT = (REMA_OUT)?	1'b0 : W_REG_WRITE;
	
	assign REG3_ADDR_OUT = 5'b0;
	assign REG4_ADDR_OUT = 	(REMA_OUT)?	{W_HLS_MA, W_BLS_MA, W_RNL_MA, W_MEMREAD_MA, W_MEMWRITE_MA}: 5'b0;
	
	// If REMA, then register address is the address of the register to be written, otherwise,
	//	if grant is high, register address is the address of the register to be read from
	assign REG1_ADDR_OUT = 	(REMA_OUT)?	W_WRREG_MA: (GR_IN? W_RDREG1_ADDR: 5'b0);
	
	
	// Using this register to write the new value of the stack pointer (W_RT_REG) that comes from the ex stage
	//assign REG2_ADDR_OUT = 	(REMA_OUT)?	{W_MEXT_MA, W_LC_MA,1'b0, W_MEMTOREG_MA, W_REGWRITE_MA}: (/*GR_IN? W_RDREG2_ADDR:*/ 5'b0);
	assign REG2_ADDR_OUT = 	REMA_OUT ?	{W_MEXT_MA, W_LC_MA,1'b0, W_MEMTOREG_MA, W_REGWRITE_MA}: W_RT_REG;
	
	
	
	//assign NEXT_PC_OUT = 	(EN_IN)?		W_NEXT_PC_ID: (REMA_OUT)?	W_PC_MA: 32'b0;
	//assign NEXT_PC_OUT =	(EN_IN)?	W_NEXT_PC_ID: (PCNEXT_OUT)?	W_NEXT_PC_EX: (REMA_OUT)?	W_PC_MA: 32'hffffffff;
	
	assign NEXT_PC_OUT =	EN_IN?	W_NEXT_PC_ID: (PCNEXT_OUT ?	W_NEXT_PC_EX: (REMA_OUT ?	W_PC_MA: 32'hffffffff));
	
	assign REG1_DATA_OUT = 	(REMA_OUT)?	W_RESULT_MA: 32'b0;
	
	// Using this register to write the new value of the stack pointer (W_REG_DATA) that comes from the ex stage
	assign REG2_DATA_OUT = 	(REMA_OUT)?	W_REG2DATA_MA: W_REG_DATA;
	
	
	assign REWB_OUT = 1'b0;
	assign EXC_OUT = 1'b0;
	assign MEXC_OUT = 1'b0;
	assign ACK_OUT = W_ACK;
	
	
	ext_id_stage my_id(.CLK_IN(CLK_IN), .RESET_IN(RESET_IN), .EN_IN(EN_IN), .INSTR_IN(INSTR_IN), .PC_IN(PC_IN),
					.RI_OUT(RI_OUT), .PI_OUT(PI_OUT), .EX_LD_OUT(W_EX_LD_ID), .REGWRITE_OUT(W_REGWRITE_ID),
					.RS_OUT(W_RS_ID), .RT_OUT(W_RT_ID), .IMMEDIATE_OUT(W_IMM_ID), 
					.PC_OUT(W_PC), .NEXT_PC_OUT(W_NEXT_PC_ID)
				);
		
	pipereg_id_ex my_toex(.CLK_IN(CLK_IN), .PCLK_IN(PCLK_IN), .RESET_IN(RESET_IN), .RESET_EX_OUT(W_RESET_EX), 
					.ACK_IN(W_ACK), .EXCEXT_IN(EXCEXT_IN), .EN_ID_IN(EN_IN), .EX_LD_ID_IN(W_EX_LD_ID), 
					.REGWRITE_ID_IN(W_REGWRITE_ID), .RS_ID_IN(W_RS_ID), .RT_ID_IN(W_RT_ID), .IMM_ID_IN(W_IMM_ID), 
					.PC_ID_IN(W_PC), .EN_EX_OUT(W_EN_EX), .EX_LD_EX_OUT(W_EX_LD_EX), .REGWRITE_EX_OUT(W_REGWRITE_EX), 
					.RS_EX_OUT(W_RS_EX), .RT_EX_OUT(W_RT_EX), .IMM_EX_OUT(W_IMM_EX), .PC_EX_OUT(W_PC_EX)
				);

		
	ext_ex_stage my_ex(.CLK_IN(CLK_IN), .PCLK_IN(PCLK_IN), .RESET_IN(RESET_IN), .EN_IN(EN_IN), .GR_IN(GR_IN), 
				.REGRDY_IN(REGRDY_IN), .REGFULL_IN(REGFULL_IN), .EN_EX_IN(W_EN_EX), .EX_LD_EX_IN(W_EX_LD_EX), .RS_EX_IN(W_RS_EX), .RS_DATA_IN(REG1_DATA_IN),
				.IMM_EX_IN(W_IMM_EX), .RS_REG_OUT(W_RDREG1_ADDR),
				.RESULT_EX_OUT(W_RESULT_EX), .ACK_OUT(W_ACK), .DNE_OUT(W_DNE),
				.MOE_OUT(MOE_OUT), .MWE_OUT(MWE_OUT), .MLOCK_OUT(MLOCK_OUT), .MADDR_OUT(MADDR_OUT), 
				.RNL_OUT(RNL_OUT), .BLS_OUT(BLS_OUT), .HLS_OUT(HLS_OUT),
				.MDATA_IN(MDATA_IN), .MDATA_VLD_IN(MDATA_VLD_IN), 
				.PCNEXT_OUT(PCNEXT_OUT), .NEXT_PC_OUT(W_NEXT_PC_EX),
				.RT_REG_OUT(W_RT_REG), .REG_WRITE_OUT(W_REG_WRITE), .REG_DATA_OUT(W_REG_DATA)
			);
	
	pipereg_ex_topipe_ma my_toma(.CLK_IN(CLK_IN), .PCLK_IN(PCLK_IN), .RESET_IN(RESET_IN), .ACK_IN(W_ACK),
				.DNE_IN(W_DNE), .EXTNOP_EX_IN(EXTNOP_EX_IN), .EXCEXT_IN(EXCEXT_IN),
				.EN_EX_IN(W_EN_EX), .EX_LD_EX_IN(W_EX_LD_EX), .RT_EX_IN(W_RT_EX),
				.REGWRITE_EX_IN(W_REGWRITE_EX), .RESULT_EX_IN(W_RESULT_EX),
				.PC_EX_IN(W_PC_EX), .REMA_OUT(REMA_OUT), .REGWRITE_MA_OUT(W_REGWRITE_MA),
				.MEMTOREG_MA_OUT(W_MEMTOREG_MA), .MEMREAD_MA_OUT(W_MEMREAD_MA),
				.MEMWRITE_MA_OUT(W_MEMWRITE_MA), .RESULT_MA_OUT(W_RESULT_MA),
				.REG2DATA_MA_OUT(W_REG2DATA_MA), .WRREG_MA_OUT(W_WRREG_MA),
				.BLS_MA_OUT(W_BLS_MA), .HLS_MA_OUT(W_HLS_MA), .MEXT_MA_OUT(W_MEXT_MA),
				.RNL_MA_OUT(W_RNL_MA), .LC_MA_OUT(W_LC_MA), .PC_MA_OUT(W_PC_MA)
			);
		
endmodule


// Instruction decode stage for the extension. It decodes the extension instruction
//	and generates the next PC (PC after the extension instruction). For this extension,
//	the decode stage is decoding an I-type instruction.
module ext_id_stage(CLK_IN, RESET_IN, EN_IN, INSTR_IN, PC_IN, RI_OUT, PI_OUT, 
					EX_LD_OUT, REGWRITE_OUT, RS_OUT, RT_OUT, IMMEDIATE_OUT, PC_OUT, NEXT_PC_OUT);
	input CLK_IN, RESET_IN, EN_IN;
	input [31:0] INSTR_IN, PC_IN;
	output RI_OUT, PI_OUT, EX_LD_OUT, REGWRITE_OUT;
	output [4:0] RS_OUT, RT_OUT;
	output [15:0] IMMEDIATE_OUT;
	output [31:0] PC_OUT, NEXT_PC_OUT;

	wire [5:0] W_OP;

	reg [31:0] pc_reg, instr_reg;

	assign IMMEDIATE_OUT = instr_reg[15:0];
	assign RT_OUT = instr_reg[20:16];
	assign RS_OUT = instr_reg[25:21];
	assign W_OP = instr_reg[31:26];
	assign PC_OUT = pc_reg;
		
	// Current extension replaces three instructions. So, new PC = Current PC + 16
	assign NEXT_PC_OUT =	(RESET_IN == 1'b0)?	32'b0: pc_reg + 4;

	// RI = 0 indicates that the extension has recognized the instruction
	//	and wants to execute it. This signal is passed on to the co-processor (cp0)
	// @BS: Changed the reset value of RI from 0 to 1
	assign RI_OUT = (RESET_IN == 1'b0)?	1'b1: ((W_OP == `OP7C) ? 1'b0: 1'b1);

	// PI = 0 indicates that the current extension is not passive (not acting in the background)
	//	and requires other resources. This signal is passed on to the co-processor (cp0)
	assign PI_OUT = (RESET_IN == 1'b0)?	1'b0: 1'b0;

	// EX_LD = 1 indicates that the current extension has recognized the instruction as an
	//	extension load instruction. This signal is used to enable the execution of the extension
	assign EX_LD_OUT = (RESET_IN && EN_IN && (W_OP == `OP7C)) ? 1'b1: 1'b0;

	assign REGWRITE_OUT = (RESET_IN && EN_IN && (W_OP == `OP7C)) ? 1'b1: 1'b0;
		
	always@(posedge CLK_IN) begin
		if (RESET_IN == 1'b0) begin
			pc_reg <= 32'b0;
			instr_reg <= 32'b0;
		end
		else begin
			pc_reg <= PC_IN;
			instr_reg <= INSTR_IN;
		end
	end
endmodule


//This is the pipeline register between the ID and the EX stages of the extension.
module pipereg_id_ex(CLK_IN, PCLK_IN, RESET_IN, RESET_EX_OUT, ACK_IN, EXCEXT_IN, EN_ID_IN, EX_LD_ID_IN, 
							REGWRITE_ID_IN, RS_ID_IN, RT_ID_IN, IMM_ID_IN, PC_ID_IN, EN_EX_OUT, EX_LD_EX_OUT, REGWRITE_EX_OUT,  
							RS_EX_OUT, RT_EX_OUT, IMM_EX_OUT, PC_EX_OUT);
	input CLK_IN, RESET_IN, ACK_IN, EXCEXT_IN, EN_ID_IN, EX_LD_ID_IN, REGWRITE_ID_IN;
	input PCLK_IN;
	input [4:0] RS_ID_IN, RT_ID_IN;
	input [31:0] PC_ID_IN;
	input [15:0] IMM_ID_IN;
	output RESET_EX_OUT, EN_EX_OUT, EX_LD_EX_OUT, REGWRITE_EX_OUT;
	output [4:0] RS_EX_OUT, RT_EX_OUT;
	output [31:0] PC_EX_OUT;
	output [15:0] IMM_EX_OUT;

	reg reset_reg;
	reg [1:0] pclkcnt;
	reg [60:0] id_ex;
	
	initial begin
		reset_reg = 1'b0;
		pclkcnt = 2'b0;
		id_ex = 61'b0;
	end
	
	assign RESET_EX_OUT = reset_reg;
	
	assign IMM_EX_OUT = id_ex[60:45];
	assign EN_EX_OUT = id_ex[44];
	assign EX_LD_EX_OUT = id_ex[43];
	assign REGWRITE_EX_OUT = id_ex[42];
	assign RS_EX_OUT =	id_ex[41:37];
	assign RT_EX_OUT =	id_ex[36:32];
	assign PC_EX_OUT =	id_ex[31:0];
									
	always@(posedge CLK_IN) begin
		pclkcnt <= {pclkcnt[0],PCLK_IN};
	end

	always@(posedge CLK_IN) begin
		case(pclkcnt)
			2'b01:	begin
							reset_reg <= RESET_IN;
						end
			default:	begin
						end
		endcase
	end
	
	always@(posedge CLK_IN) begin
		casex({pclkcnt,RESET_EX_OUT,ACK_IN,EXCEXT_IN})
			5'bxx0xx:	begin
								id_ex <= 61'b0;
							end
			// Rising edge of PCLK, no reset (RESET = 1), extension is not currently executing something else (ACK = 0)
			//	and there is no exception in the main pipeline
			5'b01100:	begin
								id_ex <= {IMM_ID_IN, EN_ID_IN, EX_LD_ID_IN, REGWRITE_ID_IN, RS_ID_IN, RT_ID_IN, PC_ID_IN};
			            end
			5'b01110:	begin
			            end
			5'b011x1:	begin
								id_ex <= 61'b0;
							end
			default:		begin
							end
		endcase
	end
endmodule

// Execution stage of the extension. This is where the extension indicates that it has recognized the
//	instruction and requests for resources. All the handshaking (RI, EN, ACK, GR) is done in this module
module ext_ex_stage(CLK_IN, PCLK_IN, RESET_IN, EN_IN, GR_IN, REGRDY_IN, REGFULL_IN, EN_EX_IN, EX_LD_EX_IN, RS_EX_IN, RS_DATA_IN, 
					IMM_EX_IN, RS_REG_OUT, RESULT_EX_OUT, ACK_OUT, DNE_OUT,
					MOE_OUT, MWE_OUT, MLOCK_OUT, MADDR_OUT, RNL_OUT, BLS_OUT, HLS_OUT,
		         MDATA_IN, MDATA_VLD_IN,
					PCNEXT_OUT, NEXT_PC_OUT,
					RT_REG_OUT, REG_WRITE_OUT, REG_DATA_OUT
					);
	input CLK_IN, RESET_IN, EN_IN, GR_IN, REGRDY_IN, REGFULL_IN, EN_EX_IN, EX_LD_EX_IN;
	input PCLK_IN;
	input [4:0] RS_EX_IN;      	// Source register number
	input [15:0] IMM_EX_IN;			// 16 bit immediate value for address offset
	input [31:0] RS_DATA_IN;		// 32 bit data read from the source register
	input [31:0] MDATA_IN;     	// Data read from the memory
	input MDATA_VLD_IN;        	// Data valid signal
	
	output [4:0] RS_REG_OUT;			// RS register number to the TISA for register read
	output ACK_OUT, DNE_OUT;				
	output [31:0] RESULT_EX_OUT;		// Output to be written back. This is needed only if it is a
											// read-modify-write type instruction (Load - Increment - Store)
	
	output MOE_OUT, MWE_OUT, MLOCK_OUT;          // Memory read control signals
	output RNL_OUT, BLS_OUT, HLS_OUT;   
	output [31:0] MADDR_OUT;                     // Memory read address
	
	// Newly added for this extension
	output PCNEXT_OUT;
	output [31:0] NEXT_PC_OUT;
	
	
	output [4:0] RT_REG_OUT; 	// Details of the register to be written to
	output REG_WRITE_OUT;
	output [31:0] REG_DATA_OUT;

	reg ack_reg, dne_reg;
	reg [1:0] pclkcnt;
	reg [1:0] encnt;
	reg [3:0] state_hs;		// State of the handshaking FSM
	reg [31:0] result_reg;
	
	reg MOE_OUT, MWE_OUT, MLOCK_OUT, RNL_OUT, BLS_OUT, HLS_OUT;
	//reg [31:0] MADDR_OUT;
	
	reg mdne;   //Local register used to latch the memory done signal
	reg pcnext_reg;
	reg [31:0] new_pc_reg;
	
	// Newly added for this extension
	reg [31:0] sp_reg;	//Register to store the new value of SP 
	//reg [4:0] rt_reg;
	reg reg_write_reg;
		
	reg initiate_reg_write;
	reg reg_wr_state;
	
	//reg reg_write_dne;
	
	reg [31:0] maddr_reg;
	
	initial
	begin
		pclkcnt = 2'b0;
		encnt = 2'b00;
		state_hs = 4'b0;
		ack_reg = 1'b0;
		dne_reg = 1'b1;
		result_reg = 32'b0;
		mdne = 1'b1;
		pcnext_reg = 1'b0;
		new_pc_reg = 32'b0;
		
		// Newly added for this extension
		sp_reg = 32'b0;
		reg_write_reg = 1'b0;
		//rt_reg = 5'b0;
		
		initiate_reg_write = 1'b0;
		reg_wr_state = 1'b0;
		
		//reg_write_dne = 1'b0;
		maddr_reg = 32'b0;
	   RNL_OUT = 1'b0;
	   BLS_OUT = 1'b0;
		HLS_OUT = 1'b0;
		MOE_OUT = 1'b0;
		MWE_OUT = 1'b0;
	end
	
	assign RESULT_EX_OUT = (RESET_IN & EN_EX_IN & EX_LD_EX_IN)? result_reg: 32'b0;
	assign ACK_OUT = (RESET_IN == 1'b0)? 1'b0: ack_reg;
	
	//@Changed
	//assign DNE_OUT = (RESET_IN == 1'b0)? 1'b1: (dne_reg & reg_write_dne);
	assign DNE_OUT = (RESET_IN == 1'b0)? 1'b1: dne_reg;
		
	//Once the grant signal arrives from the pipeline arbiter, we can place the register
	// numbers on the output ports to read the register values
	assign RS_REG_OUT = GR_IN ? RS_EX_IN: 5'b0;
	
	// Newly added for this extension
	assign PCNEXT_OUT = pcnext_reg;
	
	assign NEXT_PC_OUT = new_pc_reg;
	//assign NEXT_PC_OUT = {new_pc_reg[7:0], new_pc_reg[15:8], new_pc_reg[23:16], new_pc_reg[31:24]};
	
	assign REG_DATA_OUT = sp_reg;
	
	//assign RT_REG_OUT = rt_reg;
	assign RT_REG_OUT = GR_IN ? RS_EX_IN: 5'b0;	// Register to be written out is same as the register to be read from
																// Writing is done only after the register has been read. This is 
																// controlled by the state machine at the end of the module
	assign REG_WRITE_OUT = reg_write_reg;
	
	assign MADDR_OUT = maddr_reg;
	//assign MADDR_OUT = {1'b0, maddr_reg[30:0]};
	
	
	// Requesting the resources by This is the state machine the requests the arbiter for resources. 
	// All the handshaking (RI, EN, ACK, GR) is done here
	always@(posedge CLK_IN) begin
		if(RESET_IN == 1'b0) begin
			ack_reg <= 1'b0;
			dne_reg <= 1'b1;
			state_hs <= 4'b0;
			mdne <= 1'b1;
			result_reg <= 32'b0;
			
			// Newly added for this extension
			sp_reg <= 32'b0;
			reg_write_reg <= 1'b0;
			//rt_reg = 5'b0;
			initiate_reg_write <= 1'b0;
		   RNL_OUT <= 1'b0;
		   BLS_OUT <= 1'b0;
			HLS_OUT <= 1'b0;
			MOE_OUT <= 1'b0;
			MWE_OUT <= 1'b0;
		end
		//Changed
		else if (~EN_IN & GR_IN & dne_reg & ack_reg) begin		// If the current extension has the control (ACK = 1) and has
			ack_reg <= 1'b0;												// finished execution then lower the ACK and release the resources
		end																	
		else begin
			case(state_hs)
				4'b0000:	begin
								if(encnt == 2'b01) begin	// If the extension is enabled by the pipeline arbiter
									dne_reg <= 1'b0;			// it lowers the done signal and raises the ACK signal
									ack_reg <= 1'b1;			// and then waits for the grant to come
									state_hs <= 4'b0001;
								end
							end
				4'b0001:	begin									// Waiting for grant signal to arrive. Once that arrives, we
								if(GR_IN) begin				// move on to the next state and wait for 4 cycles for the 
									state_hs <= 4'b0010;		// data to arrive from the register file
								end
							end
				4'b0010:	begin
								state_hs <= 4'b0011;
							end
				4'b0011: 	begin
								state_hs <= 4'b0100;
							end
				4'b0100: 	begin
								state_hs <= 4'b0101;
							end
				4'b0101: 	begin   	//@BS: Added another state - now waiting for 5 clk cycles
										// for the data to come from the register file
								state_hs <= 4'b0110;
							end
				4'b0110: begin		// Read the register values here and compute effective address
								maddr_reg <= RS_DATA_IN + IMM_EX_IN - 4'h8;
								sp_reg <= RS_DATA_IN + IMM_EX_IN;
								if (GR_IN & REGRDY_IN & ~REGFULL_IN) begin
									reg_write_reg <= 1'b1;
									state_hs <= 4'b0111;
								end
							end
				4'b0111: 	begin
								// Removing the register write request after one clock cycle.
								reg_write_reg <= 1'b0;
								// Place memory read request on rising edge of PCLK
								if (pclkcnt == 2'b01) begin   			
								   RNL_OUT <= 1'b0;
								   BLS_OUT <= 1'b0;
								   HLS_OUT <= 1'b0;
								   MOE_OUT <= 1'b1;
								   MWE_OUT <= 1'b0;
								   mdne <= 1'b0;			      // This is used to indicate that a memory request has
																// been initiated from the extension

                          	state_hs <= 4'b1000;
								end
							 end				  
				4'b1000: 	begin
				            if (pclkcnt == 2'b00) begin   // Remove memory read request after the falling edge of PCLK
				               MOE_OUT <= 1'b0;
				               if(MDATA_VLD_IN && ~mdne) begin	// Look for MDATA_VLD_IN signal only after initiating the request 
																				// and after the falling edge of PCLK
								      result_reg <= MDATA_IN;  // Assign the input data to the result register
								      mdne <= 1'b1;						      // Latch the memory done signal (MDATA_VLD_IN)
										
										// Newly added for this extension
										new_pc_reg <= MDATA_IN;
										state_hs <= 4'b1001;
									end
								end
							 end
				4'b1001: 	begin
				             if (pclkcnt == 2'b10) begin   // Raise pcnext signal on the next falling edge of PCLK
							       pcnext_reg <= 1'b1;
							       state_hs <= 4'b1010;
								end
							 end
				4'b1010: 	begin
								if ((pclkcnt == 2'b01) && (mdne == 1'b1)) begin
									pcnext_reg <= 1'b0;         // Lower the pcnext on the rising edge of PCLK
									                           // @BS: DNE must go high only after rising edge of
									dne_reg <= 1'b1;				        // PCLK has appeared, so that REMA can be generated
					            state_hs <= 4'b0000;         // on the subsequent falling edge of PCLK. On next
								end									   // rising edge of PCLK, grant is removed.
							end
				default: begin
								state_hs <= 4'b0000;
							end
			endcase
		end
	end
	
	always@(posedge CLK_IN)
	begin
		pclkcnt <= {pclkcnt[0],PCLK_IN};
		encnt <= {encnt[0], EN_EX_IN};	
	end
endmodule



module pipereg_ex_topipe_ma(CLK_IN, PCLK_IN, RESET_IN, ACK_IN, DNE_IN, EXTNOP_EX_IN, EXCEXT_IN, EN_EX_IN, EX_LD_EX_IN, RT_EX_IN,
								REGWRITE_EX_IN, RESULT_EX_IN, PC_EX_IN, REMA_OUT, REGWRITE_MA_OUT, MEMTOREG_MA_OUT, MEMREAD_MA_OUT, MEMWRITE_MA_OUT,
								RESULT_MA_OUT, REG2DATA_MA_OUT, WRREG_MA_OUT, BLS_MA_OUT, HLS_MA_OUT, MEXT_MA_OUT, RNL_MA_OUT, LC_MA_OUT, PC_MA_OUT
							);
	input CLK_IN, RESET_IN, ACK_IN, DNE_IN, EXTNOP_EX_IN, EXCEXT_IN, EN_EX_IN, EX_LD_EX_IN, REGWRITE_EX_IN;
	input PCLK_IN;
	input [4:0] RT_EX_IN;
	input [31:0] RESULT_EX_IN, PC_EX_IN;
	output REMA_OUT, REGWRITE_MA_OUT, MEMTOREG_MA_OUT, MEMREAD_MA_OUT, MEMWRITE_MA_OUT, BLS_MA_OUT, HLS_MA_OUT, MEXT_MA_OUT, RNL_MA_OUT, LC_MA_OUT;
	output [4:0] WRREG_MA_OUT;			// Register where the data should be written
	output [31:0] RESULT_MA_OUT; 		// Data to be written to through the first register port
	output [31:0] REG2DATA_MA_OUT;	// Data to be written to through the second register port
	output [31:0] PC_MA_OUT;

	wire W_EN_MA, W_EX_MR_MA;

	reg reset_reg, rema_reg;
	reg [1:0] pclkcnt;
	reg [71:0] ex_ma;

	initial
	begin
		rema_reg = 1'b0;
		reset_reg = 1'b0;
		pclkcnt = 2'b0;
		ex_ma = 72'b0;
	end

	assign RESET_MA = reset_reg;

	always@(posedge CLK_IN)
	begin
		pclkcnt <= {pclkcnt[0],PCLK_IN};
	end

	always@(posedge CLK_IN)
	begin
		case(pclkcnt)
			2'b01:	begin
							reset_reg <= RESET_IN;
						end
			default:	begin
						end
		endcase
	end

	assign REMA_OUT = (rema_reg & W_EN_MA & W_EX_MR_MA);
	assign W_EN_MA =	ex_ma[71];
	assign W_EX_MR_MA =	ex_ma[70];
	assign REGWRITE_MA_OUT = ex_ma[69];
	assign MEMTOREG_MA_OUT = 1'b0;
	assign MEMREAD_MA_OUT = 1'b0;
	assign MEMWRITE_MA_OUT = 1'b0;
	assign BLS_MA_OUT = 1'b0;
	assign HLS_MA_OUT = 1'b0;
	assign MEXT_MA_OUT = 1'b0;
	assign RNL_MA_OUT = 1'b0;
	assign LC_MA_OUT =	1'b0;
	assign WRREG_MA_OUT = ex_ma[68:64];
	assign RESULT_MA_OUT = ex_ma[63:32];
	assign REG2DATA_MA_OUT = 32'b0;
	assign PC_MA_OUT =	ex_ma[31:0];

	always@(posedge CLK_IN)
	begin
		casex({pclkcnt, RESET_IN, EXTNOP_EX_IN, rema_reg, ACK_IN, DNE_IN, EXCEXT_IN})
			// Reset state
			8'bxx0xxxxx:	begin
									rema_reg <= 1'b0;
									ex_ma <= 72'b0;
								end
			
			//If ACK = 1 and DNE = 1 (i.e. extension is running and is done) then latch the result data
			8'bxx1x0110:	begin
									ex_ma <= {EN_EX_IN, EX_LD_EX_IN, REGWRITE_EX_IN, RT_EX_IN, RESULT_EX_IN, PC_EX_IN};
								end
			
			// On the falling edge of PCLK, raise REMA signal so that data can enter the MA phase
			// of the main pipeline
			8'b101100x0:	begin
									rema_reg <= 1'b1;
								end
			
			//On the next rising edge of PCLK, remove REMA signal
			8'b011x1xx0:	begin
									rema_reg <= 1'b0;
									ex_ma <= 72'b0;
								end
			//Exception state
			8'b011xxxx1:	begin
									rema_reg <= 1'b0;
									ex_ma <= 72'b0;
								end
			default:			begin
								end
		endcase
	end
endmodule

