/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	mmlite_div64													*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"			/* Constant Definitiions */

`define SLL128	6'b011110	/* Extension Opcode */

module mmlite_div64(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				EN,					/* Enable */
	input				EXCEXT,				/* Exception Flush */
	input				EXTNOP_MA,			/* Extension Bubble in Memory Access Phase */
	input				GR,					/* Grant Pipeline Resources */
	input [31:0]	INSTR,				/* Current Instruction */
	input [31:0]	PC,					/* Current PC External */
	input				PCLK,					/* Pipeline Clock */
	input [31:0]	RDREG1DATA,			/* Register Read Port 1 Register Data */
	input [31:0]	RDREG2DATA,			/* Register Read Port 2 Register Data */
	input [31:0]	RDREG3DATA,			/* Register Read Port 3 Register Data */
	input [31:0]	RDREG4DATA,			/* Register Read Port 4 Register Data */
	input				REGEMPTY,			/* Register Write Buffer Empty */
	input				REGFULL,				/* Register Write Buffer Full */
	input				REGRDY,				/* Register Write Buffer Ready */
	input				RESET,				/* System Reset */
	/* OUTPUT PORTS */
	output 			ACK,					/* Enable Acknowledged */
	output [31:0]	EXTADD,				/* Extension Address */
												/* Multiplexed: */
												/* 	Next PC */
												/* 	Exception Address */
												/* 	PC Memory Access Phase */
	output 			PCNEXT,				/* Conditional PC Update */
	output [4:0]	RDREG1,				/* Register Read Port 1 Register Number */
												/* Multiplexed: */
												/* 	Register Read Port 1 Register Number */
												/* 	Register Write Port 1 Register Number */
												/* 	Write Register Memory Access Phase */
	output [4:0]	RDREG2,				/* Register Read Port 2 Register Number */
												/* Multiplexed: */
												/* 	Register Read Port 2 Register Number */
												/* 	Register Write Port 2 Register Number */
												/*		<0> Register Write Enable Memory Access Phase */
												/* 	<1> Memory to Register Memory Acess Phase */
	output [4:0]	RDREG3,				/* Register Read Port 3 Register Number */
												/* Multiplexed: */
												/* 	Register Read Port 3 Register Number */
	output [4:0]	RDREG4,				/* Register Read Port 4 Register Number Internal */
												/* Multiplexed: */
												/* 	Register Read Port 4 Register Number */
												/* 	<1:0> Data Address [1:0] Memory Access Phase */
												/*		<2> Right/Left Unaligned Load/Store Memory Access Phase */
												/* 	<3> Byte/Halfword Load/Store Memory Acess Phase */
	output 			REGWRITE1,			/* Register Write Port 1 Write Enable */
	output 			REGWRITE2,			/* Register Write Port 2 Write Enable */
	output 			REWB,					/* Re-enter at Writeback */
	output 			RI,					/* Reserved/Recognized Instruction */
	output [31:0]	WRDATA1,				/* Register Write Port 1 Data Internal */
												/* Multiplexed: */
												/* 	Register Write Port 1 Data */
												/* 	ALU Result Memory Access Phase */
	output [31:0]	WRDATA2				/* Register Write Port 2 Data Internal */
												/* Multiplexed: */
												/* 	Register Write Port 2 Data */
												/* 	Memory Data Out Memory Access Phase */
	);
	
/*****Signals****************************************************************/

	wire [31:0]	ALURESULT_WB;		/* ALU Result to Writeback Phase */
	wire 			BHLS_WB;				/* Byte/Halfword Load/Store to Writeback Phase */
	wire [31:0]	CJMPADD;				/* Conditional Jump address to offset from Current PC */
	wire [15:0]	DIMM_EX;				/* Data Immediate Execute Phase */
	wire [15:0]	DIMM_ID;				/* Data Immediate Instruction Decode Phase */
	wire [1:0]	DMADD_WB;			/* Least Significant Bits of Data Address to Writeback Phase */
	wire [31:0]	DMDATAOUT_WB;		/* Memory Data Out to Writeback Phase */
	wire 			DNE;					/* Execution Done */
	wire 			EN_EX;				/* Enable Execute Phase */
	wire [31:0]	JMPADD;				/* Jump address to end of basic block */
	wire 			MEMTOREG_WB;		/* Memory to Register to Writeback Phase */
	wire [31:0]	PC_EX;				/* PC Execute Phase */
	wire [31:0]	PC_WB;				/* PC to Writeback Phase */
	wire [4:0]	RD_EX;				/* Destination Register Execution Phase */
	wire [4:0]	RDREG1_EX;			/* Register Read Port 1 Register Number Execute Phase */
	wire [31:0]	RDREG1DATA_EX;		/* Register Read Port 1 Register Data Execute Phase */
	wire [4:0]	RDREG2_EX;			/* Register Read Port 2 Register Number Execute Phase */
	wire [31:0]	RDREG2DATA_EX;		/* Register Read Port 2 Register Data Execute Phase */
	wire [4:0]	RDREG3_EX;			/* Register Read Port 3 Register Number Execute Phase */
	wire [4:0]	RDREG4_EX;			/* Register Read Port 4 Register Number Execute Phase */
	wire			REGWRITE_EX;		/* Register Write Execute Phase */
	wire			REGWRITE_ID;		/* Register Write Instruction Decode Phase */
	wire			REGWRITE_WB;		/* Register Write to Writeback Phase */
	wire			RESET_EX;			/* Reset Execute Phase */
	wire [31:0]	RESULT_EX;			/* Result Execution Phase */
	wire			RNL_WB;				/* Right/Left Unaligned Load/Store to Writeback Phase */
	wire [4:0]	RS_EX;				/* Operand Register 1 Execute Phase */
	wire [4:0]	RS_ID;				/* Operand Register 1 Instruction Decode Phase */
	wire [4:0]	RT_EX;				/* Operand Register 2 Execute Phase */
	wire [4:0]	RT_ID;				/* Operand Register 2 Instruction Decode Phase */
	wire 			SLL128_EX;			/* Shift Left Logical 128 bits Execute Phase */
	wire 			SLL128_ID;			/* Shift Left Logical 128 bits Instruction Decode Phase */
	wire [31:0]	WRDATA1_EX;			/* Register Write Port 1 Data Execute Phase */
	wire [31:0]	WRDATA2_EX;			/* Register Write Port 2 Data Execute Phase */
	wire [4:0]	WRREG_WB;			/* Write Register Number to Writeback Phase */
	wire [4:0]	WRREG1_EX;			/* Register Write Port 1 Register Number Execute Phase */
	wire [4:0]	WRREG2_EX;			/* Register Write Port 2 Register Number Execute Phase */
	wire [31:0]	VIRPC;				/* */

/*****Registers****************************************************************/

	reg en_reg;	/* Enable */
	reg gr_reg;	/* Grant Pipeline Resources */

/*****Initialization****************************************************************/

	initial
	begin
		en_reg = 1'b0;
		gr_reg = 1'b0;
	end
	
/*********************************************************************/

	assign EXTADD	=	(EXCEXT)?					VIRPC:
							(en_reg)?					JMPADD:
							(PCNEXT)?					CJMPADD:
							(REWB)?						PC_WB:
															32'hffffffff;
	assign RDREG1	=			(gr_reg & REGWRITE1)?	WRREG1_EX:
					(EXCEXT)?					5'b0:
							(REWB & gr_reg)?			WRREG_WB:
							(gr_reg)?					RDREG1_EX:
															5'b11111;	
	assign RDREG2	=			(gr_reg & REGWRITE2)?	WRREG2_EX:
					(EXCEXT)?					5'b0:
							(REWB & gr_reg)?			{3'b0,MEMTOREG_WB,REGWRITE_WB}:
							(gr_reg)?					RDREG2_EX:
															5'b11111;
	assign RDREG3	=			(REWB & gr_reg)?			5'b0:
					(EXCEXT)?					5'b0:
							(gr_reg)?					RDREG3_EX:
															5'b11111;
	assign RDREG4	=			(REWB & gr_reg)?			{1'b0,BHLS_WB,RNL_WB,DMADD_WB}:
					(EXCEXT)?					5'b0:
							(gr_reg)?					RDREG4_EX:
															5'b11111;
	assign WRDATA1 = 				(gr_reg & REGWRITE1)?	WRDATA1_EX:
				(EXCEXT)?					32'b0:
							(REWB)?						ALURESULT_WB:
															32'hffffffff;
	assign WRDATA2 = 				(gr_reg & REGWRITE2)?	WRDATA2_EX:
				(EXCEXT)?					32'b0:
							(REWB)?						DMDATAOUT_WB:
															32'hffffffff;

/*****Instruction Decode****************************************************************/

	mmldiv64_id id(
		.CLK(CLK),
		.DIMM(DIMM_ID),
		.EN(EN),
		.JMPADD(JMPADD),
		.INSTR(INSTR),
		.PC(PC),
		.REGWRITE(REGWRITE_ID),
		.RESET(RESET),
		.RI(RI),
		.RS(RS_ID),
		.RT(RT_ID),
		.SLL128(SLL128_ID)
		);

/*****Instruction Decode -> Execute****************************************************************/

	mmldiv64_toex to_ex(
		.ACK(ACK),
		.CLK(CLK),
		.DIMM_EX(DIMM_EX),		
		.DIMM_ID(DIMM_ID),
		.EN_EX(EN_EX),
		.EN_ID(EN),
		.EXCEXT(EXCEXT),
		.PC_EX(PC_EX),
		.PC_ID(PC),
		.PCLK(PCLK),
		.RDREG1DATA_EX(RDREG1DATA_EX),
		.RDREG1DATA_ID(RDREG1DATA),
		.RDREG2DATA_EX(RDREG2DATA_EX),
		.RDREG2DATA_ID(RDREG2DATA),
		.REGWRITE_EX(REGWRITE_EX),
		.REGWRITE_ID(REGWRITE_ID),
		.RESET(RESET),
		.RESET_EX(RESET_EX),
		.RS_EX(RS_EX),
		.RS_ID(RS_ID),
		.RT_EX(RT_EX),
		.RT_ID(RT_ID),
		.SLL128_ID(SLL128_ID),
		.SLL128_EX(SLL128_EX)
		);
		
/*****Execute****************************************************************/

	mmldiv64_ex ex(
		.ACK(ACK),
		.DIMM(DIMM_EX),
		.DNE(DNE),
		.CLK(CLK),
		.CJMPADD(CJMPADD),
		.EN(EN_EX),
		.EXTNOP_MA(EXTNOP_MA),
		.GR(GR),
		.PC(PC_EX),
		.PCLK(PCLK),
		.PCNEXT(PCNEXT),
		.RD(RD_EX),
		.RDREG1(RDREG1_EX),
		.RDREG1DATA(RDREG1DATA),
		.RDREG1DATA_ID(RDREG1DATA_EX),
		.RDREG2(RDREG2_EX),
		.RDREG2DATA(RDREG2DATA),
		.RDREG2DATA_ID(RDREG2DATA_EX),
		.RDREG3(RDREG3_EX),
		.RDREG3DATA(RDREG3DATA),
		.RDREG4(RDREG4_EX),
		.RDREG4DATA(RDREG4DATA),
		.REGEMPTY(REGEMPTY),
		.REGFULL(REGFULL),
		.REGRDY(REGRDY),
		.REGWRITE1(REGWRITE1),
		.REGWRITE2(REGWRITE2),
		.RESET(RESET_EX),
		.RESULT(RESULT_EX),
		.RS(RS_EX),
		.RT(RT_EX),
		.SLL128(SLL128_EX),
		.WRDATA1(WRDATA1_EX),
		.WRDATA2(WRDATA2_EX),
		.WRREG1(WRREG1_EX),
		.WRREG2(WRREG2_EX),
		.EXC(EXCEXT),
		.VIRPC(VIRPC)
		);

/*****Execute -> to Writeback****************************************************************/

	mmldiv64_topipe_wb to_wb(
		.ACK(ACK),
		.ALURESULT_WB(ALURESULT_WB),
		.BHLS_WB(BHLS_WB),
		.CLK(CLK),
		.DMADD_WB(DMADD_WB),
		.DMDATAOUT_WB(DMDATAOUT_WB),
		.DNE(DNE),
		.EN_EX(EN_EX),
		.EXCEXT(EXCEXT),
		.EXTNOP_MA(EXTNOP_MA),
		.PC_EX(PC_EX),
		.PC_WB(PC_WB),
		.PCLK(PCLK),
		.MEMTOREG_WB(MEMTOREG_WB),
		.RD_EX(RD_EX),
		.REGWRITE_EX(REGWRITE_EX),
		.REGWRITE_WB(REGWRITE_WB),
		.RESET(RESET),
		.RESULT_EX(RESULT_EX),
		.REWB(REWB),
		.RNL_WB(RNL_WB),
		.WRREG_WB(WRREG_WB)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			en_reg <= 1'b0;
			gr_reg <= 1'b0;
		end
		else
		begin
			en_reg <= EN;
			gr_reg <= GR;
		end
	end

endmodule

/*****Instruction Decode****************************************************************/

module mmldiv64_id(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CLK,			/* System Clock 50 - 100 MHZ */
	input 			EN,			/* Enable */
	input [31:0]	INSTR,		/* Current Instruction */
	input [31:0]	PC,			/* Current PC */
	input 			RESET,		/* System Reset */
	/* OUTPUT PORTS */
	output [15:0]	DIMM,			/* Data Immediate */
	output [31:0]	JMPADD,		/* Jump address to end of basic block */
	output 			REGWRITE,	/* Register Write */
	output 			RI,			/* Reserved/Recognized Instruction */
	output [4:0]	RS,			/* Operand Register 1 */
	output [4:0]	RT,			/* Operand Register 2 */
	output 			SLL128		/* Shift Left Logical 128 bits */
	);

/*****Signals****************************************************************/
	
	wire [5:0] OP;	/* Operation Code */
	
/*****Registers****************************************************************/

	reg 			en_reg;		/* Enable */
	reg [31:0]	instr_reg;	/* Current Instruction */
	reg [31:0]	jmpadd_reg;	/* Jump address to end of basic block */
	reg [31:0]	pc_reg;		/* Current PC */

/*****Initialization****************************************************************/

	initial 
	begin
		en_reg = 1'b0;
		instr_reg = 32'b0;
		jmpadd_reg = 32'b0;
		pc_reg = 32'b0;
	end

/*********************************************************************/

	assign DIMM		=	instr_reg[15:0];
	assign JMPADD	=	jmpadd_reg;
	assign OP		=	instr_reg[31:26];
	assign RS		=	instr_reg[25:21];
	assign RT		=	instr_reg[20:16];

/*****Data Path Control****************************************************************/

	mmldiv64_dpc dpc(
		.CLK(CLK),
		.EN(en_reg),
		.OP(OP),
		.REGWRITE(REGWRITE),
		.RESET(RESET),
		.RI(RI),
		.SLL128(SLL128)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			en_reg <= 1'b0;
			instr_reg <= 32'b0;
			jmpadd_reg <= 32'b0;
			pc_reg <= 32'b0;
		end
		else
		begin
			en_reg <= EN;
			instr_reg <= INSTR;
			jmpadd_reg <= pc_reg + 52;
			pc_reg <= PC;
		end
	end

endmodule

/*****Data Path Control****************************************************************/

module mmldiv64_dpc(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 		CLK,			/* System Clock 50 - 100 MHZ */
	input 		EN,			/* Enable */
	input [5:0]	OP,			/* Operation Code */
	input 		RESET,		/* System Reset */
	/* OUTPUT PORTS */
	output 		REGWRITE,	/* Register Write */
	output 		RI,			/* Reserved/Recognized Instruction */
	output 		SLL128		/* Shift Left Logical 128 bits */
	);

/*****Registers****************************************************************/

	reg regwrite_reg;	/* Register Write */
	reg ri_reg;			/* Reserved/Recognized Instruction */
	reg sll128_reg;	/* Shift Left Logical 128 bits */

/*****Initialization****************************************************************/

	initial
	begin
		regwrite_reg = 1'b0;
		ri_reg = 1'b0;
		sll128_reg = 1'b0;
	end

/*********************************************************************/

	assign REGWRITE	=	regwrite_reg;
	assign RI			=	ri_reg;
	assign SLL128		=	sll128_reg;

/*********************************************************************/

	/* Reserved/Recognized Instruction */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)		ri_reg <= 1'b0;
		else
		begin
			if (OP == `SLL128)	ri_reg <= 1'b0;
			else						ri_reg <= 1'b1;
		end
	end

	/* Shift Left Logical 128 bits */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			sll128_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)			sll128_reg <= 1'b0;
			else
			begin
				if (OP == `SLL128)	sll128_reg <= 1'b1;
				else 						sll128_reg <= 1'b0;
			end
		end
	end

	/* Register Write */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			regwrite_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)			regwrite_reg <= 1'b0;
			else
			begin
				if (OP == `SLL128)	regwrite_reg <= 1'b1;
				else 						regwrite_reg <= 1'b0;
			end
		end
	end

endmodule

/*****Instruction Decode -> Execute****************************************************************/

module mmldiv64_toex(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			ACK,				/* Enable Acknowledged */
	input 			CLK,				/* System Clock 50 - 100 MHZ */
	input [15:0]	DIMM_ID,			/* Data Immediate Instruction Decode Phase */
	input 			EN_ID,			/* Enable Instruction Decode Phase */
	input 			EXCEXT,			/* Exception Flush */
	input [31:0]	PC_ID,			/* Current PC Decode Phase */
	input 			PCLK,				/* Pipeline Clock */
	input [31:0]	RDREG1DATA_ID,	/* Register Read Port 1 Register Data Instruction Decode Phase */
	input [31:0]	RDREG2DATA_ID,	/* Register Read Port 2 Register Data Instruction Decode Phase */
	input 			REGWRITE_ID,	/* Register Write Instruction Decode Phase*/
	input 			RESET,			/* System Reset */
	input [4:0]		RS_ID,			/* Operand Register 1 Instruction Decode Phase */
	input [4:0]		RT_ID,			/* Operand Register 2 Instruction Decode Phase */
	input 			SLL128_ID,		/* Shift Left Logical 128 bits Instruction Decode Phase */
	/* OUTPUT PORTS */
	output [15:0]	DIMM_EX,			/* Data Immediate Execute Phase */
	output 			EN_EX,			/* Enable Execute Phase */
	output [31:0]	PC_EX,			/* Current PC Instruction Decode Phase */
	output [31:0]	RDREG1DATA_EX,	/* Register Read Port 1 Register Data Execute Phase */
	output [31:0]	RDREG2DATA_EX,	/* Register Read Port 2 Register Data Execute Phase */
	output 			REGWRITE_EX,	/* Register Write Execute Phase*/
	output 			RESET_EX,		/* Reset Execute Phase */
	output [4:0]	RS_EX,			/* Operand Register 1 Execute Phase */
	output [4:0]	RT_EX,			/* Operand Register 2 Execute Phase */
	output 			SLL128_EX		/* Shift Left Logical 128 bits Execute Phase */
	);
	
/*****Registers****************************************************************/

	reg [124:0]	id_ex;		/* Instruction Decode -> Execute Pipeline Register */
	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Execute Phase */

/*****Initialization****************************************************************/

	initial
	begin
		id_ex = 125'b0;
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_EX		=	reset_reg;
	assign EN_EX			=	id_ex[124];			//EN_ID;
	assign SLL128_EX		=	id_ex[123];			//SLL128_ID;
	assign REGWRITE_EX	=	id_ex[122];			//REGWRITE_ID;
	assign RS_EX 			=	id_ex[121:117];	//RS_ID;
	assign RT_EX 			=	id_ex[116:112];	//RT_ID;
	assign DIMM_EX 		=	id_ex[111:96];		//DIMM_ID;
	assign PC_EX			=	id_ex[95:64];		//PC_ID;
	assign RDREG1DATA_EX	=	id_ex[63:32];		//RDREG1DATA_ID;
	assign RDREG2DATA_EX	=	id_ex[31:0];		//RDREG2DATA_ID

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

	always@(posedge CLK)
	begin
		/* Instruction Decode -> Execute Pipeline Register */
		casex({pclkcnt,RESET_EX,ACK,EXCEXT})
			5'bxx0xx	:	begin
								/* Reset */
								id_ex <= 109'b0;
							end
			5'b011x1	:	begin
								/* Exception in Pipeline, Flush */
								id_ex <= 109'b0;
							end
			5'bxx110	:	begin
								/* Hold state during Execute Phase */
							end
			5'b01100	:	begin
								/* Clocking the Pipeline */
								id_ex <= {EN_ID,SLL128_ID,REGWRITE_ID,RS_ID,RT_ID,DIMM_ID,PC_ID,RDREG1DATA_ID,RDREG2DATA_ID};
							end
			default	:	begin
								/* NOP */
							end
		endcase
	end
endmodule

/*****Execute****************************************************************/

module mmldiv64_ex(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CLK,					/* System Clock 50 - 100 MHZ */
	input [15:0]	DIMM,					/* Data Immediate */
	input 			EN,					/* Enable  */
	input 			EXTNOP_MA,			/* Extension Bubble in Memory Access Phase */
	input 			GR,					/* Grant Pipeline Resources */
	input [31:0]	PC,					/* Current PC */
	input 			PCLK,					/* Pipeline Clock */
	input [31:0]	RDREG1DATA,			/* Register Read Port 1 Register Data */
	input [31:0]	RDREG1DATA_ID,		/* Register Read Port 1 Register Data Instruction Decode Phase */
	input [31:0]	RDREG2DATA,			/* Register Read Port 2 Register Data */
	input [31:0]	RDREG2DATA_ID,		/* Register Read Port 2 Register Data Instruction Decode Phase */
	input [31:0]	RDREG3DATA,			/* Register Read Port 3 Register Data */
	input [31:0]	RDREG4DATA,			/* Register Read Port 4 Register Data */
	input 			REGEMPTY,			/* Register Write Buffer Empty */
	input 			REGFULL,				/* Register Write Buffer Full */
	input 			REGRDY,				/* Register Write Buffer Ready */
	input 			RESET,				/* System Reset */
	input [4:0] 	RS,					/* Operand Register 1 */
	input [4:0] 	RT,					/* Operand Register 2 */
	input 			SLL128,				/* Shift Left Logical 128 bits */
	input				EXC,					/* */
	/* OUTPUT PORTS */
	output 			ACK,					/* Enable Acknowledged */
	output [31:0] 	CJMPADD,				/* Conditional Jump address to offset from Current PC */
	output 			DNE,					/* Execution Done */
	output 			PCNEXT,				/* Conditional PC Update */
	output [4:0] 	RD,					/* Destination Register */
	output 			REGWRITE1,			/* Register Write Port 1 Write Enable */
	output 			REGWRITE2,			/* Register Write Port 2 Write Enable */
	output [4:0] 	RDREG1,				/* Register Read Port 1 Register Number */
	output [4:0] 	RDREG2,				/* Register Read Port 2 Register Number */
	output [4:0] 	RDREG3,				/* Register Read Port 3 Register Number */
	output [4:0] 	RDREG4,				/* Register Read Port 4 Register Number */
	output [31:0] 	RESULT,				/* Result */
	output [31:0] 	WRDATA1,				/* Register Write Port 1 Data */
	output [31:0] 	WRDATA2,				/* Register Write Port 2 Data */
	output [4:0] 	WRREG1,				/* Register Write Port 1 Register Number */
	output [4:0] 	WRREG2,				/* Register Write Port 2 Register Number */
	output [31:0]	VIRPC					/* */
	);

/*****Registers****************************************************************/
	
	reg 			ack_reg;			/* Enable Acknowledged */
	reg [31:0]	cjmpadd_reg;	/* Conditional Jump address to offset from Current PC */
	reg [15:0]	dimm_reg;		/* Data Immediate */
	reg [31:0]	dimmsext;		/* Data Immediate Sign Extended */
	reg 			dne_reg;			/* Execution Done */
	reg 			en_reg;			/* Enable  */
	reg 			extnop_ma_reg;	/* Extension Bubble in Memory Access Phase */
	reg 			gr_reg;			/* Grant Pipeline Resources */
	reg [31:0]	pc_reg;			/* Current PC */
	reg [1:0]	pclkcnt;			/* Pipeline Clock edge detection */
	reg 			pcnext_reg;		/* Conditional PC Update */
	reg [4:0]	r1_reg;			/* Operand Register R1 */
	reg [31:0]	r1data_reg;		/* Operand Register R1 Data */
	reg [4:0]	r2_reg;			/* Operand Register R2 */
	reg [31:0]	r2data_reg;		/* Operand Register R2 Data */
	reg [4:0]	r3_reg;			/* Operand Register R3 */
	reg [31:0]	r3data_reg;		/* Operand Register R3 Data */
	reg [4:0]	r4_reg;			/* Operand Register R4 */
	reg [31:0]	r4data_reg;		/* Operand Register R4 Data */
	reg [4:0]	r5_reg;			/* Operand Register R5 */
	reg [31:0]	r5data_reg;		/* Operand Register R5 Data */
	reg [4:0]	r6_reg;			/* Operand Register R6 */
	reg [31:0]	r6data_reg;		/* Operand Register R6 Data */
	reg [2:0]	regcount;		/* Register Data Wait Count */
	reg 			regempty_reg;	/* Register Write Buffer Empty */
	reg			regfull_reg;	/* Register Write Buffer Full */
	reg			regrdy_reg;		/* Register Write Buffer Ready */
	reg			regwrite1_reg;	/* Register Write Port 1 Write Enable */
	reg			regwrite2_reg;	/* Register Write Port 2 Write Enable */
	reg			sll128_reg;		/* Shift Left Logical 128 bits */
	reg [4:0]	state;			/* Execution State Variable */
	reg [31:0]	wrdata1_reg;	/* Register Write Port 1 Data */
	reg [31:0]	wrdata2_reg;	/* Register Write Port 2 Data */
	reg [4:0]	wrreg1_reg;		/* Register Write Port 1 Register Number */
	reg [4:0]	wrreg2_reg;		/* Register Write Port 2 Register Number */
	reg [31:0]	virpc_reg;		/* */

/*****Parameters****************************************************************/

	parameter regwait = 5;

/*****Initialization****************************************************************/

	initial
	begin
		ack_reg = 1'b0;
		cjmpadd_reg = 32'b0;
		dimm_reg = 16'b0;
		dimmsext = 32'b0;
		dne_reg = 1'b1;
		en_reg = 1'b0;
		extnop_ma_reg = 1'b0;
		gr_reg = 1'b0;
		pc_reg = 32'b0;
		pclkcnt = 2'b0;
		pcnext_reg = 1'b0;
		r1_reg = 5'b0;
		r1data_reg = 32'b0;
		r2_reg = 5'b0;
		r2data_reg = 32'b0;
		r3_reg = 5'b0;
		r3data_reg = 32'b0;
		r4_reg = 5'b0;
		r4data_reg = 32'b0;
		r5_reg = 5'b0;
		r5data_reg = 32'b0;
		r6_reg = 5'b0;
		r6data_reg = 32'b0;
		regcount = 3'b0;
		regempty_reg = 1'b0;
		regfull_reg = 1'b0;
		regrdy_reg = 1'b0;
		regwrite1_reg = 1'b0;
		regwrite2_reg = 1'b0;
		sll128_reg = 1'b0;
		state = 5'b0;
		wrdata1_reg = 32'b0;
		wrdata2_reg = 32'b0;
		wrreg1_reg = 5'b0;
		wrreg2_reg = 5'b0;
		virpc_reg = 32'b0;
	end

/*********************************************************************/

	assign ACK 			= 	ack_reg;
	assign DNE 			= 	dne_reg;
	assign PCNEXT 		= 	pcnext_reg;
	assign REGWRITE1 	= 	regwrite1_reg;
	assign REGWRITE2	= 	regwrite2_reg;
	assign RDREG1 		= 	r1_reg;
	assign RDREG2 		= 	r3_reg;
	assign RDREG3 		= 	r5_reg;
	assign RDREG4 		= 	r6_reg;
	assign WRREG1 		= 	wrreg1_reg;
	assign WRREG2 		= 	wrreg2_reg;
	assign WRDATA1 	= 	wrdata1_reg;
	assign WRDATA2 	= 	wrdata2_reg;
	assign CJMPADD 	= 	cjmpadd_reg;
	assign RD 			= 	r3_reg;
	assign RESULT 		= 	r3data_reg;
	assign VIRPC		=	virpc_reg;

/*********************************************************************/

	/* Pipeline Clock Edge Detection */
	always@(posedge CLK)
	begin
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			en_reg <= 1'b0;
			sll128_reg <= 1'b0;
			gr_reg <= 1'b0;
			regrdy_reg <= 1'b0;
			regfull_reg <= 1'b0;
			regempty_reg <= 1'b0;
			extnop_ma_reg <= 1'b0;
		end
		else
		begin
			en_reg <= EN;
			sll128_reg <= SLL128;
			gr_reg <= GR;
			regrdy_reg <= REGRDY;
			regfull_reg <= REGFULL;
			regempty_reg <= REGEMPTY;
			extnop_ma_reg <= EXTNOP_MA;
		end
	end

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			ack_reg <= 1'b0;
		end
		else
		begin
			if (ack_reg & dne_reg && (pclkcnt == 2'b01))
			begin
				/* Lower ACK */
				ack_reg <= 1'b0;
			end
			else if (~dne_reg & ~ack_reg)
			begin
				/* Raise ACK */
				ack_reg <= 1'b1;
			end
		end
	end

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			state <= 5'b0;
			regcount <= 3'b0;
			dne_reg <= 1'b1;
			pcnext_reg <= 1'b0;
			r1_reg <= 5'b0;
			r2_reg <= 5'b0;
			r3_reg <= 5'b0;
			r4_reg <= 5'b0;
			r5_reg <= 5'b0;
			r6_reg <= 5'b0;
			r1data_reg <= 32'b0;
			r2data_reg <= 32'b0;
			r3data_reg <= 32'b0;
			r4data_reg <= 32'b0;
			r5data_reg <= 32'b0;
			r6data_reg <= 32'b0;
			dimm_reg <= 16'b0;
			pc_reg <= 32'b0;
			dimmsext <= 32'b0;
			regwrite1_reg <= 1'b0;
			regwrite2_reg <= 1'b0;
			wrreg1_reg <= 5'b0;
			wrreg2_reg <= 5'b0;
			wrdata1_reg <= 32'b0;
			wrdata2_reg <= 32'b0;
			cjmpadd_reg <= 32'b0;
			virpc_reg <= 32'b0;
		end
		else
		begin
			//if (en_reg) //$display("\nState %x", state);
			case(state)
				5'b00000	:	begin
									/* Initial State */
									regcount <= 3'b0;
									dne_reg <= 1'b1;
									pcnext_reg <= 1'b0;
									r1_reg <= 5'b0;
									r2_reg <= 5'b0;
									r3_reg <= 5'b0;
									r4_reg <= 5'b0;
									r5_reg <= 5'b0;
									r6_reg <= 5'b0;
									r1data_reg <= 32'b0;
									r2data_reg <= 32'b0;
									r3data_reg <= 32'b0;
									r4data_reg <= 32'b0;
									r5data_reg <= 32'b0;
									r6data_reg <= 32'b0;
									dimm_reg <= 16'b0;
									pc_reg <= 32'b0;
									dimmsext <= 32'b0;
									regwrite1_reg <= 1'b0;
									regwrite2_reg <= 1'b0;
									wrreg1_reg <= 5'b0;
									wrreg2_reg <= 5'b0;
									wrdata1_reg <= 32'b0;
									wrdata2_reg <= 32'b0;
									cjmpadd_reg <= 32'b0;
									if (en_reg & sll128_reg)
									begin
										/* Start Execution */
										state <= 5'b00001;
									end
								end
				5'b00001	:	begin
									/* Set Register Read */
									dne_reg <= 1'b0;
									r1_reg <= RT+1;
									r2_reg <= RT;
									r3_reg <= RT+3;
									r4_reg <= RS;
									r5_reg <= RS+1;
									r6_reg <= RS+2;
									dimm_reg <= DIMM;
									pc_reg <= PC;
									virpc_reg <= PC;
									//$display("R1 %x", r1_reg);
									//$display("R2 %x", r2_reg);
									//$display("R3 %x", r3_reg);
									//$display("R4 %x", r4_reg);
									//$display("R5 %x", r5_reg);
									//$display("R6 %x", r6_reg);
									//$display("DIMM %x", dimm_reg);
									//$display("PC %x", pc_reg);
									//$display("VIRPC %x", virpc_reg);
									//$display("CJMPADD %x", cjmpadd_reg);
									state <= 5'b00010;
								end
				5'b00010	:	begin
									/* Wait for Pipeline Resources */
									if (gr_reg)
									begin
										/* Wait for Data */
										if (regcount == regwait)
										begin
											/* Data Ready */
											state <= 4'b0011;
										end
										else
										begin
											/* Waiting for Data */
											regcount <= regcount + 1;
										end
									end
								end
				5'b00011	:	begin
									/* Latch Data */
									r1data_reg <= RDREG1DATA;
									r2data_reg <= RDREG2DATA_ID;
									r3data_reg <= RDREG2DATA;
									r4data_reg <= RDREG1DATA_ID;
									r5data_reg <= RDREG3DATA;
									r6data_reg <= RDREG4DATA;
									//$display("R1 %x %x", r1_reg, r1data_reg);
									//$display("R2 %x %x", r2_reg, r2data_reg);
									//$display("R3 %x %x", r3_reg, r3data_reg);
									//$display("R4 %x %x", r4_reg, r4data_reg);
									//$display("R5 %x %x", r5_reg, r5data_reg);
									//$display("R6 %x %x", r6_reg, r6data_reg);
									state <= 5'b00100;
								end
				5'b00100	:	begin
									/* Shift */
									{r1data_reg,r2data_reg,r4data_reg,r5data_reg} <= {r1data_reg,r2data_reg,r4data_reg,r5data_reg}<<1;
									//$display("R1 %x %x", r1_reg, r1data_reg);
									//$display("R2 %x %x", r2_reg, r2data_reg);
									//$display("R3 %x %x", r3_reg, r3data_reg);
									//$display("R4 %x %x", r4_reg, r4data_reg);
									//$display("R5 %x %x", r5_reg, r5data_reg);
									//$display("R6 %x %x", r6_reg, r6data_reg);
									state <= 5'b00101;
								end
				5'b00101	:	begin
									/* Branch Test */
									if (r1data_reg < r6data_reg)
									begin
										/* Fall Through */
										r3data_reg <= 32'b1;
										//$display("R1 %x %x", r1_reg, r1data_reg);
										//$display("R2 %x %x", r2_reg, r2data_reg);
										//$display("R3 %x %x", r3_reg, r3data_reg);
										//$display("R4 %x %x", r4_reg, r4data_reg);
										//$display("R5 %x %x", r5_reg, r5data_reg);
										//$display("R6 %x %x", r6_reg, r6data_reg);
										state <= 5'b01010;
									end
									else
									begin
										/* Update PC Needed */
										r3data_reg <= 32'b0;
										//$display("R1 %x %x", r1_reg, r1data_reg);
										//$display("R2 %x %x", r2_reg, r2data_reg);
										//$display("R3 %x %x", r3_reg, r3data_reg);
										//$display("R4 %x %x", r4_reg, r4data_reg);
										//$display("R5 %x %x", r5_reg, r5data_reg);
										//$display("R6 %x %x", r6_reg, r6data_reg);
										state <= 5'b00110;
									end
								end
				5'b00110	:	begin
									/* Sign Extend Data Immediate */
									dimmsext <= {{16{dimm_reg[15]}},dimm_reg};
									//$display("DIMM %x", dimm_reg);
									//$display("PC %x", pc_reg);
									//$display("VIRPC %x", virpc_reg);
									//$display("CJMPADD %x", cjmpadd_reg);
									state <= 5'b00111;
								end
				5'b00111	:	begin
									/* Calculate New PC */
									cjmpadd_reg <= dimmsext + pc_reg + 4;
									//$display("DIMM %x", dimm_reg);
									//$display("PC %x", pc_reg);
									//$display("VIRPC %x", virpc_reg);
									//$display("CJMPADD %x", cjmpadd_reg);
									state <= 5'b01000;
								end
				5'b01000	:	begin
									/* Wait for Negedge of PCLK */
									if (gr_reg & (pclkcnt == 2'b10))
									begin
										/* Raise PCNEXT */
										pcnext_reg <= 1'b1;
										state <= 5'b01001;
									end
								end
				5'b01001	:	begin
									/* Wait for Posedge of PCLK */
									if (gr_reg & (pclkcnt == 2'b01))
									begin
										/* Lower PCNEXT */
										pcnext_reg <= 1'b0;
										state <= 5'b01010;
									end
								end
				5'b01010	:	begin
									/* Set Register Write */
									wrreg1_reg <= r1_reg;
									wrreg2_reg <= r2_reg ;
									wrdata1_reg <= r1data_reg;
									wrdata2_reg <= r2data_reg;
									//$display("WR1 %x %x", wrreg1_reg, wrdata1_reg);
									//$display("WR2 %x %x", wrreg2_reg, wrdata2_reg);
									//$display("PC %x", pc_reg);
									//$display("VIRPC %x", virpc_reg);
									//$display("CJMPADD %x", cjmpadd_reg);
									state <= 5'b01011;
								end
				5'b01011	:	begin
									/* Wait for Write Buffer Ready */
									if (gr_reg & regrdy_reg & ~regfull_reg) 
									begin
										/* Raise Regwrite */
										regwrite1_reg <= 1'b1;
										regwrite2_reg <= 1'b1;
										virpc_reg <= pc_reg + 28 /*+4*/;
										//$display("WR1 %x %x", wrreg1_reg, wrdata1_reg);
										//$display("WR2 %x %x", wrreg2_reg, wrdata2_reg);
										//$display("PC %x", pc_reg);
										//$display("VIRPC %x", virpc_reg);
										//$display("CJMPADD %x", cjmpadd_reg);
										state <= 5'b01100;
									end
								end
				5'b01100	:	begin
									/* Lower Regwrite */
									regwrite1_reg <= 1'b0;
									regwrite2_reg <= 1'b0;
									state <= 5'b01101;
								end
				5'b01101	:	begin
									/* Set Register Write */
									wrreg1_reg <= r4_reg;
									wrreg2_reg <= r5_reg;
									wrdata1_reg <= r4data_reg;
									wrdata2_reg <= r5data_reg;
									//$display("WR1 %x %x", wrreg1_reg, wrdata1_reg);
									//$display("WR2 %x %x", wrreg2_reg, wrdata2_reg);
									//$display("PC %x", pc_reg);
									//$display("VIRPC %x", virpc_reg);
									//$display("CJMPADD %x", cjmpadd_reg);
									state <= 5'b01110;
								end
				5'b01110	:	begin
									/* Wait for Write Buffer Ready */
									if (gr_reg & regrdy_reg & ~regfull_reg) 
									begin
										/* Raise Regwrite */
										regwrite1_reg <= 1'b1;
										regwrite2_reg <= 1'b1;
										if (r3data_reg == 32'b0)
										begin
											virpc_reg <= cjmpadd_reg /*+ 4*/;
										end
										else
										begin
											virpc_reg <= pc_reg + 52 /*+ 4*/;
										end
										//$display("WR1 %x %x", wrreg1_reg, wrdata1_reg);
										//$display("WR2 %x %x", wrreg2_reg, wrdata2_reg);
										//$display("PC %x", pc_reg);
										//$display("VIRPC %x", virpc_reg);
										//$display("CJMPADD %x", cjmpadd_reg);
										state <= 5'b01111;
									end
								end
				5'b01111	:	begin
									/* Lower Regwrite */
									regwrite1_reg <= 1'b0;
									regwrite2_reg <= 1'b0;
									state <= 5'b10000;
								end
				5'b10000	:	begin
									/* Wait for Writes to Finish */
									if (gr_reg & regempty_reg & extnop_ma_reg)
									begin
										/* Done */
										dne_reg <= 1'b1;
										state <= 5'b10001;
									end
								end
				5'b10001	:	begin
									/* Wait for instruction to Pipe out */
									if (~en_reg)
									begin
										/* Reset to Initial State */
										state <= 5'b00000;
									end
								end
				5'b10010	:	begin
									pcnext_reg <= 1'b0;
									dne_reg <= 1'b1;
									regwrite1_reg <= 1'b0;
									regwrite2_reg <= 1'b0;
									//$stop;
									if (~EXC)
									begin
										state <= 5'b0;
									end
								end
			endcase
			if (EXC & en_reg) 
			begin
				state <= 5'b10010;
			end
		end
	end

endmodule

/*****Execute -> to Writeback****************************************************************/

module mmldiv64_topipe_wb(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			ACK,				/* Enable Acknowledged */
	input 			CLK,				/* System Clock 50 - 100 MHZ */
	input 			DNE,				/* Execution Done */
	input 			EN_EX,			/* Enable Execute Phase */
	input 			EXCEXT,			/* Exception Flush */
	input 			EXTNOP_MA,		/* Extension Bubble in Memory Access Phase */
	input [31:0]	PC_EX,			/* Current PC Execute Phase */
	input 			PCLK,				/* Pipeline Clock */
	input [4:0]		RD_EX,			/* Destination Register Execution Phase */
	input				REGWRITE_EX,	/* Register Write Execute Phase */
	input 			RESET,			/* System Reset */
	input [31:0]	RESULT_EX,		/* Result Execution Phase */
	/* OUTPUT PORTS */
	output [31:0]	ALURESULT_WB,	/* ALU Result to Writeback Phase */
	output 			BHLS_WB,			/* Byte/Halfword Load/Store to Writeback Phase */
	output [1:0]	DMADD_WB,		/* Least Significant Bits of Data Address to Writeback Phase */
	output [31:0]	DMDATAOUT_WB,	/* Memory Data Out to Writeback Phase */
	output 			MEMTOREG_WB,	/* Memory to Register to Writeback Phase */
	output [31:0]	PC_WB,			/* Current PC to Writeback Phase */
	output 			REGWRITE_WB,	/* Register Write to Writeback Phase */
	output 			REWB,				/* Re-enter at Writeback */
	output 			RNL_WB,			/* Right/Left Unaligned Load/Store to Writeback Phase */
	output [4:0]	WRREG_WB			/* Write Register Number to Writeback Phase */
	);
	
/*****Signals****************************************************************/

	wire EN_WB;		/* Enable to Writeback Phase */
	wire RESET_WB;	/* Reset to Writeback Phase */

/*****Registers****************************************************************/

	reg [70:0] ex_wb;		/* Execute -> to Writeback Pipeline Register */
	reg [1:0] pclkcnt;	/* Pipeline Clock edge detection */
	reg reset_reg;			/* Reset to Writeback Phase */
	reg rewb_reg;			/* Re-enter at Writeback */

/*****Initialization****************************************************************/

	initial
	begin
		ex_wb = 71'b0;
		pclkcnt = 2'b0;
		rewb_reg = 1'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_WB		=	reset_reg;
	assign REWB 			= 	rewb_reg & EN_WB;
	assign EN_WB 			=	ex_wb[70];		//EN_EX;
	assign REGWRITE_WB 	=	ex_wb[69];		//REGWRITE_EX;
	assign MEMTOREG_WB 	=	1'b0;
	assign RNL_WB 			=	1'b0;
	assign BHLS_WB 		=	1'b0;
	assign DMADD_WB 		=	2'b0;
	assign WRREG_WB 		=	ex_wb[68:64];	//RD_EX;
	assign ALURESULT_WB	=	ex_wb[63:32];	//RESULT_EX;
	assign DMDATAOUT_WB 	=	32'b0;
	assign PC_WB			=	ex_wb[31:0];	//PC_EX;

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

	always@(posedge CLK)
	begin
		/* Execute -> to Memory Access Pipeline Register */
		casex({pclkcnt,RESET_WB,EXTNOP_MA,rewb_reg,ACK,DNE,EXCEXT})
			8'bxx0xxxxx	:	begin
									/* Reset */
									rewb_reg <= 1'b0;
									ex_wb <= 71'b0;
								end
			8'b011xxxx1	:	begin
									/* Exception in Pipeline, Flush */
									rewb_reg <= 1'b0;
									ex_wb <= 71'b0;
								end
			8'bxx1x0110	:	begin
									/* Latch Data and Control after Execution Finishes */
									ex_wb <= {EN_EX,REGWRITE_EX,RD_EX,RESULT_EX,PC_EX};
								end
			8'b101100x0	:	begin
									/* Raise REWB at next Negedge of PCLK after ACK Lowers */
									rewb_reg <= 1'b1;
								end
			8'b011x1xx0	:	begin
									/* Lower REWB at next Posedge and reset register */
									rewb_reg <= 1'b0;
									ex_wb <= 71'b0;
								end
			default	:		begin
									/* NOP */
								end
		endcase
	end

endmodule
