/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	timer_controller_ext											*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	Peripheral Extension, Timer							*/
/*																							*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module timer_controller_ext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				SRT,			/* Start Transaction */
	input				TMRCLK,		/* Timer Clock 10 MHZ */
	input				WE,			/* Write Enable */
	input				BAT_EN,		/* */
	input				EXTCLK_EN,	/* */
	input				LDEXT,		/* */
	/* OUTPUT PORTS */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output [31:0]	DATA_OUT,	/* Data Bus to Data Path Memory Interface */
	output			DNE,			/* Transaction Done */
	output [31:0]	STATUS,		/* Control Register */
	output			TDIRQ,		/* Down Counter Interrupt */
	output			TFIRQ,		/* Free Counter Interrupt */
	output			PRESENT,	/* */
	output			WANT_INTR,	/* */
	output			PRIVILEGE,	/* */
	output			MY_ADDR,	/* */
	output			IRQ,		/* */
	/* IO PORTS */
	inout				CLK_EXT		/* External Clock */
	);

/*****Signals****************************************************************/

	wire [3:0]	BM;			/* Byte Selects */
	wire [31:0]	DATA_IN_C;	/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;	/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;	/* Endian Flipped Outgoing Data */
	wire			MEN;			/* Transaction Enabled for Timer Configuration Space */
	wire [31:0]	REG_M;		/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;		/* Selected Configuration Register Data */
	wire [31:0]	WMASK_IN;	/* Incoming Data Write Mask */
	wire [31:0]	WMASK_REG;	/* Configuration Register Data Write Mask */
	wire 			BEN;	/* */
	wire 			CEN;	/* */
	wire [31:0]	BASE_ADDR;	/* */
	wire [31:0]	MAX_ADDR;	/* */
	wire 			AEN;	/* */
	wire [15:0]	MADDR;	/* */
	wire [15:0] CBBASE_ADDR;	/* */
	wire [15:0] CBMAX_ADDR;		/* */

/*****Registers****************************************************************/
	
	reg [31:0]	basetagr;		/* BASETAG Register */
	reg			clr;				/* Clear Interrupt */
	reg [31:0]	control;			/* Control Register */
	reg [31:0]	data_in_m_reg;	/* Masked Incoming Data */
	reg [3:0]	dnecnt;			/* Transaction Progress Counter */
	reg			dner;				/* Configuration Space Transaction Done */
	reg [63:0]	downcount;		/* Down Counter */
	reg [63:0]	freecount;		/* Free Counter */
	reg			irq_td;			/* Down Counter Interrupt */
	reg			irq_tf;			/* Free Counter Interrupt */
	reg [63:0]	loaddown;		/* Load Down Counter */
	reg [63:0]	loadfree;		/* Load Free Counter */
	reg			men_reg;			/* Transaction Enabled for Timer Configuration Space */
	reg			ovf_td;			/* Down Counter Overflow */
	reg			ovr_td;			/* Down Counter Overrun */
	reg [31:0]	reg_m_0_reg;	/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;		/* Masked Configuration Register Data */
	reg			rst;				/* Reset Counters */
	reg			updateh_td;		/* Update Down Counter High */
	reg			updateh_tf;		/* Update Free Counter High */
	reg			updatel_td;		/* Update Down Counter Low */
	reg			updatel_tf;		/* Update Free Counter Low */
	reg			present_reg;	/* */
	reg [3:0]	state;		/* */
	reg [31:0]	bat0_reg;		/* */
	reg [31:0]	size0_reg;		/* */
	reg [31:0]	bat1_reg;		/* */
	reg [31:0]	size1_reg;		/* */
	reg [31:0]	bat2_reg;		/* */
	reg [31:0]	size2_reg;		/* */
	reg [31:0]	bat3_reg;		/* */
	reg [31:0]	size3_reg;		/* */
	reg [31:0]	bat4_reg;		/* */
	reg [31:0]	size4_reg;		/* */
	reg 			ben_reg;	/* */
	reg 			cen_reg;	/* */
	reg			aen_reg;	/* */

/*****Parameters****************************************************************/

	/* Operation Register Addresses */
	parameter	BASETAGREG =		16'h0000;	/* BASETAG Register Relative Address */
	parameter	CONTROLREG =		16'h0004;	/* Control Register Relative Address */
	parameter	FREECOUNT_HIGH =	16'h0008;	/* Free Counter High Register Relative Address */
	parameter	FREECOUNT_LOW =	16'h000C;	/* Free Counter Low Register Relative Address */
	parameter	DOWNCOUNT_HIGH =	16'h0010;	/* Down Counter High Register Relative Address */
	parameter	DOWNCOUNT_LOW =	16'h0014;	/* Down Counter Low Register Relative Address */
	parameter	RESERVED0REG =	16'h0018;
	parameter	RESERVED1REG = 	16'h001c;

	/* BASETAG Register */
	parameter	BASEl =				31;			/* BASE Bit Range Left */
	parameter	BASEr =				16;			/* BASE Bit Range Right */
	parameter 	TAGl =				15;			/* TAG Bit Range Left */
	parameter	TAGr =				0;				/* TAG Bit Range Right */
	parameter	TAG =					7;				/* TAG Value */
	
	/* Control Register */
	parameter	TD_OVERRUN =		11;			/* Down Counter Overrun */
	parameter	TD_OVERFLOW =		10;			/* Down Counter Overflow */
	parameter	TF_INT =				9;				/* Free Counter Interrupt */
	parameter	TD_INT =				8;				/* Down Counter Interrupt */
	parameter	T_CLKSELl =			7;				/* Clock Select Bit Range Left */
	parameter	T_CLKSELr =			6;				/* Clock Select Bit Range Right */
	parameter	TF_INT_ENABLE =	5;				/* Free Counter Interrupt Enable */
	parameter	T_RESET =			4;				/* Soft Reset */
	parameter	T_CLKO =				3;				/* Output Clock */
	parameter	TD_INT_ENABLE =	1;				/* Down Counter Interrupt Enable */ 
	parameter	T_ENABLE =			0;				/* Timer Enable */

	/* Configuration Register Addressess */
	parameter	TAGSTATUSREG =	16'h0008;
	parameter	BAT_OR_SIZE0 =	16'h000c;
	parameter	BAT_OR_SIZE1 =	16'h0010;
	parameter	BAT_OR_SIZE2 =	16'h0014;
	parameter	BAT_OR_SIZE3 =	16'h0018;
	parameter	BAT_OR_SIZE4 =	16'h001c;

	/* TAGSTATUS Register */
	parameter	EC_STATUSl =	23;
	parameter	EC_STATUSr =	16;
	parameter	TAG_STATUS_OFF =	16'h0000;
	parameter	EC_ABSENT =		8'h00;
	parameter	EC_CONFIG =		8'h01;
	parameter	EC_RUN =		8'h02;
	parameter	EC_PM0 =		8'h03;
	parameter	EC_PM1 =		8'h04;
	parameter	EC_PM2 =		8'h05;

	parameter	PSIZE = 32'h20;
	
/*****Initialization****************************************************************/

	initial
	begin
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
		clr = 1'b0;
		control = 32'b0;
		dnecnt = 4'b1111;
		dner = 1'b1;
		downcount = 64'b0;
		freecount = 64'b0;
		irq_td = 1'b0;
		irq_tf = 1'b0;
		loaddown = 64'b0;
		loadfree = 64'b0;
		men_reg = 1'b0;
		ovf_td = 1'b0;
		ovr_td = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
		rst = 1'b1;
		updatel_td = 1'b0;
		updatel_tf = 1'b0;
		updateh_td = 1'b0;
		updateh_tf = 1'b0;
		present_reg = 1'b0;
		state = 3'b0;
		bat0_reg = 32'b0;
		size0_reg = PSIZE;
		bat1_reg = 32'b0;
		size1_reg = 32'b0;
		bat2_reg = 32'b0;
		size2_reg = 32'b0;
		bat3_reg = 32'b0;
		size3_reg = 32'b0;
		bat4_reg = 32'b0;
		size4_reg = 32'b0;
		ben_reg = 1'b0;
		cen_reg = 1'b0;
		aen_reg = 1'b0;
	end

/*********************************************************************/
	
	assign BASETAG = basetagr;
	assign CLK_EXT = (control[T_CLKO])?TMRCLK:1'bz;
	assign DNE = dner;
	assign STATUS = control;
	assign TDIRQ = irq_td & control[T_ENABLE] & control[TD_INT_ENABLE] & ~clr;
	assign TFIRQ = irq_tf & control[T_ENABLE] & control[TF_INT_ENABLE] & ~clr;
	//assign PRESENT = present_reg;
	assign IRQ = TDIRQ | TFIRQ;
	assign WANT_INTR = present_reg;
	assign PRIVILEGE = 1'b0;
	assign MY_ADDR = (men_reg | ben_reg | cen_reg | aen_reg);

/*****Address Decoding****************************************************************/

	assign BASE_ADDR = {bat0_reg[BASEl:1],1'b0};
	assign MAX_ADDR = ({bat0_reg[BASEl:1],1'b0} + PSIZE);
	assign CBBASE_ADDR = TAGSTATUSREG;
	assign CBMAX_ADDR = BAT_OR_SIZE4;
	assign MADDR = ADDR_IN[BASEr-1:0] - {bat0_reg[BASEr-1:1],1'b0};

	assign MEN = (basetagr[EC_STATUSl:EC_STATUSr] == EC_RUN) & (BASE_ADDR <= ADDR_IN) & (MAX_ADDR > ADDR_IN) & bat0_reg[0] & ~BAT_EN;
	assign BEN = (basetagr[EC_STATUSl:EC_STATUSr] == EC_RUN) & (CBBASE_ADDR <= ADDR_IN[BASEr-1:0]) & (CBMAX_ADDR >= ADDR_IN[BASEr-1:0]) & BAT_EN;
	assign CEN = (basetagr[EC_STATUSl:EC_STATUSr] == EC_CONFIG) & (CBBASE_ADDR <= ADDR_IN[BASEr-1:0]) & (CBMAX_ADDR >= ADDR_IN[BASEr-1:0]) & BAT_EN;
	assign AEN = (basetagr[EC_STATUSl:EC_STATUSr] == EC_ABSENT) & (CBBASE_ADDR == ADDR_IN[BASEr-1:0]) & BAT_EN;

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/
	
	assign REG_M_0 = (men_reg & (MADDR == BASETAGREG))?(basetagr & 32'h0000ffff):32'bz;
	assign REG_M_0 = (men_reg & (MADDR == CONTROLREG))?control:32'bz;
	assign REG_M_0 = (men_reg & OE & (MADDR == FREECOUNT_HIGH))?freecount[63:32]:32'bz;
	assign REG_M_0 = (men_reg & OE & (MADDR == FREECOUNT_LOW))?freecount[31:0]:32'bz;
	assign REG_M_0 = (men_reg & WE & (MADDR == FREECOUNT_HIGH))?loadfree[63:32]:32'bz;
	assign REG_M_0 = (men_reg & WE & (MADDR == FREECOUNT_LOW))?loadfree[31:0]:32'bz;
	assign REG_M_0 = (men_reg & OE & (MADDR == DOWNCOUNT_HIGH))?downcount[63:32]:32'bz;
	assign REG_M_0 = (men_reg & OE & (MADDR == DOWNCOUNT_LOW))?downcount[31:0]:32'bz;
	assign REG_M_0 = (men_reg & WE & (MADDR == DOWNCOUNT_HIGH))?loaddown[63:32]:32'bz;
	assign REG_M_0 = (men_reg & WE & (MADDR == DOWNCOUNT_LOW))?loaddown[31:0]:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RESERVED0REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RESERVED1REG))?32'b0:32'bz;

	assign REG_M_0 = ((ben_reg | cen_reg | aen_reg) & (ADDR_IN[BASEr-1:0] == TAGSTATUSREG))?basetagr:32'bz;

	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE0))?bat0_reg:32'bz;
	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE1))?bat1_reg:32'bz;
	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE2))?bat2_reg:32'bz;
	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE3))?bat3_reg:32'bz;
	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE4))?bat4_reg:32'bz;

	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE0))?size0_reg:32'bz;
	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE1))?size1_reg:32'bz;
	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE2))?size2_reg:32'bz;
	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE3))?size3_reg:32'bz;
	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE4))?size4_reg:32'bz;

	assign DATA_OUT = (OE & MY_ADDR)?DATA_OUT_C:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed1(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*********************************************************************/

	key ky(
		.CLK(CLK),
		.CLK_EN(EXTCLK_EN),
		.EXT_LD(LDEXT),
		.PRESENT(present_reg),
		.RESET(RESET),
		.PRESENT_K(PRESENT)
		);

/*********************************************************************/
		
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
			ben_reg = 1'b0;
			cen_reg = 1'b0;
			aen_reg = 1'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
			ben_reg = BEN;
			cen_reg = CEN;
			aen_reg = AEN;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,control[T_RESET],dnecnt[3],dner})
			4'b0xxx	:		begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
								clr = 1'b1;
								control = 32'b0;
								dner = 1'b1;
								loaddown = 64'b0;
								loadfree = 64'b0;
								updateh_td = 1'b0;
								updateh_tf = 1'b0;
								updatel_td = 1'b0;
								updatel_tf = 1'b0;
								rst = 1'b1;
								present_reg = 1'b0;
								state = 3'b0;
								bat0_reg = 32'b0;
								size0_reg = PSIZE;
								bat1_reg = 32'b0;
								size1_reg = 32'b0;
								bat2_reg = 32'b0;
								size2_reg = 32'b0;
								bat3_reg = 32'b0;	
								size3_reg = 32'b0;
								bat4_reg = 32'b0;
								size4_reg = 32'b0;
							end
			4'b11xx	:		begin
								/* Soft Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
								clr = 1'b1;
								control = 32'b0;
								dner = 1'b1;
								loaddown = 64'b0;
								loadfree = 64'b0;
								updateh_td = 1'b0;
								updateh_tf = 1'b0;
								updatel_td = 1'b0;
								updatel_tf = 1'b0;
								rst = 1'b1;
								present_reg = 1'b0;
								state = 3'b0;
								bat0_reg = 32'b0;
								size0_reg = PSIZE;
								bat1_reg = 32'b0;
								size1_reg = 32'b0;
								bat2_reg = 32'b0;
								size2_reg = 32'b0;
								bat3_reg = 32'b0;	
								size3_reg = 32'b0;
								bat4_reg = 32'b0;
								size4_reg = 32'b0;
							end
			4'b1000	:		begin
								/* Update Registers */
								dner = 1'b1;
								if (men_reg & OE)
								begin
									casex(MADDR)
										CONTROLREG	:	begin
														/* Clear Interrupt */
														if (irq_td || irq_tf) clr = 1'b1;
													end
										default	:	begin
														/* Do Nothing */
														/* Address not Recognized */
													end
									endcase
								end
								if (men_reg & WE)
								begin
									casex(MADDR)
										BASETAGREG		:	begin
															/* Update Basetag Register */
															basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
														end
										CONTROLREG		:	begin
															/* Update Control Register */
															control = {(data_in_m_reg[31:12] | reg_m_reg[31:12]),control[11:8],(data_in_m_reg[7:0] | reg_m_reg[7:0])};
														end
										DOWNCOUNT_HIGH	:	begin
															/* Update Down Counter High Register */
															loaddown[63:32] = data_in_m_reg | reg_m_reg;
															updateh_td = 1'b1;
														end
										DOWNCOUNT_LOW	:	begin
															/* Update Down Counter Low Register */
															loaddown[31:0] = data_in_m_reg | reg_m_reg;
															updatel_td = 1'b1;
														end
										FREECOUNT_HIGH	:	begin
															/* Update Free Counter High Register */
															loadfree[63:32] = data_in_m_reg | reg_m_reg;
															updateh_tf = 1'b1;
														end
										FREECOUNT_LOW	:	begin
															/* Update Free Counter Low Register */
															loadfree[31:0] = data_in_m_reg | reg_m_reg;
															updatel_tf = 1'b1;
														end
										default		:	begin
															/* Do Nothing */
															/* Address not Recognized */
														end
									endcase
								end
								if ((cen_reg | ben_reg) & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										TAGSTATUSREG	:	begin
															basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
														end
										BAT_OR_SIZE0	:	begin
															bat0_reg = data_in_m_reg | reg_m_reg;
														end
										BAT_OR_SIZE1	:	begin
															bat1_reg = data_in_m_reg | reg_m_reg;
														end
										BAT_OR_SIZE2	:	begin
															bat2_reg = data_in_m_reg | reg_m_reg;
														end
										BAT_OR_SIZE3	:	begin
															bat3_reg = data_in_m_reg | reg_m_reg;
														end
										BAT_OR_SIZE4	:	begin
															bat4_reg = data_in_m_reg | reg_m_reg;
														end										
										default		:	begin
														end
									endcase
								end
								if (aen_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										TAGSTATUSREG	:	begin
															basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
														end
										default		:	begin
														end
									endcase
								end
								if ((men_reg | ben_reg | cen_reg | aen_reg) & OE)
								begin
									//$display("Reading %x %x %x", ADDR_IN, reg_m_0_reg, DATA_OUT);
									//if (men_reg) //$stop;
								end
								if ((men_reg | ben_reg | cen_reg | aen_reg) & WE)
								begin
									//$display("Writing %x %x %x", ADDR_IN, DATA_IN_C, (data_in_m_reg | reg_m_reg));
									//$stop;
								end 
							end
			4'b10x1	:		begin
								/* Wait for incoming request */
								if (SRT & (men_reg | ben_reg | cen_reg | aen_reg))
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
								casex(state)
									4'b0000	:	begin
													basetagr = 32'b0;
													basetagr[TAGl:TAGr] = TAG;
													basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
													present_reg = 1'b0;
													dner = 1'b1;
													clr = 1'b1;
													control = 32'b0;
													loaddown = 64'b0;
													loadfree = 64'b0;
													updateh_td = 1'b0;
													updateh_tf = 1'b0;
													updatel_td = 1'b0;
													updatel_tf = 1'b0;
													rst = 1'b1;
													bat0_reg = 32'b0;
													size0_reg = PSIZE;
													bat1_reg = 32'b0;
													size1_reg = 32'b0;
													bat2_reg = 32'b0;
													size2_reg = 32'b0;
													bat3_reg = 32'b0;	
													size3_reg = 32'b0;
													bat4_reg = 32'b0;
													size4_reg = 32'b0;
													if (EXTCLK_EN)
													begin
														basetagr[EC_STATUSl:EC_STATUSr] = EC_CONFIG;
														present_reg = 1'b1;
														state = 4'b1001;
													end
												end
									4'b1000	:	begin
													basetagr = 32'b0;
													basetagr[TAGl:TAGr] = TAG;
													basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
													present_reg = 1'b0;
													dner = 1'b1;
													clr = 1'b1;
													control = 32'b0;
													loaddown = 64'b0;
													loadfree = 64'b0;
													updateh_td = 1'b0;
													updateh_tf = 1'b0;
													updatel_td = 1'b0;
													updatel_tf = 1'b0;
													rst = 1'b1;
													bat0_reg = 32'b0;
													size0_reg = PSIZE;
													bat1_reg = 32'b0;
													size1_reg = 32'b0;
													bat2_reg = 32'b0;
													size2_reg = 32'b0;
													bat3_reg = 32'b0;	
													size3_reg = 32'b0;
													bat4_reg = 32'b0;
													size4_reg = 32'b0;
													state = {1'b1,basetagr[EC_STATUSr+2:EC_STATUSr]};
												end
									4'b1001	:	begin
													state = {1'b1,basetagr[EC_STATUSr+2:EC_STATUSr]};
												end
									4'b1010	:	begin
													state = {1'b1,basetagr[EC_STATUSr+2:EC_STATUSr]};
													if (clr && (~irq_td && ~irq_tf))
													begin
														/* Interrupt Cleared */
														clr = 1'b0;
													end
													if ((downcount == 64'b0) && (freecount == 64'b0) && (rst))
													begin
														/* Reset Counters */
														rst = 1'b0;
													end
													if (downcount[31:0] == loaddown[31:0])
													begin
														/* Down Counter Low Register Updated */
														updatel_td = 1'b0;
													end
													if (freecount[31:0] == loadfree[31:0])
													begin
														/* Free Counter Low Register Updated */
														updatel_tf = 1'b0;
													end
													if (downcount[63:32] == loaddown[63:32])
													begin
														/* Down Counter High Register Updated */
														updateh_td = 1'b0;
													end
													if (freecount[63:32] == loadfree[63:32])
													begin
														/* Free Counter High Register Updated */
														updateh_tf = 1'b0;
													end
													if (irq_td)
													begin
														/* Down Counter Interrupt */
														control[TD_INT] = 1'b1;
													end
													else
													begin
														control[TD_INT] = 1'b0;
													end
													if (irq_tf)
													begin
														/* Free Counter Interrupt */
														control[TF_INT] = 1'b1;
													end
													else
													begin
														control[TF_INT] = 1'b0;
													end
													if (ovf_td)
													begin
														/* Down Counter Overflow */
														control[TD_OVERFLOW] = 1'b1;
													end
													else
													begin
														control[TD_OVERFLOW] = 1'b0;
													end
													if (ovr_td)
													begin
														/* Down Counter Overrun */
														control[TD_OVERRUN] = 1'b1;
													end
													else
													begin
														control[TD_OVERRUN] = 1'b0;
													end
												end
									default	:	begin
													state = 4'b0;
												end
								endcase
							end
			default	:		begin
								/* Do Nothing */
							end
		endcase
	end
	
	/* Free Counter */
	always@(posedge TMRCLK)
	begin
		if (rst)
		begin
			/* Reset */
			irq_tf = 1'b0;
			freecount = 64'b0;
		end
		else
		begin
			if (updatel_tf || updateh_tf)
			begin
				/* Update Counter */
				if (updatel_tf) freecount[31:0] = loadfree[31:0];
				if (updateh_tf) freecount[63:32] = loadfree[63:32];
			end
			else
			begin
				if (control[T_ENABLE])
				begin
					if ((clr) && (irq_tf))
					begin
						/* Clear Interrupt */
						irq_tf = 1'b0;
					end
					if (freecount == 64'hffffffffffffffff)
					begin
						/* Set Interrupt */
						irq_tf = 1'b1;
					end
					/* Increment Counter */
					freecount = freecount + 1;
				end
			end
		end
	end
	
	/* Down Counter */
	always@(posedge TMRCLK)
	begin
		if (rst)
		begin
			/* Reset */
			irq_td = 1'b0;
			ovr_td = 1'b0;
			ovf_td = 1'b0;
			downcount = 64'b0;
		end
		else
		begin
			if (updatel_td || updateh_td)
			begin
				/* Update Counter */
				if (updatel_td) downcount[31:0] = loaddown[31:0];
				if (updateh_td) downcount[63:32] = loaddown[63:32];
			end
			else
			begin
				if (control[T_ENABLE])
				begin
					if ((clr) && (irq_td))
					begin
						/* Clear Interrupt */
						irq_td = 1'b0;
						ovf_td = 1'b0;
						ovr_td = 1'b0;
					end
					if (ovf_td & (downcount == loaddown))
					begin
						/* Set Overrun */
						ovr_td = 1'b1;
					end
					if (irq_td & (downcount == loaddown))
					begin
						/* Set Overflow */
						ovf_td = 1'b1;
					end
					if (downcount == 64'b0)
					begin
						/* Set Interrupt */
						irq_td = 1'b1;
					end
					/* Decrement Counter */
					downcount = downcount - 1;
				end
			end
		end
	end
	
endmodule
