/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Giovanni Busonera													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	usart_controller_ext											*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	Peripheral Extension, USART							*/
/*																							*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module usart_controller_ext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				RX,			/* Serial Receive */
	input				SRT,			/* Start Transaction */
	input				USARTCLK,	/* USART Clock 100 MHZ */
	input				WE,			/* Write Enable */
	input				BAT_EN,		/* */
	input				EXTCLK_EN,	/* */
	input				LDEXT,		/* */
	/* OUTPUT PORTS */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output [31:0]	DATA_OUT,	/* Data Bus to Data Path Memory Interface */
	output			DNE,			/* Transaction Done */
	output			SCLK,			/* Baud Clock */
	output [31:0]	STATUS,		/* Control Register */
	output			TX,			/* Serial Transmit */
	output			USART_IRQ,	/* USART Interrupt */
	output			PRESENT,	/* */
	output			WANT_INTR,	/* */
	output			PRIVILEGE,	/* */
	output			MY_ADDR	/* */
	);

/*****Signals****************************************************************/

	wire			BAUD_CLK;	/* Baud Clock */
	wire [3:0]	BM;			/* Byte Selects */
	wire [31:0]	DATA_IN_C;	/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;	/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;	/* Endian Flipped Outgoing Data */
	wire [31:0]	REG_M;		/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;		/* Selected Configuration Register Data */
	wire [7:0]	RXDATAW;		/* Received Data */
	wire			RXRDY;		/* Receiver Ready */
	wire			TXRDY;		/* Transmitter Ready */
	wire [31:0]	WMASK_IN;	/* Incoming Data Write Mask */
	wire [31:0]	WMASK_REG;	/* Configuration Register Data Write Mask */
	wire 			TXBSY;	/* */
	wire 			BEN;	/* */
	wire 			CEN;	/* */
	wire [31:0]	BASE_ADDR;	/* */
	wire [31:0]	MAX_ADDR;	/* */
	wire 			AEN;	/* */
	wire [15:0]	MADDR;	/* */
	wire [15:0] CBBASE_ADDR;	/* */
	wire [15:0] CBMAX_ADDR;		/* */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;			/* BASETAG Register */
	reg [31:0]	channelstatus;		/* Channel Status Register */
	reg [31:0]	control;				/* Control Register */
	reg [31:0]	data_in_m_reg;		/* Masked Incoming Data */
	reg [3:0]	dnecnt;				/* Transaction Progress Counter */
	reg			dner;					/* Configuration Space Transaction Done */
	reg [31:0]	intrdisable;		/* Interrupt Disable Register */
	reg [31:0]	intrenable;			/* Interrupt Enable Register */
	reg [31:0]	intrmask;			/* Interrupt Mask Register */
	reg			men_reg;				/* Transaction Enabled for USART Configuration Space */
	reg [31:0]	reg_m_0_reg;		/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;			/* Masked Configuration Register Data */
	reg			rxd;					/* Received Data flag */
	reg [31:0]	rxdata;				/* Receive Register */
	reg			tsrt;					/* Transmit Start */
	reg [31:0]	txdata;				/* Transmit Register */
	reg [1:0]	txrdycnt;			/* Transmiter Ready Edge Detection */
	reg			present_reg;	/* */
	reg [3:0]	state;		/* */
	reg [31:0]	bat0_reg;		/* */
	reg [31:0]	size0_reg;		/* */
	reg [31:0]	bat1_reg;		/* */
	reg [31:0]	size1_reg;		/* */
	reg [31:0]	bat2_reg;		/* */
	reg [31:0]	size2_reg;		/* */
	reg [31:0]	bat3_reg;		/* */
	reg [31:0]	size3_reg;		/* */
	reg [31:0]	bat4_reg;		/* */
	reg [31:0]	size4_reg;		/* */
	reg 			ben_reg;	/* */
	reg 			cen_reg;	/* */
	reg			aen_reg;	/* */
	reg [1:0]	rdycnt;

/*****Parameters****************************************************************/

	/* Operation Register Addresses */
	parameter	BASETAGREG =			16'h0000;	/* BASETAG Register Relative Address */
	parameter	CONTROLREG =			16'h0004;	/* Control Register Relative Address */
	parameter	INTRENABLEREG =		16'h0008;	/* Interrupt Enable Register Relative Address */
	parameter	INTRDISABLEREG =		16'h000c;	/* Interrupt Disable Register Relative Address */
	parameter	INTRMASK =				16'h0010;	/* Interrupt Mask Register Relative Address */
	parameter	CHANNELSTATUSREG =	16'h0014;	/* Channel Status Register Relative Address */
	parameter	RXDATAREG = 			16'h0018;	/* Recieve Register Relative Address */
	parameter	TXDATAREG =				16'h001c;	/* Transmit Register Relative Address */
	parameter	BAUDREG =				16'h0020;	/* Baud Register Relative Address */
	parameter	TIMEOUTREG =			16'h0024;	/* Timeout Register Relative Address */
	parameter	RESERVED0REG =			16'h0028;
	parameter	RESERVED1REG =			16'h002c;
	parameter	RESERVED2REG =			16'h0030;
	parameter	RESERVED3REG =			16'h0034;
	parameter	RESERVED4REG =			16'h0038;
	parameter	RESERVED5REG =			16'h003c;

	/* BASETAG Register */
	parameter	BASEl =					31;			/* BASE Bit Range Left */
	parameter	BASEr =					16;			/* BASE Bit Range Right */
	parameter	TAGl = 					15;			/* TAG Bit Range Left */
	parameter	TAGr = 					0;				/* TAG Bit Range Right */
	parameter	TAG =						6;				/* TAG Value */

	/* Control Register */
	parameter	CHMODEl =				31;			/* Channel Mode Bit Range Left */
	parameter	CHMODEr =				30;			/* Channel Mode Bit Range Right */
	parameter	NBSTOPl =				29;			/* Stop Bits Bit Range Left */
	parameter	NBSTOPr =				28;			/* Stop Bits Bit Range Right */
	parameter	USCLKSl =				27;			/* Use Clock Bit Range Left */
	parameter	USCLKSr =				24;			/* Use Clock Bit Range Right */
	parameter	CHRLl =					23;			/* Character Length Bit Range Left */
	parameter	CHRLr =					22;			/* Character Length Bit Range Right */
	parameter	PARl = 					21;			/* Parity Type Bit Range Left */
	parameter	PARr = 					19;			/* Parity Type Bit Range Right */
	parameter	CLKO =					18;			/* Output Clock */
	parameter	MODE9 =					17;			/* 9 bit Mode */
	parameter	SENDA = 					16;			/* Send Address */
	parameter	STTTO = 					11;			/* Start Time-Out */
	parameter	STPBRK =					10;			/* Stop Break */
	parameter	STTBRK = 				9;				/* Start Break */
	parameter	RSTSTA = 				8;				/* Reset Status Bits */
	parameter	TXDIS = 					7;				/* Disable Transmitter */
	parameter	TXEN = 					6;				/* Enable Transmitter */
	parameter	RXDIS =					5;				/* Disable Receiver */
	parameter	RXEN	=					4;				/* Enable Receiver */
	parameter	RSTTX = 					3;				/* Reset Transmitter */
	parameter	RSTRX = 					2;				/* Reset Receiver */
	parameter	RST = 					0;				/* Reset Usart */

	/* Interrupt, Channel Status Register */
	parameter	TXEMPTY =				9;				/* Transmitter Empty */
	parameter	TIMEOUTINT =			8;				/* Time-out */
	parameter	PAREINT =				7;				/* Parity Error */
	parameter	FRAMEINT =				6;				/* Frame Error */
	parameter	OVRE =					5;				/* Overrun Error */
	parameter	ENDTX =					4;				/* End of Transmit */
	parameter	ENDRX =					3;				/* End of Receive */
	parameter	RXBRK =					2;				/* Recieve Break */
	parameter	TXRDYINT =				1;				/* Transmitter Ready */
	parameter	RXRDYINT = 				0;				/* Receiver Ready */

	/* Configuration Register Addressess */
	parameter	TAGSTATUSREG =	16'h0008;
	parameter	BAT_OR_SIZE0 =	16'h000c;
	parameter	BAT_OR_SIZE1 =	16'h0010;
	parameter	BAT_OR_SIZE2 =	16'h0014;
	parameter	BAT_OR_SIZE3 =	16'h0018;
	parameter	BAT_OR_SIZE4 =	16'h001c;

	/* TAGSTATUS Register */
	parameter	EC_STATUSl =	23;
	parameter	EC_STATUSr =	16;
	parameter	TAG_STATUS_OFF =	16'h0000;
	parameter	EC_ABSENT =		8'h00;
	parameter	EC_CONFIG =		8'h01;
	parameter	EC_RUN =		8'h02;
	parameter	EC_PM0 =		8'h03;
	parameter	EC_PM1 =		8'h04;
	parameter	EC_PM2 =		8'h05;

	parameter	PSIZE = 32'h40;

/*****Variables****************************************************************/

	integer i;

/*****Initialization****************************************************************/

	initial
	begin
		channelstatus = 32'b0;
		control = 32'b0;
		control[CHRLl:CHRLr] = 2'b11;
		control[PARl:PARr] = 3'b100;
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		dnecnt = 4'b1111;
		dner = 1'b1;
		intrdisable = 32'b0;
		intrenable = 32'b0;
		intrmask = 32'b0;
		men_reg = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
		rxd = 1'b0;
		rxdata = 32'b0;
		tsrt = 1'b0;
		txdata = 32'b0;
		txrdycnt = 2'b0;
		present_reg = 1'b0;
		state = 3'b0;
		bat0_reg = 32'b0;
		size0_reg = PSIZE;
		bat1_reg = 32'b0;
		size1_reg = 32'b0;
		bat2_reg = 32'b0;
		size2_reg = 32'b0;
		bat3_reg = 32'b0;
		size3_reg = 32'b0;
		bat4_reg = 32'b0;
		size4_reg = 32'b0;
		ben_reg = 1'b0;
		cen_reg = 1'b0;
		aen_reg = 1'b0;
		rdycnt = 2'b11;
	end

/*********************************************************************/

	assign BASETAG		=	basetagr;
	assign DNE			=	dner;
	assign SCLK			=	BAUD_CLK;
	assign STATUS		=	channelstatus;
	assign USART_IRQ	=	(| (channelstatus & intrmask));
	assign TXRDY		=	~TXBSY;
	assign WANT_INTR = present_reg;
	assign PRIVILEGE = 1'b0;
	assign MY_ADDR = (men_reg | ben_reg | cen_reg | aen_reg);

/*****Address Decoding****************************************************************/

	assign BASE_ADDR = {bat0_reg[BASEl:1],1'b0};
	assign MAX_ADDR = ({bat0_reg[BASEl:1],1'b0} + PSIZE);
	assign CBBASE_ADDR = TAGSTATUSREG;
	assign CBMAX_ADDR = BAT_OR_SIZE4;
	assign MADDR = ADDR_IN[BASEr-1:0] - {bat0_reg[BASEr-1:1],1'b0};

	assign MEN = (basetagr[EC_STATUSl:EC_STATUSr] == EC_RUN) & (BASE_ADDR <= ADDR_IN) & (MAX_ADDR > ADDR_IN) & bat0_reg[0] & ~BAT_EN;
	assign BEN = (basetagr[EC_STATUSl:EC_STATUSr] == EC_RUN) & (CBBASE_ADDR <= ADDR_IN[BASEr-1:0]) & (CBMAX_ADDR >= ADDR_IN[BASEr-1:0]) & BAT_EN;
	assign CEN = (basetagr[EC_STATUSl:EC_STATUSr] == EC_CONFIG) & (CBBASE_ADDR <= ADDR_IN[BASEr-1:0]) & (CBMAX_ADDR >= ADDR_IN[BASEr-1:0]) & BAT_EN;
	assign AEN = (basetagr[EC_STATUSl:EC_STATUSr] == EC_ABSENT) & (CBBASE_ADDR == ADDR_IN[BASEr-1:0]) & BAT_EN;

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG	=	{{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN	=	{{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M	=	DATA_IN_C & WMASK_IN;
	assign REG_M		=	reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/
	
	assign REG_M_0 = (men_reg & (MADDR == BASETAGREG))?(basetagr & 32'h0000ffff):32'bz;
	assign REG_M_0 = (men_reg & (MADDR == CONTROLREG))?control:32'bz;
	assign REG_M_0 = (men_reg & OE & (MADDR == INTRENABLEREG))?intrmask:32'bz;
	assign REG_M_0 = (men_reg & WE & (MADDR == INTRENABLEREG))?intrenable:32'bz;
	assign REG_M_0 = (men_reg & OE & (MADDR == INTRDISABLEREG))?intrmask:32'bz;
	assign REG_M_0 = (men_reg & WE & (MADDR == INTRDISABLEREG))?intrdisable:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == CHANNELSTATUSREG))?channelstatus:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RXDATAREG))?rxdata:32'bz;
	assign REG_M_0 = (men_reg & OE & (MADDR == TXDATAREG))?32'hffffffff:32'bz;
	assign REG_M_0 = (men_reg & WE & (MADDR == TXDATAREG))?txdata:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == INTRMASK))?intrmask:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == BAUDREG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == TIMEOUTREG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RESERVED0REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RESERVED1REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RESERVED2REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RESERVED3REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RESERVED4REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (MADDR == RESERVED5REG))?32'b0:32'bz;

	assign REG_M_0 = ((ben_reg | cen_reg | aen_reg) & (ADDR_IN[BASEr-1:0] == TAGSTATUSREG))?basetagr:32'bz;

	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE0))?bat0_reg:32'bz;
	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE1))?bat1_reg:32'bz;
	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE2))?bat2_reg:32'bz;
	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE3))?bat3_reg:32'bz;
	assign REG_M_0 = (ben_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE4))?bat4_reg:32'bz;

	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE0))?size0_reg:32'bz;
	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE1))?size1_reg:32'bz;
	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE2))?size2_reg:32'bz;
	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE3))?size3_reg:32'bz;
	assign REG_M_0 = (cen_reg & (ADDR_IN[BASEr-1:0] == BAT_OR_SIZE4))?size4_reg:32'bz;

	assign DATA_OUT = (OE & MY_ADDR)?DATA_OUT_C:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed1(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*********************************************************************/

	key ky(
		.CLK(CLK),
		.CLK_EN(EXTCLK_EN),
		.EXT_LD(LDEXT),
		.PRESENT(present_reg),
		.RESET(RESET),
		.PRESENT_K(PRESENT)
		);

/*****USART****************************************************************/

	uart_ext u0(
		.clk(USARTCLK),
		.rst(~(RESET & ~control[RST])),
		.RxD(RX),
		.TxD_Start(tsrt),
		.TxD_Data(txdata[7:0]),
		.TxD(TX),
		.Ready(RXRDY),
		.busy(TXBSY),
		.Byte_Data(RXDATAW)
		);


/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
			ben_reg = 1'b0;
			cen_reg = 1'b0;
			aen_reg = 1'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
			ben_reg = BEN;
			cen_reg = CEN;
			aen_reg = AEN;
		end
	end

	always@(posedge CLK)
	begin
		if (rdycnt == 2'b01)
		begin
			/* Received Byte */
			rxd = 1'b1;
		end
		if (tsrt & ~TXRDY)
		begin
			/* Transmitter Started */
			tsrt = 1'b0;
		end
		rdycnt = {rdycnt[0],RXRDY};
		casex({RESET,control[RST],dnecnt[3],dner})
			4'b0xxx	:		begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
								dner = 1'b1;
								channelstatus = 32'b0;
								control = 32'b0;
								control[CHRLl:CHRLr] = 2'b11;
								control[PARl:PARr] = 3'b100;
								intrdisable = 32'b0;
								intrenable = 32'b0;
								intrmask = 32'b0;
								rxd = 1'b0;
								rxdata = 32'b0;
								tsrt = 1'b0;
								txrdycnt = 2'b0;
								txdata = 32'b0;
								present_reg = 1'b0;
								state = 3'b0;
								bat0_reg = 32'b0;
								size0_reg = PSIZE;
								bat1_reg = 32'b0;
								size1_reg = 32'b0;
								bat2_reg = 32'b0;
								size2_reg = 32'b0;
								bat3_reg = 32'b0;	
								size3_reg = 32'b0;
								bat4_reg = 32'b0;
								size4_reg = 32'b0;
							end
			4'b11xx	:		begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
								dner = 1'b1;
								channelstatus = 32'b0;
								control = 32'b0;
								control[CHRLl:CHRLr] = 2'b11;
								control[PARl:PARr] = 3'b100;
								intrdisable = 32'b0;
								intrenable = 32'b0;
								intrmask = 32'b0;
								rxd = 1'b0;
								rxdata = 32'b0;
								tsrt = 1'b0;
								txrdycnt = 2'b0;
								txdata = 32'b0;
								present_reg = 1'b0;
								state = 3'b0;
								bat0_reg = 32'b0;
								size0_reg = PSIZE;
								bat1_reg = 32'b0;
								size1_reg = 32'b0;
								bat2_reg = 32'b0;
								size2_reg = 32'b0;
								bat3_reg = 32'b0;	
								size3_reg = 32'b0;
								bat4_reg = 32'b0;
								size4_reg = 32'b0;
							end
			4'b1000	:		begin
								/* Update Registers */
								dner = 1'b1;
								if (men_reg & OE)
								begin
									casex(MADDR)
										RXDATAREG	:	begin
														/* Clear Receiver Ready */
														channelstatus[RXRDYINT] = 1'b0;
													end
										default	:	begin
														/* Do Nothing */
														/* Address not Recognized */
													end
									endcase
								end
								if (men_reg & WE)
								begin
									casex(MADDR)
										BASETAGREG		:	begin
															/* Update Basetag Register */
															basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
														end
										CONTROLREG		:	begin
															/* Update Control Register */
															control = data_in_m_reg | reg_m_reg;
														end
										INTRENABLEREG	:	begin
															/* Update Interrupt Enable Register */
															intrenable = data_in_m_reg | reg_m_reg;
														end
										INTRDISABLEREG	:	begin
															/* Update Interrupt Disable Register */
															intrdisable = data_in_m_reg | reg_m_reg;
														end
										TXDATAREG		:	begin
															/* Update Transmit Register */
															txdata = data_in_m_reg | reg_m_reg;
															/* Start Transmitter */
															tsrt = 1'b1;
														end
										default		:	begin
															/* Do Nothing */
															/* Address not Recognized */
														end
									endcase
								end
								if ((cen_reg | ben_reg) & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										TAGSTATUSREG	:	begin
															basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
														end
										BAT_OR_SIZE0	:	begin
															bat0_reg = data_in_m_reg | reg_m_reg;
														end
										BAT_OR_SIZE1	:	begin
															bat1_reg = data_in_m_reg | reg_m_reg;
														end
										BAT_OR_SIZE2	:	begin
															bat2_reg = data_in_m_reg | reg_m_reg;
														end
										BAT_OR_SIZE3	:	begin
															bat3_reg = data_in_m_reg | reg_m_reg;
														end
										BAT_OR_SIZE4	:	begin
															bat4_reg = data_in_m_reg | reg_m_reg;
														end										
										default		:	begin
														end
									endcase
								end
								if (aen_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										TAGSTATUSREG	:	begin
															basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
														end
										default		:	begin
														end
									endcase
								end
								//if ((WE | OE) & (men_reg /*| ben_reg | cen_reg | aen_reg*/))
								//begin
								//	$stop;
								//end
							end
			4'b10x1	:		begin
								/* Wait for incoming request */
								if (SRT & (men_reg | ben_reg | cen_reg | aen_reg))
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
								casex(state)
									4'b0000	:	begin
													basetagr = 32'b0;
													basetagr[TAGl:TAGr] = TAG;
													basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
													dner = 1'b1;
													channelstatus = 32'b0;
													control = 32'b0;
													control[CHRLl:CHRLr] = 2'b11;
													control[PARl:PARr] = 3'b100;
													intrdisable = 32'b0;
													intrenable = 32'b0;
													intrmask = 32'b0;
													rxd = 1'b0;
													rxdata = 32'b0;
													tsrt = 1'b0;
													txrdycnt = 2'b0;
													txdata = 32'b0;
													present_reg = 1'b0;
													state = 3'b0;
													bat0_reg = 32'b0;
													size0_reg = PSIZE;
													bat1_reg = 32'b0;
													size1_reg = 32'b0;
													bat2_reg = 32'b0;
													size2_reg = 32'b0;
													bat3_reg = 32'b0;	
													size3_reg = 32'b0;
													bat4_reg = 32'b0;
													size4_reg = 32'b0;
													if (EXTCLK_EN)
													begin
														basetagr[EC_STATUSl:EC_STATUSr] = EC_CONFIG;
														present_reg = 1'b1;
														state = 4'b1001;
													end
												end
									4'b1000	:	begin
													basetagr = 32'b0;
													basetagr[TAGl:TAGr] = TAG;
													basetagr[EC_STATUSl:EC_STATUSr] = EC_ABSENT;
													dner = 1'b1;
													channelstatus = 32'b0;
													control = 32'b0;
													control[CHRLl:CHRLr] = 2'b11;
													control[PARl:PARr] = 3'b100;
													intrdisable = 32'b0;
													intrenable = 32'b0;
													intrmask = 32'b0;
													rxd = 1'b0;
													rxdata = 32'b0;
													tsrt = 1'b0;
													txrdycnt = 2'b0;
													txdata = 32'b0;
													present_reg = 1'b0;
													state = 3'b0;
													bat0_reg = 32'b0;
													size0_reg = PSIZE;
													bat1_reg = 32'b0;
													size1_reg = 32'b0;
													bat2_reg = 32'b0;
													size2_reg = 32'b0;
													bat3_reg = 32'b0;	
													size3_reg = 32'b0;
													bat4_reg = 32'b0;
													size4_reg = 32'b0;
													state = {1'b1,basetagr[EC_STATUSr+2:EC_STATUSr]};
												end
									4'b1001	:	begin
													state = {1'b1,basetagr[EC_STATUSr+2:EC_STATUSr]};
												end
									4'b1010	:	begin
													state = {1'b1,basetagr[EC_STATUSr+2:EC_STATUSr]};
													if (RXRDY & channelstatus[RXRDYINT])
													begin
														/* Overrun Error */
														channelstatus[OVRE] = 1'b1;
													end
													if (rxd)
													begin
														/* Recieved Byte */
														rxd = 1'b0;
														channelstatus[RXRDYINT] = 1'b1;
														/* Update Receive Register */
														rxdata[7:0] = RXDATAW;
													end
													else
													begin
														channelstatus[ENDRX] = 1'b0;
													end
													if (TXRDY)
													begin
														/* Transmitter Ready */
														channelstatus[TXRDYINT] = 1'b1;
													end
													else
													begin
														channelstatus[TXEMPTY] = 1'b0;
														channelstatus[ENDTX] = 1'b0;
														channelstatus[TXRDYINT] = 1'b0;
													end
													if (control[RSTSTA])
													begin
														/* Reset Status Bits */
														channelstatus[PAREINT] = 1'b0;
														channelstatus[FRAMEINT] = 1'b0;
														channelstatus[OVRE] = 1'b0;
														channelstatus[RXBRK] = 1'b0;
														control[RSTSTA] = 1'b0;
													end
													/* Update Interrupt Mask */
													for (i=0; i<32; i=i+1)
													begin
														if (intrenable[i]) intrmask[i] = 1'b1;
													end
													intrenable = 32'b0;
													for (i=0; i<32; i=i+1)
													begin
														if (intrdisable[i]) intrmask[i] = 1'b0;
													end
													intrdisable = 32'b0;
												end
									default	:	begin
													state = 4'b0;
												end
								endcase
							end
			default	:		begin
								/* Do Nothing */
							end
		endcase
	end

endmodule