/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	TISA																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	TISA encapsulates all the modules in the eMIPS microprocessor 		*/
/*	system that makes up the Trusted ISA.  These modules include the	*/
/*	pipeline stages and supporting logic as well as the memory			*/
/*	subsystem including peripheral drivers and busses.						*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	eBug Extension, Extension Peripherals, Bug		*/
/*							Fixes, see Manual											*/
/*																							*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"	/* Constant Definitiions */

`define NVALUES		74
`define test_size		0
`define test_code		1
`define instruction	2
`define r0			3
`define r1			4
`define r31			34
`define hireg		36
`define loreg		35
`define cp0_0		37
`define cp0_31		68
`define pc			69
`define membase		70
`define memcount		71
`define memvalue		72
`define memvalue2		73

module TISA(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			ACK_EXT0e,				/* Extension 0 Enable Acknowledged External */
	input 			BLS_EXT0_IN,			/* Extension 0 Byte Load/Store External */
	input 			BRAMCLK,					/* Blockram Clock 50 - 100 MHZ */
	input 			BRDY,						/* System Ace Buffer Ready */
	input 			CLK_IN_PER,				/* System Ace Clock 33 MHZ Buffered */
	input 			DCMLOCK,					/* Digital Clock Manager Lock */
	input 			EXC_EXT0_IN,			/* Extension 0 Exception M External */
	input 			EXC_EXT0e,				/* Extension 0 Exception External */
	input [31:0]	EXTADD_EXT0e,			/* Extension 0 Extension Address External */
													/* Multiplexed: */
													/*	Extension 0 Next PC */
													/*	Extension 0 Exception Address */
													/*	Extension 0 PC Execute Phase */
													/*	Extension 0 PC Memory Access Phase */
	input 			FLASHCLK,				/* Flash Memory Interface Clock 50 - 100 MHZ */
	input [31:0]	GPIO_OUT,				/* GPIO data out of tristate buffer in to FPGA */
	input 			HLS_EXT0_IN,			/* Extension 0 Halfword Load/Store External */
	input 			INT_EXT0_IN,			/* Extension 0 Interrupt External */
	input 			IRQ_SYSACE,				/* System Ace Interrupt Request */
	input [31:0]	MADDR_EXT0_IN,			/* Extension 0 Memory Address External */
	input [31:0]	MDATA_EXT0_IN,			/* Extension 0 Memory Data Out External */
													/* Multiplexed: */
													/* 	Extension 0 Memory Data Out */
													/* 	Extension 0 Peripheral Memory Data Out */
	input [31:0]	MDATA_OUT,				/* Memory data bus out of tristate buffer in to FPGA */
	input [3:0]		MDATAP_OUT,				/* Memory data parity bus out of tristate buffer in to FPGA */
	input				MEMCLK,					/* Memory Interface to Data Path Clock 50 - 100 MHZ */
	input				MEMCLK2,					/* Memory Interface to Peripherals Clock 50 - 100 MHZ */
	input				MLOCK_EXT0_IN,			/* Extension 0 Memory Automic Lock External */
	input				MOE_EXT0_IN,			/* Extension 0 Memory Output Enable External */
	input				MWE_EXT0_IN,			/* Extension 0 Memory Write Enable External */
	input				PCNEXT_EXT0e,			/* Extension 0 Conditional PC Update External */
	input [15:0]	PDATA_OUT,				/* Peripheral data bus out of tristate buffer in to FPGA */
	input				PER0_DNE_IN,			/* */
	input				PER0_MY_ADDR_IN,		/* */
	input				PER0_PRESENT_IN,		/* */
	input				PER0_PRIVILEGE_IN,	/* */
	input				PER0_WANTS_INTR_IN,	/* */
	input				PI_EXT0e,				/* Extension 0 Passive/Parrallel Instruction External */
	input [4:0]		RDREG1_EXT0e,			/* Extension 0 Register Read Port 1 Register Number External */
													/* Multiplexed: */
													/* 	Extension 0 Register Read Port 1 Register Number */
													/* 	Extension 0 Register Write Port 1 Register Number */
													/* 	Extension 0 Write Register Execute Phase */
													/* 	Extension 0 Write Register Memory Access Phase */
	input [4:0]		RDREG2_EXT0e,			/* Extension 0 Register Read Port 2 Register Number External */
													/* Multiplexed: */
													/* 	Extension 0 Register Read Port 2 Register Number */
													/* 	Extension 0 Register Write Port 2 Register Number */
													/* 	<0> Extension 0 Register Write Enable Execute Phase */
													/*		<0> Extension 0 Register Write Enable Memory Access Phase */
													/* 	<1> Extension 0 Memory to Register Execute Phase */
													/* 	<1> Extension 0 Memory to Register Memory Acess Phase */
													/* 	<3> Extension 0 Load/Store Conditional Execute Phase */
													/* 	<4> Extension 0 Extended Memory Data Execute Phase */
	input [4:0]		RDREG3_EXT0e,			/* Extension 0 Register Read Port 3 Register Number External */
													/* Multiplexed: */
													/* 	Extension 0 Register Read Port 3 Register Number */
													/* 	Extension 0 Exception Code [4:0] */
	input [4:0]		RDREG4_EXT0e,			/* Extension 0 Register Read Port 4 Register Number External */
													/* Multiplexed: */
													/* 	Extension 0 Register Read Port 4 Register Number */
													/* 	<0> Extension 0 Memory Write Execute Phase */
													/* 	<1> Extension 0 Memory Read Execute Phase */
													/* 	<1:0> Extension 0 Data Address [1:0] Memory Access Phase */
													/*		<2> Extension 0 Right/Left Unaligned Load/Store Execute Phase */
													/*		<2> Extension 0 Right/Left Unaligned Load/Store Memory Access Phase */
													/* 	<3> Extension 0 Byte Load/Store Execute Phase */
													/* 	<3> Extension 0 Byte/Halfword Load/Store Memory Acess Phase */
													/* 	<4> Extension 0 Halfword Load/Store Execute */
	input				REGWRITE1_EXT0e,		/* Extension 0 Register Write Port 1 Write Enable External */
	input				REGWRITE2_EXT0e,		/* Extension 0 Register Write Port 2 Write Enable External */
	input				REMA_EXT0e,				/* Extension 0 Re-enter at Memory Access External */
	input				RESET,					/* Hardware Reset */
	input				REWB_EXT0e,				/* Extension 0 Re-enter at Writeback External */
	input				RI_EXT0e,				/* Extension 0 Reserved/Recognized Instruction External */
	input				RNL_EXT0_IN,			/* Extension 0 Memory Right/Left Unaligned Load/Store External */
	input				RX,						/* System Serial Recieve */
	input				SRAMCLK,					/* SRAM Clock Internal 50 - 100 MHZ */
	input				SYSCLK,					/* System Clock 50 - 100 MHZ */
	input				TMRCLK,					/* Timer Clock 10 MHZ */
	input				USARTCLK,				/* USART Clock 100 MHZ */
	input [31:0]	WRDATA1_EXT0e,			/* Extension 0 Register Write Port 1 Data External */
													/* Multiplexed: */
													/* 	Extension 0 Register Write Port 1 Data */
													/* 	Extension 0 ALU Result Execute Phase */
													/* 	Extension 0 ALU Result Memory Access Phase */
	input [31:0]	WRDATA2_EXT0e,			/* Extension 0 Register Write Port 2 Data External */
													/* Multiplexed: */
													/* 	Extension 0 Register Write Port 2 Data */
													/* 	Extension 0 Register 2 Data Execute Phase */
													/* 	Extension 0 Memory Data Out Memory Access Phase */
	input				fspecial_EXT0e,		/* */
	/* OUTPUT PORTS */
	output [31:0]	ADDR_EXT0_OUT,			/* Extension 0 Peripheral Memory Addresss External */
													/* Multiplexed: */
													/* 	Extension 0 Peripheral Memory Addresss */
													/* 	Extension 0 Memory Address Monitor */
	output 			BYTE,						/* Flash Byte Mode */
	output [1:0]	BYTES_EXT0_OUT,		/* */
	output 			BYTESEL,					/* Flash Byte Select */
	output			EXTCLK_EN_EXT0e,	/* */
	output 			CE_FLASH,				/* Flash Chip Enable */
	output 			EMPTY_RG_EXT0e,		/* Extension 0 Register Write Buffer Empty External */
	output 			EN_EXT0e,				/* Extension 0 Enable External */
	output 			ENEXT_EXT0e,			/* Extension 0 Execution Enable External */
	output 			EXCEXT_EXT0e,			/* Extension 0 Exception Flush External */
	output 			EXTNOP_EX_EXT0e,		/* Extension 0 Extension Bubble in Execute Phase External */
	output 			EXTNOP_MA_EXT0e,		/* Extension 0 Extension Bubble in Memory Access Phase External */
	output 			FULL_RG_EXT0e,			/* Extension 0 Register Write Buffer Full External */
	output [31:0]	GPIO_IN,					/* GPIO data in to tristate buffer out to pins */
	output [31:0]	GPIO_TR,					/* GPIO data pin direction */
	output			GR_EXT0e,				/* Extension 0 Grant Pipeline Resources External */
	output [31:0]	INSTR_EXT0e,			/* Extension 0 Current Instruction External */
	output			LDEXT_EXT0e,				/* Extension 0 Extension Loaded External */
	output [23:0]	MADDR_OUT,				/* Memory Address Bus (SRAM & Flash) */
	output [31:0]	MDATA_EXT0_OUT,		/* Extension 0 Memory Data In External */
													/* Multiplexed: */
													/* 	Extension 0 Memory Data In */
													/* 	Extension 0 Peripheral Memory Data In */
													/* 	Extension 0 Memory Data Monitor */
	output [31:0]	MDATA_IN,				/* Memory data bus in to tristate buffer out to pins */
	output [31:0]	MDATA_TR,				/* Memory data pin direction */
	output [3:0]	MDATAP_IN,				/* Memory data parity bus in to tristate buffer out to pins */
	output [3:0]	MDATAP_TR,				/* Memory data parity pin direction */
	output			MDATAVLD_EXT0_OUT,	/* Extension 0 Memory Data Valid External */
													/* Multiplexed: */
													/* 	Extension 0 Memory Data Valid */
													/* 	Extension 0 Memory Data Valid Monitor */
	output			MODE,						/* SRAM Mode */
	output			NADVLD,					/* SRAM ADV/LD */
	output [3:0]	NBW,						/* SRAM Byte Write Enable */
	output			NCE_SRAM,				/* SRAM Chip Enable */
	output			NCE_SYSACE,				/* System Ace Chip Enable */
	output			NOE_MEM,					/* Memory Output Enable (SRAM & Flash) */
	output			NOE_PER,					/* Peripheral Output Enable (SYSACE) */
	output			NRP,						/* Flash Hardware Reset */
	output			NWE_MEM,					/* Memory Write Enable (SRAM & Flash) */
	output			NWE_PER,					/* Peripheral Write Enable (SYSACE) */
	output			OE_EXT0_OUT,			/* Extension 0 Peripheral Output Enable External */
													/* Multiplexed: */
													/* 	Extension 0 Peripheral Output Enable */
													/* 	Extension 0 Memory Output Enable Monitor */
	output [6:0]	PADDR_OUT,				/* Peripheral Address Bus (SYSACE) */
	output [31:0]	PC_EXT0e,				/* Extension 0 Current PC External */
	output 			PCLK_EXT0e,				/* Extension 0 Pipeline Clock External */
	output [15:0]	PDATA_IN,				/* Peripheral data bus in to tristate buffer out to pins */
	output [15:0]	PDATA_TR,				/* Peripheral data pin direction */
	output			PER0_BAT_EN_OUT,		/* */
	output			PER0_MSRT_OUT,	/* */
	output [31:0]	RDREG1DATA_EXT0e,		/* Extension 0 Register Read Port 1 Register Data External */
	output [31:0]	RDREG2DATA_EXT0e,		/* Extension 0 Register Read Port 2 Register Data External */
	output [31:0]	RDREG3DATA_EXT0e,		/* Extension 0 Register Read Port 3 Register Data External */
	output [31:0]	RDREG4DATA_EXT0e,		/* Extension 0 Register Read Port 4 Register Data External */
	output			REGRDY_EXT0e,			/* Extension 0 Register Write Buffer Ready External */
	output			REGWRITE1_RG_EXT0e,	/* Extension 0 Register Write Port 1 Write Enable Monitor External */
	output			REGWRITE2_RG_EXT0e,	/* Extension 0 Register Write Port 2 Write Enable Monitor External */
	output			RESET_EXT0e,			/* Extension 0 System Reset External */
	output			SCLK,						/* Baud Clock */
	output			TX,						/* System Serial Transmit */
	output			WE_EXT0_OUT,			/* Extension 0 Peripheral Write Enable External */
													/* Multiplexed: */
													/* 	Extension 0 Peripheral Write Enable */
													/* 	Extension 0 Memory Write Enable Monitor */
	output [31:0]	WRDATA1_RG_EXT0e,		/* Extension 0 Register Write Port 1 Data Monitor External */
	output [31:0]	WRDATA2_RG_EXT0e,		/* Extension 0 Register Write Port 2 Data Monitor External */
	output [4:0]	WRREG1_RG_EXT0e,		/* Extension 0 Register Write Port 1 Register Number Monitor External */
	output [4:0]	WRREG2_RG_EXT0e,		/* Extension 0 Register Write Port 2 Register Number Monitor External */
	output			ZZ,
	/* INOUT PORTS */
	inout				CLK_EXT					/* External Clock */
	);
	
/*****Signals****************************************************************/
	
	/* GLOBAL SIGNALS */
	wire 			DNE;					/* Memory Done */
	wire		 	PCLK;					/* Pipeline Clock */
	wire 			RESET_G;				/* Debounced Global Reset */
	wire 			RST;					/* Soft Reset */

	/* PIPELINE ARBITER SIGNALS */
	wire 			ACK_EXT0_PA;		/* Extension 0 Enable Acknowledged */
	wire 			ACK_EXT1_PA;		/* Extension 1 Enable Acknowledged */
	wire 			EN_EXT0_PA;			/* Extension 0 Enable */
	wire 			EN_EXT1_PA;			/* Extension 1 Enable */
	wire 			EN_ID_PA;			/* TISA Enable */
	wire [31:0]	EXTCTR_PA;			/* Extension Control Register */
	wire 			EXTSTALL_PA;		/* Stall for Extension */
	wire 			GR_EXT0_PA;			/* Extension 0 Grant Pipeline Resources */
	wire 			GR_EXT1_PA;			/* Extension 1 Grant Pipeline Resources */
	wire 			PI_EXT0_PA;			/* Extension 0 Passive/Parrallel Instruction */
	wire 			PI_EXT1_PA;			/* Extension 1 Passive/Parrallel Instruction */
	wire 			RESET_PA;			/* Reset Pipeline Arbiter */
	wire 			RI_ID_PA;			/* TISA Reserved/Recognized Instruction */
	wire 			RI_EXT0_PA;			/* Extension 0 Reserved/Recognized Instruction */
	wire 			RI_EXT1_PA;			/* Extension 1 Reserved/Recognized Instruction */
	wire 			RI_PA;				/* Pipeline Arbiter Reserved/Recognized Instruction */
	wire 			STALL_PA;			/* Pipeline Stall */

	/* INSTRUCTION FETCH SIGNALS */
	wire 			EN_EXT_IF;			/* Extension Enable */
	wire 			EN_ID_IF;			/* TISA Enable */
	wire			EXC_IF;				/* Instruction Fetch Exception */
	wire [31:0]	EXCADDR_IF;			/* Exception Jump Address */
	wire [6:0]	EXCCODE_IF;			/* Instruction Fetch Exception Code */
	wire 			EXCE_IF;				/* Instruction Fetch Exception Flush */
	wire [31:0]	EXTADD_EXT_IF;		/* Extension Jump Address */
	wire 			GR_EXT_IF;			/* Extension Grant Pipeline Resources */
	wire 			FLUSH_IF;			/* Instruction Fetch Flush */
	wire [31:0]	INSTR_IF;			/* Fetched Instruction */
	wire [31:0]	INSTREDF_IF;		/* Instruction Fetch Data from Memory */
	wire [31:0]	JMPADD_IF;			/* Branch Jump Address */
	wire 			KU_IF;				/* Kernel/User Mode */
	wire [31:0]	PC_IF;				/* Instruction Fetch Current PC */
	wire [31:0]	PC4_IF;				/* Instruction Fetch Currect PC plus 4 */
	wire 			PCNEXT_EXT_IF;		/* Extension Conditional PC Update */
	wire 			PCSRC_IF;			/* Branch Address Selection Switch */
	wire [31:0]	PPC_IF;				/* Instruction Fetch Current PC to Memory */
	wire 			RE_IF;				/* Reverse Endian */
	wire 			RESET_IF;			/* Reset Instruction Fetch */
	wire 			STALL_IF;			/* Pipeline Stall */

	/* INSTRUCTION DECODE SIGNALS */
	wire [31:0]	ALUA_ID;				/* ALU Operand A */
	wire [31:0]	ALUB_ID;				/* ALU Operand B */
	wire [5:0]	ALUOP_ID;			/* ALU Opcode */
	wire 			ALUSRC_ID;			/* ALU Operand B Switch */
	wire 			BLS_ID;				/* Byte Load/Store */
	wire 			BRANCH_ID;			/* Instruction Decode Branch */
	wire 			BUBBLE_ID;			/* Bubble in Instruction Decode */
	wire 			CMOV_ID;				/* Conditional Move */
	wire 			COND_ID;				/* Conditional Operation */
	wire 			CPREGWRITE_ID;		/* Coprocessor Register Write Enable */
	wire [3:0]	CP_ID;				/* Enabled Coprocessors */
	wire 			CP0_ID;				/* Coprocessor 0 Operation */
	wire 			EN_EXT_ID;			/* Extension Enable */
	wire 			EN_ID;				/* TISA Enable */
	wire 			EXC_ID;				/* Instruction Decode Exception */
	wire 			EXCE_ID;				/* Instruction Decode Exception Flush */
	wire [6:0]	EXCCODE_ID;			/* Instruction Decode Exception Code */
	wire 			EXTNOP_ID;			/* Instruction Decode Extension NOP */
	wire [31:0]	EXCPC_ID;			/* Instruction Decode Exception PC */
	wire 			FLUSH_ID;			/* Instruction Decode Flush */
	wire [1:0]	FORWARD_A_ID;		/* Instruction Decode Data Forward Switch Operand A */
	wire [1:0]	FORWARD_B_ID;		/* Instruction Decode Data Forward Switch Operand B */
	wire 			HLS_ID;				/* Halfword Load/Store */
	wire [31:0]	IMM_ID;				/* Data Immediate */
	wire [31:0]	INSTR_ID;			/* Instruction Decode Instruction */
	wire [31:0]	JMPADD_ID;			/* Branch Jump Address */
	wire 			KU_ID;				/* Kernel/User Mode */
	wire 			LINK_ID;				/* Linked Branch */
	wire 			LUI_ID;				/* Load Upper Immediate */
	wire 			LC_ID;				/* Load/Store Conditional */
	wire 			MEMREAD_ID;			/* Memory Read */
	wire 			MEMTOREG_ID;		/* Memory to Regiser */
	wire 			MEMWRITE_ID;		/* Memory Write */
	wire 			MEXT_ID;				/* Sign Extend Memory Data */
	wire 			MULT_ID;				/* Instruction Decode Multiplication Unit Operation */
	wire [31:0]	PC_ID;				/* Instruction Decode Current PC */
	wire [31:0]	PC4_ID;				/* Instruction Decode Current PC plus 4 */
	wire 			PCSRC_ID;			/* Branch Address Selection Switch */
	wire [4:0]	RD_ID;				/* Destination Register Number */
	wire [31:0]	RDDATA_MA_ID;		/* Data forwarded from Memory Access Phase */
	wire [31:0]	RDDATA_WB_ID;		/* Data forwarded from Writeback Phase */
	wire [31:0]	REG1DATA_ID;		/* Data from Register Read Port 1 */
	wire [31:0]	REG2DATA_ID;		/* Data from Register Read Port 2 */
	wire 			REGDST_ID;			/* Destination Register Switch */
	wire 			REGWRITE_ID;		/* Register Write */
	wire 			RESET_ID;			/* Reset Instruction Decode */
	wire 			RFE_ID;				/* Return From Exception */
	wire 			RI_ID;				/* TISA Reserved/Recognized Instruction */
	wire 			RI_PA_ID;			/* Pipeline Arbiter Reserved/Recognized Instruction */
	wire 			RNL_ID;				/* Right/Left Unaligned Load/Store */
	wire [4:0]	RS_ID;				/* Instruction Decode TISA Operand Register Number 1 */
	wire [4:0]	RS0_ID;				/* Raw Operand Register Number 1 */
	wire [4:0]	RT_ID;				/* Instruction Decode TISA Operand Register Number 2 */
	wire [4:0]	RT0_ID;				/* Raw Operand Register Number 1 */
	wire 			STALL_ID;			/* Pipeline Stall */

	/* EXECUTE SIGNALS */
	wire [31:0]	ALU_B_EX;			/* ALU Operand B */
	wire [5:0]	ALUOP_EX;			/* ALU Opcode */
	wire [31:0]	ALURESULT_EX;		/* ALU Result */
	wire 			ALUSRC_EX;			/* ALU Operand B Switch */
	wire 			BLS_EX;				/* Byte Load/Store */
	wire 			BRANCH_EX;			/* Branch */
	wire 			BUBBLE_EX;			/* Bubble in Execute */
	wire 			CMOV_EX;				/* Conditional Move */
	wire 			COND_EX;				/* Conditional Operation */
	wire 			CP0_EX;				/* Coprocessor 0 Operation */
	wire [31:0]	CP0RDDATA1_EX;		/* Data from CP0 Register Read Port 1 */
	wire 			CP0REGWRITE_EX;	/* Coprocessor 0 Register Write Enable */
	wire 			CPREGWRITE_EX;		/* Coprocessor Register Write Enable */
	wire 			CREGWRITE_EX;		/* Conditional Register Write */
	wire 			EXC_EX;				/* Execute Exception */
	wire [6:0]	EXCCODE_EX;			/* Execute Exception Code */
	wire 			EXCE_EX;				/* Execute Exception Flush */
	wire 			EXTNOP_EX;			/* Execute Extension NOP */
	wire [1:0]	FORWARD_A_EX;		/* Execute Data Forward Switch Operand A */
	wire [1:0]	FORWARD_B_EX;		/* Execute Data Forward Switch Operand B */
	wire 			HLS_EX;				/* Halfword Load/Store */
	wire [31:0]	IMM_EX;				/* Data Immediate */
	wire 			LC_EX;				/* Load/Store Conditional */
	wire 			LINK_EX;				/* Linked Branch */
	wire 			LUI_EX;				/* Load Upper Immediate */
	wire 			MEMREAD_EX;			/* Execute Memory Read */
	wire 			MEMTOREG_EX;		/* Memory to Regiser */
	wire 			MEMWRITE_EX;		/* Memory Write */
	wire 			MEXT_EX;				/* Sign Extend Memory Data */
	wire 			MULTRDY_EX;			/* Multiplication Unit Ready */
	wire [31:0]	PC_EX;				/* Execute Currect PC */
	wire [31:0]	PC4_EX;				/* Execute Currect PC plus 4*/
	wire [4:0]	RD_EX;				/* Destination Register */
	wire [31:0]	RDDATA_MA_EX;		/* Data forwarded from Memory Access Phase */
	wire [31:0]	RDDATA_WB_EX;		/* Data forwarded from Writeback Phase */
	wire [31:0]	REG1DATA_EX;		/* Data from Register Read Port 1 */
	wire [31:0]	REG2DATA_EX;		/* Data from Register Read Port 2 */
	wire 			REGDST_EX;			/* Destination Register Switch */
	wire 			REGWRITE_EX;		/* Register Write */
	wire 			RESET_EX;			/* Reset Execute */
	wire 			RFE_EX;				/* Return From Exception */
	wire 			RNL_EX;				/* Right/Left Unaligned Load/Store */
	wire [4:0]	RS_EX;				/* Execute TISA Operand Register Number 1 */
	wire [4:0]	RT_EX;				/* Execute TISA Operand Register Number 2 */
	wire [4:0]	WRREG_EX;			/* Execute TISA Destination Register Number */
	wire [31:0] HI_EX;				/* */
	wire [31:0]	LO_EX;				/* */


	/* MEMORY ACCESS SIGNALS */
	wire [31:0]	ALURESULT_MA;		/* ALU Result */
	wire 			BLS_MA;				/* Memory Access Byte Load/Store */
	wire 			BHLS_MA;				/* Byte/Halfword Load/Store */
	wire 			BRANCH_MA;			/* Branch */
	wire 			BUBBLE_MA; 			/* Bubble in Memory Access */
	wire 			CMEMWRITE_MA;		/* Conditional Memory Write */
	wire [31:0]	DMADD_MA;			/* Memory Access Data Memory Address */
	wire [31:0]	DMDATAOUT_MA;		/* Memory Data Out */
	wire [31:0]	DMDATAOUTEDF_MA;	/* Memory Accesss Memory Data Out from Memory */
	wire 			EXC_MA;				/* Memory Access Exception */
	wire [6:0]	EXCCODE_MA;			/* Memory Access Exception Code */
	wire 			EXCE_MA;				/* Memory Access Exception Flush */
	wire 			EXTNOP_MA;			/* Memory Access Extension NOP */
	wire 			HLS_MA;				/* Memory Access Halfword Load/Store */
	wire 			LC_MA;				/* Load/Store Conditional */
	wire [31:0]	MEMDATAINEDF_MA;	/* Memory Access Memory Data In to Memory */
	wire 			MEMREAD_MA;			/* Memory Access Memory Read */
	wire 			MEMTOREG_MA;		/* Memory to Regiser */
	wire 			MEMWRITE_MA;		/* Memory Access Memory Write */
	wire 			MEXT_MA;				/* Sign Extend Memory Data */
	wire [31:0]	PC_MA;				/* Memory Access Currect PC */
	wire 			RE_MA;				/* Reverse Endian */
	wire [31:0]	REG2DATA_MA;		/* Data from Register Read Port 2 */
	wire 			REGWRITE_MA;		/* Memory Access Register Write */
	wire 			RESET_MA;			/* Reset Memory Access */
	wire 			RNL_MA;				/* Memory Access Right/Left Unaligned Load/Store */
	wire [4:0]	WRREG_MA;			/* Memory Access TISA Destination Register Number */
	wire			ALS_MA;

	/* WRITE BACK SIGNALS */
	wire [31:0]	ALURESULT_WB;		/* ALU Result */
	wire 			BHLS_WB;				/* Writeback Byte/Halfword Load/Store */
	wire 			BRANCH_WB;			/* Branch */
	wire 			BUBBLE_WB;			/* Bubble in Writeback */
	wire [1:0] 	DMADD_WB;			/* Writeback Least Significant bits of the Data Memory Address */
	wire [31:0]	DMDATAOUT_WB;		/* Memory Data Out */
	wire 			EXTNOP_WB;			/* Writeback Extension NOP */
	wire 			MEMTOREG_WB;		/* Memory to Regiser */
	wire [31:0]	PC_WB;				/* Writeback Currect PC */
	wire 			REGWRITE_WB;		/* Writeback Register Write */
	wire 			RESET_WB;			/* Reset Writeback */
	wire 			RNL_WB;				/* Writeback Right/Left Unaligned Load/Store */
	wire [4:0]	WRREG_WB;			/* Writeback TISA Destination Register Number */
	wire [31:0] WRREGDATA_WB;		/* Register Write Port Data */

	/* EXTENSION SIGNALS */
	wire			W_PER0_BAT_EN;		/* */
	wire			W_PER1_BAT_EN;		/* */
	wire 			PER0_PRESENT_LK;	/* */

	/* DATA FORWARD SIGNALS */
	wire [1:0]	FORWARD_A_EX_DF;	/* Execute Data Forward Switch Operand A */
	wire [1:0]	FORWARD_A_ID_DF;	/* Instruction Decode Data Forward Switch Operand A */
	wire [1:0]	FORWARD_B_EX_DF;	/* Execute Data Forward Switch Operand b */
	wire [1:0]	FORWARD_B_ID_DF;	/* Instruction Decode Data Forward Switch Operand B */
	wire [4:0]	RD_MA_DF;			/* Memory Access TISA Destination Register Number */
	wire [4:0]	RD_WB_DF;			/* Writeback TISA Destination Register Number */
	wire 			RESET_DF;			/* Reset Data Forward */
	wire 			REGWRITE_MA_DF;	/* Memory Access Register Write */
	wire 			REGWRITE_WB_DF;	/* Writeback Register Write */
	wire [4:0]	RS_EX_DF;			/* Execute TISA Operand Register Number 1 */
	wire [4:0]	RS_ID_DF;			/* Instruction Decode TISA Operand Register Number 1 */
	wire [4:0]	RT_EX_DF;			/* Execute TISA Operand Register Number 2 */
	wire [4:0]	RT_ID_DF;			/* Instruction Decode TISA Operand Register Number 2 */

	/* HAZARD SIGNALS */
	wire 			BRANCH_ID_HZ;		/* Instruction Decode Branch */
	wire 			EXTSTALL_HZ;		/* Stall for Extension */
	wire 			IRQ0_HZ;				/* Interrupt */
	wire 			MEMREAD_EX_HZ;		/* Execute Memory Read */
	wire 			MEMREAD_MA_HZ;		/* Memory Access Memory Read */
	wire 			MULT_HZ;				/* Instruction Decode Multiplication Unit Operation */
	wire 			MULTRDY_HZ;			/* Multiplication Unit Ready */
	wire 			RESET_HZ;			/* Reset Hazard */
	wire [4:0]	RS_ID_HZ;			/* Instruction Decode TISA Operand Register Number 1 */
	wire [4:0]	RT_ID_HZ;			/* Instruction Decode TISA Operand Register Number 2 */
	wire [4:0]	RTD_EX_HZ;			/* Execute TISA Destination Register Number */
	wire [4:0]	RTD_MA_HZ;			/* Memory Access TISA Destination Register Number */
	wire 			STALL_HZ;			/* Pipeline Stall */
	wire			LC_EX_HZ;		/* */

	/* MEMORY SIGNALS */
	wire 			BLS_MEM;				/* Memory Access Byte Load/Store */
	wire [31:0]	DMADD_MEM;			/* Memory Access Data Memory Address */
	wire [31:0]	DMDATAIN_MEM;		/* Memory Access Memory Data In to Memory */
	wire [31:0]	DMDATAOUT_MEM;		/* Memory Accesss Memory Data Out from Memory */
	wire 			EXC_IF_MEM;			/* Instruction Fetch Exception Flush */
	wire 			EXC_MA_MEM;			/* Memory Access Exception Flush */
	wire 			FLUSH_MEM;			/* Instruction Decode Flush */
	wire 			HLS_MEM;				/* Memory Access Halfword Load/Store */
	wire [31:0]	INSTR_MEM;			/* Instruction Fetch Data from Memory */
	wire 			IRQ0_MEM;			/* Interrupt */
	wire 			MEMREAD_MEM;		/* Memory Access Memory Read */
	wire 			MEMWRITE_MEM;		/* Memory Access Memory Write */
	wire [31:0]	PC_MEM;				/* Instruction Fetch Current PC to Memory */
	wire 			RESET_MEM;			/* Reset Memory */
	wire 			RNL_MEM;				/* Memory Access Right/Left Unaligned Load/Store */
	wire			STALL_MEM;			/* Pipeline Stall */
	wire [31:0]	W_MADDR_EXT;		/* Extension Memory Address */
	wire [31:0]	W_MADDR_PER;		/* Extension Peripheral Memory Addresss */
	wire [31:0]	W_MADDR_SNOOP;		/* Extension Memory Address Monitor */
	wire 			W_BLS_EXT;			/* Extension Byte Load/Store */
	wire 			W_EXC_EXT;			/* Extension Exception M */
	wire [31:0]	W_EXT_MDATA;		/* Extension Memory Data In */
	wire 			W_EXT_MDATAVLD;	/* Extension Memory Data Valid */
	wire 			W_HLS_EXT;			/* Extension Halfword Load/Store */
	wire [31:0] W_MDATA_EXT;		/* Extension Memory Data Out */
											/* Multiplexed: */
											/* 	Extension Memory Data Out */
											/* 	Extension Peripheral Memory Data Out */
	wire [31:0]	W_MDATA_PER;		/* Extension Peripheral Memory Data In */
	wire [31:0]	W_MDATA_SNOOP;		/* Extension Memory Data Monitor */
	wire 			W_MDATAVLD_SNOOP;	/* Extension Memory Data Valid Monitor */
	wire 			W_MOE_EXT;			/* Extension Memory Output Enable */
	wire 			W_MOE_PER;			/* Extension Peripheral Output Enable */
	wire 			W_MOE_SNOOP;		/* Extension Memory Output Enable Monitor */
	wire 			W_MLOCK_EXT;		/* Extension Memory Automic Lock */
	wire 			W_MWE_EXT;			/* Extension Memory Write Enable */
	wire 			W_MWE_PER;			/* Extension Peripheral Write Enable */
	wire 			W_MWE_SNOOP;		/*	Extension Memory Write Enable Monitor */
	wire			W_PER0_BATEN_MEM;	/* */
	wire 			W_RNL_EXT;			/* Extension Memory Right/Left Unaligned Load/Store */
	wire			MSRT_PER_MEM;			/* */
	wire			PER0_DNE_MEM;			/* */
	wire			PER0_MY_ADDR_MEM;		/* */
	wire 			PER0_PRESENT_MEM;		/* */
	wire			PER0_PRIVILEGE_MEM;	/* */
	wire			PER0_WANTS_INTR_MEM;	/* */
	wire			PER1_DNE_MEM;			/* */
	wire			PER1_MY_ADDR_MEM;		/* */
	wire 			PER1_PRESENT_MEM;		/* */
	wire			PER1_PRIVILEGE_MEM;	/* */
	wire			PER1_WANTS_INTR_MEM;	/* */
	wire [1:0]	MBYTES_PER_MEM;		/* */
	wire			INT_EXT0_MEM;			/* */
	wire			INT_EXT1_MEM;			/* */
	wire 			RE_MEM;					/* */
	wire [31:0]	MDATA_PER_EXT_MEM;	/* */
	wire 			BUBBLE_WB_MEM;			/* */
	wire [31:0]	PC_WB_MEM;				/* */
	wire 			PER_EXT0_MEM;			/* */
	wire			PER_EXT1_MEM;			/* */
	wire			LC_MEM;					/* */
	wire			ALS_MEM;					/* */

	/* REGISTER SIGNALS */
	wire			BHLS_RG;				/* Byte/Halfword Load/Store */
	wire [1:0]	DMADD_RG;			/* Least Significant bits of the Data Memory Address */
	wire 			EMPTY_RG;			/* Register Write Buffer Empty */
	wire 			FULL_RG;				/* Register Write Buffer Full */
	wire 			GR_RG;				/* Extension Grant Pipeline Resources */
	wire [4:0]	RD_MA_RG;			/* Memory Access TISA Destination Register Number */
	wire [4:0]	RD_WB_RG;			/* Writeback TISA Destination Register Number */
	wire [31:0]	RDDATA_MA_RG;		/* Data forwarded from Memory Access Phase */
	wire [31:0]	RDDATA_WB_RG;		/* Data forwarded from Writeback Phase */
	wire [4:0]	RDREG1_RG;			/* Register Read Port 1 Register Number */
	wire [31:0]	RDREG1DATA_RG;		/* Register Read Port 1 Register Data */
	wire [4:0]	RDREG2_RG;			/* Register Read Port 2 Register Number */
	wire [31:0]	RDREG2DATA_RG;		/* Register Read Port 2 Register Data */
	wire [4:0]	RDREG3_RG;			/* Register Read Port 3 Register Number */
	wire [31:0]	RDREG3DATA_RG;		/* Register Read Port 3 Register Data */
	wire [4:0]	RDREG4_RG;			/* Register Read Port 4 Register Number */
	wire [31:0]	RDREG4DATA_RG;		/* Register Read Port 4 Register Data */
	wire 			RE_RG;				/* Reverse Endian */
	wire 			REGRDY_RG;			/* Register Write Buffer Ready */
	wire 			REGWRITE1_RG;		/* Register Write Port 1 Write Enable */
	wire 			REGWRITE2_RG;		/* Register Write Port 2 Write Enable */
	wire 			REGWRITE_MA_RG;	/* Memory Access Register Write */
	wire 			REGWRITE_WB_RG;	/* Writeback Register Write */
	wire 			RESET_RG;			/* Reset Registers */
	wire 			RNL_RG;				/* Right/Left Unaligned Load/Store */
	wire [31:0]	WRDATA1_RG;			/* Register Write Port 1 Data */
	wire [31:0]	WRDATA2_RG;			/* Register Write Port 2 Data */
	wire [4:0]	WRREG1_RG;			/* Register Write Port 1 Register Number */
	wire [4:0]	WRREG2_RG;			/* Register Write Port 2 Register Number */
	wire [31:0]	HI_RG;				/* */
	wire [31:0]	LO_RG;				/* */
	wire 			FSPECIAL_RG;		/* */
	wire [31:0]	FPSTATUS_RG;		/* */
	wire [31:0]	FPIMP_RG;			/* */
	wire [31:0]	STATUS_RG;			/* */
	wire [31:0]	BADVADDR_RG;		/* */
	wire [31:0]	CAUSE_RG;			/* */

	/* COPROCESSOR 0 SIGNALS */
	wire 			BD_EX_CP0;			/* Execute Branch Delay Instruction */
	wire 			BD_EXT_CP0;			/* Extension Branch Delay Instruction */
	wire 			BD_ID_CP0;			/* Instruction Decode Branch Delay Instruction */
	wire 			BD_MA_CP0;			/* Memory Access Branch Delay Instruction */
	wire 			BD_WB_CP0;			/* Writeback Branch Delay Instruction */
	wire 			BUBBLE_EX_CP0;		/* Bubble in Execute */
	wire 			BUBBLE_ID_CP0;		/* Bubble in Instruction Decode */
	wire 			BUBBLE_MA_CP0;		/* Bubble in Memory Access */
	wire [3:0] 	CP_CP0;				/* Enabled Coprocessors */
	wire [31:0]	CP0RDDATA1_CP0;	/* Data from CP0 Register Read Port 1 */
	wire [4:0] 	CP0RDREG_CP0;		/* CP0 Register Read Port 1 Register Number */
	wire 			CP0REGWRITE_CP0;	/* Coprocessor Register Write Enable */
	wire [4:0] 	CP0WRREG_CP0;		/* CP0 Register Write Port Register Number */
	wire [31:0] CP0WRREGDATA_CP0;	/* CP0 Register Write Port Data */
	wire [3:0] 	EXC_CP0;				/* TISA Pipeline Exception Flush */
	wire 			EXC_EX_CP0;			/* Execute Exception */
	wire 			EXC_EXT_CP0;		/* Extension Exception */
	wire 			EXC_ID_CP0;			/* Instruction Decode Exception */
	wire 			EXC_IF_CP0;			/* Instruction Fetch Exception */
	wire 			EXC_MA_CP0;			/* Memory Access Exception */
	wire [31:0]	EXCADD_EX_CP0;		/* Execute Exception Address/Data */
	wire [31:0] EXCADD_EXT_CP0;	/* Extension Exception Address/Data */
	wire [31:0] EXCADD_ID_CP0;		/* Instruction Decode Exception Address/Data */
	wire [31:0] EXCADD_IF_CP0;		/* Instruction Fetch Exception Address/Data */
	wire [31:0] EXCADD_MA_CP0;		/* Memory Access Exception Address/Data */
	wire [31:0] EXCADDR_CP0;		/* Exception Jump Address */
	wire 			EXCEXT_CP0;			/* Extension Exception Flush */
	wire [6:0]	EXCCODE_EX_CP0;	/* Execute Exception Code */
	wire [6:0]	EXCCODE_EXT_CP0;	/* Extension Exception Code */
	wire [6:0]	EXCCODE_ID_CP0;	/* Instruction Decode Exception Code */
	wire [6:0]	EXCCODE_IF_CP0;	/* Instruction Fetch Exception Code */
	wire [6:0]	EXCCODE_MA_CP0;	/* Memory Access Exception Code */
	wire [31:0]	EXCPC_EX_CP0;		/* Execute Exception PC */
	wire [31:0]	EXCPC_EXT_CP0;		/* Extension Exception PC */
	wire [31:0]	EXCPC_ID_CP0;		/* Instruction Decode Exception PC */
	wire [31:0]	EXCPC_IF_CP0;		/* Instruction Fetch Exception PC */
	wire [31:0]	EXCPC_MA_CP0;		/* Memory Access Exception PC */
	wire [31:0]	EXTCTR_CP0;			/* Extension Control Register */
	wire 			EXTSTALL_CP0;		/* Stall for Extension */
	wire 			FLUSH_CP0;			/* Instruction Fetch Flush */
	wire 			IRQ0_CP0;			/* Interrupt */
	wire 			KU_CP0;				/* Kernel/User Mode */
	wire 			RE_CP0;				/* Reverse Endian */
	wire 			REMA_CP0;			/* Extension Re-enter at Memory Access */
	wire 			RESET_CP0;			/* Reset CP0 */
	wire 			REWB_CP0;			/* Extension Re-enter at Writeback */
	wire 			RFE_CP0;				/* Return from Exception */
	wire 			STALL_CP0;			/* Pipeline Stall */
	wire			EXEN_CP0;			/* */
	wire [31:0]	STATUS_CP0;			/* */
	wire [31:0]	BADVADDR_CP0;		/* */
	wire [31:0]	CAUSE_CP0;			/* */
	
/*****Parameters****************************************************************/

	parameter	EXT0		=	0;		/* */
	parameter	EXT1		=	8;		/* */
	parameter	EXT2		=	16;	/* */
	parameter	EXT3		=	24;	/* */
	parameter	EXTCLKEN	=	7;		/* */
	parameter	EXTPRIV	=	6;		/* */
	parameter	EXTPER	=	5;		/* */
	parameter	EXTPRr	=	4;		/* */
	parameter	EXTPRl	=	3;		/* */
	parameter	EXTTR		=	2;		/* */
	parameter	EXTLD		=	1;		/* */
	parameter	EXTEN		=	0;		/* */

/*****Debug****************************************************************/

	/*`include "decoder.v"*/

	//DEBUG SIGNALS
	/*integer dbflag, i, hwint;
	reg regflag, MDNER;
	reg MEMREAD_MEM_WB, MEMWRITE_MEM_WB;
	reg EXCFLAG, EXCFLAG0, EXCFLAG1;
	reg IDEXC, HWEXC;
	reg [1:0] pclkcnt, dnecnt, regcnt;
	reg [1:0] CPEXC;
	reg [31:0] INSTR_EX, INSTR_MA, INSTR_WB, DMADD_MEM_WB;
	reg [31:0] DMDATAIN_MEM_WB, DMDATAOUT_MEM_WB, HI_MA;
	reg [31:0] LO_MA, HI_WB, LO_WB;
	reg [31:0] ticks;
	reg  [31:0] dbmem[`NVALUES-1 : 0];
	reg [31:0] cp0regfile_MA[0:31], cp0regfile_WB[0:31];
	wire REGDNE, GIANO_STALL, GIANO_STALL2;
	wire [31:0] cp0regfile[0:31];
	wire [31:0] regfile[0:31];*/

	/*initial
	begin
		dbmem[`test_size] = 0;
		dbflag = 1'b0;
		pclkcnt = 2'b11;
		dnecnt = 2'b11;
		regcnt = 2'b11;
		regflag = 1'b1;
		DMADD_MEM_WB = 32'b0;
		DMDATAIN_MEM_WB = 32'b0;
		DMDATAOUT_MEM_WB = 32'b0;
		MEMREAD_MEM_WB = 1'b0;
		MEMWRITE_MEM_WB = 1'b0;
		INSTR_WB = 32'b0;
		INSTR_MA = 32'b0;
		INSTR_EX = 32'b0;
		EXCFLAG = 1'b0;
		EXCFLAG0 = 1'b0;
		EXCFLAG1 = 1'b0;
		IDEXC = 1'b0;
		HWEXC = 1'b0;
		CPEXC = 3'b0;
		ticks = 32'b0;
		hwint = 0;
		MDNER = 1'b1;
	end*/

	/*always@(posedge SYSCLK)
	begin
		pclkcnt <= {pclkcnt[0],PCLK};
		dnecnt <= {dnecnt[0],DNE};
	end*/

	/*always@(posedge PCLK)
	begin
		DMADD_MEM_WB <= DMADD_MEM;
		DMDATAIN_MEM_WB <= DMDATAIN_MEM;
		DMDATAOUT_MEM_WB <= DMDATAOUT_MEM;
		MEMREAD_MEM_WB <= MEMREAD_MEM;
		MEMWRITE_MEM_WB <= MEMWRITE_MEM;
		INSTR_WB <= INSTR_MA;
		INSTR_MA <= INSTR_EX;
		if (~EXTSTALL_PA)
		begin
			INSTR_EX <= INSTR_ID;
		end
		HI_WB <= HI_MA;
		HI_MA <= HI_EX;
		LO_WB <= LO_MA;
		LO_MA <= LO_EX;
		CPEXC <= {CPEXC[0],IDEXC};
		for (i = 0; i < 32; i=i+1)
		begin
			cp0regfile_WB[i] <= cp0regfile[i];
		end
	end*/

	/*always@(posedge SYSCLK)
	begin
		if ((dnecnt == 2'b01) && (pclkcnt == 2'b00) && (~BUBBLE_WB))
		begin
			regflag <= 1'b1;
			$NextOracleData(dbmem); 
			if (dbmem[`test_size] == 0)
			begin
				$display("Buffer Overrun!!!");
				$stop;
			end
			//dbflag <= 1'b0;
			if (PC_WB != dbmem[`pc])
			begin
				$display("ERROR in PC");
				dbflag <= 1'b1;
			end
			if (INSTR_WB != dbmem[`instruction])
			begin
				$display("ERROR in Instruction");
				dbflag <= 1'b1;
			end
			if (MULTRDY_EX) 
			begin
				if (HI_WB != dbmem[`hireg])
				begin
					if (HI_EX != dbmem[`hireg])
					begin
						$display("ERROR in HI");
						dbflag <= 1'b1;
					end
				end
				if (LO_WB != dbmem[`loreg])
				begin
					if (LO_EX != dbmem[`loreg])
					begin
						$display("ERROR in LO");
						dbflag <= 1'b1;
					end
				end
			end
			for (i = 0; i < 32; i=i+1)
			begin
				
				if ((cp0regfile[i] != dbmem[`cp0_0+i]) && (HWEXC == 0))
				begin
					$display("ERROR in CPREG %d", i);
					dbflag <= 1'b1;
				end
				else if ((cp0regfile_WB[i] != dbmem[`cp0_0+i]) && (HWEXC == 1))
				begin
					$display("ERROR in CPREG %d", i);
					dbflag <= 1'b1;
				end
			end
			for (i = 0; i < 32; i=i+1)
			begin
				if (regfile[i] != dbmem[`r0+i])
				begin
					$display("ERROR in REG %d", i);
					dbflag <= 1'b1;
				end
			end
			if (dbflag)
			begin
				$display("\nPC %x", PC_WB);
				$display("INSTR %x", INSTR_WB);
				decode_instr(1'b0,1'b0,INSTR_WB,PC_WB);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					regfile[0],regfile[1],regfile[2],
					regfile[3],regfile[4],regfile[5],
					regfile[6],regfile[7]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					regfile[8],regfile[9],regfile[10],
					regfile[11],regfile[12],regfile[13],
					regfile[14],regfile[15]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					regfile[16],regfile[17],regfile[18],
					regfile[19],regfile[20],regfile[21],
					regfile[22],regfile[23]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					regfile[24],regfile[25],regfile[26],
					regfile[27],regfile[28],regfile[29],
					regfile[30],regfile[31]);
				$display("Updated %d = %x", 
					WRREG1_RG, regfile[WRREG1_RG]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					cp0regfile_WB[0],cp0regfile_WB[1],
					cp0regfile_WB[2],cp0regfile_WB[3],
					cp0regfile_WB[4],cp0regfile_WB[5],
					cp0regfile_WB[6],cp0regfile_WB[7]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					cp0regfile_WB[8],cp0regfile_WB[9],
					cp0regfile_WB[10],cp0regfile_WB[11],
					cp0regfile_WB[12],cp0regfile_WB[13],
					cp0regfile_WB[14],cp0regfile_WB[15]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					cp0regfile_WB[16],cp0regfile_WB[17],
					cp0regfile_WB[18],cp0regfile_WB[19],
					cp0regfile_WB[20],cp0regfile_WB[21],
					cp0regfile_WB[22],cp0regfile_WB[23]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					cp0regfile_WB[24],cp0regfile_WB[25],
					cp0regfile_WB[26],cp0regfile_WB[27],
					cp0regfile_WB[28],cp0regfile_WB[29],
					cp0regfile_WB[30],cp0regfile_WB[31]);
				$display(" ");
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					cp0regfile[0],cp0regfile[1],
					cp0regfile[2],cp0regfile[3],
					cp0regfile[4],cp0regfile[5],
					cp0regfile[6],cp0regfile[7]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					cp0regfile[8],cp0regfile[9],
					cp0regfile[10],cp0regfile[11],
					cp0regfile[12],cp0regfile[13],
					cp0regfile[14],cp0regfile[15]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					cp0regfile[16],cp0regfile[17],
					cp0regfile[18],cp0regfile[19],
					cp0regfile[20],cp0regfile[21],
					cp0regfile[22],cp0regfile[23]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					cp0regfile[24],cp0regfile[25],
					cp0regfile[26],cp0regfile[27],
					cp0regfile[28],cp0regfile[29],
					cp0regfile[30],cp0regfile[31]);
				$display("HI %x", HI_WB);
				$display("LO %x", LO_WB);
				if (MEMREAD_MEM_WB)
				begin
					$display("LOAD %x = %x",
						DMADD_MEM_WB, DMDATAOUT_MEM_WB);
				end
				if (MEMWRITE_MEM_WB)
				begin
					$display("STORE %x = %x",
						DMADD_MEM_WB, DMDATAIN_MEM_WB);
				end
				$display("\nORACLE DATA");
				$display("PC %x", dbmem[`pc]);
				$display("INSTR %x", dbmem[`instruction]);
				decode_instr(1'b0,1'b0,dbmem[`instruction],
					dbmem[`pc]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					dbmem[`r0+0],dbmem[`r0+1],dbmem[`r0+2],
					dbmem[`r0+3],dbmem[`r0+4],dbmem[`r0+5],
					dbmem[`r0+6],dbmem[`r0+7]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					dbmem[`r0+8],dbmem[`r0+9],
					dbmem[`r0+10],dbmem[`r0+11],
					dbmem[`r0+12],dbmem[`r0+13],
					dbmem[`r0+14],dbmem[`r0+15]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					dbmem[`r0+16],dbmem[`r0+17],
					dbmem[`r0+18],dbmem[`r0+19],
					dbmem[`r0+20],dbmem[`r0+21],
					dbmem[`r0+22],dbmem[`r0+23]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					dbmem[`r0+24],dbmem[`r0+25],
					dbmem[`r0+26],dbmem[`r0+27],
					dbmem[`r0+28],dbmem[`r0+29],
					dbmem[`r0+30],dbmem[`r0+31]);
				$display("");
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					dbmem[`cp0_0+0],dbmem[`cp0_0+1],
					dbmem[`cp0_0+2],dbmem[`cp0_0+3],
					dbmem[`cp0_0+4],dbmem[`cp0_0+5],
					dbmem[`cp0_0+6],dbmem[`cp0_0+7]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					dbmem[`cp0_0+8],dbmem[`cp0_0+9],
					dbmem[`cp0_0+10],dbmem[`cp0_0+11],
					dbmem[`cp0_0+12],dbmem[`cp0_0+13],
					dbmem[`cp0_0+14],dbmem[`cp0_0+15]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					dbmem[`cp0_0+16],dbmem[`cp0_0+17],
					dbmem[`cp0_0+18],dbmem[`cp0_0+19],
					dbmem[`cp0_0+20],dbmem[`cp0_0+21],
					dbmem[`cp0_0+22],dbmem[`cp0_0+23]);
				$display("%x\t%x\t%x\t%x\t%x\t%x\t%x\t%x",
					dbmem[`cp0_0+24],dbmem[`cp0_0+25],
					dbmem[`cp0_0+26],dbmem[`cp0_0+27],
					dbmem[`cp0_0+28],dbmem[`cp0_0+29],
					dbmem[`cp0_0+30],dbmem[`cp0_0+31]);
				$display("");
				$display("HI %x", dbmem[`hireg]);
				$display("LO %x", dbmem[`loreg]);
				$display(" ");
				//if (dbflag) $stop;
			end
			else
			begin
				$display(" ");
				$display("CHECKED!!!");
			end
		end
		else
		begin
			dbflag <= 1'b0;
		end
	end*/

	/*always@(posedge SYSCLK)
	begin
		if ((pclkcnt == 2'b00) && (dnecnt == 2'b01) && (RESET_G))
		begin
			$display(" ");
			decode_instr(1'b0,1'b0,INSTR_IF,PC_IF);
			decode_instr(1'b0,1'b0,INSTR_ID,PC_ID);
			decode_instr(1'b0,1'b0,INSTR_EX,PC_EX);
			decode_instr(1'b0,1'b0,INSTR_MA,PC_MA);
			//$display(" ");
			decode_instr(1'b0,1'b0,INSTR_WB,PC_WB);
			//decode_instr(1'b0,1'b0,dbmem[`instruction],dbmem[`pc]);
			//if (dbflag) $stop;
			//$stop;
		end
	end*/

/*********************************************************************/

/*********************************************************************/

	assign PER0_BAT_EN_OUT	=	W_PER0_BAT_EN;

/*****Clockmaster****************************************************************/
	
	clockmaster clkmas(
		.CLK(SYSCLK),
		.DCMLOCK(DCMLOCK),
		.MDNE(DNE),
		.PCLK(PCLK),
		.RESET(RESET),
		.RST(RST),
		.RESET_G(RESET_G)
		);
		
/*****Pipeline Arbiter****************************************************************/

	topa to_pa(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_PA(RESET_PA),

		//INPUTS ID
		.RI_ID(RI_ID),
		//OUTPUTS ID
		.RI_ID_PA(RI_ID_PA),

		//INPUTS EXT
		.ACK_EXT0(ACK_EXT0e),
		.ACK_EXT1(1'b0),				/* GND EXT1 Signals */
		.PI_EXT0(PI_EXT0e),
		.PI_EXT1(1'b0),				/* GND EXT1 Signals */
		.RI_EXT0(RI_EXT0e),
		.RI_EXT1(1'b1),				/* VCC EXT1 Signals */
		//OUTPUTS EXT
		.ACK_EXT0_PA(ACK_EXT0_PA),
		.ACK_EXT1_PA(ACK_EXT1_PA),
		.PI_EXT0_PA(PI_EXT0_PA),
		.PI_EXT1_PA(PI_EXT1_PA),
		.RI_EXT0_PA(RI_EXT0_PA),
		.RI_EXT1_PA(RI_EXT1_PA),

		//INPUTS HZ
		.STALL_HZ(STALL_HZ),
		//OUTPUTS HZ
		.STALL_PA(STALL_PA),

		//INPUTS CP0
		.EXTCTR_CP0(EXTCTR_CP0),
		//OUTPUTS CP0
		.EXTCTR_PA(EXTCTR_PA)
		);

	pipeline_arbiter pa(
		.ACK_EXT0(ACK_EXT0_PA),
		.ACK_EXT1(ACK_EXT1_PA),
		.CLK(SYSCLK),
		.EN_EXT0(EN_EXT0_PA),
		.EN_EXT1(EN_EXT1_PA),
		.EN_ID(EN_ID_PA),
		.EXT0CTR(EXTCTR_PA[7:0]),
		.EXT1CTR(EXTCTR_PA[15:8]),
		.EXTSTALL(EXTSTALL_PA),
		.GR_EXT0(GR_EXT0_PA),
		.GR_EXT1(GR_EXT1_PA),
		.PCLK(PCLK),
		.PI_EXT0(PI_EXT0_PA),
		.PI_EXT1(PI_EXT1_PA),
		.RESET(RESET_PA),
		.RI_EXT0(RI_EXT0_PA),
		.RI_EXT1(RI_EXT1_PA),
		.RI_ID(RI_ID_PA),
		.RI_PA(RI_PA),
		.STALL(STALL_PA),
		.EXCEXT(EXCEXT_CP0)
		);

/*****Instruction Fetch****************************************************************/

	toif to_if(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_IF(RESET_IF),

		//INPUTS PA
		.EN_EXT0_PA(EN_EXT0_PA),
		.EN_EXT1_PA(EN_EXT1_PA),
		.EN_ID_PA(EN_ID_PA),
		.GR_EXT0_PA(GR_EXT0_PA),
		.GR_EXT1_PA(GR_EXT1_PA),
		//OUTPUTS PA
		.EN_ID_IF(EN_ID_IF),
		.EN_EXT_IF(EN_EXT_IF),
		.GR_EXT_IF(GR_EXT_IF),

		//INPUTS ID
		.FLUSH_ID(FLUSH_ID),
		.JMPADD_ID(JMPADD_ID),
		.PCSRC_ID(PCSRC_ID),
		//OUTPUTS ID
		.FLUSH_IF(FLUSH_IF),
		.JMPADD_IF(JMPADD_IF),
		.PCSRC_IF(PCSRC_IF),

		//INPUTS EXT
		.EXTADD_EXT0(EXTADD_EXT0e),
		.EXTADD_EXT1(32'b0),				/* GND EXT1 Signals */
		.PCNEXT_EXT0(PCNEXT_EXT0e),
		.PCNEXT_EXT1(1'b0),				/* GND EXT1 Signals */
		//OUTPUTS EXT
		.EXTADD_EXT_IF(EXTADD_EXT_IF),
		.PCNEXT_EXT_IF(PCNEXT_EXT_IF),

		//INPUTS HZ
		.STALL_HZ(STALL_HZ),
		//OUTPUTS HZ
		.STALL_IF(STALL_IF),

		//INPUTS MEM
		.INSTR_MEM(INSTR_MEM),
		//OUTPUTS MEM
		.INSTR_IF(INSTREDF_IF),

		//INPUTS CP0
		.EXC_CP0(EXC_CP0[0]),
		.EXCADDR_CP0(EXCADDR_CP0),
		.KU_CP0(KU_CP0),
		.RE_CP0(RE_CP0),
		//OUTPUTS CP0
		.EXC_IF(EXCE_IF),
		.EXCADDR_IF(EXCADDR_IF),
		.KU_IF(KU_IF),
		.RE_IF(RE_IF)
		);

	instruction_fetch inf(
		.CLK(SYSCLK),
		/*.EDF(RE_IF),*/
		.EN_EXT(EN_EXT_IF),
		.EXCADDR(EXCADDR_IF),
		.EXC(EXC_IF),
		.EXCCODE(EXCCODE_IF),
		.EXCE(EXCE_IF),
		.EXTADD_EXT(EXTADD_EXT_IF),
		.GR_EXT(GR_EXT_IF),
		.INSTR(INSTR_IF),
		.INSTR_EDF(INSTREDF_IF),
		.JMPADD(JMPADD_IF),
		.KU(KU_IF),.EN_ID(EN_ID_IF),
		.PC(PC_IF),
		.PC4(PC4_IF),
		.PCLK(PCLK),
		.PCNEXT_EXT(PCNEXT_EXT_IF),
		.PCSRC(PCSRC_IF),
		.PPC(PPC_IF),
		.RESET(RESET_IF),
		.STALL(STALL_IF)
		);

/*****Instruction Decode****************************************************************/

	toid to_id(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_ID(RESET_ID),

		//INPUTS PA
		.GR_EXT0_PA(GR_EXT0_PA),
		.GR_EXT1_PA(GR_EXT1_PA),		

		//INPUTS IF
		.EXCE(EXCE_IF),
		.FLUSH(FLUSH_IF),
		.INSTR_IF(INSTR_IF),
		.PC_IF(PC_IF),
		.PC4_IF(PC4_IF),
		.STALL(STALL_HZ),
		//OUTPUTS IF
		.BUBBLE_ID(BUBBLE_ID),
		.INSTR_ID(INSTR_ID),
		.PC_ID(PC_ID),
		.PC4_ID(PC4_ID),

		//INPUTS MA
		.ALURESULT_MA(ALURESULT_MA),
		//OUTPUTS MA
		.RDDATA_MA_ID(RDDATA_MA_ID),

		//INPUTS WB
		.WRREGDATA_WB(WRREGDATA_WB),
		//OUTPUTS WB
		.RDDATA_WB_ID(RDDATA_WB_ID),

		//INPUTS EXT
		.PCNEXT_EXT0(PCNEXT_EXT0e),
		.PCNEXT_EXT1(1'b0),					/* GND EXT1 Signals */

		//INPUTS RG
		.REG1DATA_RG(RDREG1DATA_RG),
		.REG2DATA_RG(RDREG2DATA_RG),
		//OUTPUTS RG
		.REG1DATA_ID(REG1DATA_ID),
		.REG2DATA_ID(REG2DATA_ID),

		//INPUTS DF
		.FORWARD_A_DF(FORWARD_A_ID_DF),
		.FORWARD_B_DF(FORWARD_B_ID_DF),
		//OUTPUTS DF
		.FORWARD_A_ID(FORWARD_A_ID),
		.FORWARD_B_ID(FORWARD_B_ID),

		//INPUTS HZ
		.STALL_HZ(STALL_HZ),
		//OUTPUTS HZ
		.STALL_ID(STALL_ID),

		//INPUTS CP0
		.EXC_CP0(EXC_CP0[1]),
		.CP_CP0(CP_CP0),
		.KU_CP0(KU_CP0),
		//OUTPUTS CP0
		.EXCE_ID(EXCE_ID),
		.CP_ID(CP_ID),
		.KU_ID(KU_ID),

		//INPUTS PA
		.EN_EXT0_PA(EN_EXT0_PA),
		.EN_EXT1_PA(EN_EXT1_PA),
		.EN_ID_PA(EN_ID_PA),
		.RI_PA(RI_PA),
		//OUTPUTS PA
		.EN_EXT_ID(EN_EXT_ID),
		.EN_ID(EN_ID),
		.RI_PA_ID(RI_PA_ID)
		);

	instruction_decode id(
		.ALU_A(ALUA_ID),
		.ALU_B(ALUB_ID),
		.ALUOP(ALUOP_ID),
		.ALUSRC(ALUSRC_ID),
		.BLS(BLS_ID),
		.BRANCH(BRANCH_ID),
		.CLK(SYSCLK),
		.CMOV(CMOV_ID),
		.COND(COND_ID),
		.CP0(CP0_ID),
		.CPREGWRITE(CPREGWRITE_ID),
		.EN(EN_ID),
		.EN_EXT(EN_EXT_ID),
		.EXC(EXC_ID),
		.EXCCODE(EXCCODE_ID),
		.EXCPC(EXCPC_ID),
		.EXTNOP(EXTNOP_ID),
		.FLUSH(FLUSH_ID),
		.FORWARD_A(FORWARD_A_ID),
		.FORWARD_B(FORWARD_B_ID),
		.HLS(HLS_ID),
		.IMM(IMM_ID),
		.INSTR(INSTR_ID),
		.JMPADD(JMPADD_ID),
		.KU(KU_ID),.CP(CP_ID),
		.LC(LC_ID),
		.LINK(LINK_ID),
		.LUI(LUI_ID),
		.MEMREAD(MEMREAD_ID),
		.MEMTOREG(MEMTOREG_ID),
		.MEMWRITE(MEMWRITE_ID),
		.MEXT(MEXT_ID),
		.MULT(MULT_ID),
		.PC(PC_ID),
		.PC4(PC4_ID),
		.PCSRC(PCSRC_ID),
		.RDDATA_MA(RDDATA_MA_ID),
		.RDDATA_WB(RDDATA_WB_ID),
		.RD(RD_ID),
		.REG1DATA(REG1DATA_ID),
		.REG2DATA(REG2DATA_ID),
		.REGDST(REGDST_ID),
		.REGWRITE(REGWRITE_ID),
		.RESET(RESET_ID),
		.RFE(RFE_ID),
		.RI(RI_ID),
		.RI_PA(RI_PA_ID),
		.RNL(RNL_ID),
		.RS(RS_ID),
		.RS0(RS0_ID),
		.RT(RT_ID),
		.RT0(RT0_ID),
		.STALL(STALL_ID)
		);

/*****Execute****************************************************************/

	toex to_ex(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_EX(RESET_EX),

		//INPUTS ID
		.ALUOP_ID(ALUOP_ID),
		.ALUSRC_ID(ALUSRC_ID),
		.BLS_ID(BLS_ID),
		.BRANCH_ID(BRANCH_ID),
		.BUBBLE_ID(BUBBLE_ID),
		.CMOV_ID(CMOV_ID),
		.COND_ID(COND_ID),
		.CPREGWRITE_ID(CPREGWRITE_ID),
		.CP0_ID(CP0_ID),
		.EXCE(EXCE_ID),
		.EXTNOP_ID(EXTNOP_ID),
		.HLS_ID(HLS_ID),
		.IMM_ID(IMM_ID),
		.LC_ID(LC_ID),
		.LINK_ID(LINK_ID),
		.LUI_ID(LUI_ID),
		.MEMREAD_ID(MEMREAD_ID),
		.MEMTOREG_ID(MEMTOREG_ID),
		.MEMWRITE_ID(MEMWRITE_ID),
		.MEXT_ID(MEXT_ID),
		.PC_ID(PC_ID),
		.PC4_ID(PC4_ID),
		.RD_ID(RD_ID),
		.REG1DATA_ID(ALUA_ID),
		.REG2DATA_ID(ALUB_ID),
		.REGDST_ID(REGDST_ID),
		.REGWRITE_ID(REGWRITE_ID),
		.RFE_ID(RFE_ID),
		.RNL_ID(RNL_ID),
		.RS_ID(RS_ID),
		.RT_ID(RT_ID),
		//OUTPUTS ID
		.ALUOP_EX(ALUOP_EX),
		.ALUSRC_EX(ALUSRC_EX),
		.BLS_EX(BLS_EX),
		.BRANCH_EX(BRANCH_EX),
		.BUBBLE_EX(BUBBLE_EX),
		.EXTNOP_EX(EXTNOP_EX),
		.CMOV_EX(CMOV_EX),
		.COND_EX(COND_EX),
		.CPREGWRITE_EX(CPREGWRITE_EX),
		.CP0_EX(CP0_EX),
		.HLS_EX(HLS_EX),
		.IMM_EX(IMM_EX),
		.LC_EX(LC_EX),
		.LINK_EX(LINK_EX),
		.LUI_EX(LUI_EX),
		.MEMREAD_EX(MEMREAD_EX),
		.MEMTOREG_EX(MEMTOREG_EX),
		.MEMWRITE_EX(MEMWRITE_EX),
		.MEXT_EX(MEXT_EX),
		.PC_EX(PC_EX),
		.PC4_EX(PC4_EX),
		.RD_EX(RD_EX),
		.RFE_EX(RFE_EX),
		.REG1DATA_EX(REG1DATA_EX),
		.REG2DATA_EX(REG2DATA_EX),
		.REGDST_EX(REGDST_EX),
		.REGWRITE_EX(REGWRITE_EX),
		.RNL_EX(RNL_EX),
		.RS_EX(RS_EX),
		.RT_EX(RT_EX),

		//INPUTS MA
		.ALURESULT_MA(ALURESULT_MA),
		//OUTPUTS MA
		.RDDATA_MA_EX(RDDATA_MA_EX),

		//INPUTS WB
		.WRREGDATA_WB(WRREGDATA_WB),
		//OUTPUTS WB
		.RDDATA_WB_EX(RDDATA_WB_EX),

		//INPUTS DF
		.FORWARD_A_DF(FORWARD_A_EX_DF),
		.FORWARD_B_DF(FORWARD_B_EX_DF),
		//OUTPUTS DF
		.FORWARD_A_EX(FORWARD_A_EX),
		.FORWARD_B_EX(FORWARD_B_EX),

		//INPUTS HZ
		.STALL(STALL_HZ),

		//INPUTS CP0
		.CP0RDDATA1_CP0(CP0RDDATA1_CP0),
		.EXC_CP0(EXC_CP0[2]),
		//OUTPUTS CP0
		.CP0RDDATA1_EX(CP0RDDATA1_EX),
		.EXCE_EX(EXCE_EX)
		);

	execute ex(
		.ALU_B(ALU_B_EX),
		.ALUOP(ALUOP_EX),
		.ALURESULT(ALURESULT_EX),
		.ALUSRC(ALUSRC_EX),
		.CLK(SYSCLK),
		.CMOV(CMOV_EX),
		.COND(COND_EX),
		.CP0(CP0_EX),
		.CP0REGWRITE(CP0REGWRITE_EX),
		.CP0RGDATA(CP0RDDATA1_EX),
		.CPREGWRITE(CPREGWRITE_EX),
		.CREGWRITE(CREGWRITE_EX),
		.EXC(EXC_EX),
		.EXCCODE(EXCCODE_EX),
		.FORWARD_A(FORWARD_A_EX),
		.FORWARD_B(FORWARD_B_EX),
		.IMM(IMM_EX),
		.LINK(LINK_EX),
		.LUI(LUI_EX),
		.MULT_RDY(MULTRDY_EX),
		.PCLK(PCLK),
		.PC4(PC4_EX),
		.RD(RD_EX),
		.RDDATA_MA(RDDATA_MA_EX),
		.RDDATA_WB(RDDATA_WB_EX),
		.REG1DATA(REG1DATA_EX),
		.REG2DATA(REG2DATA_EX),
		.REGDST(REGDST_EX),
		.REGWRITE(REGWRITE_EX),
		.RESET(RESET_EX),
		.RT(RT_EX),
		.WRREG(WRREG_EX),
		.HI(HI_EX),
		.LO(LO_EX)
		);

/*****Memory Access****************************************************************/

	toma to_ma(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_MA(RESET_MA),

		//INPUTS PA
		.GR_EXT0_PA(GR_EXT0_PA),
		.GR_EXT1_PA(GR_EXT1_PA), 

		//INPUTS EX
		.ALURESULT_EX(ALURESULT_EX),
		.BLS_EX(BLS_EX),
		.BRANCH_EX(BRANCH_EX),
		.BUBBLE_EX(BUBBLE_EX),
		.HLS_EX(HLS_EX),
		.EXCE(EXCE_EX),
		.EXTNOP_EX(EXTNOP_EX),
		.LC_EX(LC_EX),
		.MEMREAD_EX(MEMREAD_EX),
		.MEMTOREG_EX(MEMTOREG_EX),
		.MEMWRITE_EX(MEMWRITE_EX),
		.MEXT_EX(MEXT_EX),
		.PC_EX(PC_EX),
		.REG2DATA_EX(ALU_B_EX),
		.REGWRITE_EX(CREGWRITE_EX),
		.RNL_EX(RNL_EX),
		.WRREG_EX(WRREG_EX),
		//OUTPUTS EX
		.ALURESULT_MA(ALURESULT_MA),
		.BLS_MA(BLS_MA),
		.BRANCH_MA(BRANCH_MA),
		.BUBBLE_MA(BUBBLE_MA),
		.EXTNOP_MA(EXTNOP_MA),
		.HLS_MA(HLS_MA),
		.LC_MA(LC_MA),
		.MEMREAD_MA(MEMREAD_MA),
		.MEMTOREG_MA(MEMTOREG_MA),
		.MEMWRITE_MA(MEMWRITE_MA),
		.MEXT_MA(MEXT_MA),
		.PC_MA(PC_MA),
		.REG2DATA_MA(REG2DATA_MA),
		.REGWRITE_MA(REGWRITE_MA),
		.RNL_MA(RNL_MA),
		.WRREG_MA(WRREG_MA),

		//INPUTS EXT0
		.ALURESULT_EXT0(WRDATA1_EXT0e),
		.BLS_EXT0(RDREG4_EXT0e[3]),
		.BRANCH_EXT0(1'b1),						/* Extension always a Branch */
		.BUBBLE_EXT0(1'b0),						/* Extension is not a Bubble */
		.HLS_EXT0(RDREG4_EXT0e[4]),
		.EXTNOP_EXT0(1'b0),						/* Extension is not a Extension NOP */
		.LC_EXT0(RDREG2_EXT0e[3]),
		.MEMREAD_EXT0(RDREG4_EXT0e[1]),
		.MEMTOREG_EXT0(RDREG2_EXT0e[1]),
		.MEMWRITE_EXT0(RDREG4_EXT0e[0]),
		.MEXT_EXT0(RDREG2_EXT0e[4]),
		.PC_EXT0(EXTADD_EXT0e),
		.REMA_EXT0(REMA_EXT0e),
		.REG2DATA_EXT0(WRDATA2_EXT0e),
		.REGWRITE_EXT0(RDREG2_EXT0e[0]),
		.RNL_EXT0(RDREG4_EXT0e[2]),
		.WRREG_EXT0(RDREG1_EXT0e),

		//INPUTS EXT1
		.ALURESULT_EXT1(32'b0),					/* GND EXT1 Signals */
		.BLS_EXT1(1'b0),							/* GND EXT1 Signals */
		.BRANCH_EXT1(1'b1),						/* VCC EXT1 Signals */
		.BUBBLE_EXT1(1'b0),						/* GND EXT1 Signals */
		.EXTNOP_EXT1(1'b0),						/* GND EXT1 Signals */
		.HLS_EXT1(1'b0),							/* GND EXT1 Signals */
		.LC_EXT1(1'b0),							/* GND EXT1 Signals */
		.MEMREAD_EXT1(1'b0),						/* GND EXT1 Signals */
		.MEMTOREG_EXT1(1'b0),					/* GND EXT1 Signals */
		.MEMWRITE_EXT1(1'b0),					/* GND EXT1 Signals */
		.MEXT_EXT1(1'b0),							/* GND EXT1 Signals */
		.PC_EXT1(32'b0),							/* GND EXT1 Signals */
		.REG2DATA_EXT1(32'b0),					/* GND EXT1 Signals */
		.REGWRITE_EXT1(1'b0),					/* GND EXT1 Signals */
		.REMA_EXT1(1'b0),							/* GND EXT1 Signals */
		.RNL_EXT1(1'b0),							/* GND EXT1 Signals */
		.WRREG_EXT1(5'b0),						/* GND EXT1 Signals */

		//INPUTS MEM
		.DMDATAOUT_MEM(DMDATAOUT_MEM),
		.ALS_MEM(ALS_MEM),
		//OUTPUTS MEM
		.DMDATAOUT_MA(DMDATAOUTEDF_MA),
		.ALS_MA(ALS_MA),

		//INPUTS CP0
		.EXC_CP0(EXC_CP0[3]),
		.RE_CP0(RE_CP0),
		//OUTPUTS CP0
		.EXCE_MA(EXCE_MA),
		.RE_MA(RE_MA)
		);

	memory_access ma(
		.BLS(BLS_MA),
		.BHLS(BHLS_MA),
		.CLK(SYSCLK),
		.CMEMWRITE(CMEMWRITE_MA),
		.EDF(RE_MA),
		.EXC(EXC_MA),
		.EXCCODE(EXCCODE_MA),
		.EXCE(EXCE_MA),
		.HLS(HLS_MA),
		.KU(KU_CP0),
		.LC(LC_MA),
		.MEMDATAIN(REG2DATA_MA),
		.MEMDATAIN_EDF(MEMDATAINEDF_MA),
		.MEMDATAOUT(DMDATAOUT_MA),
		.MEMDATAOUT_EDF(DMDATAOUTEDF_MA),
		.MEMREAD(MEMREAD_MA),
		.MEMWRITE(MEMWRITE_MA),
		.MEXT(MEXT_MA),
		.PADDR(DMADD_MA),
		.PCLK(PCLK),
		.RESET(RESET_MA),
		.RNL(RNL_MA),
		.VADDR(ALURESULT_MA),
		.ALS(ALS_MA)
		);

/*****Writeback****************************************************************/

	towb to_wb(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_WB(RESET_WB),

		//INPUTS PA
		.GR_EXT0_PA(GR_EXT0_PA),
		.GR_EXT1_PA(GR_EXT1_PA), 

		//INPUTS MA
		.ALURESULT_MA(ALURESULT_MA),
		.BHLS_MA(BHLS_MA),
		.BRANCH_MA(BRANCH_MA),
		.BUBBLE_MA(BUBBLE_MA),
		.DMADD_MA(ALURESULT_MA[1:0]),
		.DMDATAOUT_MA(DMDATAOUT_MA),
		.EXCE(EXCE_MA),
		.EXTNOP_MA(EXTNOP_MA),
		.MEMTOREG_MA(MEMTOREG_MA),
		.PC_MA(PC_MA),
		.REGWRITE_MA(REGWRITE_MA),
		.RNL_MA(RNL_MA),
		.WRREG_MA(WRREG_MA),
		//OUTPUTS MA
		.ALURESULT_WB(ALURESULT_WB),
		.BHLS_WB(BHLS_WB),
		.BRANCH_WB(BRANCH_WB),
		.BUBBLE_WB(BUBBLE_WB),
		.DMADD_WB(DMADD_WB),
		.DMDATAOUT_WB(DMDATAOUT_WB),
		.EXTNOP_WB(EXTNOP_WB),
		.MEMTOREG_WB(MEMTOREG_WB),
		.PC_WB(PC_WB),
		.REGWRITE_WB(REGWRITE_WB),
		.RNL_WB(RNL_WB),
		.WRREG_WB(WRREG_WB),

		//INPUTS EXT0
		.ALURESULT_EXT0(WRDATA1_EXT0e),
		.BHLS_EXT0(RDREG4_EXT0e[3]),
		.BRANCH_EXT0(1'b1),						/* Extension always a Branch */
		.BUBBLE_EXT0(1'b0),						/* Extension is not a Bubble */
		.DMADD_EXT0(RDREG4_EXT0e[1:0]),
		.DMDATAOUT_EXT0(WRDATA2_EXT0e),
		.EXTNOP_EXT0(1'b0),						/* Extension is not a Extension NOP */
		.MEMTOREG_EXT0(RDREG2_EXT0e[1]),
		.PC_EXT0(EXTADD_EXT0e),
		.REGWRITE_EXT0(RDREG2_EXT0e[0]),
		.REWB_EXT0(REWB_EXT0e),
		.RNL_EXT0(RDREG4_EXT0e[2]),
		.WRREG_EXT0(RDREG1_EXT0e),

		//INPUTS EXT1
		.ALURESULT_EXT1(32'b0),					/* GND EXT1 Signals */
		.BHLS_EXT1(1'b0),							/* GND EXT1 Signals */
		.BRANCH_EXT1(1'b1),						/* VCC EXT1 Signals */
		.BUBBLE_EXT1(1'b0),						/* GND EXT1 Signals */
		.DMDATAOUT_EXT1(32'b0),					/* GND EXT1 Signals */
		.DMADD_EXT1(2'b0),						/* GND EXT1 Signals */
		.EXTNOP_EXT1(1'b0),						/* GND EXT1 Signals */
		.MEMTOREG_EXT1(1'b0),					/* GND EXT1 Signals */
		.PC_EXT1(32'b0),							/* GND EXT1 Signals */
		.REGWRITE_EXT1(1'b0),					/* GND EXT1 Signals */
		.REWB_EXT1(1'b0),							/* GND EXT1 Signals */
		.RNL_EXT1(1'b0),							/* GND EXT1 Signals */
		.WRREG_EXT1(5'b0)							/* GND EXT1 Signals */
		);

	write_back wb(
		.ALURESULT(ALURESULT_WB),
		.CLK(SYSCLK),
		.DMDATAOUT(DMDATAOUT_WB),
		.MEMTOREG(MEMTOREG_WB),
		.RESET(RESET_WB),
		.WRREGDATA(WRREGDATA_WB)
		);

/*****Extension Interface****************************************************************/

	/* Extension 0 */
	toext to_ext(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.PCLK_EXT(PCLK_EXT0e),
		.RESET_EXT(RESET_EXT0e),

		//INPUTS PA
		.EN_PA(EN_EXT0_PA),
		.GR_PA(GR_EXT0_PA),
		//OUTPUS PA
		.EN_EXT(EN_EXT0e),
		.GR_EXT(GR_EXT0e),
		
		//INPUTS ID
		.INSTR_ID(INSTR_ID),
		.PC_ID(PC_ID),
		//OUTPUTS ID
		.INSTR_EXT(INSTR_EXT0e),
		.PC_EXT(PC_EXT0e),
		
		//INPUTS EX
		.EXTNOP_EX(EXTNOP_EX),
		//OUTPUTS EX
		.EXTNOP_EX_EXT(EXTNOP_EX_EXT0e),
		
		//INPUTS MA
		.EXTNOP_MA(EXTNOP_MA),
		//OUTPUTS MA
		.EXTNOP_MA_EXT(EXTNOP_MA_EXT0e),
		
		//INPUTS RG
		.EMPTY_RG(EMPTY_RG),
		.FULL_RG(FULL_RG),
		.RDREG1DATA_RG(RDREG1DATA_RG),
		.RDREG2DATA_RG(RDREG2DATA_RG),
		.RDREG3DATA_RG(RDREG3DATA_RG),
		.RDREG4DATA_RG(RDREG4DATA_RG),
		.REGRDY_RG(REGRDY_RG),
		.REGWRITE1_RG(REGWRITE1_RG),
		.REGWRITE2_RG(REGWRITE2_RG),
		.WRDATA1_RG(WRDATA1_RG),
		.WRDATA2_RG(WRDATA2_RG),
		.WRREG1_RG(WRREG1_RG),
		.WRREG2_RG(WRREG2_RG),
		//OUTPUTS RG
		.EMPTY_RG_EXT(EMPTY_RG_EXT0e),
		.FULL_RG_EXT(FULL_RG_EXT0e),
		.RDREG1DATA_EXT(RDREG1DATA_EXT0e),
		.RDREG2DATA_EXT(RDREG2DATA_EXT0e),
		.RDREG3DATA_EXT(RDREG3DATA_EXT0e),
		.RDREG4DATA_EXT(RDREG4DATA_EXT0e),
		.REGRDY_EXT(REGRDY_EXT0e),
		.REGWRITE1_RG_EXT(REGWRITE1_RG_EXT0e),
		.REGWRITE2_RG_EXT(REGWRITE2_RG_EXT0e),
		.WRDATA1_RG_EXT(WRDATA1_RG_EXT0e),
		.WRDATA2_RG_EXT(WRDATA2_RG_EXT0e),
		.WRREG1_RG_EXT(WRREG1_RG_EXT0e),
		.WRREG2_RG_EXT(WRREG2_RG_EXT0e),
		
		//INPUTS CP0
		.ENEXT_CP0(EXTCTR_CP0[EXT0+EXTEN]),
		.EXCEXT_CP0(EXCEXT_CP0),
		.LDEXT_CP0(EXTCTR_CP0[EXT0+EXTLD]),
		.PER_EXT_IN(EXTCTR_CP0[EXT0+EXTPER]),
		.PRIV_EXT_IN(EXTCTR_CP0[EXT0+EXTPRIV]),
		.EXTCLK_EN_IN(EXTCTR_CP0[EXT0+EXTCLKEN]),
		//OUTPUTS CP0
		.ENEXT_EXT(ENEXT_EXT0e),
		.EXCEXT_EXT(EXCEXT_EXT0e),
		.LDEXT_EXT(LDEXT_EXT0e),
		.EXTCLK_EN_OUT(EXTCLK_EN_EXT0e),

		//INPUTS MEM
		.BAT_EN_IN(W_PER0_BATEN_MEM),
		.EXT_MDATA_IN(W_EXT_MDATA),
		.EXT_MDATAVLD_IN(W_EXT_MDATAVLD),
		.PER_MADDR_IN(W_MADDR_PER),
		.PER_MDATA_IN(W_MDATA_PER),
		.PER_MOE_IN(W_MOE_PER),
		.PER_MWE_IN(W_MWE_PER),
		.SNOOP_MADDR_IN(W_MADDR_SNOOP),
		.SNOOP_MDATA_IN(W_MDATA_SNOOP),
		.SNOOP_MDATAVLD_IN(W_MDATAVLD_SNOOP), 
		.SNOOP_MOE_IN(W_MOE_SNOOP),
		.SNOOP_MWE_IN(W_MWE_SNOOP), 
		.SRT_IN(MSRT_PER_MEM),
		.BYTES_IN(MBYTES_PER_MEM),
		//OUTPUTS MEM
		.BAT_EN_OUT(W_PER0_BAT_EN),
		.EXT_ADDR_OUT(ADDR_EXT0_OUT),
		.EXT_DATA_OUT(MDATA_EXT0_OUT),
		.EXT_DATAVLD_OUT(MDATAVLD_EXT0_OUT),
		.EXT_OE_OUT(OE_EXT0_OUT),
		.EXT_WE_OUT(WE_EXT0_OUT),		
		.SRT_OUT(PER0_MSRT_OUT),
		.BYTES_OUT(BYTES_EXT0_OUT)
		);
		
	lock lk(
		.CLK(SYSCLK),
		.CLK_EN(EXTCLK_EN_EXT0e),
		.LD_EXT(LDEXT_EXT0e),
		.RESET(RESET_G),
		.PRESENT_K(PER0_PRESENT_IN),
		.PRESENT_LK(PER0_PRESENT_LK)
		);
		
/*****Data Forward****************************************************************/
		
	todf to_df(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_DF(RESET_DF),

		//INPUTS EX
		.RS_EX(RS_EX),
		.RT_EX(RT_EX),
		//OUTPUTS EX
		.RS_EX_DF(RS_EX_DF),
		.RT_EX_DF(RT_EX_DF),

		//INPUTS ID
		.RS_ID(RS_ID),
		.RT_ID(RT_ID),
		//OUTPUTS ID
		.RS_ID_DF(RS_ID_DF),
		.RT_ID_DF(RT_ID_DF),

		//INPUTS MA
		.REGWRITE_MA(REGWRITE_MA),
		.WRREG_MA(WRREG_MA),
		//OUTPUTS MA
		.RD_MA_DF(RD_MA_DF),
		.REGWRITE_MA_DF(REGWRITE_MA_DF),

		//INPUTS WB
		.REGWRITE_WB(REGWRITE_WB),
		.WRREG_WB(WRREG_WB),
		//OUTPUTS WB
		.RD_WB_DF(RD_WB_DF),
		.REGWRITE_WB_DF(REGWRITE_WB_DF)
		);

	dataforward df(
		.CLK(SYSCLK),
		.FORWARD_A_EX(FORWARD_A_EX_DF),
		.FORWARD_A_ID(FORWARD_A_ID_DF),
		.FORWARD_B_EX(FORWARD_B_EX_DF),
		.FORWARD_B_ID(FORWARD_B_ID_DF),
		.RD_MA(RD_MA_DF),
		.RD_WB(RD_WB_DF),
		.REGWRITE_MA(REGWRITE_MA_DF),
		.REGWRITE_WB(REGWRITE_WB_DF),
		.RESET(RESET_DF),
		.RS_EX(RS_EX_DF),
		.RS_ID(RS_ID_DF),
		.RT_EX(RT_EX_DF),
		.RT_ID(RT_ID_DF)
		);

/*****Hazard****************************************************************/

	tohz to_hz(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_HZ(RESET_HZ),

		//INPUTS PA
		.EXTSTALL_PA(EXTSTALL_PA),
		//OUTPUTS PA
		.EXTSTALL_HZ(EXTSTALL_HZ),

		//INPUTS ID
		.BRANCH_ID(BRANCH_ID),
		.MULT_ID(MULT_ID),
		.RS_ID(RS_ID),
		.RT_ID(RT_ID),
		//OUTPUTS ID
		.BRANCH_ID_HZ(BRANCH_ID_HZ),
		.MULT_HZ(MULT_HZ),
		.RS_ID_HZ(RS_ID_HZ),
		.RT_ID_HZ(RT_ID_HZ),

		//INPUTS EX
		.MEMREAD_EX(MEMREAD_EX),
		.MULTRDY_EX(MULTRDY_EX),
		.RTD_EX(WRREG_EX),
		.LC_EX(LC_EX),
		//OUTPUTS EX
		.MEMREAD_EX_HZ(MEMREAD_EX_HZ),
		.MULTRDY_HZ(MULTRDY_HZ),
		.RTD_EX_HZ(RTD_EX_HZ),
		.LC_EX_HZ(LC_EX_HZ),

		//INPUTS MA
		.MEMREAD_MA(MEMREAD_MA),
		.RTD_MA(WRREG_MA),
		//OUTPUTS MA
		.MEMREAD_MA_HZ(MEMREAD_MA_HZ),
		.RTD_MA_HZ(RTD_MA_HZ),

		//INPUTS MEM
		.IRQ0_MEM(IRQ0_MEM),
		//OUTPUTS MEM
		.IRQ0_HZ(IRQ0_HZ)
		);

	hazard hz(
		.BRANCH_ID(BRANCH_ID_HZ),
		.CLK(SYSCLK),
		.CP0REGWRITE_EX(CP0REGWRITE_EX),
		.EXTSTALL(EXTSTALL_HZ),
		.IRQ0(IRQ0_HZ),
		.MEMREAD_EX(MEMREAD_EX_HZ),
		.MEMREAD_MA(MEMREAD_MA_HZ),
		.MULT_ID(MULT_HZ),
		.MULT_RDY(MULTRDY_HZ),
		.RESET(RESET_HZ),
		.RS_ID(RS_ID_HZ),
		.RT_ID(RT_ID_HZ),
		.RTD_EX(RTD_EX_HZ),
		.RTD_MA(RTD_MA_HZ),
		.STALL(STALL_HZ),
		.LC_EX(LC_EX_HZ)
		);

/*****Registers****************************************************************/

	torg to_rg(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_RG(RESET_RG),

		//INPUTS PA
		.GR_EXT0_PA(GR_EXT0_PA),
		.GR_EXT1_PA(GR_EXT1_PA),
		//OUTPUT PA
		.GR_RG(GR_RG),
		
		//INPUTS ID	
		.RS_ID(RS0_ID),
		.RT_ID(RT0_ID),

		//INPUTS EX
		.HI_EX(HI_EX),
		.LO_EX(LO_EX),
		.FPSTATUS_EX(32'hffffffff),
		.FPIMP_EX(32'hffffffff),
		//OUTPUTS EX
		.HI_RG(HI_RG),
		.LO_RG(LO_RG),
		.FPSTATUS_RG(FPSTATUS_RG),
		.FPIMP_RG(FPIMP_RG),

		//INPUTS MA
		.RD_MA(WRREG_MA),
		.RDDATA_MA(ALURESULT_MA),
		.REGWRITE_MA(REGWRITE_MA),
		//OUTPUTS MA
		.RD_MA_RG(RD_MA_RG),
		.RDDATA_MA_RG(RDDATA_MA_RG),
		.REGWRITE_MA_RG(REGWRITE_MA_RG),

		//INPUTS WB
		.BHLS_WB(BHLS_WB),
		.DMADD_WB(DMADD_WB),
		.REGWRITE_WB(REGWRITE_WB),
		.RNL_WB(RNL_WB),
		.WRREG_WB(WRREG_WB),
		.WRREGDATA_WB(WRREGDATA_WB),
		//OUTPUTS WB
		.BHLS_RG(BHLS_RG),
		.DMADD_RG(DMADD_RG),
		.RD_WB_RG(RD_WB_RG),
		.RDDATA_WB_RG(RDDATA_WB_RG),
		.REGWRITE_WB_RG(REGWRITE_WB_RG),
		.RNL_RG(RNL_RG),

		//INPUTS EXT0
		.RDREG1_EXT0(RDREG1_EXT0e),
		.RDREG2_EXT0(RDREG2_EXT0e),
		.RDREG3_EXT0(RDREG3_EXT0e),
		.RDREG4_EXT0(RDREG4_EXT0e),
		.REGWRITE1_EXT0(REGWRITE1_EXT0e),
		.REGWRITE2_EXT0(REGWRITE2_EXT0e),
		.WRDATA1_EXT0(WRDATA1_EXT0e),
		.WRDATA2_EXT0(WRDATA2_EXT0e),
		.WRREG1_EXT0(RDREG1_EXT0e),
		.WRREG2_EXT0(RDREG2_EXT0e),
		.FSPECIAL_EXT0(fspecial_EXT0e),
		//INPUTS EXT1
		.RDREG1_EXT1(5'b0),						/* GND EXT1 Signals */
		.RDREG2_EXT1(5'b0),						/* GND EXT1 Signals */
		.RDREG3_EXT1(5'b0),						/* GND EXT1 Signals */
		.RDREG4_EXT1(5'b0),						/* GND EXT1 Signals */
		.REGWRITE1_EXT1(1'b0),					/* GND EXT1 Signals */
		.REGWRITE2_EXT1(1'b0),					/* GND EXT1 Signals */
		.WRDATA1_EXT1(32'b0),					/* GND EXT1 Signals */
		.WRDATA2_EXT1(32'b0),					/* GND EXT1 Signals */
		.WRREG1_EXT1(5'b0),						/* GND EXT1 Signals */
		.WRREG2_EXT1(5'b0),						/* GND EXT1 Signals */
		.FSPECIAL_EXT1(1'b0),					/* GND EXT1 Signals */
		//OUTPUTS EXT
		.RDREG1_RG(RDREG1_RG),
		.RDREG2_RG(RDREG2_RG),
		.RDREG3_RG(RDREG3_RG),
		.RDREG4_RG(RDREG4_RG),
		.REGWRITE1_RG(REGWRITE1_RG),
		.REGWRITE2_RG(REGWRITE2_RG),
		.WRDATA1_RG(WRDATA1_RG),
		.WRDATA2_RG(WRDATA2_RG),
		.WRREG1_RG(WRREG1_RG),
		.WRREG2_RG(WRREG2_RG),
		.FSPECIAL_RG(FSPECIAL_RG),
	
		//INPUTS CP0
		.RE_CP0(RE_CP0),
		.STATUS_CP0(STATUS_CP0),
		.BADVADDR_CP0(BADVADDR_CP0),
		.CAUSE_CP0(CAUSE_CP0),
		//OUTPUTS CP0
		.RE_RG(RE_RG),
		.STATUS_RG(STATUS_RG),
		.BADVADDR_RG(BADVADDR_RG),
		.CAUSE_RG(CAUSE_RG)
		);

	registerfile4 regs(
		.BHLS1(BHLS_RG),
		.CLK(SYSCLK),
		.DMADD(DMADD_RG),
		.EDF(RE_RG),
		.EMPTY_WRBUF(EMPTY_RG),
		.FULL_WRBUF(FULL_RG),
		.GR(GR_RG),
		.PCLK(PCLK),
		.RD_MA(RD_MA_RG),
		.RD_WB(RD_WB_RG),
		.RDDATA_MA(RDDATA_MA_RG),
		.RDDATA_WB(RDDATA_WB_RG),
		.RDDATA1(RDREG1DATA_RG),
		.RDDATA2(RDREG2DATA_RG),
		.RDDATA3(RDREG3DATA_RG),
		.RDDATA4(RDREG4DATA_RG),
		.RDREG1(RDREG1_RG),
		.RDREG2(RDREG2_RG),
		.RDREG3(RDREG3_RG),
		.RDREG4(RDREG4_RG),
		.REGRDY(REGRDY_RG),
		.REGWRITE_MA(REGWRITE_MA_RG),
		.REGWRITE_WB(REGWRITE_WB_RG),
		.REGWRITE1(REGWRITE1_RG),
		.REGWRITE2(REGWRITE2_RG),
		.RESET(RESET_RG),
		.RNL1(RNL_RG),
		.WRDATA1(WRDATA1_RG),
		.WRDATA2(WRDATA2_RG),		
		.WRREG1(WRREG1_RG),
		.WRREG2(WRREG2_RG),
		.FSPECIAL(FSPECIAL_RG),
		.STATUS_REGISTER(STATUS_RG),
		.BADVADDR_REGISTER(BADVADDR_RG),
		.CAUSE_REGISTER(CAUSE_RG),
		.FPSTATUS_REGISTER(FPSTATUS_RG),
		.FPIMP_REGISTER(FPIMP_RG),
		.MULT_HI(HI_RG),
		.MULT_LO(LO_RG)
		/*,
		//Debug Ports
		.REGDNE(REGDNE),
		.regfile00(regfile[0]),
		.regfile01(regfile[1]), 
		.regfile02(regfile[2]),
		.regfile03(regfile[3]),
		.regfile04(regfile[4]),
		.regfile05(regfile[5]),
		.regfile06(regfile[6]),
		.regfile07(regfile[7]),
		.regfile08(regfile[8]),
		.regfile09(regfile[9]),
		.regfile10(regfile[10]),
		.regfile11(regfile[11]),
		.regfile12(regfile[12]),
		.regfile13(regfile[13]),
		.regfile14(regfile[14]),
		.regfile15(regfile[15]),
		.regfile16(regfile[16]),
		.regfile17(regfile[17]),
		.regfile18(regfile[18]),
		.regfile19(regfile[19]),
		.regfile20(regfile[20]),
		.regfile21(regfile[21]),
		.regfile22(regfile[22]),
		.regfile23(regfile[23]),
		.regfile24(regfile[24]),
		.regfile25(regfile[25]),
		.regfile26(regfile[26]),
		.regfile27(regfile[27]),
		.regfile28(regfile[28]),
		.regfile29(regfile[29]),
		.regfile30(regfile[30]),
		.regfile31(regfile[31])*/
		);

/*****Coprocessor 0****************************************************************/

	tocp0 to_cp0(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_CP0(RESET_CP0),

		//INPUTS PA
		.EXTSTALL_PA(EXTSTALL_PA),
		.GR_EXT0_PA(GR_EXT0_PA),
		.GR_EXT1_PA(GR_EXT1_PA),
		//OUTPUTS PA
		.EXTSTALL_CP0(EXTSTALL_CP0),

		//INPUTS IF
		.EXC_IF(EXC_IF),
		.EXCADD_IF(PC_IF),
		.EXCCODE_IF(EXCCODE_IF),
		.EXCPC_IF(PC_IF),
		//OUTPUTS IF
		.EXCADD_IF_CP0(EXCADD_IF_CP0),
		.EXC_IF_CP0(EXC_IF_CP0),
		.EXCCODE_IF_CP0(EXCCODE_IF_CP0),
		.EXCPC_IF_CP0(EXCPC_IF_CP0),

		//INPUTS ID
		.BRANCH_ID(BRANCH_ID),
		.BUBBLE_ID(BUBBLE_ID),
		.EXC_ID(EXC_ID),
		.EXCADD_ID(32'b0),
		.EXCCODE_ID(EXCCODE_ID),
		.EXCPC_ID(EXCPC_ID),
		.FLUSH_ID(FLUSH_ID),
		//OUTPUTS ID
		.BD_ID_CP0(BD_ID_CP0),
		.BUBBLE_ID_CP0(BUBBLE_ID_CP0),
		.EXC_ID_CP0(EXC_ID_CP0),
		.EXCADD_ID_CP0(EXCADD_ID_CP0),
		.EXCCODE_ID_CP0(EXCCODE_ID_CP0),
		.EXCPC_ID_CP0(EXCPC_ID_CP0),
		.FLUSH_CP0(FLUSH_CP0),

		//INPUTS EX
		.BRANCH_EX(BRANCH_EX),
		.BUBBLE_EX(BUBBLE_EX),
		.EXC_EX(EXC_EX),
		.EXCADD_EX(32'b0),
		.EXCCODE_EX(EXCCODE_EX),
		.EXCPC_EX(PC_EX),
		.CP0RDREG_EX(RD_EX),
		.CP0REGWRITE_EX(CP0REGWRITE_EX),
		.CP0WRREG_EX(WRREG_EX),
		.CP0WRREGDATA_EX(ALURESULT_EX),
		.RFE_EX(RFE_EX),
		//OUTPUTS EX
		.BD_EX_CP0(BD_EX_CP0),
		.BUBBLE_EX_CP0(BUBBLE_EX_CP0),
		.EXC_EX_CP0(EXC_EX_CP0),
		.EXCADD_EX_CP0(EXCADD_EX_CP0),
		.EXCCODE_EX_CP0(EXCCODE_EX_CP0),
		.EXPC_EX_CP0(EXCPC_EX_CP0),
		.CP0RDREG_CP0(CP0RDREG_CP0),
		.CP0REGWRITE_CP0(CP0REGWRITE_CP0),
		.CP0WRREG_CP0(CP0WRREG_CP0),
		.CP0WRREGDATA_CP0(CP0WRREGDATA_CP0),
		.RFE_CP0(RFE_CP0),

		//INPUTS MA
		.BRANCH_MA(BRANCH_MA),
		.BUBBLE_MA(BUBBLE_MA),
		.EXC_MA(EXC_MA),
		.EXCADD_MA(ALURESULT_MA),
		.EXCCODE_MA(EXCCODE_MA),
		.EXCPC_MA(PC_MA),
		//OUTPUTS MA
		.BD_MA_CP0(BD_MA_CP0),
		.BUBBLE_MA_CP0(BUBBLE_MA_CP0),
		.EXC_MA_CP0(EXC_MA_CP0),
		.EXCADD_MA_CP0(EXCADD_MA_CP0),
		.EXCCODE_MA_CP0(EXCCODE_MA_CP0),
		.EXPC_MA_CP0(EXCPC_MA_CP0),

		//INPUTS WB
		.BRANCH_WB(BRANCH_WB),
		//OUTPUTS WB
		.BD_WB_CP0(BD_WB_CP0),

		//INPUTS EXT0
		.BD_EXT0(RDREG4_EXT0e[2]),
		.CP0RDREG_EXT0(5'b0),
		.CP0REGWRITE_EXT0(1'b0),
		.CP0WRREG_EXT0(5'b0),
		.CP0WRREGDATA_EXT0(32'b0),
		.EXC_EXT0(EXC_EXT0e),
		.EXCADD_EXT0(32'b0),
		.EXCCODE_EXT0({RDREG4_EXT0e[1:0],RDREG3_EXT0e}),
		.EXCPC_EXT0(EXTADD_EXT0e),
		.REMA_EXT0(REMA_EXT0e),
		.REWB_EXT0(REWB_EXT0e),
		.RFE_EXT0(1'b0),

		//INPUTS EXT1
		.BD_EXT1(1'b0),					/* GND EXT1 Signals */
		.CP0RDREG_EXT1(5'b0),			/* GND EXT1 Signals */
		.CP0REGWRITE_EXT1(1'b0),		/* GND EXT1 Signals */
		.CP0WRREG_EXT1(5'b0),			/* GND EXT1 Signals */
		.CP0WRREGDATA_EXT1(32'b0),		/* GND EXT1 Signals */
		.EXC_EXT1(1'b0),					/* GND EXT1 Signals */
		.EXCADD_EXT1(32'b0),				/* GND EXT1 Signals */
		.EXCCODE_EXT1(7'b0),				/* GND EXT1 Signals */
		.EXCPC_EXT1(32'b0),				/* GND EXT1 Signals */
		.REMA_EXT1(1'b0),					/* GND EXT1 Signals */
		.REWB_EXT1(1'b0),					/* GND EXT1 Signals */
		.RFE_EXT1(1'b0),					/* GND EXT1 Signals */

		//OUTPUS EXT
		.BD_EXT_CP0(BD_EXT_CP0),
		.EXC_EXT_CP0(EXC_EXT_CP0),
		.EXCADD_EXT_CP0(EXCADD_EXT_CP0),
		.EXCCODE_EXT_CP0(EXCCODE_EXT_CP0),
		.EXPC_EXT_CP0(EXCPC_EXT_CP0),
		.REMA_CP0(REMA_CP0),
		.REWB_CP0(REWB_CP0),

		//INPUTS HZ
		.STALL_HZ(STALL_HZ),
		//OUTPUTS HZ
		.STALL_CP0(STALL_CP0),

		//INPUTS MEM
		.IRQ0_MEM(IRQ0_MEM),
		//OUTPUTS MEM
		.IRQ0_CP0(IRQ0_CP0)
		);

	coprocessor0_sim cp0(
		.BD_EX(BD_EX_CP0),
		.BD_EXT(BD_EXT_CP0),
		.BD_ID(BD_ID_CP0),
		.BD_MA(BD_MA_CP0),
		.BD_WB(BD_WB_CP0),
		.BUBBLE_EX(BUBBLE_EX_CP0),
		.BUBBLE_ID(BUBBLE_ID_CP0),
		.BUBBLE_MA(BUBBLE_MA_CP0),
		.CLK(SYSCLK),
		.CP(CP_CP0),
		.EXC(EXC_CP0),
		.EXC_EX(EXC_EX_CP0),
		.EXC_EXT(EXC_EXT_CP0),
		.EXC_ID(EXC_ID_CP0),
		.EXC_IF(EXC_IF_CP0),
		.EXC_MA(EXC_MA_CP0),
		.EXCADD_CI_EX(EXCPC_EX_CP0),
		.EXCADD_CI_EXT(EXCPC_EXT_CP0),
		.EXCADD_CI_ID(EXCPC_ID_CP0),
		.EXCADD_CI_IF(EXCPC_IF_CP0),
		.EXCADD_CI_MA(EXCPC_MA_CP0),
		.EXCADD_CD_EX(EXCADD_EX_CP0),
		.EXCADD_CD_EXT(EXCADD_EXT_CP0),
		.EXCADD_CD_ID(EXCADD_ID_CP0),
		.EXCADD_CD_IF(EXCADD_IF_CP0),
		.EXCADD_CD_MA(EXCADD_MA_CP0),
		.EXCADD_R(EXCADDR_CP0),
		.EXCCODE_EX(EXCCODE_EX_CP0),
		.EXCCODE_EXT(EXCCODE_EXT_CP0),
		.EXCCODE_ID(EXCCODE_ID_CP0),
		.EXCCODE_IF(EXCCODE_IF_CP0),
		.EXCCODE_MA(EXCCODE_MA_CP0),
		.EXCEXT(EXCEXT_CP0),
		.EXTCTR(EXTCTR_CP0),
		.EXTSTALL(EXTSTALL_CP0),
		.IRQ0(1'b0),							/* Interrupt Line Unused */
		.IRQ1(1'b0),							/* Interrupt Line Unused */
		.IRQ2(1'b0),							/* Interrupt Line Unused */
		.IRQ3(1'b0),							/* Interrupt Line Unused */
		.IRQ4(1'b0),							/* Interrupt Line Unused */
		.IRQ5(IRQ0_CP0),
		.KU(KU_CP0),
		.PCLK(PCLK),
		.RDDATA1(CP0RDDATA1_CP0),
		.RE(RE_CP0),
		.REGWRITE(CP0REGWRITE_CP0),
		.RESET(RESET_CP0),
		.RST(RST),
		.RDREG1(CP0RDREG_CP0),
		.RDREG2(5'b0),							/* Read Port Unused */
		.RFE(RFE_CP0),
		.WRDATA(CP0WRREGDATA_CP0),
		.WRREG(CP0WRREG_CP0),
		.EXEN(EXEN_CP0),
		.STATUS(STATUS_CP0),
		.BADVADDR(BADVADDR_CP0),
		.CAUSE(CAUSE_CP0)
		/*,
		//Debug Ports
		.cp0regfile00(cp0regfile[0]),
		.cp0regfile01(cp0regfile[1]),
		.cp0regfile02(cp0regfile[2]),
		.cp0regfile03(cp0regfile[3]),
		.cp0regfile04(cp0regfile[4]),
		.cp0regfile05(cp0regfile[5]),
		.cp0regfile06(cp0regfile[6]),
		.cp0regfile07(cp0regfile[7]),
		.cp0regfile08(cp0regfile[8]),
		.cp0regfile09(cp0regfile[9]),
		.cp0regfile10(cp0regfile[10]),
		.cp0regfile11(cp0regfile[11]),
		.cp0regfile12(cp0regfile[12]),
		.cp0regfile13(cp0regfile[13]),
		.cp0regfile14(cp0regfile[14]),
		.cp0regfile15(cp0regfile[15]),
		.cp0regfile16(cp0regfile[16]),
		.cp0regfile17(cp0regfile[17]),
		.cp0regfile18(cp0regfile[18]),
		.cp0regfile19(cp0regfile[19]),
		.cp0regfile20(cp0regfile[20]),
		.cp0regfile21(cp0regfile[21]),
		.cp0regfile22(cp0regfile[22]),
		.cp0regfile23(cp0regfile[23]),
		.cp0regfile24(cp0regfile[24]),
		.cp0regfile25(cp0regfile[25]),
		.cp0regfile26(cp0regfile[26]),
		.cp0regfile27(cp0regfile[27]),
		.cp0regfile28(cp0regfile[28]),
		.cp0regfile29(cp0regfile[29]),
		.cp0regfile30(cp0regfile[30]),
		.cp0regfile31(cp0regfile[31])*/
		);

/*****Memory****************************************************************/

	tomem to_mem(
		.CLK(SYSCLK),
		.PCLK(PCLK),
		.RESET(RESET_G),
		.RESET_MEM(RESET_MEM),

		//INPUTS PA
		.GR_EXT0_IN(GR_EXT0_PA),
		.GR_EXT1_IN(GR_EXT1_PA),

		//INPUTS IF
		.PC_IF(PPC_IF),
		//OUTPUTS IF
		.PC_MEM(PC_MEM),

		//INPUTS ID
		.FLUSH_ID(FLUSH_ID),
		//OUTPUTS ID
		.FLUSH_MEM(FLUSH_MEM),

		//INPUTS MA
		.BLS_MA(BLS_MA),
		.DMADD_MA(DMADD_MA),
		.DMDATAIN_MA(MEMDATAINEDF_MA),
		.HLS_MA(HLS_MA),
		.MEMREAD_MA(MEMREAD_MA),
		.MEMWRITE_MA(CMEMWRITE_MA),
		.RNL_MA(RNL_MA),
		.LC_MA(LC_MA),
		//OUTPUTS MA
		.BLS_MEM(BLS_MEM),
		.DMADD_MEM(DMADD_MEM),
		.DMDATAIN_MEM(DMDATAIN_MEM),
		.HLS_MEM(HLS_MEM),
		.MEMWRITE_MEM(MEMWRITE_MEM),
		.MEMREAD_MEM(MEMREAD_MEM),
		.RNL_MEM(RNL_MEM),
		.LC_MEM(LC_MEM),

		//INPUTS WB
		.BUBBLE_WB(BUBBLE_WB),
		.PC_WB(PC_WB),
		//OUTPUTS WB
		.BUBBLE_WB_MEM(BUBBLE_WB_MEM),
		.PC_WB_MEM(PC_WB_MEM),

		//INPUTS HZ
		.STALL_HZ(STALL_HZ),
		//OUTPUTS HZ
		.STALL_MEM(STALL_MEM),

		//INPUTS CP0
		.EXC_IF_CP0(EXC_CP0[0]),
		.EXC_MA_CP0(EXC_CP0[3]),
		.EXT0_PER_IN(EXTCTR_CP0[EXT0+EXTPER]),
		.EXT1_PER_IN(1'b0),						/* GND EXT1 Signals */
		.RE_CP0(RE_CP0),
		//OUTPUTS CP0
		.EXC_IF_MEM(EXC_IF_MEM),
		.EXC_MA_MEM(EXC_MA_MEM),
		.RE_MEM(RE_MEM),
		.PER_EXT0_MEM(PER_EXT0_MEM),
		.PER_EXT1_MEM(PER_EXT1_MEM),

		//INPUTS EXT0
		.BLS_EXT0_IN(BLS_EXT0_IN),
		.EXC_EXT0_IN(EXC_EXT0_IN),
		/*.EXTPER0_DNE_IN(PER0_DNE_IN),*/
		/*.EXTPER0_MY_ADDR_IN(PER0_MY_ADDR_IN),*/
		.HLS_EXT0_IN(HLS_EXT0_IN), 
		.MADDR_EXT0_IN(MADDR_EXT0_IN),
		.MDATA_EXT0_IN(MDATA_EXT0_IN),
		.MLOCK_EXT0_IN(MLOCK_EXT0_IN),
		.MOE_EXT0_IN(MOE_EXT0_IN),
		.MWE_EXT0_IN(MWE_EXT0_IN),
		.RNL_EXT0_IN(RNL_EXT0_IN),
		.PER0_DNE_IN(PER0_DNE_IN),
		.PER0_MY_ADDR_IN(PER0_MY_ADDR_IN),
		.PER0_PRESENT_IN(PER0_PRESENT_LK),
		.PER0_PRIVILEGE_IN(PER0_PRIVILEGE_IN),
		.PER0_WANTS_INTR_IN(PER0_WANTS_INTR_IN),
		.INT_EXT0_IN(INT_EXT0_IN),
		//OUTPUTS EXT0
		.PER0_DNE_OUT(PER0_DNE_MEM),
		.PER0_MY_ADDR_OUT(PER0_MY_ADDR_MEM),
		.PER0_PRESENT_OUT(PER0_PRESENT_MEM),
		.PER0_PRIVILEGE_OUT(PER0_PRIVILEGE_MEM),
		.PER0_WANTS_INTR_OUT(PER0_WANTS_INTR_MEM),
		.INT_EXT0_OUT(INT_EXT0_MEM),

		//INPUTS EXT1
		.BLS_EXT1_IN(1'b0),						/* GND EXT1 Signals */
		.EXC_EXT1_IN(1'b0),						/* GND EXT1 Signals */
		/*.EXTPER1_DNE_IN(1'b0),					/* GND EXT1 Singals */
		/*.EXTPER1_MY_ADDR_IN(1'b0),				/* GND EXT1 Signals */
		.HLS_EXT1_IN(1'b0),						/* GND EXT1 Signals */
		.MADDR_EXT1_IN(32'b0),					/* GND EXT1 Signals */
		.MDATA_EXT1_IN(32'b0),					/* GND EXT1 Signals */
		.MLOCK_EXT1_IN(1'b0),					/* GND EXT1 Signals */
		.MOE_EXT1_IN(1'b0),						/* GND EXT1 Signals */
		.MWE_EXT1_IN(1'b0),						/* GND EXT1 Signals */
		.RNL_EXT1_IN(1'b0),						/* GND EXT1 Signals */
		.PER1_DNE_IN(1'b1),
		.PER1_MY_ADDR_IN(1'b0),
		.PER1_PRESENT_IN(1'b0),
		.PER1_PRIVILEGE_IN(1'b0),
		.PER1_WANTS_INTR_IN(1'b0),
		.INT_EXT1_IN(1'b0),
		//OUTPUTS EXT1
		.PER1_DNE_OUT(PER1_DNE_MEM),
		.PER1_MY_ADDR_OUT(PER1_MY_ADDR_MEM),
		.PER1_PRESENT_OUT(PER1_PRESENT_MEM),
		.PER1_PRIVILEGE_OUT(PER1_PRIVILEGE_MEM),
		.PER1_WANTS_INTR_OUT(PER1_WANTS_INTR_MEM),
		.INT_EXT1_OUT(INT_EXT1_MEM),
		
		//OUTPUTS EXT
		.BLS_EXT_OUT(W_BLS_EXT), 
		.EXC_EXT_OUT(W_EXC_EXT),
		.HLS_EXT_OUT(W_HLS_EXT),
		.MADDR_EXT_OUT(W_MADDR_EXT),
		.MDATA_EXT_OUT(W_MDATA_EXT),
		.MLOCK_EXT_OUT(W_MLOCK_EXT),
		.MOE_EXT_OUT(W_MOE_EXT),
		.MWE_EXT_OUT(W_MWE_EXT),
		.RNL_EXT_OUT(W_RNL_EXT),
		.MDATA_OUT_PER_EXT(MDATA_PER_EXT_MEM),

		//INPUTS MEM
		.BAT_EN_EXT0_IN(W_PER0_BATEN_MEM),
		.BAT_EN_EXT1_IN(1'b0)					/* GND EXT1 Signals */
		);

	memory_interface mem(
		.BLS(BLS_MEM),
		.BLS_EXT_IN(W_BLS_EXT),
		.BRAMCLK(BRAMCLK),
		.BRDY(BRDY),
		.BYTE(BYTE),
		.BYTESEL(BYTESEL),
		.CE_FLASH(CE_FLASH),
		.CLK(SYSCLK),
		.CLK_EXT(CLK_EXT),
		.CLK_IN_PER(CLK_IN_PER),
		.DM_ADDR(DMADD_MEM),
		.DM_DATA_IN(DMDATAIN_MEM),
		.DM_DATA_OUT(DMDATAOUT_MEM),
		.DM_OE(MEMREAD_MEM),
		.DM_WE(MEMWRITE_MEM),
		.DNE(DNE),
		.EXC_EXT_MEM_IN(W_EXC_EXT),
		.EXC_IF_MEM(EXC_IF_MEM),
		.EXC_MA_MEM(EXC_MA_MEM),
		.FLASHCLK(FLASHCLK),
		.FLUSH(FLUSH_MEM),
		.GPIO_IN(GPIO_IN),
		.GPIO_OUT(GPIO_OUT),
		.GPIO_TR(GPIO_TR),
		.HLS(HLS_MEM),
		.HLS_EXT_IN(W_HLS_EXT),
		.IM_ADDR(PC_MEM),
		.IM_DATA_OUT(INSTR_MEM),
		.IM_OE(EXEN_CP0),
		.INT_EXT0_IN(INT_EXT0_MEM),
		.INT_EXT1_IN(INT_EXT1_MEM),
		.IRQ(IRQ0_MEM),
		.IRQ_SYSACE(IRQ_SYSACE),
		.MADDR_EXT_IN(W_MADDR_EXT),
		.MADDR_OUT(MADDR_OUT),
		.MADDR_PER_OUT(W_MADDR_PER),
		.MADDR_SNOOP_OUT(W_MADDR_SNOOP),
		.MBYTES_PER_OUT(MBYTES_PER_MEM),
		.MDATA_EXT_IN(W_MDATA_EXT),
		.MDATA_EXT_OUT(W_EXT_MDATA),
		.MDATA_IN(MDATA_IN),
		.MDATA_OUT(MDATA_OUT),
		.MDATA_PER_OUT(W_MDATA_PER),
		.MDATA_SNOOP_OUT(W_MDATA_SNOOP),
		.MDATA_TR(MDATA_TR),
		.MDATAP_IN(MDATAP_IN),
		.MDATAP_OUT(MDATAP_OUT),
		.MDATAP_TR(MDATAP_TR),
		.MDATAVLD_EXT_OUT(W_EXT_MDATAVLD),
		.MDATAVLD_SNOOP_OUT(W_MDATAVLD_SNOOP),
		.MEMCLK(MEMCLK),
		.MEMCLK2(MEMCLK2),
		.MLOCK_EXT_IN(W_MLOCK_EXT),
		.MOE_EXT_IN(W_MOE_EXT),
		.MOE_PER_OUT(W_MOE_PER),
		.MOE_SNOOP_OUT(W_MOE_SNOOP),
		.MODE(MODE),
		.MSRT_PER_OUT(MSRT_PER_MEM),
		.MWE_EXT_IN(W_MWE_EXT),
		.MWE_PER_OUT(W_MWE_PER), 
		.MWE_SNOOP_OUT(W_MWE_SNOOP),
		.NADVLD(NADVLD),
		.NBW(NBW),
		.NCE_SRAM(NCE_SRAM),
		.NCE_SYSACE(NCE_SYSACE),
		.NCEN(NCEN),
		.NOE_MEM(NOE_MEM),
		.NOE_PER(NOE_PER),
		.NRP(NRP),
		.NWE_MEM(NWE_MEM),
		.NWE_PER(NWE_PER),
		.PADDR_OUT(PADDR_OUT),
		.PC(PC_WB),
		.PCLK(PCLK),
		.PDATA_IN(PDATA_IN),
		.PDATA_OUT(PDATA_OUT),
		.PDATA_TR(PDATA_TR),
		.PER0_BAT_EN_OUT(W_PER0_BATEN_MEM),
		.PER0_DNE_IN(PER0_DNE_MEM),
		.PER0_PRESENT_IN(PER0_PRESENT_MEM),
		.PER0_PRIVILEGE_IN(PER0_PRIVILEGE_MEM),
		.PER0_WANTS_INTR_IN(PER0_WANTS_INTR_MEM),
		//.PER1_BAT_EN_OUT(),							/* GND EXT1 Signals */
		.PER1_DNE_IN(PER1_DNE_MEM),								/* GND EXT1 Signals */
		.PER1_PRESENT_IN(PER1_PRESENT_MEM),							/* GND EXT1 Signals */
		.PER1_PRIVILEGE_IN(PER1_PRIVILEGE_MEM),						/* GND EXT1 Signals */
		.PER1_WANTS_INTR_IN(PER1_WANTS_INTR_MEM),						/* GND EXT1 Signals */
		.RESET(RESET_MEM),
		.RNL(RNL_MEM),
		.RNL_EXT_IN(W_RNL_EXT),
		.RX(RX),
		.SCLK(SCLK),
		.SRAMCLK(SRAMCLK),
		.STALL(STALL_MEM),
		.TMRCLK(TMRCLK),
		.TX(TX),
		.USARTCLK(USARTCLK),
		.ZZ(ZZ),
		.RE(RE_MEM),
		.MDATA_PER_EXT_IN(MDATA_PER_EXT_MEM),
		.BUBBLE_WB(BUBBLE_WB_MEM),
		.PC_WB(PC_WB_MEM),
		.PER_EXT0(PER_EXT0_MEM),
		.PER_EXT1(PER_EXT1_MEM),
		.LC(LC_MEM),
		.ALS(ALS_MEM)
		);

endmodule
