/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Nathaniel Lee Lynch												*/
/*					Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	alu																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"	/* Constant Definitiions */

module alu(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	A,				/* ALU Operand A */
	input [31:0]	B,				/* ALU Operand A */
	input 			CARRY_IN,	/* Carry In */
	input 			CLK,			/* System Clock 50 - 100 MHZ */
	input [5:0]		CTR,			/* ALU Control/Opcode */
	input 			RESET,		/* System Reset */
	/* OUTPUT PORTS */
	output 			CARRY_OUT,	/* Carry Out */
	output 			OVERFLOW,	/* Overflow 1 */
	output 			OVERFLOW2,	/* Overflow 2 */
	output [31:0]	RESULT,		/* ALU Result */
	output 			ZERO			/* ALU Result is Zero */
	);

/*****Signals****************************************************************/

	wire [32:0]	ADD;		/* Signed Add Result */
	wire			ADDO;		/* Signed Add Overflow */
	wire [32:0]	ADDU;		/* Unsigned Add Result */
	wire [32:0]	AND;		/* And Result */
	wire [32:0]	NOR;		/* Nor Result */
	wire [32:0]	OR;		/* Or Result */
	wire [32:0]	SLT;		/* Set Less Than Result */
	wire [32:0]	SLTU;		/* Unsigned Set Less Than Result */
	wire [32:0]	SUB;		/* Signed Sub Result */
	wire 			SUBO;		/* Signed Sub Overflow */
	wire [32:0]	SUBU;		/* Unsigned Sub Result */
	wire [32:0]	XOR;		/* Xor Result */

/*****Registers****************************************************************/
	
	reg [32:0]	add_reg;		/* Add Result */
	reg 			addo_reg;	/* Signed Add Overflow */
	reg [32:0]	and_reg;		/* And Result */
	reg [32:0]	nor_reg;		/* Nor Result */
	reg [32:0]	or_reg;		/* Or Result */
	reg 			slt_reg;		/* Set Less Than Result */
	reg 			sltu_reg;	/* Unsigned Set Less Than Result */
	reg [32:0]	sub_reg;		/* Sub Result */
	reg 			subo_reg;	/* Signed Sub Overflow */
	reg [32:0]	xor_reg;		/* Xor Result */
	reg 			zero_reg;	/* ALU Result is Zero */

/*****Initialization****************************************************************/

	initial
	begin
		add_reg = 33'b0;
		addo_reg = 1'b0;
		and_reg = 33'b0;
		slt_reg = 1'b0;
		sltu_reg = 1'b0;
		sub_reg = 33'b0;
		subo_reg = 1'b0;
		nor_reg = 33'b0;
		or_reg = 33'b0;
		xor_reg = 33'b0;
		zero_reg = 1'b0;
	end

/*********************************************************************/

	assign ADD		=	add_reg;
	assign ADDO		=	addo_reg;
	assign ADDU		=	add_reg;
	assign AND		=	and_reg;
	assign SLT		=	{`BLANK32,slt_reg};
	assign SLTU		=	{`BLANK32,sltu_reg};
	assign SUB		=	sub_reg;
	assign SUBO		=	subo_reg;
	assign SUBU		=	sub_reg;
	assign NOR		=	nor_reg;
	assign OR		=	or_reg;
	assign XOR		=	xor_reg;
	assign ZERO		=	zero_reg;

/*********************************************************************/

	assign {CARRY_OUT,RESULT,OVERFLOW,OVERFLOW2}	=
		(CTR == `ADD)?		{ADD,ADDO,ADDO}:
		(CTR == `ADDU)?	{1'b0,ADDU[31:0],ADDU[32],1'b0}:
		(CTR == `SUB)?		{SUB,SUBO,SUBO}:
		(CTR == `SUBU)?	{1'b0,SUBU[31:0],SUBU[32],1'b0}:
		(CTR == `AND)?		{AND,2'b0}:
		(CTR == `OR)?		{OR,2'b0}:
		(CTR == `XOR)?		{XOR,2'b0}:
		(CTR == `NOR)?		{NOR,2'b0}:
		(CTR == `SLT)?		{SLT,2'b0}:
		(CTR == `SLTU)?	{SLTU,2'b0}:
		(CTR == `NOP)?		{1'b0,A,2'b0}:
								`BLANK35;

/*********************************************************************/

	/* Add Result */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	add_reg <= 33'b0;
		else 						add_reg <= A + B + CARRY_IN;
	end
	
	/* Signed Add Overflow */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	addo_reg <= 1'b0;
		else 						addo_reg <= (A[31] & B[31] & ~ADD[31]) | (~A[31] & ~B[31] & ADD[31]);
	end

	/* Sub Result */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	sub_reg <= 33'b0;
		else						sub_reg <= A - B;
	end

	/* Signed Sub Overflow */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	subo_reg <= 1'b0;
		else 						subo_reg <= (~A[31] & B[31] & SUB[31]) | (A[31] & ~B[31] & ~SUB[31]);
	end

	/* And Result */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	and_reg <= 33'b0;
		else 						and_reg <= {1'b0,(A & B)};
	end

	/* Or Result */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	or_reg <= 33'b0;
		else						or_reg <= {1'b0,(A | B)};
	end

	/* Nor Result */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	nor_reg <= 33'b0;
		else 						nor_reg <= {1'b0,~(A | B)};
	end

	/* Xor Result */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	xor_reg <= 33'b0;
		else						xor_reg <= {1'b0,(A ^ B)};
	end

	/* Set Less Than */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						slt_reg <= 1'b0;
		else
		begin
			if ({A[31],B[31]}== 2'b11)			slt_reg <= (A < B);
			else if ({A[31],B[31]}== 2'b00)	slt_reg <= (A < B);
			else if ({A[31],B[31]}== 2'b10)	slt_reg <= 1'b1;
			else										slt_reg <= 1'b0;
		end
	end

	/* Unsigned Set Less Than */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	sltu_reg <= 1'b0;
		else 						sltu_reg <= (A < B);
	end

	/* ALU Result is Zero */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	zero_reg <= 1'b0;
		else 						zero_reg <= ~| RESULT;
	end

endmodule
