/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Nathaniel Lee Lynch												*/
/*					Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	alu_full															*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"	/* Constant Definitiions */

module alu_full(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	A,				/* ALU Operand A */
	input [31:0]	B,				/* ALU Operand B */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input [5:0]		CTR,			/* ALU Control/Opcode */
	input				PCLK,			/* Pipeline Clock */
	input				RESET,		/* System Reset */
	input [4:0]		SHAMT,		/* Shift Amount */
	/* OUTPUT PORTS */
	output			CARRY_OUT,	/* Carry Out */
	output			OVERFLOW,	/* Overflow 1 */
	output			OVERFLOW2,	/* Overflow 2 */
	output			MULT_RDY,	/* Multiplication Unit Ready */
	output [31:0]	RESULT,		/* ALU Result */
	output			ZERO,			/* ALU Result is Zero */
	/* DEBUG PORTS */
	output [31:0]	HI,			/* Multiplication Unit Hi Register */
	output [31:0]	LO				/* Multiplication Unit Lo Register */
	);

/*****Signals****************************************************************/

	wire [31:0] RESULT_A;	/* ALU Result */
	wire [31:0] RESULT_M;	/* Multiplication Result */
	wire [31:0] RESULT_S;	/* Shift Result */

/*****Registers****************************************************************/
	
	reg [31:0] result_a_reg;	/* ALU Result */
	reg [31:0] result_m_reg;	/* Multiplication Result */
	reg [31:0] result_s_reg;	/* Shift Result */

/*****Initialization****************************************************************/
	
	initial
	begin
		result_a_reg = 32'b0; 
		result_m_reg = 32'b0; 
		result_s_reg = 32'b0;
	end

/*********************************************************************/

	assign RESULT =	(CTR[5:4] == 2'b10)?	result_a_reg:
							(CTR[5:4] == 2'b01)?	result_m_reg:
							(CTR[5:4] == 2'b00)?	result_s_reg:
							`BLANK32;

/*****Arithmetic Logic Unit****************************************************************/
				
	alu alu0(
		.A(A),
		.B(B),
		.CARRY_IN(1'b0),
		.CARRY_OUT(CARRY_OUT),
		.CLK(CLK),
		.CTR(CTR),
		.OVERFLOW(OVERFLOW),
		.OVERFLOW2(OVERFLOW2),
		.RESET(RESET),
		.RESULT(RESULT_A),
		.ZERO(ZERO)
		);

/*****Multiplication Unit****************************************************************/

	multiplier mult0(
		.A(A),
		.B(B),
		.CLK(CLK),
		.CTR(CTR),
		.PCLK(PCLK),
		.RDY(MULT_RDY),
		.RESET(RESET),
		.RESULT(RESULT_M),
		//Debug Ports
		.HI(HI),
		.LO(LO)
		);

/*****Shift Unit****************************************************************/

	shift shift0(
		.CLK(CLK),
		.RESET(RESET),
		.IN(B),
		.SHAMT(SHAMT),
		.A(A[4:0]),
		.CTR(CTR),
		.OUT(RESULT_S)
		);
		
/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			result_a_reg <= 32'b0; 
			result_m_reg <= 32'b0; 
			result_s_reg <= 32'b0;
		end
		else
		begin
			/* Latch Signals */
			result_a_reg <= RESULT_A; 
			result_m_reg <= RESULT_M; 
			result_s_reg <= RESULT_S;
		end
	end

endmodule
