/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	blockram_controller											*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/* This module interfaces to a Xilinx Singleport BlockRam IPCORE		*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module blockram_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,			/* Address Bus from Data Path Memory Interface */
	input				BRAMCLK,			/* BlockRam Clock 50 - 100 MHZ */
	input [1:0]		BYTES,			/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,			/* Configuration Base Address */
	input				CLK,				/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,			/* Data Bus from Data Path Memory Interface */ 
	input [31:0]	DATA_MEM_OUT,	/* Data Bus from BlockRam */
	input				OE,				/* Output Enable */
	input				RESET,			/* System Reset */
	input				SRT,				/* Start Transaction */
	input				WE,				/* Write Enable */
	/* OUTPUT PORTS */
	output [6:0]	ADDR_OUT,		/* Address Bus to BlockRam */
	output [31:0]	BASETAG,			/* BASETAG Register */
	output			CLK_OUT,			/* Clock to BlockRam 50 - 100 MHZ */
	output [31:0]	DATA_MEM_IN,	/* Data Bus to BlockRam */
	output [31:0]	DATA_OUT,		/* Data Bus to Data Path Memory Interface */
	output			DNE,				/* Transaction Done */
	output			MWE,				/* BlockRam Write Enable */
	output			MOE,				/* BlockRam Output Enable */
	output [31:0]	STATUS			/* Control Register */
	);

/*****Signals****************************************************************/
	
	wire [31:0]	BASE_ADDR;		/* BlockRam Base Address */
	wire [3:0]	BM;				/* Byte Selects */
	wire [31:0]	DATA_BRAM;		/* Data from BlockRam */
	wire [31:0]	DATA_IN_C;		/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;		/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;		/* Endian Flipped Outgoing Data */
	wire			DNE_BRAM;		/* BlockRam Interface Transaction Done */
	wire			EN;				/* Transaction Enabled for BlockRam Address Space */
	wire [31:0]	MAX_ADDR;		/* BlockRam Max Address */
	wire			MEN;				/* Transaction Enabled for BlockRam Configuration Space */
	wire [31:0]	REG_M;			/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;			/* Selected Configuration Register Data */
	wire [31:0]	WMASK_IN;		/* Incoming Data Write Mask */
	wire [31:0]	WMASK_REG;		/* Configuration Register Data Write Mask */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;		/* BASETAG Register */
	reg [31:0]	control;			/* Control Register */
	reg [31:0]	data_in_m_reg;	/* Masked Incoming Data */
	reg [3:0]	dnecnt;			/* Transaction Progress Counter */
	reg			dner;				/* Configuration Space Transaction Done */
	reg			en_reg;			/* Transaction Enabled for BlockRam Address Space */
	reg			men_reg;			/* Transaction Enabled for BlockRam Configuration Space */
	reg [31:0]	reg_m_0_reg;	/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;		/* Masked Configuration Register Data */

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter	BASETAGREG =	16'h0000;	/* BASETAG Register Relative Address */
	parameter	CONTROLREG =	16'h0004;	/* Control Register Relative Address */	

	/* BASETAG Register */
	parameter	BASEl =			31;			/* BASE Bit Range Left */
	parameter	BASEr =			16;			/* BASE Bit Range Right */
	parameter	TAGl = 			15;			/* TAG Bit Range Left */
	parameter	TAGr = 			0;				/* TAG Bit Range Right */
	parameter	TAG =				0;				/* TAG Value */
	parameter	DEFAULTBASE =	16'hffff;	/* Default Base Value */

	/* Control Register */
	parameter	SIZEl = 			31;			/* Size Bit Range Left */
	parameter	SIZEr = 			16;			/* Size Bit Range Right */
	parameter	BUS32 =			15;			/* 32-bit wide Bus */
	parameter	BUS16 =			14;			/* 16-bit wide Bus */
	parameter	BUS8 = 			13;			/* 8-bit wide Bus */
	parameter	RST = 			12;			/* Soft Reset */
	parameter	DEFAULTSIZE =	16'h0200;	/* Default Size Value */


/*****Initialization****************************************************************/

	initial
	begin
		basetagr = 32'b0;
		basetagr[BASEl:BASEr] = DEFAULTBASE;
		basetagr[TAGl:TAGr] = TAG;
		control = 32'b0;
		control[SIZEl:SIZEr] = DEFAULTSIZE;
		control[BUS32] = 1'b1;
		dnecnt = 4'b1111;
		dner = 1'b1;
		en_reg = 1'b0;
		men_reg = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
	end

/*********************************************************************/

	assign BASETAG = basetagr;
	assign DNE = DNE_BRAM & dner;
	assign STATUS = control;

/*****Address decoding****************************************************************/

	assign BASE_ADDR = (basetagr[BASEl:BASEr]<<BASEr);
	assign MAX_ADDR = ((basetagr[BASEl:BASEr]<<BASEr) | control[SIZEl:SIZEr] | 32'h0000ffff);

	assign EN = (ADDR_IN >= BASE_ADDR) & (ADDR_IN < MAX_ADDR);
	assign MEN = (CADDR[BASEl:BASEr] == ADDR_IN[BASEl:BASEr]);

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/

	assign DATA_OUT = (OE & en_reg)?DATA_BRAM:32'bz;
	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BASETAGREG)?basetagr:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == CONTROLREG)?control:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed1(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*****Blockram Interface****************************************************************/

	blockram_interface bi(
		.ADDR_IN(ADDR_IN[8:0]),
		.ADDR_OUT(ADDR_OUT),
		.BYTES(BYTES),
		.CLK(BRAMCLK),
		.CLK_OUT(CLK_OUT),
		.DATA_IN(DATA_IN),
		.DATA_MEM_IN(DATA_MEM_IN),
		.DATA_MEM_OUT(DATA_MEM_OUT),
		.DATA_OUT(DATA_BRAM),
		.DNE(DNE_BRAM),
		.OE(OE & en_reg),
		.MOE(MOE),
		.MWE(MWE),
		.WE(WE & en_reg),
		.RESET(RESET & ~control[RST]),
		.SRT(SRT & en_reg)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			men_reg = 1'b0;
			en_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			data_in_m_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			dnecnt = {dnecnt[2:0],dner};
			men_reg = MEN;
			en_reg = EN;
			reg_m_0_reg = REG_M_0;
			data_in_m_reg = DATA_IN_M;
			reg_m_reg = REG_M;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,control[RST],dnecnt[3],dner})
			4'b0xxx	:	begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[BASEl:BASEr] = DEFAULTBASE;
								basetagr[TAGl:TAGr] = TAG;
								control = 32'b0;
								control[SIZEl:SIZEr] = DEFAULTSIZE;
								control[BUS32] = 1'b1;
								dner = 1'b1;
							end
			4'b11xx	:	begin
								/* Software Reset */
								basetagr = 32'b0;
								basetagr[BASEl:BASEr] = DEFAULTBASE;
								basetagr[TAGl:TAGr] = TAG;
								control = 32'b0;
								control[SIZEl:SIZEr] = DEFAULTSIZE;
								control[BUS32] = 1'b1;
								dner = 1'b1;
							end
			4'b1000	:	begin
								/* Update Configuration Registers */
								dner = 1'b1;
								if (men_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										CONTROLREG	:	begin
																/* Update Control Register */
																control = data_in_m_reg | reg_m_reg;
															end
										BASETAGREG	:	begin
																/* Update Basetag Register */
																basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
															end
										default		:	begin
																/* Do Nothing */
																/* Address not Recognized */
															end
									endcase
								end
							end
			4'b10x1	:	begin
								/* Wait for incoming request */
								if (SRT & men_reg)
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
							end
			default	:	begin
								/* Do Nothing */
							end
		endcase
	end
	
endmodule

/*****Blockram Interface****************************************************************/

module blockram_interface(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [8:0]		ADDR_IN,			/* Address Bus from BlockRam Controller */
	input [1:0]		BYTES,			/* Number of Bytes to Read/Write */
	input				CLK,				/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,			/* Data Bus from BlockRam Controller */
	input [31:0]	DATA_MEM_OUT,	/* Data Bus from BlockRam */
	input				OE,				/* Output Enable */
	input				RESET,			/* System Reset */
	input				SRT,				/* Start Transaction */
	input				WE,				/* Write Enable */
	/* OUTPUT PORTS */
	output [6:0]	ADDR_OUT,		/* Address Bus to BlockRam */
	output			CLK_OUT,			/* Clock to BlockRam 50 - 100 MHZ */
	output [31:0]	DATA_MEM_IN,	/* Data Bus to BlockRam */
	output [31:0]	DATA_OUT,		/* Data Bus to BlockRam Controller */
	output			DNE,				/* Transaction Done */
	output			MWE,				/* BlockRam Write Enable */
	output			MOE				/* BlockRam Output Enable */
	);

/*****Signals****************************************************************/

	wire [3:0]	BM;			/* Byte Selects */
	wire [31:0]	DATA_IN_M;	/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_M;	/* Masked BlockRam Data */
	wire [31:0]	WMASK_IN;	/* Incoming Data Write Mask */
	wire [31:0]	WMASK_REG;	/* BlockRam Data Write Mask */

/*****Registers****************************************************************/

	reg [8:0]	addr_out_r;			/* Currect Address */
	reg [1:0]	bytes_r;				/* Number of Bytes to Read/Write */
	reg [31:0]	data_in_r;			/* Incoming Data to be Written */
	reg [31:0]	data_in_m_reg;		/* Incoming Data Masked */
	reg [31:0]	data_out_r;			/* BlockRam Data */
	reg [31:0]	data_out_m_reg;	/* Masked BlockRam Data */
	reg			dne_r;				/* Transaction Done */
	reg			oe_r;					/* BlockRam Output Enable */
	reg			we_r;					/* BlockRam Write Enable */

/*****Initialization****************************************************************/

	initial
	begin
		addr_out_r = 9'h1ff;
		bytes_r = 2'b0;
		data_in_r = 32'b0;
		data_in_m_reg = 32'b0;
		data_out_r = 32'b0;
		data_out_m_reg = 32'b0;
		dne_r = 1'b1;
		we_r = 1'b0;
		oe_r = 1'b0;
	end

/*********************************************************************/

	assign DNE = dne_r;
	assign MWE = we_r;
	assign MOE = oe_r;
	assign CLK_OUT = CLK;
	assign ADDR_OUT = addr_out_r[8:2];
	assign DATA_OUT = data_out_r;
	assign DATA_MEM_IN = data_in_r;

/*****Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN & WMASK_IN;
	assign DATA_OUT_M = DATA_MEM_OUT & WMASK_REG;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(addr_out_r[1:0]),
		.BYTES(bytes_r),
		.BW(BM)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			data_out_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			data_out_m_reg = DATA_OUT_M;
		end
	end

	always@(posedge CLK)
	begin
		case(RESET)
			1'b0		:	begin
								/* Reset */
								addr_out_r = 9'h1ff;
								bytes_r = 2'b0;
								data_in_r = 32'b0;
								data_out_r = 32'b0;
								dne_r = 1'b1;
								we_r = 1'b0;
								oe_r = 1'b0;
							end
			default	:	begin
								/* Interfacing Blockram */
								casex({dne_r,SRT,WE,OE,we_r,oe_r})
									6'b10xxxx	:	begin
															/* Done Waiting For Next Request */
															we_r = 1'b0;
															oe_r = 1'b0;
														end
									6'b11xxxx	:	begin
															/* Initializing */ 
															addr_out_r = ADDR_IN;
															bytes_r = BYTES;
															dne_r = 1'b0;
														end
									6'b0x1000	:	begin
															/* Writing */
															data_in_r = data_in_m_reg | data_out_m_reg;
															we_r = 1'b1;
														end
									6'bxxxx1x	:	begin 
															/* Finish Writing */
															we_r = 1'b0;
															dne_r = 1'b1;
														end
									6'b0x0100	:	begin
															/* Reading */
															oe_r = 1'b1;
														end
									6'bxxxxx1	:	begin
															/* Finish Reading */
															data_out_r = DATA_MEM_OUT;
															oe_r = 1'b0;
															dne_r = 1'b1;
														end
								endcase
						end
		endcase
	end
	
endmodule
