/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	clockmaster														*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module clockmaster(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input CLK,					/* System Clock 50 - 100 MHZ */
	input DCMLOCK,				/* Digital Clock Manager Lock */
	input MDNE,					/* Memory Done */
	input RESET,				/* Hardware Reset */
	input RST,					/* Soft Reset */
	/* OUTPUT PORTS */
	output PCLK,				/* Pipeline Clock */
	output RESET_G,			/* Debounced Global Reset */
	/* DEBUG PORTS */
	output GIANO_STALL,		/* Giano Signal */
	output GIANO_STALL2		/* Giano Signal */
	);

/*****Registers****************************************************************/

	reg			dcmlock_reg;		/* Digital Clock Manager Lock */
	reg [3:0]	dnecount;			/* Memory Done Counter */
	reg			mdne_reg;			/* Memory Done */
	reg [1:0]	mdnecnt;				/* Memory Done Edge Dectection */
	reg			pclk_reg;			/* Pipeline Clock */
	reg [15:0]	resetcount;			/* Reset Debounce Counter */
	reg			resetreg;			/* Debounced Global Reset */
	reg			rst_reg;				/* Soft Reset */

	reg			giano_stall_reg;	/* Giano Signal */
	reg			giano_stall2_reg;	/* Giano Signal */

/*****Parameters****************************************************************/

	parameter perioddiv2 	=	4'b1000;	
	parameter perioddiv		=	4'b1000;
	parameter slack 			=	4'b0100;

/*****Initialization****************************************************************/

	initial
	begin
		dcmlock_reg = 1'b0;
		dnecount = 4'b0;
		mdne_reg = 1'b0;
		mdnecnt = 2'b11;
		pclk_reg = 1'b0;
		resetcount = 16'b0;
		resetreg = 1'b0;
		rst_reg = 1'b0;

		giano_stall_reg = 1'b1;
		giano_stall2_reg = 1'b1;
	end

/*********************************************************************/

	assign PCLK 			= 	pclk_reg;
	assign RESET_G 		= 	resetreg;

	assign GIANO_STALL 	= 	giano_stall_reg;
	assign GIANO_STALL2 	=	giano_stall2_reg;

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Memory Done Edge Dectection */
		mdnecnt <= {mdnecnt[0],(MDNE | ~resetreg)};
	end

	always@(posedge CLK)
	begin
		/* Latch Inputs */
		dcmlock_reg <= DCMLOCK;
		mdne_reg <= MDNE;
		rst_reg <= RST;
	end

	always@(posedge CLK)
	begin
		case(mdne_reg | ~resetreg)
			1'b1		:	begin
								/* If Memory Operation DNE allow time for Pipeline Stages to Complete */
								if (dnecount == perioddiv2)
								begin
									/* Toggle Pipeline Clock */
									pclk_reg <= ~pclk_reg;
									giano_stall_reg <= 1'b1;
									dnecount <= 4'b1111;
								end
								else if (dnecount == perioddiv)
								begin
									giano_stall_reg <= 1'b0;
								end
								giano_stall2_reg <= 1'b1;
								dnecount <= dnecount + 1;
							end
			default	:	begin
								/* Wait for Memory Operations to finish */
								if ((dnecount < (perioddiv - slack)))
								begin
									dnecount <= dnecount + 1;
									giano_stall2_reg <= 1'b1;
								end
								else
								begin
									giano_stall2_reg <= 1'b0;
								end
							end
		endcase
	end

	always@(posedge CLK)
	begin
		/* Debounce Reset Button */
		if ((RESET == 1'b0) || (rst_reg == 1'b1) || (dcmlock_reg == 1'b0))
		begin
			resetreg <= 1'b0;
			resetcount <= 0;
		end
		else
		begin
			/* Synchronize Reset to Pipeline Clock */
			if ((~PCLK) && (resetcount[15]))
			begin
				resetreg <= 1'b1;
			end
			if (~resetcount[15])
			begin
				resetcount <= resetcount + 1;
			end
		end
	end

endmodule
