/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	coprocessor0_sim												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"

/*****Coprocessor0 Default Values****************************************************************/

/* 13 Cause Register Definitions */
`define	Cause		5'b1101	/* 13 Cause Register */
`define	BD			1'b1		/* Branch Delay */
`define	CE			2'b0		/* Coprocessor Error */
`define	IP			6'b0		/* Interrupts Pending */
`define	Sw			2'b0		/* Software Interruputs Pending */
`define	ExcCode 	4'b0		/* Exception Code */

/* 14 Exception Program Counter Definitions */
`define	EPC		5'b1110	/* 14 Exception Program Counter */

/* 12 Status Register Definitions */
`define	Status	5'b1100	/* 12 Status Register */
`define	CU0		4'h1		/* Enable Coprocessor0 */
`define	CU1		4'h2		/* Enable Coprocessor1 */
`define	CU2		4'b4		/* Enable Coprocessor2 */
`define	CU3		4'b8		/* Enable Coprocessor3 */
`define	CU			`CU0		/* Enabled Coprocessors */
`define	RP			1'b0		/* Reduced Power */
`define	FR			1'b0		/* Floating Point Register Enable */
`define	RE			1'b1		/* Reverse Endian */
`define 	BEV		1'b1		/* Bootstrap Exception Vector */
`define	TS			1'b1		/* TLB Shutdown */
`define	PE			1'b0		/* Parity Error */
`define	CM			1'b0		/* Cache Miss */
`define	PZ			1'b0		/* Parity Zero */
`define	SwC		1'b0		/* Swap Caches */
`define	IsC		1'b0		/* Isolate Cache */
`define	SWRST		1'b0		/* Software Reset */
`define	IntMask	8'hff		/* Interrupt Mask */
`define	KUo		1'b0		/* Kernel/User Mode Old */
`define	IEo		1'b0		/* Interrupt Enable Old */
`define	KUp		1'b0		/* Kernal/User Mode Previous */
`define	IEp		1'b0		/* Interrupt Enable Previous */
`define	KUc		1'b0		/* Kernal/User Mode Current */
`define	IEc		1'b0		/* Interrupt Enable Current */

/* 8 Bad Virtual Address Register Definitions */
`define	BadVAddr	5'b1000	/* 8 BadVAddr Register */

/* 4 Context Register Definitions */
`define	Context	5'b100	/* 4 Context Register */
`define	PTEBase	11'b0		/* Page Table Entry Base */
`define	BadVPN	19'b0		/* Bad Virtual Page Number */

/* 15 Processor Revision Identifier Register Definitions */
`define	PRId		5'b01111	/* 15 Processor Revision Identifier Register */
`define	Imp		8'h4		/* Implementation Number */
`define	Rev		8'h0		/* Revision Number */

/* 10 EntryHI Register Definitions */
`define	EntryHI	5'b1010 /* 10 EntryHI Register */

/* 2 EntryLO Register Definitions */
`define	EntryLO	5'b10		/* 2 EntryLO Register */

/* 0 Index Register Definitions */
`define	Index		5'b0		/* 0 Index Register */

/* 1 Random Register Definitions */
`define	Random	5'b1		/* 1 Random Register */

/* 16 Extension Control Register Definitions*/
`define	Ext		5'b10000	/* 16 Extension Control Register */

/* Exception Vectors */
//`define	RESET_EV		32'h82000100
`define	RESET_EV		32'hffff0000			/* Reset Exception Vector */
`define	RESET_EV100	`RESET_EV | 32'h100	/* Reset Exception Vector | 0x100 */
`define	RESET_EV180	`RESET_EV | 32'h180	/* Reset Exception Vector | 0x180 */
`define	UTLBM_EV		32'h80000000			/* User Mode TLB Miss Exception Vector */
`define	GenExcEV		32'h80000080			/* General Exception Vector */

module coprocessor0_sim(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BD_EX,			/* Branch Delay Execute Phase */
	input				BD_EXT,			/* Branch Delay Extension */
	input				BD_ID,			/* Branch Delay Instruction Decode Phase */
	input				BD_MA,			/* Branch Delay Memory Access */
	input				BD_WB,			/* Branch Delay Writeback Phase */
	input				BUBBLE_EX,		/* Bubble Execute Phase */
	input				BUBBLE_ID,		/* Bubble Instruction Decode Phase */
	input				BUBBLE_MA,		/* Bubble Memory Access Phase */
	input				CLK,				/* System Clock 50 - 100 MHZ */
	input				EXC_EX,			/* Exception Execute Phase */
	input				EXC_EXT,			/* Exception Extension Phase */
	input				EXC_ID,			/* Exception Instruction Decode */
	input				EXC_IF,			/* Exception Instruction Fetch */
	input				EXC_MA,			/* Exception Memory Access */
	input [31:0]	EXCADD_CD_EX,	/* Exception Address Data Execute Phase */
	input [31:0]	EXCADD_CD_EXT,	/* Exception Address Data Extension */
	input [31:0]	EXCADD_CD_ID,	/* Exception Address Data Instruction Decode Phase */
	input [31:0]	EXCADD_CD_IF,	/* Exception Address Data Instruction Fetch Phase */
	input [31:0]	EXCADD_CD_MA,	/* Exception Address Data Memory Address Phase */
	input [31:0]	EXCADD_CI_EX,	/* Exception Address Instruction Execute Phase */
	input [31:0]	EXCADD_CI_EXT,	/* Exception Address Instruction Extension */
	input [31:0]	EXCADD_CI_ID,	/* Exception Address Instruction Instruction Decode Phase */
	input [31:0]	EXCADD_CI_IF,	/* Exception Address Instruction Instruction Fetch Phase */
	input [31:0]	EXCADD_CI_MA,	/* Exception Address Instruction Memory Access Phase */
	input [6:0]		EXCCODE_EX,		/* Exception Code Execute Phase */
	input [6:0]		EXCCODE_EXT,	/* Exception Code Extension */
	input [6:0]		EXCCODE_ID,		/* Exception Code Instruction Decode */
	input [6:0]		EXCCODE_IF,		/* Exception Code Instruction Fetch */
	input [6:0]		EXCCODE_MA,		/* Exception Code Memory Access */
	input				EXTSTALL,		/* Stall for Extension */
	input				IRQ0,				/* Hardware Interrupt 0 */
	input				IRQ1,				/* Hardware Interrupt 1 */
	input				IRQ2,				/* Hardware Interrupt 2 */
	input				IRQ3,				/* Hardware Interrupt 3 */
	input				IRQ4,				/* Hardware Interrupt 4 */
	input				IRQ5,				/* Hardware Interrupt 5 */
	input				PCLK,				/* Pipeline Clock */
	input [4:0]		RDREG1,			/* Register Read Port 1 Register Number */
	input [4:0]		RDREG2,			/* Register Read Port 2 Register Number */
	input				REGWRITE,		/* Register Write Enable */
	input				RESET,			/* System Reset */
	input				RFE,				/* Return from Exception */
	input [31:0]	WRDATA,			/* Register Write Port Data */
	input [4:0]		WRREG,			/* Register Write Port Register Number */
	/* OUTPUT PORTS */
	output [3:0]	CP,				/* Enabled Coprocessors */
	output [3:0]	EXC,				/* Exception Flush */
	output [31:0]	EXCADD_R,		/* Exception Jump Address */
	output			EXCEXT,			/* Exception Flush Extension */
	output [31:0]	EXTCTR,			/* Extension Control Register */
	output			KU,				/* Kernel/User Mode */
	output [31:0]	RDDATA1,			/* Register Read Port 1 Data */
	output [31:0]	RDDATA2,			/* Register Read Port 2 Data */
	output			RE,				/* Reverse Endian */
	output			RST,				/* Soft Reset */
	output			EXEN,				/* */
	output [31:0]	STATUS,			/* Coprocessor0 Register 12, Status Register */
	output [31:0]	BADVADDR,		/* */
	output [31:0]	CAUSE,			/* Coprocessor0 Register 13, Cause Register */
	/* DEBUG PORTS */
	output [31:0]	cp0regfile00,	/* Coprocessor0 Register 0, Index Register */
	output [31:0]	cp0regfile01,	/* Coprocessor0 Register 1, Random Register */
	output [31:0]	cp0regfile02,	/* Coprocessor0 Register 2, EntryLO Register */
	output [31:0]	cp0regfile03,	/* Coprocessor0 Register 3 */
	output [31:0]	cp0regfile04,	/* Coprocessor0 Register 4, Context */
	output [31:0]	cp0regfile05,	/* Coprocessor0 Register 5 */
	output [31:0]	cp0regfile06,	/* Coprocessor0 Register 6 */
	output [31:0]	cp0regfile07,	/* Coprocessor0 Register 7 */
	output [31:0]	cp0regfile08,	/* Coprocessor0 Register 8, Bad Virtual Address Register */
	output [31:0]	cp0regfile09,	/* Coprocessor0 Register 9 */
	output [31:0]	cp0regfile10,	/* Coprocessor0 Register 10, EntryHI Register */
	output [31:0]	cp0regfile11,	/* Coprocessor0 Register 11 */
	output [31:0]	cp0regfile12,	/* Coprocessor0 Register 12, Status Register */
	output [31:0]	cp0regfile13,	/* Coprocessor0 Register 13, Cause Register */
	output [31:0]	cp0regfile14,	/* Coprocessor0 Register 14, Exception Program Counter */
	output [31:0]	cp0regfile15,	/* Coprocessor0 Register 15, Processor Revision Identifier Register */
	output [31:0]	cp0regfile16,	/* Coprocessor0 Register 16, Extension Control Register */
	output [31:0]	cp0regfile17,	/* Coprocessor0 Register 17 */
	output [31:0]	cp0regfile18,	/* Coprocessor0 Register 18 */
	output [31:0]	cp0regfile19,	/* Coprocessor0 Register 19 */
	output [31:0]	cp0regfile20,	/* Coprocessor0 Register 20 */
	output [31:0]	cp0regfile21,	/* Coprocessor0 Register 21 */
	output [31:0]	cp0regfile22,	/* Coprocessor0 Register 22 */
	output [31:0]	cp0regfile23,	/* Coprocessor0 Register 23 */
	output [31:0]	cp0regfile24,	/* Coprocessor0 Register 24 */
	output [31:0]	cp0regfile25,	/* Coprocessor0 Register 25 */
	output [31:0]	cp0regfile26,	/* Coprocessor0 Register 26 */
	output [31:0]	cp0regfile27,	/* Coprocessor0 Register 27 */
	output [31:0]	cp0regfile28,	/* Coprocessor0 Register 28 */
	output [31:0]	cp0regfile29,	/* Coprocessor0 Register 29 */
	output [31:0]	cp0regfile30,	/* Coprocessor0 Register 30 */
	output [31:0]	cp0regfile31	/* Coprocessor0 Register 31 */
	);

/*****Signals****************************************************************/

	wire [31:0]	CONTEXT;			/* Coprocessor0 Register 4, Context */
	wire [3:0]	EXC0;				/* Exception Flush Asynchronous */
	wire [31:0]	EXCADD_CD;		/* Exception Address Data */
	wire [31:0]	EXCADD_CI;		/* Exception Address Instruction */
	wire [31:0]	EXCADD_CI_INT;	/* Exception Address Instruciton Interrupt */
	wire [31:0]	EXCADD_R_0;		/* Exception Jump Address Asynchronous */
	wire [7:0]	EXCCODE;			/* Exception Code */
	wire [7:0]	EXCCODE_INT;	/* Exception Code Interrupt */
	wire 			EXCEXT0;			/* Exception Flush Extenion Asynchronous */
	wire 			EXCFLAG;			/* Exception Detection Flag */
	wire [7:0]	INT_E;			/* Asserted Hardware Interrupts Masked by Global Interrupt Enable */
	wire [7:0]	INT_EN;			/* Global Interrupt Enable x8 */
	wire [7:0]	INT_M;			/* Asserted Hardware Interrupts Masked by Interrupt Mask Register */
	wire [7:0]	INT_S;			/* Interrupt Signal Bus */
	wire [6:0]	INTCODE;			/* Interrupt Exception Code */
	wire [31:0]	PRID;				/* Coprocessor0 Register 15, Processor Revision Identifier Register */
	wire [31:0]	RDDATA1_0;		/* Register Read Port 1 Data Asynchronous */
	wire [31:0]	RDDATA2_0;		/* Register Read Port 2 Data Asynchronous */
	wire 			RESET_INT;		/* Reset Exception Flag */

/*****Registers****************************************************************/
	
	reg [31:0]	BadVAddrreg;			/* Coprocessor0 Register 8, Bad Virtual Address Register */
	reg [18:0]	BadVPNreg;				/* Bad Virtual Page Number */
	reg			bd_ex_reg;				/* Branch Delay Execute Phase */
	reg			bd_ext_reg;				/* Branch Delay Extension */
	reg			bd_id_reg;				/* Branch Delay Instruction Decode Phase */
	reg			bd_ma_reg;				/* Branch Delay Instruction Fetch Phase */
	reg			bd_wb_reg;				/* Branch Delay Writeback Phase */
	reg			BDreg;					/* Branch Delay */
	reg			BEVreg;					/* Bootstrap Exception Vector */
	reg			bubble_ex_reg;			/* Bubble Execute Phase */
	reg			bubble_id_reg;			/* Bubble Instruction Decode Phase */
	reg			bubble_ma_reg;			/* Bubble Memory Access Phase */
	reg [1:0]	CEreg;					/* Coprocessor Error */
	reg			CMreg;					/* Cache Miss */
	reg [3:0]	CUreg;					/* Enabled Coprocessors */
	reg [31:0]	EntryHIreg;				/* Coprocessor0 Register 10, EntryHI Register */
	reg [31:0]	EntryLOreg;				/* Coprocessor0 Register 2, EntryLO Register */
	reg [31:0]	EPCreg;					/* Coprocessor0 Register 14, Exception Program Counter */
	reg			exc_ex_reg;				/* Exception Execute Phase */
	reg			exc_ext_reg;			/* Exception Extension */
	reg			exc_id_reg;				/* Exception Instruction Decode Phase */
	reg			exc_if_reg;				/* Exception Instruction Fetch Phase */
	reg			exc_ma_reg;				/* Exception Memory Access Phase */
	reg [31:0]	excadd_cd_ex_reg;		/* Exception Address Data Execute Phase */
	reg [31:0]	excadd_cd_ext_reg;	/* Exception Address Data Extension */
	reg [31:0]	excadd_cd_id_reg;		/* Exception Address Data Instruction Decode Phase */
	reg [31:0]	excadd_cd_if_reg;		/* Exception Address Data Instruction Fetch Phase */
	reg [31:0]	excadd_cd_ma_reg;		/* Exception Address Data Memory Access Phase */
	reg [31:0]	EXCADD_CDreg;			/* Exception Address Data */
	reg [31:0]	excadd_ci_ex_reg;		/* Exception Address Instruction Execute Phase */
	reg [31:0]	excadd_ci_ext_reg;	/* Exception Address Instruction Extension */
	reg [31:0]	excadd_ci_id_reg;		/* Exception Address Instruction Instruction Decode Phase */
	reg [31:0]	excadd_ci_if_reg;		/* Exception Address Instruction Instruction Fetch Phase */
	reg [31:0]	excadd_ci_int_reg;	/* Exception Address Instruction Interrupt */
	reg [31:0]	excadd_ci_ma_reg;		/* Exception Address Instruction Memory Access Phase */
	reg [31:0]	EXCADD_CIreg;			/* Exception Address Instruction */
	reg [6:0]	exccode_ex_reg;		/* Exception Code Execute Phase */
	reg [6:0]	exccode_ext_reg;		/* Exception Code Extension */
	reg [6:0]	exccode_id_reg;		/* Exception Code Instruction Decode Phase */
	reg [6:0]	exccode_if_reg;		/* Exception Code Instruction Fetch Phase */
	reg [7:0]	exccode_int_reg; 		/* Exception Code Interrupt */
	reg [6:0]	exccode_ma_reg;		/* Exception Code Memory Access Phase */
	reg [7:0]	EXCCODE_reg;			/* Exception Code */
	reg [3:0]	ExcCodereg;				/* Exception Code Cause Register */
	reg			EXCEXTreg;				/* Exception Flush Extension */
	reg			excflag_reg;			/* Exception Detection Flag */
	reg [3:0]	EXCreg;					/* Exception Flush */
	reg [31:0]	EXTCTRreg;				/* Coprocessor0 Register 16, Extension Control Register */ 
	reg			extstall_reg;			/* Stall for Extension */
	reg			FPreg;					/* Floating Point Register Enable */
	reg			IEcreg;					/* Interrupt Enable Current */
	reg			IEoreg;					/* Interrupt Enable Old */
	reg			IEpreg;					/* Interrupt Enable Previous */
	reg [7:0]	Impreg;					/* Implementation Number */
	reg [31:0]	Indexreg;				/* Coprocessor0 Register 0, Index Register */
	reg [7:0]	int_e_reg;				/* Asserted Hardware Interrupts Masked by Global Interrupt Enable */
	reg [7:0]	int_m_reg;				/* Asserted Hardware Interrupts Masked by Interrupt Mask Register */
	reg [7:0]	Int_R;					/* Asserted & Masked Interrupts Registered */
	reg [6:0]	intcode_reg;			/* Interrupt Exception Code */
	reg [7:0]	IntMaskreg;				/* Interrupt Mask Register */
	reg [5:0]	IPreg;					/* Pending Interrupts */
	reg			irq0_reg;				/* Hardware Interrupt 0 */
	reg			irq1_reg;				/* Hardware Interrupt 1 */
	reg			irq2_reg;				/* Hardware Interrupt 2 */
	reg			irq3_reg;				/* Hardware Interrupt 3 */
	reg			irq4_reg;				/* Hardware Interrupt 4 */
	reg			irq5_reg;				/* Hardware Interrupt 5 */
	reg			IsCreg;					/* Isolate Cache */
	reg			KUcreg;					/* Kernel/User Mode Current */
	reg			KUoreg;					/* Kernel/User Mode Old */
	reg			KUpreg;					/* Kernel/User Mode Previous */
	reg [1:0]	pclkcnt;					/* Pipeline Clock Edge Detection */
	reg			PEreg;					/* Parity Error */
	reg [10:0]	PTEBasereg;				/* Page Table Entry Base */
	reg			PZreg;					/* Parity Zero */
	reg [31:0]	Randomreg;				/* Coprocessor0 Register 1, Random Register */
	reg [31:0]	rddata1_reg;			/* Register Read Port 1 Data */
	reg [31:0]	rddata2_reg;			/* Register Read Port 2 Data */
	reg [4:0]	rdreg1_reg;				/* Register Read Port 1 Register Number */
	reg [4:0]	rdreg2_reg;				/* Register Read Port 2 Register Number */
	reg			REGWRITE_LAT;			/* Register Write Enable Latched */
	reg			REGWRITE_MA;			/* Register Write Enable Pipelined Memory Access Phase */
	reg			regwrite_reg;			/* Register Write Enable */
	reg			REGWRITE_WB;			/* Register Write Enable Pipelined Writeback Phase */
	reg			REreg;					/* Reverse Endian */
	reg			reset_int_reg;			/* Reset Exception Flag */
	reg [1:0]	resetcnt;				/* Reset Edge Detection */
	reg [7:0]	Revreg;					/* Revision Number */
	reg			RFE_LAT;					/* Return from Exception Latched */
	reg			RFE_MA;					/* Return from Exception Pipelined Memory Access Phase */
	reg			rfe_reg;					/* Return from Exception */
	reg			RFE_WB;					/* Return from Exception Pipelined Writeback Phase */
	reg			RPreg;					/* Reduced Power */
	reg			SwCreg;					/* Swap Caches */
	reg [1:0]	Swreg;					/* Software Interruputs Pending */
	reg			SWRSTreg;				/* Software Reset */
	reg			TSreg;					/* TLB Shutdown */
	reg [31:0]	WRDATA_LAT;				/* Register Write Port Data Latched */
	reg [31:0]	WRDATA_MA;				/* Register Write Port Data Pipelined Memory Access Phase */
	reg [31:0]	wrdata_reg;				/* Register Write Port Data */
	reg [31:0]	WRDATA_WB;				/* Register Write Port Data Pipelined Writeback Phase */
	reg [4:0]	WRREG_LAT;				/* Register Write Port Register Number Latched */
	reg [4:0]	WRREG_MA;				/* Register Write Port Register Number Pipelined Memory Access */
	reg [4:0]	wrreg_reg;				/* Register Write Port Register Number */
	reg [4:0]	WRREG_WB;				/* Register Write Port Register Number Pipelined Writeback Phase */
	reg			exen_reg;

/*****Parameters****************************************************************/

	/* 13 Cause Register Bit Locations */
	parameter	rlBD		=	31;	/* Branch Delay */
	parameter	lCE		=	29;	/* Coprocessor Error bit range left */
	parameter	rCE		=	28;	/* Coprocessor Error bit range right */
	parameter	lIP		=	15;	/* Interrupts Pending bit ragne left */
	parameter	rIP		=	10;	/* Interrupts Pending bit ragne right */
	parameter	lSw		=	9;		/* Software Interruputs Pending bit range left */
	parameter	rSw		=	8;		/* Software Interruputs Pending bit range right */
	parameter	lExcCode	=	5;		/* Exception Code bit range left */
	parameter	rExcCode	=	2;		/* Exception Code bit range right */

	/* 12 Status Register Bit Locations */
	parameter	lCU		=	31;	/* Enabled Coprocessors bit range left */
	parameter	rCU		=	28;	/* Enabled Coprocessors bit range right */
	parameter	rlRP		=	27;	/* Reduced Power */
	parameter	rlFP		=	26;	/* Floating Point Register Enable */
	parameter	rlRE		=	25;	/* Reverse Endian */
	parameter	rlBEV		=	22;	/* Bootstrap Exception Vector */
	parameter	rlTS		=	21;	/* TLB Shutdown */
	parameter	rlPE		=	20;	/* Parity Error */
	parameter	rlCM		=	19;	/* Cache Miss */
	parameter	rlPZ		=	18;	/* Parity Zero */
	parameter	rlSwC		=	17;	/* Swap Caches */
	parameter	rlIsC		=	16;	/* Isolate Cache */
	parameter	lIntMask	=	15;	/* Interrupt Mask bit range left */
	parameter	rIntMask	=	8;		/* Interrupt Mask bit range right */
	parameter	rlSWRST	=	7;		/* Software Reset */
	parameter	rlKUo		=	5;		/* Kernel/User Mode Old */
	parameter	rlIEo		=	4;		/* Interrupt Enable Old */
	parameter	rlKUp		=	3;		/* Kernal/User Mode Previous */
	parameter	rlIEp		=	2;		/* Interrupt Enable Previous */
	parameter	rlKUc		=	1;		/* Kernal/User Mode Current */
	parameter	rlIEc		=	0;		/* Interrupt Enable Current */

	/* 4 Context Register Bit Locations */
	parameter	lPTEBase	=	31;	/* Page Table Entry Base bit range left */
	parameter	rPTEBase	=	21;	/* Page Table Entry Base bit range right */
	parameter	lBadVPN	=	20;	/* Bad Virtual Page Number bit range left */
	parameter	rBadVPN	=	2;		/* Bad Virtual Page Number bit range right */

	/* 15 Processor Revision Identifier Bit Locations */
	parameter	lImp		=	15;	/* Implementation Number bit range left */
	parameter	rImp		=	8;		/* Implementation Number bit range right */
	parameter	lRev		=	7;		/* Revision Number bit range left */
	parameter	rRev		=	0;		/* Revision Number bit range right */

	/* 16 Extension Register */
	parameter	EXT0		= 	0;		/* Extension Zero base location */ 
	parameter	EXT1		=	8;		/* Extension One base location */
	parameter	EXT2		=	16;	/* Extension Two base location */
	parameter	EXT3		=	24;	/* Extension Three base location */
	parameter	EXTPRr	=	4;		/* Extension Prioriy bit range left */
	parameter	EXTPRl	=	3;		/* Extension Prioriy bit range right */
	parameter	EXTTR		=	2;		/* Extension Trap Enable */
	parameter	EXTLD		=	1;		/* Extension Loaded */
	parameter	EXTEN		=	0;		/* Extension Enabled */

/*****Initialization****************************************************************/
	
	initial
	begin
		BadVAddrreg = 32'b0;
		BadVPNreg = `BadVPN;
		bd_ex_reg = 1'b0;
		bd_ext_reg = 1'b0;
		bd_id_reg = 1'b0;
		bd_ma_reg = 1'b0;
		bd_wb_reg = 1'b0;
		BDreg = `BD;
		BEVreg = `BEV;
		bubble_ex_reg = 1'b0;
		bubble_id_reg = 1'b0;
		bubble_ma_reg = 1'b0;
		CEreg = `CE;
		CMreg = `CM;
		CUreg = `CU;
		EntryHIreg = 32'b0;
		EntryLOreg = 32'b0;
		EPCreg = 32'hfffffffc;
		exc_ex_reg = 1'b0;
		exc_ext_reg = 1'b0;
		exc_id_reg = 1'b0;
		exc_if_reg = 1'b0;
		exc_ma_reg = 1'b0;
		excadd_cd_ext_reg = 32'b0;
		excadd_cd_ex_reg = 32'b0;
		excadd_cd_id_reg = 32'b0;
		excadd_cd_if_reg = 32'b0;
		excadd_cd_ma_reg = 32'b0;
		excadd_ci_ex_reg = 32'b0;
		excadd_ci_ext_reg = 32'b0;
		excadd_ci_id_reg = 32'b0;
		excadd_ci_if_reg = 32'b0;
		excadd_ci_int_reg = 32'b0;
		excadd_ci_ma_reg = 32'b0;
		exccode_ex_reg = 7'b0;
		exccode_ext_reg = 7'b0;
		exccode_id_reg = 7'b0;
		exccode_if_reg = 7'b0;
		exccode_int_reg = 8'b0;
		exccode_ma_reg = 7'b0;
		ExcCodereg = `ExcCode;
		EXCEXTreg = 1'b0;
		excflag_reg = 1'b0;
		EXCreg = 4'b0;
		EXTCTRreg = 32'b0;
		extstall_reg = 1'b0;
		FPreg = `FR;
		IEcreg = `IEc;
		IEoreg = `IEo;
		IEpreg = `IEp;
		Impreg = `Imp;
		Indexreg = 32'b0;
		int_e_reg = 8'b0;
		int_m_reg = 8'b0;
		Int_R = 8'b0;
		intcode_reg = 7'b0;
		IntMaskreg = `IntMask;
		IPreg = `IP;
		irq0_reg = 1'b0;
		irq1_reg = 1'b0;
		irq2_reg = 1'b0;
		irq3_reg = 1'b0;
		irq4_reg = 1'b0;
		irq5_reg = 1'b0;
		IsCreg = `IsC;
		KUcreg = `KUc;
		KUoreg = `KUo;
		KUpreg = `KUp;
		pclkcnt = 2'b0;
		PEreg = `PE;
		PTEBasereg = `PTEBase;
		PZreg = `PZ;
		Randomreg = 32'h00003f00;
		rddata1_reg = 32'b0;
		rddata2_reg = 32'b0;
		rdreg1_reg = 5'b0;
		rdreg2_reg = 5'b0;
		REGWRITE_MA = 1'b0;
		regwrite_reg = 1'b0;
		REGWRITE_WB = 1'b0;
		REreg = `RE;
		reset_int_reg = 1'b0;
		resetcnt = 2'b0;
		Revreg = `Rev;
		RFE_MA = 1'b0;
		rfe_reg = 1'b0;
		RFE_WB = 1'b0;
		RPreg = `RP;
		SwCreg = `SwC;
		Swreg = `Sw;
		SWRSTreg = `SWRST;
		TSreg = `TS;
		WRDATA_MA = 32'b0;
		wrdata_reg = 32'b0;
		WRDATA_WB = 32'b0;
		WRREG_MA = 5'b0;
		wrreg_reg = 5'b0;
		WRREG_WB = 5'b0;
		RFE_LAT = 1'b0;
		WRDATA_LAT = 32'b0;
		WRREG_LAT = 5'b0;
		REGWRITE_LAT = 1'b0;
		exen_reg = 1'b0;
	end

/*****Debug Port Assignments****************************************************************/

	assign cp0regfile00 = Indexreg;
	assign cp0regfile01 = Randomreg;
	assign cp0regfile02 = EntryLOreg;
	assign cp0regfile03 = 32'b0;
	assign cp0regfile04 = CONTEXT;
	assign cp0regfile05 = 32'b0;
	assign cp0regfile06 = 32'b0;
	assign cp0regfile07 = 32'b0;
	assign cp0regfile08 = BadVAddrreg;
	assign cp0regfile09 = 32'b0;
	assign cp0regfile10 = EntryHIreg;
	assign cp0regfile11 = 32'b0;
	assign cp0regfile12 = STATUS;
	assign cp0regfile13 = CAUSE;
	assign cp0regfile14 = EPCreg;
	assign cp0regfile15 = PRID;
	assign cp0regfile16 = EXTCTRreg;
	assign cp0regfile17 = 32'b0;
	assign cp0regfile18 = 32'b0;
	assign cp0regfile19 = 32'b0;
	assign cp0regfile20 = 32'b0;
	assign cp0regfile21 = 32'b0;
	assign cp0regfile22 = 32'b0;
	assign cp0regfile23 = 32'b0;
	assign cp0regfile24 = 32'b0;
	assign cp0regfile25 = 32'b0;
	assign cp0regfile26 = 32'b0;
	assign cp0regfile27 = 32'b0;
	assign cp0regfile28 = 32'b0;
	assign cp0regfile29 = 32'b0;
	assign cp0regfile30 = 32'b0;
	assign cp0regfile31 = 32'b0;

/*********************************************************************/

	assign BADVADDR	=	BadVAddrreg;

/*****Exception Processing Logic****************************************************************/

	assign EXEN =				exen_reg;
	assign RESET_INT =		(resetcnt == 2'b01)?	1'b1:
															1'b0;
	assign CP =					CUreg;
	assign KU =					KUcreg;
	assign RE =					REreg;
	assign RST = 				SWRSTreg;
	assign EXTCTR = 			EXTCTRreg;
	assign EXCADD_CI	=		(exc_ma_reg == 1'b1)?	excadd_ci_ma_reg:
									(exc_ex_reg == 1'b1)?	excadd_ci_ex_reg:
									(exc_ext_reg == 1'b1)?	excadd_ci_ext_reg:
									(exc_id_reg == 1'b1)?	excadd_ci_id_reg:
									(exc_if_reg == 1'b1)?	excadd_ci_if_reg:
									(Int_R > 0)?				excadd_ci_int_reg:
																	32'b0;
	assign EXCADD_CI_INT = 	(~bubble_ma_reg)?						excadd_ci_ma_reg:
									(~bubble_ex_reg)?						excadd_ci_ex_reg:
									(bubble_ex_reg & extstall_reg)?	excadd_ci_ext_reg:
									(~bubble_id_reg)?						excadd_ci_id_reg:
																				excadd_ci_if_reg;

	assign EXCADD_CD	=		(exc_ma_reg == 1'b1)?	excadd_cd_ma_reg:
									(exc_ex_reg == 1'b1)?	excadd_cd_ex_reg:
									(exc_ext_reg == 1'b1)?	excadd_cd_ext_reg:
									(exc_id_reg == 1'b1)?	excadd_cd_id_reg:
									(exc_if_reg == 1'b1)?	excadd_cd_if_reg:
																	32'b0;
	assign INTCODE =			(Int_R[0] == 1'b1)?	`Int0c:
									(Int_R[1] == 1'b1)?	`Int1c:
									(Int_R[2] == 1'b1)?	`Int2c:
									(Int_R[3] == 1'b1)?	`Int3c:
									(Int_R[4] == 1'b1)?	`Int4c:
									(Int_R[5] == 1'b1)?	`Int5c:
									(Int_R[6] == 1'b1)?	`Int6c:
									(Int_R[7] == 1'b1)?	`Int7c:
																7'h7f;
	assign EXCCODE_INT = 	(~bubble_ma_reg)?						{bd_wb_reg,intcode_reg}:
									(~bubble_ex_reg)?						{bd_ma_reg,intcode_reg}:
									(bubble_ex_reg & extstall_reg)?	{bd_ext_reg,intcode_reg}:
									(~bubble_id_reg)?						{bd_ex_reg,intcode_reg}:
																				{bd_id_reg,intcode_reg};
	assign EXCCODE = 			(reset_int_reg == 1'b1)?	{1'b0,`RSTc}:
									(exc_ma_reg == 1'b1)?		{bd_wb_reg,exccode_ma_reg}:
									(exc_ex_reg == 1'b1)?		{bd_ma_reg,exccode_ex_reg}:
									(exc_ext_reg == 1'b1)?		{bd_ext_reg,exccode_ext_reg}:
									(exc_id_reg == 1'b1)?		{bd_ex_reg,exccode_id_reg}:
									(exc_if_reg == 1'b1)?		{bd_id_reg,exccode_if_reg}:
									(Int_R > 0)?					exccode_int_reg:
																		8'hff;
	assign EXC =				EXCreg;
	assign EXC0 =				(reset_int_reg == 1'b1)?										4'b1111:
									(exc_ma_reg == 1'b1)?											4'b1111:
									(exc_ex_reg == 1'b1)? 											4'b0111:
									(exc_ext_reg == 1'b1)?											4'b0111:
									((exc_id_reg == 1'b1) && (exccode_id_reg == `Bpc))?	4'b0001:
									((exc_id_reg == 1'b1) && (exccode_id_reg == `SYSc))?	4'b0001:
									(exc_id_reg == 1'b1)?											4'b0011:
									(exc_if_reg == 1'b1)?											4'b0001:
									(Int_R > 0)?														4'b1111:
																											4'b0000;
	assign EXCEXT =			EXCEXTreg;
	assign EXCEXT0 =			(reset_int_reg == 1'b1)?	1'b1:
									(exc_ma_reg == 1'b1)?		1'b1:
									(exc_ex_reg == 1'b1)? 		1'b1:
									(exc_ext_reg == 1'b1)?		1'b1:
									(Int_R > 0)?					1'b1:
																		1'b0;		
	assign INT_S =				{irq5_reg,irq4_reg,irq3_reg,irq2_reg,irq1_reg,irq0_reg,Swreg};
	assign INT_EN =			{IEcreg,IEcreg,IEcreg,IEcreg,IEcreg,IEcreg,IEcreg,IEcreg};
	assign INT_E =				INT_S & INT_EN;
	assign INT_M =				int_e_reg & IntMaskreg;
	assign EXCFLAG = 			reset_int_reg | exc_if_reg | exc_id_reg | exc_ex_reg | exc_ma_reg | exc_ext_reg | (Int_R > 0);
	assign EXCADD_R =			EXCADD_R_0;
	assign EXCADD_R_0 =		(({EXCEXTreg,EXCreg} > 5'b0) & (EXCCODE_reg[6:0] == `RSTc))?											`RESET_EV:
									(({EXCEXTreg,EXCreg} > 5'b0) & BEVreg & EXCADD_CDreg[31] & (EXCCODE_reg[6:0] == `UTLBLc))?	`RESET_EV100:
									(({EXCEXTreg,EXCreg} > 5'b0) & BEVreg & EXCADD_CDreg[31] & (EXCCODE_reg[6:0] == `UTLBSc))?	`RESET_EV100:
									(({EXCEXTreg,EXCreg} > 5'b0) & BEVreg)?																		`RESET_EV180:
									(({EXCEXTreg,EXCreg} > 5'b0) & EXCADD_CDreg[31] & (EXCCODE_reg[6:0] == `TLBLc))?					`UTLBM_EV:
									(({EXCEXTreg,EXCreg} > 5'b0) & EXCADD_CDreg[31] & (EXCCODE_reg[6:0] == `TLBSc))?					`UTLBM_EV:
									({EXCEXTreg,EXCreg} > 5'b0)?																						`GenExcEV:
																																								32'b0;

/*****Coprocessor0 Registers****************************************************************/
																																								
	assign CAUSE[rlBD] =						BDreg;
	assign CAUSE[lCE:rCE] =					CEreg;
	assign CAUSE[lIP:rIP] =					IPreg;
	assign CAUSE[lSw:rSw] =					Swreg;
	assign CAUSE[lExcCode:rExcCode] =	ExcCodereg;
	assign CAUSE[30] = 						1'b0;
	assign CAUSE[27:16] =					12'b0;
	assign CAUSE[7:6] =						2'b0;
	assign CAUSE[1:0] =						2'b0;
	
	assign STATUS[lCU:rCU] =				CUreg;
	assign STATUS[rlRP] =					RPreg;
	assign STATUS[rlFP] =					FPreg;
	assign STATUS[rlRE] =					REreg;
	assign STATUS[rlBEV] = 					BEVreg;
	assign STATUS[rlTS] =					TSreg;
	assign STATUS[rlPE] =					PEreg;
	assign STATUS[rlCM] = 					CMreg;
	assign STATUS[rlPZ] = 					PZreg;
	assign STATUS[rlSwC] =					SwCreg;
	assign STATUS[rlIsC] = 					IsCreg;
	assign STATUS[lIntMask:rIntMask] =	IntMaskreg;
	assign STATUS[rlSWRST] =				SWRSTreg;
	assign STATUS[rlKUo] =					KUoreg;
	assign STATUS[rlIEo] =					IEoreg;
	assign STATUS[rlKUp] =					KUpreg;
	assign STATUS[rlIEp] =					IEpreg;
	assign STATUS[rlKUc] =					KUcreg;
	assign STATUS[rlIEc] =					IEcreg;
	assign STATUS[24:23] =					2'b0;
	assign STATUS[6] =						1'b0;
	
	assign CONTEXT[lPTEBase:rPTEBase] =	PTEBasereg;
	assign CONTEXT[lBadVPN:rBadVPN] =	BadVPNreg;
	assign CONTEXT[1:0] =					2'b0;
	
	assign PRID[lImp:rImp] =				Impreg;
	assign PRID[lRev:rRev] =				Revreg;
	assign PRID[31:16] = 					16'b0;

/*****Register Read Ports****************************************************************/

	assign RDDATA1 =															rddata1_reg;
	assign RDDATA1_0 = 	(rdreg1_reg == WRREG_MA)?					WRDATA_MA:
								(rdreg1_reg == WRREG_WB)?					WRDATA_WB:
								((rdreg1_reg == `Status) && (RFE_MA))?	{STATUS[31:rlKUp+1],KUoreg,IEoreg,KUpreg,IEpreg}:
								((rdreg1_reg == `Status) && (RFE_WB))?	{STATUS[31:rlKUp+1],KUoreg,IEoreg,KUpreg,IEpreg}:
								(rdreg1_reg == `Status)?					STATUS:
								(rdreg1_reg == `Cause)?						CAUSE:
								(rdreg1_reg == `EPC)?						EPCreg:
								(rdreg1_reg == `BadVAddr)?					BadVAddrreg:
								(rdreg1_reg == `Context)?					CONTEXT:
								(rdreg1_reg == `PRId)?						PRID:
								(rdreg1_reg == `EntryHI)?					EntryHIreg:
								(rdreg1_reg == `EntryLO)?					EntryLOreg:
								(rdreg1_reg == `Index)?						Indexreg:
								(rdreg1_reg == `Random)?					Randomreg:
								(rdreg1_reg == `Ext)?						EXTCTRreg:
																					32'b0;	

	assign RDDATA2 =															rddata2_reg;
	assign RDDATA2_0 = 	(rdreg2_reg == WRREG_MA)?					WRDATA_MA:
								(rdreg2_reg == WRREG_WB)?					WRDATA_WB:
								((rdreg2_reg == `Status) && (RFE_MA))?	{STATUS[31:rlKUp+1],KUoreg,IEoreg,KUpreg,IEpreg}:
								((rdreg2_reg == `Status) && (RFE_WB))?	{STATUS[31:rlKUp+1],KUoreg,IEoreg,KUpreg,IEpreg}:
								(rdreg2_reg == `Status)?					STATUS:
								(rdreg2_reg == `Cause)?						CAUSE:
								(rdreg2_reg == `EPC)?						EPCreg:
								(rdreg2_reg == `BadVAddr)?					BadVAddrreg:
								(rdreg2_reg == `Context)?					CONTEXT:
								(rdreg2_reg == `PRId)?						PRID:
								(rdreg2_reg == `EntryHI)?					EntryHIreg:
								(rdreg2_reg == `EntryLO)?					EntryLOreg:
								(rdreg2_reg == `Index)?						Indexreg:
								(rdreg2_reg == `Random)?					Randomreg:
								(rdreg2_reg == `Ext)?						EXTCTRreg:
																					32'b0;

/*********************************************************************/

	/* Pipeline Clock Edge Detection */
	always@(posedge CLK)
	begin
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	/* Latch Signals */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			bd_ex_reg <= 1'b0;
			bd_ext_reg <= 1'b0;
			bd_id_reg <= 1'b0;
			bd_ma_reg <= 1'b0;
			bd_wb_reg <= 1'b0;
			bubble_ex_reg <= 1'b0;
			bubble_id_reg <= 1'b0;
			bubble_ma_reg <= 1'b0;
			exc_ex_reg <= 1'b0;
			exc_ext_reg <= 1'b0;
			exc_id_reg <= 1'b0;
			exc_if_reg <= 1'b0;
			exc_ma_reg <= 1'b0;
			excadd_cd_ex_reg <= 32'b0;
			excadd_cd_ext_reg <= 32'b0;
			excadd_cd_id_reg <= 32'b0;
			excadd_cd_if_reg <= 32'b0;
			excadd_cd_ma_reg <= 32'b0;
			excadd_ci_ex_reg <= 32'b0;
			excadd_ci_ext_reg <= 32'b0;
			excadd_ci_id_reg <= 32'b0;
			excadd_ci_if_reg <= 32'b0;
			excadd_ci_int_reg <= 32'b0;
			excadd_ci_ma_reg <= 32'b0;
			exccode_ex_reg <= 7'b0;
			exccode_ext_reg <= 7'b0;
			exccode_id_reg <= 7'b0;
			exccode_if_reg <= 7'b0;
			exccode_int_reg <= 8'b0;
			exccode_ma_reg <= 7'b0;
			excflag_reg <= 1'b0;
			extstall_reg <= 1'b0;
			int_e_reg <= 8'b0;
			int_m_reg <= 8'b0;
			intcode_reg <= 7'b0;
			irq0_reg <= 1'b0;
			irq1_reg <= 1'b0;
			irq2_reg <= 1'b0;
			irq3_reg <= 1'b0;
			irq4_reg <= 1'b0;
			irq5_reg <= 1'b0;
			rddata1_reg <= 32'b0;
			rddata2_reg <= 32'b0;
			rdreg1_reg <= 5'b0;
			rdreg2_reg <= 5'b0;
			regwrite_reg <= 1'b0;
			reset_int_reg <= 1'b0;
			rfe_reg <= 1'b0;
			wrdata_reg <= 32'b0;
			wrreg_reg <= 5'b0;
		end
		else
		begin
			bd_ex_reg <= BD_EX;
			bd_ext_reg <= BD_EXT;
			bd_id_reg <= BD_ID;
			bd_ma_reg <= BD_MA;
			bd_wb_reg <= BD_WB;
			bubble_ex_reg <= BUBBLE_EX;
			bubble_id_reg <= BUBBLE_ID;
			bubble_ma_reg <= BUBBLE_MA;
			exc_ex_reg <= EXC_EX;
			exc_ext_reg <= EXC_EXT;
			exc_id_reg <= EXC_ID;
			exc_if_reg <= EXC_IF;
			exc_ma_reg <= EXC_MA;
			excadd_cd_ex_reg <= EXCADD_CD_EX;
			excadd_ci_ext_reg <= EXCADD_CI_EXT;
			excadd_cd_id_reg <= EXCADD_CD_ID;
			excadd_cd_if_reg <= EXCADD_CD_IF;
			excadd_cd_ma_reg <= EXCADD_CD_MA;
			excadd_cd_ext_reg <= EXCADD_CD_EXT;
			excadd_ci_ex_reg <= EXCADD_CI_EX;
			excadd_ci_id_reg <= EXCADD_CI_ID;
			excadd_ci_if_reg <= EXCADD_CI_IF;
			excadd_ci_int_reg <= EXCADD_CI_INT;
			excadd_ci_ma_reg <= EXCADD_CI_MA;
			exccode_ext_reg <= EXCCODE_EXT;
			exccode_ex_reg <= EXCCODE_EX;
			exccode_id_reg <= EXCCODE_ID;
			exccode_if_reg <= EXCCODE_IF;
			exccode_int_reg <= EXCCODE_INT;
			exccode_ma_reg <= EXCCODE_MA;
			excflag_reg <= EXCFLAG;
			extstall_reg <= EXTSTALL;
			int_e_reg <= INT_E;
			int_m_reg <= INT_M;
			intcode_reg <= INTCODE;
			irq0_reg <= IRQ0;
			irq1_reg <= IRQ1;
			irq2_reg <= IRQ2;
			irq3_reg <= IRQ3;
			irq4_reg <= IRQ4;
			irq5_reg <= IRQ5;
			rddata1_reg <= RDDATA1_0;
			rddata2_reg <= RDDATA2_0;
			rdreg1_reg <= RDREG1;
			rdreg2_reg <= RDREG2;
			regwrite_reg <= REGWRITE;
			rfe_reg <= RFE;
			reset_int_reg <= RESET_INT;
			wrdata_reg <= WRDATA;
			wrreg_reg <= WRREG;
		end
	end

	/* Reset Exception Detection */
	always@(posedge CLK)
	begin
		case(RESET)
			1'b0		:	begin
								/* Reset */
								resetcnt <= 2'b0;
							end
			default	:	begin
								case({pclkcnt})
									2'b01		:	begin
														/* Synchronize to Positive edge of Pipeline Clock */
														resetcnt <= {resetcnt[0],RESET};
													end
									default	:	begin
													end
								endcase
							end
		endcase
	end
																
	always@(posedge CLK)
	begin
		case(RESET)
			1'b0		:	begin
								/* Reset */
								BadVAddrreg <= 32'b0;
								BadVPNreg <= `BadVPN;
								BEVreg <= `BEV;
								BDreg <= `BD;
								CEreg <= `CE;
								CMreg <= `CM;
								CUreg <= `CU;
								EntryHIreg <= 32'b0;
								EntryLOreg <= 32'b0;
								EPCreg <= 32'hfffffffc;
								EXCADD_CIreg <= 32'b0;
								EXCADD_CDreg <= 32'b0; 
								EXCCODE_reg <= 8'hff;
								ExcCodereg <= `ExcCode;
								EXCEXTreg <= 1'b0;
								EXCreg <= 4'b0;
								EXTCTRreg <= 32'b0;
								FPreg <= `FR;
								IEcreg <= `IEc;
								IEoreg <= `IEo;
								IEpreg <= `IEp;
								Impreg <= `Imp;
								Indexreg <= 32'b0;
								Int_R <= 8'b0;
								IntMaskreg <= `IntMask;
								IPreg <= `IP;
								IsCreg <= `IsC;
								KUcreg <= `KUc;
								KUoreg <= `KUo;
								KUpreg <= `KUp;
								PEreg <= `PE;
								PTEBasereg <= `PTEBase;
								PZreg <= `PZ;
								Randomreg <= 32'h00003f00;
								REGWRITE_WB <= 1'b0;
								REGWRITE_MA <= 1'b0;
								REreg <= `RE;
								RFE_MA <= 1'b0;
								RFE_WB <= 1'b0;
								RPreg <= `RP;
								Revreg <= `Rev;
								SwCreg <= `SwC;
								Swreg <= `Sw;
								SWRSTreg <= `SWRST;
								TSreg <= `TS;
								WRDATA_MA <= 32'b0;
								WRDATA_WB <= 32'b0;
								WRREG_MA <= 5'b0;
								WRREG_WB <= 5'b0;
								RFE_LAT <= 1'b0;
								WRDATA_LAT <= 32'b0;
								WRREG_LAT <= 5'b0;
								REGWRITE_LAT <= 1'b0;
								exen_reg <= 1'b0;
							end
			default	:	begin
								casex(pclkcnt)
									2'b01		:	begin
														/* Synchronize to Postive Edge of Pipeline Clock */
														Int_R <= int_m_reg;
														/* Pipeline Coprocessor0 Register Update */
														REGWRITE_WB <= REGWRITE_MA;
														REGWRITE_MA <= REGWRITE_LAT;
														WRREG_WB <= WRREG_MA;
														WRREG_MA <= WRREG_LAT;
														WRDATA_WB <= WRDATA_MA;
														WRDATA_MA <= WRDATA_LAT;
														RFE_WB <= RFE_MA;
														RFE_MA <= RFE_LAT;
													end
									2'b10		:	begin
														/* Synchronize to Negative Edge of Pipeline Clock */
														RFE_LAT <= rfe_reg;
														WRDATA_LAT <= wrdata_reg;
														WRREG_LAT <= wrreg_reg;
														REGWRITE_LAT <= regwrite_reg;
														if (RFE_WB == 1'b1)
														begin
															/* Return form Exception */
															IEcreg <= IEpreg;
															IEpreg <= IEoreg;
															KUcreg <= KUpreg;
															KUpreg <= KUoreg;
														end
														if (REGWRITE_WB == 1'b1)
														begin
															/* Update Coprocessor0 Register */ 
															casex(WRREG_WB)
																`Cause		:	begin
																						/* Update Cause Register */
																						Swreg <= WRDATA_WB[lSw:rSw];
																					end
																`EPC			:	begin
																						/* EPC is read only */
																					end
																`Status		:	begin
																						/* Update Status Register */
																						BEVreg <= WRDATA_WB[rlBEV];
																						CMreg <= WRDATA_WB[rlCM];
																						CUreg <= WRDATA_WB[lCU:rCU];
																						FPreg <= WRDATA_WB[rlFP];
																						IEcreg <= WRDATA_WB[rlIEc];
																						IEoreg <= WRDATA_WB[rlIEo];
																						IEpreg <= WRDATA_WB[rlIEp];
																						IntMaskreg <= WRDATA_WB[lIntMask:rIntMask];
																						IsCreg <= WRDATA_WB[rlIsC];
																						KUcreg <= WRDATA_WB[rlKUc];
																						KUoreg <= WRDATA_WB[rlKUo];
																						KUpreg <= WRDATA_WB[rlKUp];
																						PEreg <= WRDATA_WB[rlPE];
																						PZreg <= WRDATA_WB[rlPZ];
																						REreg <= WRDATA_WB[rlRE];
																						RPreg <= WRDATA_WB[rlRP];
																						SwCreg <= WRDATA_WB[rlSwC];
																						SWRSTreg <= WRDATA_WB[rlSWRST];
																						TSreg <= WRDATA_WB[rlTS];
																					end
																`BadVAddr	:	begin
																						/* BadVAddr is read only */
																					end
																`Context		:	begin
																						/* Context is read only */
																					end
																`PRId			:	begin
																						/* PRId is read only */
																					end
																`EntryHI		:	begin
																						/* Update EntryHI */
																						EntryHIreg <= WRDATA_WB;
																					end
																`EntryLO		:	begin
																						/* Update EntryLO */
																						EntryLOreg	<= WRDATA_WB;
																					end
																`Index		:	begin
																						/* Update Index */
																						Indexreg <= WRDATA_WB;
																					end
																`Random		:	begin
																						/* Random is read only */
																					end
																`Ext			:	begin
																						/* Update Ext */
																						EXTCTRreg <= WRDATA_WB;
																					end
																default		:	begin
																						/* Register Number not recognized */
																					end
															endcase
														end
														casex(excflag_reg)
															1'b1		:	begin
																				/* Exception Occurred */
																				if (EXCCODE_reg[7]) 
																				begin
																					EPCreg <= EXCADD_CIreg  - 4;
																					BDreg <= 1'b1;
																				end
																				else 
																				begin
																					EPCreg <= EXCADD_CIreg;
																					BDreg <= 1'b0;
																				end
																				KUoreg <= KUpreg;
																				KUpreg <= KUcreg;
																				KUcreg <= 1'b0; 
																				IEoreg <= IEpreg;
																				IEpreg <= IEcreg;
																				IEcreg <= 1'b0;
																				RFE_MA <= 1'b0;
																				RFE_WB <= 1'b0;
																				REGWRITE_WB <= 1'b0;
																				REGWRITE_MA <= 1'b0;
																				WRREG_WB <= 5'b0;
																				WRREG_MA <= 5'b0;
																				WRDATA_WB <= 32'b0;
																				WRDATA_MA <= 32'b0;
																			end
															default	:	begin	
																				/* No exception */
																			end
														endcase
														casex({excflag_reg,EXCCODE_reg})
															{2'b1x,`Int0c}		:	begin
																							/* Software Interrupt 0 */
																							ExcCodereg <= `Int;
																							Swreg[0] <= 1'b1;
																							//$stop;
																						end
															{2'b1x,`Int1c}		:	begin
																							/* Software Interrupt 1 */
																							ExcCodereg <= `Int;
																							Swreg[1] <= 1'b1;
																							//$stop;
																						end
															{2'b1x,`Int2c}		:	begin
																							/* Hardware Interrupt 0 */
																							ExcCodereg <= `Int;
																							IPreg[0] <= 1'b1;
																							//$stop;
																						end
															{2'b1x,`Int3c}		:	begin
																							/* Hardware Interrupt 1 */
																							ExcCodereg <= `Int;
																							IPreg[1] <= 1'b1;
																							//$stop;
																						end
															{2'b1x,`Int4c}		:	begin
																							/* Hardware Interrupt 2 */
																							ExcCodereg <= `Int;
																							IPreg[2] <= 1'b1;
																							//$stop;
																						end
															{2'b1x,`Int5c}		:	begin
																							/* Hardware Interrupt 3 */
																							ExcCodereg <= `Int;
																							IPreg[3] <= 1'b1;
																							//$stop;
																						end
															{2'b1x,`Int6c}		:	begin
																							/* Hardware Interrupt 4 */
																							ExcCodereg <= `Int;
																							IPreg[4] <= 1'b1;
																							//$stop;
																						end
															{2'b1x,`Int7c}		:	begin
																							/* Hardware Interrupt 5 */
																							ExcCodereg <= `Int;
																							IPreg[5] <= 1'b1;
																							//$stop;
																						end
															{2'b1x,`MODc}		:	begin
																							/* TLB modification exception */
																							ExcCodereg <= `MOD;
																							BadVAddrreg <= EXCADD_CDreg; 
																							BadVPNreg <= EXCADD_CDreg[30:12]; 
																							EntryHIreg <= EXCADD_CDreg;
																							//$stop; 
																						end
															{2'b1x,`TLBLc}		:	begin
																							/* TLB miss exception (Load or Instruction Fetch) */
																							ExcCodereg <= `TLBL;
																							BadVAddrreg  <= EXCADD_CDreg; 
																							BadVPNreg <= EXCADD_CDreg[30:12]; 
																							EntryHIreg <= EXCADD_CDreg; 
																							//$stop;
																						end
															{2'b1x,`TLBSc}		:	begin
																							/* TLB miss exception (Store) */
																							ExcCodereg <= `TLBS;
																							BadVAddrreg  <= EXCADD_CDreg; 
																							BadVPNreg <= EXCADD_CDreg[30:12]; 
																							EntryHIreg <= EXCADD_CDreg; 
																							//$stop;
																						end
															{2'b1x,`UTLBLc}	:	begin
																							/* UTLB miss exception (Load or Instruction Fetch) */
																							ExcCodereg <= `TLBL;
																							BadVAddrreg  <= EXCADD_CDreg; 
																							BadVPNreg <= EXCADD_CDreg[30:12]; 
																							EntryHIreg <= EXCADD_CDreg; 
																							//$stop;
																						end
															{2'b1x,`UTLBSc}	:	begin
																							/* UTLB miss exception (Store) */
																							ExcCodereg <= `TLBS;
																							BadVAddrreg  <= EXCADD_CDreg; 
																							BadVPNreg <= EXCADD_CDreg[30:12]; 
																							EntryHIreg <= EXCADD_CDreg;
																							//$stop; 
																						end
															{2'b1x,`AdELc}		:	begin
																							/* Address error exception (Load or Instruction Fetch) */
																							ExcCodereg <= `AdEL;
																							BadVAddrreg  <= EXCADD_CDreg; 
																							//$stop;
																						end
															{2'b1x,`AdESc}		:	begin
																							/* Address error exception (Store) */
																							ExcCodereg <= `AdES;
																							BadVAddrreg  <= EXCADD_CDreg; 
																							//$stop;
																						end
															{2'b1x,`IBEc}		:	begin
																							/* Bus error exception (for an instruction fetch) */
																							ExcCodereg  <= `IBE;
																							//$stop;
																						end
															{2'b1x,`DBEc}		:	begin
																							/* Bus error exception (for a data load or store) */
																							ExcCodereg <= `DBE;
																							//$stop;
																						end
															{2'b1x,`SYSc}		:	begin
																							/* Syscall exception */
																							ExcCodereg <= `SYS;
																							//$stop;
																						end
															{2'b1x,`Bpc}		:	begin
																							/* Breakpoint exception */
																							ExcCodereg <= `Bp;
																							//$stop;
																						end
															{2'b1x,`RIc}		:	begin
																							/* Reserved Instruction exception */
																							ExcCodereg <= `RI;
																							//$stop;
																						end
															{2'b1x,`CpU0c}		:	begin
																							/* Coprocessor0 Unusable exception */
																							ExcCodereg <= `CpU;
																							CEreg <= 0;
																							//$stop;  
																						end
															{2'b1x,`CpU1c}		:	begin
																							/* Coprocessor1 Unusable exception */
																							ExcCodereg <= `CpU;
																							CEreg <= 1;
																							//$stop;  
																						end
															{2'b1x,`CpU2c}		:	begin
																							/* Coprocessor2 Unusable exception */
																							ExcCodereg <= `CpU;
																							CEreg <= 2;
																							//$stop;  
																						end
															{2'b1x,`CpU3c}		:	begin
																							/* Coprocessor3 Unusable exception */
																							ExcCodereg <= `CpU;
																							CEreg <= 3;
																							//$stop;  
																						end
															{2'b1x,`Ovfc}		:	begin
																							/* Arithmetic overflow exception */
																							ExcCodereg <= `Ovf;
																							//$stop;
																						end
															{2'b1x,`RSTc}		:	begin
																							/* Reset exception */
																							SwCreg <= 1'b0; 
																							BEVreg <= 1'b1;
																							Randomreg <= 32'h3f00;
																							EPCreg <= 32'hfffffffc;
																							BDreg <= `BD;
																							CEreg <= `CE;
																							IPreg <= `IP;
																							Swreg <= `Sw;
																							ExcCodereg <= `ExcCode;
																							exen_reg <= 1'b1;
																							//$stop;
																						end
															default				:	begin
																							/* Unrecognized Exception */
																						end
														endcase
													end
								endcase
								EXCreg <= EXC0;
								EXCEXTreg <= EXCEXT0;
								EXCADD_CIreg <= EXCADD_CI;
								EXCADD_CDreg <= EXCADD_CD;
								EXCCODE_reg <= EXCCODE;
							end
		endcase
	end
	
endmodule
