/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	dataforward														*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module dataforward(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,				/* System Clock 50 - 100 MHZ */
	input				REGWRITE_MA,	/* Register Write Enable Memory Access Phase */
	input				REGWRITE_WB,	/* Register Write Enable Writeback Phase */
	input				RESET,			/* System Reset */
	input [4:0]		RS_EX,			/* Operand Register 1 Execute Phase */
	input [4:0]		RS_ID,			/* Operand Register 1 Instruction Decode Phase */
	input [4:0]		RT_EX,			/* Operand Register 2 Execute Phase */
	input [4:0]		RT_ID,			/* Operand Register 2 Instruction Decode Phase */
	input [4:0]		RD_MA,			/* Destination Register Memory Access Phase */
	input [4:0]		RD_WB,			/* Destination Register Writeback Phase */
	/* OUTPUT PORTS */
	output [1:0]	FORWARD_A_EX,	/* Operand 1 Data Forward Switch Execute Phase */
	output [1:0]	FORWARD_A_ID,	/* Operand 1 Data Forward Switch Instruction Decode Phase */
	output [1:0]	FORWARD_B_EX,	/* Operand 2 Data Forward Switch Execute Phase */
	output [1:0]	FORWARD_B_ID	/* Operand 2 Data Forward Switch Instruction Decode Phase */
	);

/*****Registers****************************************************************/

	reg [1:0]	forward_a_ex_reg;	/* Operand 1 Data Forward Switch Execute Phase */
	reg [1:0]	forward_a_id_reg;	/* Operand 1 Data Forward Switch Instruction Decode Phase */
	reg [1:0]	forward_b_ex_reg;	/* Operand 2 Data Forward Switch Execute Phase */
	reg [1:0]	forward_b_id_reg;	/* Operand 2 Data Forward Switch Instruction Decode Phase */
	reg			regwrite_ma_reg;	/* Register Write Enable Memory Access Phase */
	reg			regwrite_wb_reg;	/* Register Write Enable Writeback Phase */
	reg [4:0]	rs_ex_reg;			/* Operand Register 1 Execute Phase */
	reg [4:0]	rs_id_reg;			/* Operand Register 1 Instruction Decode Phase */
	reg [4:0]	rt_ex_reg;			/* Operand Register 2 Execute Phase */
	reg [4:0]	rt_id_reg;			/* Operand Register 2 Instruction Decode Phase */
	reg [4:0]	rd_ma_reg;			/* Destination Register Memory Access Phase */
	reg [4:0]	rd_wb_reg;			/* Destination Register Writeback Phase */

/*****Initialization****************************************************************/

	initial
	begin
		forward_a_ex_reg = 2'b0;
		forward_a_id_reg = 2'b0;
		forward_b_ex_reg = 2'b0;
		forward_b_id_reg = 2'b0;
		regwrite_ma_reg = 1'b0;
		regwrite_wb_reg = 1'b0;
		rs_ex_reg = 5'b0;
		rs_id_reg = 5'b0;
		rt_ex_reg = 5'b0;
		rt_id_reg = 5'b0;
		rd_ma_reg = 5'b0;
		rd_wb_reg = 5'b0;
	end

/*********************************************************************/

	assign FORWARD_A_ID =	(RESET == 0)?		2'b00:
															forward_a_id_reg;
	assign FORWARD_B_ID = 	(RESET == 0)?		2'b00:
															forward_b_id_reg;
	assign FORWARD_A_EX =	(RESET == 0)?		2'b00:
															forward_a_ex_reg;
	assign FORWARD_B_EX = 	(RESET == 0)?		2'b00:
															forward_b_ex_reg;

/*********************************************************************/

	/* Latch Signals */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			regwrite_ma_reg <= 1'b0;
			regwrite_wb_reg <= 1'b0;
			rs_ex_reg <= 5'b0;
			rs_id_reg <= 5'b0;
			rt_ex_reg <= 5'b0;
			rt_id_reg <= 5'b0;
			rd_ma_reg <= 5'b0;
			rd_wb_reg <= 5'b0;
		end
		else
		begin
			regwrite_ma_reg <= REGWRITE_MA;
			regwrite_wb_reg <= REGWRITE_WB;
			rs_ex_reg <= RS_EX;
			rs_id_reg <= RS_ID;
			rt_ex_reg <= RT_EX;
			rt_id_reg <= RT_ID;
			rd_ma_reg <= RD_MA;
			rd_wb_reg <= RD_WB;
		end
	end

	/* Operand 1 Data Forward Switch Instruction Decode Phase */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						forward_a_id_reg <= 2'b0;
		else
		begin
			if (rs_id_reg == 5'b0)				forward_a_id_reg <= 2'b0;
			else
			begin
				if (rs_id_reg == rd_ma_reg)	forward_a_id_reg[1] <= regwrite_ma_reg;
				else									forward_a_id_reg[1] <= 1'b0;
				if (rs_id_reg == rd_wb_reg)	forward_a_id_reg[0] <= regwrite_wb_reg & ~forward_a_id_reg[1];
				else 									forward_a_id_reg[0] <= 1'b0;
			end
		end
	end

	/* Operand 2 Data Forward Switch Instruction Decode Phase */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						forward_b_id_reg <= 2'b0;
		else
		begin
			if (rt_id_reg == 5'b0)				forward_b_id_reg <= 2'b0;
			else
			begin
				if (rt_id_reg == rd_ma_reg)	forward_b_id_reg[1] <= regwrite_ma_reg;
				else 									forward_b_id_reg[1] <= 1'b0;
				if (rt_id_reg == rd_wb_reg)	forward_b_id_reg[0] <= regwrite_wb_reg & ~forward_b_id_reg[1];
				else									forward_b_id_reg[0] <= 1'b0;
			end
		end
	end

	/* Operand 1 Data Forward Switch Execute Phase */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						forward_a_ex_reg <= 2'b0;
		else
		begin
			if (rs_ex_reg == 5'b0)				forward_a_ex_reg <= 2'b0;
			else
			begin
				if (rs_ex_reg == rd_ma_reg)	forward_a_ex_reg[1] <= regwrite_ma_reg;
				else									forward_a_ex_reg[1] <= 1'b0;
				if (rs_ex_reg == rd_wb_reg)	forward_a_ex_reg[0] <= regwrite_wb_reg & ~forward_a_ex_reg[1];
				else									forward_a_ex_reg[0] <= 1'b0;
			end
		end
	end

	/* Operand 2 Data Forward Switch Execute Phase */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						forward_b_ex_reg <= 2'b0;
		else
		begin
			if (rt_ex_reg == 5'b0)				forward_b_ex_reg <= 2'b0;
			else
			begin
				if (rt_ex_reg == rd_ma_reg)	forward_b_ex_reg[1] <= regwrite_ma_reg;
				else									forward_b_ex_reg[1] <= 1'b0;
				if (rt_ex_reg == rd_wb_reg)	forward_b_ex_reg[0] <= regwrite_wb_reg & ~forward_b_ex_reg[1];
				else									forward_b_ex_reg[0] <= 1'b0;
			end
		end
	end

endmodule
