/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	datapathcontrol												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"

module datapathcontrol(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [4:0]		BCFUNC,			/* Bi-Conditional Function Code */
	input 			CLK,				/* System Clock 50 - 100 MHZ */
	input [3:0]		CP,				/* Enabled Coprocessors */
	input [4:0]		CPFUNC,			/* Coprocessor Function Code */
	input 			EN,				/* TISA Enable */
	input 			EN_EXT,			/* Extension Enable */
	input [5:0]		FUNC,				/* Function Code */
	input 			KU,				/* Kernel/User Mode */
	input [5:0]		OP,				/* Opcode */
	input 			RESET,			/* System Reset */
	input 			RI_PA,			/* Pipeline Arbiter Reserved/Recognized Instruction */
	/* OUTPUT PORTS */
	output [5:0]	ALUOP,			/* ALU Opcode */
	output 			ALUSRC,			/* ALU Operand B Switch */
	output 			BLS,				/* Byte Load/Store */
	output 			BRANCH,			/* Branch */
	output 			CEXC,				/* Conditional Exception */
	output 			CMOV,				/* Conditional Move */
	output 			COMPRZ,			/* Compare with Zero */
	output 			COND,				/* Conditional Operation */
	output 			CP0,				/* Coprocessor 0 Operation */
	output 			CP1,				/* Coprocessor 1 Operation */
	output 			CP2,				/* Coprocessor 2 Operation */
	output 			CP3,				/* Coprocessor 3 Operation */
	output 			CPREGWRITE,		/* Coprocessor Register Write Enable */
	output 			EXC,				/* Exception */
	output 			EXT,				/* Sign Extend */
	output 			FLUSH,			/* Flush Enabled Operation */
	output 			FLUSH_EXT,		/* Flush for Extension */
	output 			GL,				/* Greaterthan/Lessthan Operation */
	output 			GLUS,				/* Greaterthan/Lessthan Unsigned */
	output 			HLS,				/* Halfword Load/Store */
	output 			JPR,				/* Jump Registered */
	output 			JUMP,				/* Jump Operation */
	output 			LC,				/* Load/Store Conditional */
	output 			LINK,				/* Linked Branch */
	output 			LNG,				/* Lessthan/Greaterthan Operation */
	output 			LUI,				/* Load Upper Immediate */
	output 			MEMREAD,			/* Memory Read */
	output 			MEMTOREG,		/* Memory to Regiser */
	output 			MEMWRITE,		/* Memory Write */
	output 			MEXT,				/* Sign Extend Memory Data */
	output 			MULT,				/* Instruction Decode Multiplication Unit Operation */
	output 			REGDST,			/* Destination Register Switch */
	output 			REGWRITE,		/* Register Write */
	output 			RFE,				/* Return From Exception */
	output 			RI,				/* Reserved/Recognized Instruction */
	output 			RNL				/* Right/Left Unaligned Load/Store */
	);

/*****Registers****************************************************************/

	reg [5:0]	aluop_reg;			/* ALU Opcode */
	reg 			alusrc_reg;			/* ALU Operand B Switch */
	reg 			bls_reg;				/* Byte Load/Store */
	reg 			branch_reg;			/* Branch */
	reg 			cexc_reg;			/* Conditional Exception */
	reg 			cmov_reg;			/* Conditional Move */
	reg 			comprz_reg;			/* Compare with Zero */
	reg 			cond_reg;			/* Conditional Operation */
	reg 			cp0_reg;				/* Coprocessor 0 Operation */
	reg 			cp1_reg;				/* Coprocessor 1 Operation */
	reg 			cp2_reg;				/* Coprocessor 2 Operation */
	reg 			cp3_reg;				/* Coprocessor 3 Operation */
	reg 			cpregwrite_reg;	/* Coprocessor Register Write Enable */
	reg 			exc_reg;				/* Exception */
	reg 			ext_reg;				/* Sign Extend */
	reg 			flush_reg;			/* Flush Enabled Operation */
	reg 			flush_ext_reg;		/* Flush for Extension */
	reg 			gl_reg;				/* Greaterthan/Lessthan Operation */
	reg 			glus_reg;			/* Greaterthan/Lessthan Unsigned */
	reg 			hls_reg;				/* Halfword Load/Store */
	reg 			jpr_reg;				/* Jump Registered */
	reg 			jump_reg;			/* Jump Operation */
	reg 			lc_reg;				/* Load/Store Conditional */
	reg 			link_reg;			/* Linked Branch */
	reg 			lng_reg;				/* Lessthan/Greaterthan Operation */
	reg 			lui_reg;				/* Load Upper Immediate */
	reg 			memread_reg;		/* Memory Read */
	reg 			memtoreg_reg;		/* Memory to Regiser */
	reg 			memwrite_reg;		/* Memory Write */
	reg 			mext_reg;			/* Sign Extend Memory Data */
	reg 			mult_reg;			/* Instruction Decode Multiplication Unit Operation */
	reg 			regdst_reg;			/* Destination Register Switch */
	reg 			regwrite_reg;		/* Register Write */
	reg 			rfe_reg;				/* Return From Exception */
	reg 			ri_reg;				/* Reserved/Recognized Instruction */
	reg 			rnl_reg;				/* Right/Left Unaligned Load/Store */

/*****Initialization****************************************************************/

	initial
	begin
		aluop_reg = 6'b0;				/* ALU Opcode */
		alusrc_reg = 1'b0;			/* ALU Operand B Switch */
		bls_reg = 1'b0;				/* Byte Load/Store */
		branch_reg = 1'b0;			/* Branch */
		cexc_reg = 1'b0;				/* Conditional Exception */
		cmov_reg = 1'b0;				/* Conditional Move */
		comprz_reg = 1'b0;			/* Compare with Zero */
		cond_reg = 1'b0;				/* Conditional Operation */
		cp0_reg = 1'b0;				/* Coprocessor 0 Operation */
		cp1_reg = 1'b0;				/* Coprocessor 1 Operation */
		cp2_reg = 1'b0;				/* Coprocessor 2 Operation */
		cp3_reg = 1'b0;				/* Coprocessor 3 Operation */
		cpregwrite_reg = 1'b0;		/* Coprocessor Register Write Enable */
		exc_reg = 1'b0;				/* Exception */
		ext_reg = 1'b0;				/* Sign Extend */
		flush_reg = 1'b0;				/* Flush Enabled Operation */
		flush_ext_reg = 1'b0;		/* Flush for Extension */
		gl_reg = 1'b0;					/* Greaterthan/Lessthan Operation */
		glus_reg = 1'b0;				/* Greaterthan/Lessthan Unsigned */
		hls_reg = 1'b0;				/* Halfword Load/Store */
		jpr_reg = 1'b0;				/* Jump Registered */
		jump_reg = 1'b0;				/* Jump Operation */
		lc_reg = 1'b0;					/* Load/Store Conditional */
		link_reg = 1'b0;				/* Linked Branch */
		lng_reg = 1'b0;				/* Lessthan/Greaterthan Operation */
		lui_reg = 1'b0;				/* Load Upper Immediate */
		memread_reg = 1'b0;			/* Memory Read */
		memtoreg_reg = 1'b0;			/* Memory to Regiser */
		memwrite_reg = 1'b0;			/* Memory Write */
		mext_reg = 1'b0;				/* Sign Extend Memory Data */
		mult_reg = 1'b0;				/* Instruction Decode Multiplication Unit Operation */
		regdst_reg = 1'b0;			/* Destination Register Switch */
		regwrite_reg = 1'b0;			/* Register Write */
		rfe_reg = 1'b0;				/* Return From Exception */
		ri_reg = 1'b0;					/* Reserved/Recognized Instruction */
		rnl_reg = 1'b0;				/* Right/Left Unaligned Load/Store */
	end

/*********************************************************************/
	
	assign ALUOP		=	aluop_reg;			/* ALU Opcode */
	assign ALUSRC		=	alusrc_reg;			/* ALU Operand B Switch */
	assign BLS			=	bls_reg;				/* Byte Load/Store */
	assign BRANCH		=	branch_reg;			/* Branch */
	assign CEXC			=	cexc_reg;			/* Conditional Exception */
	assign CMOV			=	cmov_reg;			/* Conditional Move */
	assign COMPRZ		=	comprz_reg;			/* Compare with Zero */
	assign COND			=	cond_reg;			/* Conditional Operation */
	assign CP0			=	cp0_reg;				/* Coprocessor 0 Operation */
	assign CP1			=	cp1_reg;				/* Coprocessor 1 Operation */
	assign CP2			=	cp2_reg;				/* Coprocessor 2 Operation */
	assign CP3			=	cp3_reg;				/* Coprocessor 3 Operation */
	assign CPREGWRITE	=	cpregwrite_reg;	/* Coprocessor Register Write Enable */
	assign EXC			=	exc_reg;				/* Exception */
	assign EXT			=	ext_reg;				/* Sign Extend */
	assign FLUSH		=	flush_reg;			/* Flush Enabled Operation */
	assign FLUSH_EXT	=	flush_ext_reg;		/* Flush for Extension */
	assign GL			=	gl_reg;				/* Greaterthan/Lessthan Operation */
	assign GLUS			=	glus_reg;			/* Greaterthan/Lessthan Unsigned */
	assign HLS			=	hls_reg;				/* Halfword Load/Store */
	assign JPR			=	jpr_reg;				/* Jump Registered */
	assign JUMP			=	jump_reg;			/* Jump Operation */
	assign LC			=	lc_reg;				/* Load/Store Conditional */
	assign LINK			=	link_reg;			/* Linked Branch */
	assign LNG			=	lng_reg;				/* Lessthan/Greaterthan Operation */
	assign LUI			=	lui_reg;				/* Load Upper Immediate */
	assign MEMREAD		=	memread_reg;		/* Memory Read */
	assign MEMTOREG	=	memtoreg_reg;		/* Memory to Regiser */
	assign MEMWRITE	=	memwrite_reg;		/* Memory Write */
	assign MEXT			=	mext_reg;			/* Sign Extend Memory Data */
	assign MULT			=	mult_reg;			/* Instruction Decode Multiplication Unit Operation */
	assign REGDST		=	regdst_reg;			/* Destination Register Switch */
	assign REGWRITE	=	regwrite_reg;		/* Register Write */
	assign RFE			=	rfe_reg;				/* Return From Exception */
	assign RI			=	ri_reg;				/* Reserved/Recognized Instruction */
	assign RNL			=	rnl_reg;				/* Right/Left Unaligned Load/Store */

/*********************************************************************/

	/* Reserved/Recognized Instruction */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						ri_reg <= 1'b0;
		else
		begin
			if (OP == `SPEC)
			begin
				if (FUNC == `NOP)					ri_reg <= 1'b0;
				else if (FUNC == `ADD)			ri_reg <= 1'b0;
				else if (FUNC == `ADDU)			ri_reg <= 1'b0;
				else if (FUNC == `SUB)			ri_reg <= 1'b0;
				else if (FUNC == `SUBU)			ri_reg <= 1'b0;
				else if (FUNC == `AND)			ri_reg <= 1'b0;
				else if (FUNC == `OR)			ri_reg <= 1'b0;
				else if (FUNC == `XOR)			ri_reg <= 1'b0;
				else if (FUNC == `SLT)			ri_reg <= 1'b0;
				else if (FUNC == `SLTU)			ri_reg <= 1'b0;
				else if (FUNC == `NOR)			ri_reg <= 1'b0;
				else if (FUNC == `MFHI)			ri_reg <= 1'b0;
				else if (FUNC == `MFLO)			ri_reg <= 1'b0;
				else if (FUNC == `MTHI)			ri_reg <= 1'b0;
				else if (FUNC == `MTLO)			ri_reg <= 1'b0;
				else if (FUNC == `MULT) 		ri_reg <= 1'b0;
				else if (FUNC == `MULTU)		ri_reg <= 1'b0;
				else if (FUNC == `DIV)			ri_reg <= 1'b0;
				else if (FUNC == `DIVU)			ri_reg <= 1'b0;
				else if (FUNC == `SLL)			ri_reg <= 1'b0;
				else if (FUNC == `SRL)			ri_reg <= 1'b0;
				else if (FUNC == `SRA)			ri_reg <= 1'b0;
				else if (FUNC == `SLLV)			ri_reg <= 1'b0;
				else if (FUNC == `SRLV)			ri_reg <= 1'b0;
				else if (FUNC == `SRAV)			ri_reg <= 1'b0;
				else if (FUNC == `JR)			ri_reg <= 1'b0;
				else if (FUNC == `JALR)			ri_reg <= 1'b0;
				else if (FUNC == `MOVZ)			ri_reg <= 1'b0;
				else if (FUNC == `MOVN)			ri_reg <= 1'b0;
				else if (FUNC == `SYSCALL)		ri_reg <= 1'b0;
				else if (FUNC == `BREAK)		ri_reg <= 1'b0;
				else if (FUNC == `TGE)			ri_reg <= 1'b0;
				else if (FUNC == `TGEU)			ri_reg <= 1'b0;
				else if (FUNC == `TLT)			ri_reg <= 1'b0;
				else if (FUNC == `TLTU)			ri_reg <= 1'b0;
				else if (FUNC == `TEQ)			ri_reg <= 1'b0;
				else if (FUNC == `TNE)			ri_reg <= 1'b0;
				else		 							ri_reg <= 1'b1;
			end	
			else if (OP == `BCOND)
			begin
				if (BCFUNC == `BLTZ)				ri_reg <= 1'b0;
				else if (BCFUNC == `BGEZ)		ri_reg <= 1'b0;
				else if (BCFUNC == `BLTZL)		ri_reg <= 1'b0;
				else if (BCFUNC == `BGEZL)		ri_reg <= 1'b0;
				else if (BCFUNC == `BLTZAL)	ri_reg <= 1'b0;
				else if (BCFUNC == `BGEZAL)	ri_reg <= 1'b0;
				else if (BCFUNC == `BLTZALL)	ri_reg <= 1'b0;
				else if (BCFUNC == `BGEZALL)	ri_reg <= 1'b0;
				else if (BCFUNC == `TGEI)		ri_reg <= 1'b0;
				else if (BCFUNC == `TGEIU)		ri_reg <= 1'b0;
				else if (BCFUNC == `TLTI)		ri_reg <= 1'b0;
				else if (BCFUNC == `TLTIU)		ri_reg <= 1'b0;
				else if (BCFUNC == `TEQI)		ri_reg <= 1'b0;
				else if (BCFUNC == `TNEI)		ri_reg <= 1'b0;
				else 									ri_reg <= 1'b1;
			end
			else if (OP == `COP0)
			begin
				if (CPFUNC == `C0)
				begin
					if (FUNC == {1'b0,`RFE})	ri_reg <= 1'b0;
					else 								ri_reg <= 1'b1;
				end
				else if (CPFUNC == `MF)			ri_reg <= 1'b0;
				else if (CPFUNC == `MT)			ri_reg <= 1'b0;
				else 									ri_reg <= 1'b1;
			end
			else if (OP == `J)					ri_reg <= 1'b0;
			else if (OP == `JAL)					ri_reg <= 1'b0;
			else if (OP == `BEQ)					ri_reg <= 1'b0;
			else if (OP == `BNE)					ri_reg <= 1'b0;
			else if (OP == `BLEZ)				ri_reg <= 1'b0;
			else if (OP == `BGTZ)				ri_reg <= 1'b0;
			else if (OP == `ADDI)				ri_reg <= 1'b0;
			else if (OP == `ADDIU)				ri_reg <= 1'b0;
			else if (OP == `SLTI)				ri_reg <= 1'b0;
			else if (OP == `SLTIU)				ri_reg <= 1'b0;
			else if (OP == `ANDI)				ri_reg <= 1'b0;
			else if (OP == `ORI)					ri_reg <= 1'b0;
			else if (OP == `XORI)				ri_reg <= 1'b0;
			else if (OP == `LUI)					ri_reg <= 1'b0;
			else if (OP == `BEQL)				ri_reg <= 1'b0;
			else if (OP == `BNEL)				ri_reg <= 1'b0;
			else if (OP == `BLEZL)				ri_reg <= 1'b0;
			else if (OP == `BGTZL)				ri_reg <= 1'b0;
			else if (OP == `LB)					ri_reg <= 1'b0;
			else if (OP == `LH)					ri_reg <= 1'b0;
			else if (OP == `LWL)					ri_reg <= 1'b0;
			else if (OP == `LW)					ri_reg <= 1'b0;
			else if (OP == `LBU)					ri_reg <= 1'b0;
			else if (OP == `LHU)					ri_reg <= 1'b0;
			else if (OP == `LWR)					ri_reg <= 1'b0;
			else if (OP == `LWU)					ri_reg <= 1'b0;
			else if (OP == `SB)					ri_reg <= 1'b0;
			else if (OP == `SH)					ri_reg <= 1'b0;
			else if (OP == `SWL)					ri_reg <= 1'b0;
			else if (OP == `SW)					ri_reg <= 1'b0;
			else if (OP == `SWR)					ri_reg <= 1'b0;
			else if (OP == `LL)					ri_reg <= 1'b0;
			else if (OP == `SC)					ri_reg <= 1'b0;
			else 										ri_reg <= 1'b1;
		end
	end

	/* ALU Opcode */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							aluop_reg <= `NOP;
		else 
		begin
			if (EN == 1'b0)							aluop_reg <= `NOP;
			else
			begin
				if (OP == `SPEC)
				begin
					if (FUNC == `MOVN)				aluop_reg <= `SUB;
					else if (FUNC == `MOVZ)			aluop_reg <= `SUB;
					else if (FUNC == `JALR)			aluop_reg <= `ADD;
					else if (FUNC == `JR)			aluop_reg <= `NOP;
					else if (FUNC == `SYSCALL)		aluop_reg <= `NOP;
					else if (FUNC == `BREAK)		aluop_reg <= `NOP;
					else if (FUNC == `TGE)			aluop_reg <= `NOP;
					else if (FUNC == `TGEU)			aluop_reg <= `NOP;
					else if (FUNC == `TLT)			aluop_reg <= `NOP;
					else if (FUNC == `TLTU)			aluop_reg <= `NOP;
					else if (FUNC == `TEQ)			aluop_reg <= `NOP;
					else if (FUNC == `TNE)			aluop_reg <= `NOP;
					else 									aluop_reg <= FUNC;
				end
				else if (OP == `BCOND)
				begin
					if (BCFUNC == `BLTZAL)			aluop_reg <= `ADD;
					else if (BCFUNC == `BGEZAL)	aluop_reg <= `ADD;
					else if (BCFUNC == `BLTZALL)	aluop_reg <= `ADD;
					else if (BCFUNC == `BGEZALL)	aluop_reg <= `ADD;
					else 									aluop_reg <= `NOP;
				end
				else if (OP == `JAL)					aluop_reg <= `ADD;
				else if (OP == `LW)					aluop_reg <= `ADD;
				else if (OP == `LL)					aluop_reg <= `ADD;
				else if (OP == `LWU)					aluop_reg <= `ADD;
				else if (OP == `LB)					aluop_reg <= `ADD;
				else if (OP == `LBU)					aluop_reg <= `ADD;
				else if (OP == `LH)					aluop_reg <= `ADD;
				else if (OP == `LHU)					aluop_reg <= `ADD;
				else if (OP == `LWL)					aluop_reg <= `ADD;
				else if (OP == `LWR)					aluop_reg <= `ADD;
				else if (OP == `SW)					aluop_reg <= `ADD;
				else if (OP == `SC)					aluop_reg <= `ADD;
				else if (OP == `SB)					aluop_reg <= `ADD;
				else if (OP == `SH)					aluop_reg <= `ADD;
				else if (OP == `SWL)					aluop_reg <= `ADD;
				else if (OP == `SWR)					aluop_reg <= `ADD;
				else if (OP == `ADDI)				aluop_reg <= `ADD;
				else if (OP == `ADDIU)				aluop_reg <= `ADDU;
				else if (OP == `ANDI)				aluop_reg <= `AND;
				else if (OP == `ORI)					aluop_reg <= `OR;
				else if (OP == `XORI)				aluop_reg <= `XOR;
				else if (OP == `SLTI)				aluop_reg <= `SLT;
				else if (OP == `SLTIU)				aluop_reg <= `SLTU;
				else										aluop_reg <= `NOP;
			end
		end
	end

	/* Destination Register Switch */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)					regdst_reg <= 1'b0;
		else 
		begin
			if (EN == 1'b0)					regdst_reg <= 1'b0;
			else
			begin
				if (OP == `COP0)
				begin
					if (CPFUNC == `MT)		regdst_reg <= 1'b1;
					else 							regdst_reg <= 1'b0;
				end
				else if (OP == `SPEC)		regdst_reg <= 1'b1;
				else								regdst_reg <= 1'b0;
			end
		end
	end

	/* ALU Operand B Switch */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							alusrc_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							alusrc_reg <= 1'b0;
			else
			begin
				if (OP == `BCOND)
				begin
					if (BCFUNC == `TGEI)				alusrc_reg <= 1'b1;
					else if (BCFUNC == `TGEIU)		alusrc_reg <= 1'b1;
					else if (BCFUNC == `TLTI)		alusrc_reg <= 1'b1;
					else if (BCFUNC == `TLTIU)		alusrc_reg <= 1'b1;
					else if (BCFUNC == `TEQI)		alusrc_reg <= 1'b1;
					else if (BCFUNC == `TNEI)		alusrc_reg <= 1'b1;
					else 									alusrc_reg <= 1'b0;
				end
				else if (OP == `LW)					alusrc_reg <= 1'b1;
				else if (OP == `LL)					alusrc_reg <= 1'b1;
				else if (OP == `LWU)					alusrc_reg <= 1'b1;
				else if (OP == `LB)					alusrc_reg <= 1'b1;
				else if (OP == `LBU)					alusrc_reg <= 1'b1;
				else if (OP == `LH)					alusrc_reg <= 1'b1;
				else if (OP == `LHU)					alusrc_reg <= 1'b1;
				else if (OP == `LWL)					alusrc_reg <= 1'b1;
				else if (OP == `LWR)					alusrc_reg <= 1'b1;
				else if (OP == `SW)					alusrc_reg <= 1'b1;
				else if (OP == `SC)					alusrc_reg <= 1'b1;
				else if (OP == `SB)					alusrc_reg <= 1'b1;
				else if (OP == `SH)					alusrc_reg <= 1'b1;
				else if (OP == `SWL)					alusrc_reg <= 1'b1;
				else if (OP == `SWR)					alusrc_reg <= 1'b1;
				else if (OP == `ADDI)				alusrc_reg <= 1'b1;
				else if (OP == `ADDIU)				alusrc_reg <= 1'b1;
				else if (OP == `ANDI)				alusrc_reg <= 1'b1;
				else if (OP == `ORI)					alusrc_reg <= 1'b1;
				else if (OP == `XORI)				alusrc_reg <= 1'b1;
				else if (OP == `SLTI)				alusrc_reg <= 1'b1;
				else if (OP == `SLTIU)				alusrc_reg <= 1'b1;
				else										alusrc_reg <= 1'b0;
			end
		end
	end

	/* Memory to Regiser */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			memtoreg_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)			memtoreg_reg <= 1'b0;
			else
			begin
				if (OP == `LW)			memtoreg_reg <= 1'b1;
				else if (OP == `LL)	memtoreg_reg <= 1'b1;
				else if (OP == `LWU)	memtoreg_reg <= 1'b1;
				else if (OP == `LB)	memtoreg_reg <= 1'b1;
				else if (OP == `LBU)	memtoreg_reg <= 1'b1;
				else if (OP == `LH)	memtoreg_reg <= 1'b1;
				else if (OP == `LHU)	memtoreg_reg <= 1'b1;
				else if (OP == `LWL)	memtoreg_reg <= 1'b1;
				else if (OP == `LWR)	memtoreg_reg <= 1'b1;
				else if (OP == `SC)	memtoreg_reg <= 1'b1;
				else						memtoreg_reg <= 1'b0;
			end
		end
	end

	/* Register Write */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							regwrite_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							regwrite_reg <= 1'b0;
			else
			begin
				if (OP == `SPEC)
				begin
					if (FUNC == `SYSCALL)			regwrite_reg <= 1'b0;
					else if (FUNC == `BREAK)		regwrite_reg <= 1'b0;
					else if (FUNC == `TGE)			regwrite_reg <= 1'b0;
					else if (FUNC == `TGEU)			regwrite_reg <= 1'b0;
					else if (FUNC == `TLT)			regwrite_reg <= 1'b0;
					else if (FUNC == `TLTU)			regwrite_reg <= 1'b0;
					else if (FUNC == `TEQ)			regwrite_reg <= 1'b0;
					else if (FUNC == `TNE)			regwrite_reg <= 1'b0;
					else if (FUNC == `MULT)			regwrite_reg <= 1'b0;
					else if (FUNC == `MULTU)		regwrite_reg <= 1'b0;
					else if (FUNC == `DIV)			regwrite_reg <= 1'b0;
					else if (FUNC == `DIVU)			regwrite_reg <= 1'b0;
					else if (FUNC == `MTHI)			regwrite_reg <= 1'b0;
					else if (FUNC == `MTLO)			regwrite_reg <= 1'b0;
					else if (FUNC == `JR)			regwrite_reg <= 1'b0;
					else									regwrite_reg <= 1'b1;
				end
				else if (OP == `BCOND)
				begin
					if (BCFUNC == `BLTZAL)			regwrite_reg <= 1'b1;
					else if (BCFUNC == `BGEZAL)	regwrite_reg <= 1'b1;
					else if (BCFUNC == `BLTZALL)	regwrite_reg <= 1'b1;
					else if (BCFUNC == `BGEZALL)	regwrite_reg <= 1'b1;
					else 									regwrite_reg <= 1'b0;
				end
				else if (OP == `COP0)
				begin
					if (CPFUNC == `MF)				regwrite_reg <= 1'b1;
					else									regwrite_reg <= 1'b0;
				end
				else if (OP == `JAL)					regwrite_reg <= 1'b1;
				else if (OP == `LW)					regwrite_reg <= 1'b1;
				else if (OP == `LL)					regwrite_reg <= 1'b1;
				else if (OP == `LWU)					regwrite_reg <= 1'b1;
				else if (OP == `LB)					regwrite_reg <= 1'b1;
				else if (OP == `LBU)					regwrite_reg <= 1'b1;
				else if (OP == `LH)					regwrite_reg <= 1'b1;
				else if (OP == `LHU)					regwrite_reg <= 1'b1;
				else if (OP == `LWL)					regwrite_reg <= 1'b1;
				else if (OP == `LWR)					regwrite_reg <= 1'b1;
				else if (OP == `ADDI)				regwrite_reg <= 1'b1;
				else if (OP == `ADDIU)				regwrite_reg <= 1'b1;
				else if (OP == `ANDI)				regwrite_reg <= 1'b1;
				else if (OP == `ORI)					regwrite_reg <= 1'b1;
				else if (OP == `XORI)				regwrite_reg <= 1'b1;
				else if (OP == `SLTI)				regwrite_reg <= 1'b1;
				else if (OP == `SLTIU)				regwrite_reg <= 1'b1;
				else if (OP == `LUI)					regwrite_reg <= 1'b1;
				else if (OP == `SC)					regwrite_reg <= 1'b1;
				else										regwrite_reg <= 1'b0;
			end
		end
	end

	/* Coprocessor Register Write Enable */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)				cpregwrite_reg <= 1'b0;
		else 
		begin
			if (EN == 1'b0)				cpregwrite_reg <= 1'b0;
			else
			begin
				if (OP == `COP0)
				begin
					if (CPFUNC == `MT)	cpregwrite_reg <= 1'b1;
					else 						cpregwrite_reg <= 1'b0;
				end
				else							cpregwrite_reg <= 1'b0;
			end
		end
	end

	/* Coprocessor 0 Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)		cp0_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)		cp0_reg <= 1'b0;
			else
			begin
				if (OP == `COP0)	cp0_reg <= 1'b1;
				else					cp0_reg <= 1'b0;
			end
		end
	end

	/* Coprocessor 1 Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)		cp1_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)		cp1_reg <= 1'b0;
			else
			begin
										cp1_reg <= 1'b0;
			end
		end
	end

	/* Coprocessor 2 Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)		cp2_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)		cp2_reg <= 1'b0;
			else
			begin
										cp2_reg <= 1'b0;
			end
		end
	end

	/* Coprocessor 3 Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)		cp3_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)		cp3_reg <= 1'b0;
			else
			begin
										cp3_reg <= 1'b0;
			end
		end
	end

	/* Memory Read */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			memread_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)			memread_reg <= 1'b0;
			else
			begin
				if (OP == `LW)			memread_reg <= 1'b1;
				else if (OP == `LL)	memread_reg <= 1'b1;
				else if (OP == `LWU)	memread_reg <= 1'b1;
				else if (OP == `LB)	memread_reg <= 1'b1;
				else if (OP == `LBU)	memread_reg <= 1'b1;
				else if (OP == `LH)	memread_reg <= 1'b1;
				else if (OP == `LHU)	memread_reg <= 1'b1;
				else if (OP == `LWL)	memread_reg <= 1'b1;
				else if (OP == `LWR)	memread_reg <= 1'b1;
				else 						memread_reg <= 1'b0;
			end
		end
	end

	/* Memory Write */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			memwrite_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)			memwrite_reg <= 1'b0;
			else
			begin
				if (OP == `SW)			memwrite_reg <= 1'b1;
				else if (OP == `SC)	memwrite_reg <= 1'b1;
				else if (OP == `SB)	memwrite_reg <= 1'b1;
				else if (OP == `SH)	memwrite_reg <= 1'b1;
				else if (OP == `SWL)	memwrite_reg <= 1'b1;
				else if (OP == `SWR)	memwrite_reg <= 1'b1;
				else  					memwrite_reg <= 1'b0;
			end
		end
	end

	/* Branch */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							branch_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							branch_reg <= 1'b1;	/* treat extension like branch */
			else
			begin
				if (OP == `BCOND)
				begin
					if (BCFUNC == `BGEZ)				branch_reg <= 1'b1;
					else if (BCFUNC == `BLTZ)		branch_reg <= 1'b1;
					else if (BCFUNC == `BGEZL)		branch_reg <= 1'b1;
					else if (BCFUNC == `BLTZL)		branch_reg <= 1'b1;
					else if (BCFUNC == `BLTZAL)	branch_reg <= 1'b1;
					else if (BCFUNC == `BGEZAL)	branch_reg <= 1'b1;
					else if (BCFUNC == `BLTZALL)	branch_reg <= 1'b1;
					else if (BCFUNC == `BGEZALL)	branch_reg <= 1'b1;
					else 									branch_reg <= 1'b0;
				end
				else if (OP == `SPEC)
				begin
					if (FUNC == `JR)					branch_reg <= 1'b1;
					else if (FUNC == `JALR)			branch_reg <= 1'b1;
					else 									branch_reg <= 1'b0;
				end
				else if (OP == `BEQ)					branch_reg <= 1'b1;
				else if (OP == `BEQL)				branch_reg <= 1'b1;
				else if (OP == `BNE)					branch_reg <= 1'b1;
				else if (OP == `BNEL)				branch_reg <= 1'b1;
				else if (OP == `BGTZ)				branch_reg <= 1'b1;
				else if (OP == `BLEZ)				branch_reg <= 1'b1;
				else if (OP == `BLEZL)				branch_reg <= 1'b1;
				else if (OP == `BGTZL)				branch_reg <= 1'b1;
				else if (OP == `J)					branch_reg <= 1'b1;
				else if (OP == `JAL)					branch_reg <= 1'b1;
				else										branch_reg <= 1'b0;
			end
		end
	end

	/* Flush Enabled Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							flush_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							flush_reg <= 1'b0;
			else
			begin
				if (OP == `BCOND)
				begin
					if (BCFUNC == `BGEZL)			flush_reg <= 1'b1;
					else if (BCFUNC == `BLTZL)		flush_reg <= 1'b1;
					else if (BCFUNC == `BLTZALL)	flush_reg <= 1'b1;
					else if (BCFUNC == `BGEZALL)	flush_reg <= 1'b1;
					else 									flush_reg <= 1'b0;
				end
				else if (OP == `BEQL)				flush_reg <= 1'b1;
				else if (OP == `BNEL)				flush_reg <= 1'b1;
				else if (OP == `BLEZL)				flush_reg <= 1'b1;
				else if (OP == `BGTZL)				flush_reg <= 1'b1;
				else										flush_reg <= 1'b0;
			end
		end
	end

	/* Flush for Extension */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			flush_ext_reg <= 1'b0;
		else
		begin
			if (EN == 1'b1)			flush_ext_reg <= 1'b0;
			else	
			begin
				if (EN_EXT == 1'b1)	flush_ext_reg <= 1'b1;
				else 						flush_ext_reg <= 1'b0;
			end
		end
	end

	/* Instruction Decode Multiplication Unit Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						mult_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)						mult_reg <=	1'b0;
			else
			begin
				if (OP == `SPEC)
				begin
					if (FUNC == `MULT)			mult_reg <= 1'b1;
					else if (FUNC == `MULTU)	mult_reg <= 1'b1;
					else if (FUNC == `MFHI)		mult_reg <= 1'b1;
					else if (FUNC == `MFLO)		mult_reg <= 1'b1;
					else if (FUNC == `MTHI)		mult_reg <= 1'b1;
					else if (FUNC == `MTLO)		mult_reg <= 1'b1;
					else if (FUNC == `DIV)		mult_reg <= 1'b1;
					else if (FUNC == `DIVU)		mult_reg <= 1'b1;
					else 								mult_reg <= 1'b0;
				end
				else									mult_reg <= 1'b0;
			end
		end
	end

	/* Conditional Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							cond_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							cond_reg <= 1'b0;
			else
			begin
				if (OP == `BCOND)
				begin
					if (BCFUNC == `BLTZ)				cond_reg <= 1'b1;
					else if (BCFUNC == `BLTZL)		cond_reg <= 1'b1;
					else if (BCFUNC == `BLTZAL)	cond_reg <= 1'b1;
					else if (BCFUNC == `BLTZALL)	cond_reg <= 1'b1;
					else if (BCFUNC == `TLTI)		cond_reg <= 1'b1;
					else if (BCFUNC == `TLTIU)		cond_reg <= 1'b1;
					else if (BCFUNC == `TNEI)		cond_reg <= 1'b1;
					else 									cond_reg <= 1'b0;
				end
				else if (OP == `SPEC)
				begin
					if (FUNC == `MOVN)				cond_reg <= 1'b1;
					else if (FUNC == `TLT)			cond_reg <= 1'b1;
					else if (FUNC == `TLTU)			cond_reg <= 1'b1;
					else if (FUNC == `TNE)			cond_reg <= 1'b1;
					else 									cond_reg <= 1'b0;
				end
				else if (OP == `BNE)					cond_reg <= 1'b1;
				else if (OP == `BNEL)				cond_reg <= 1'b1;
				else if (OP == `BGTZ)				cond_reg <= 1'b1;
				else if (OP == `BGTZL)				cond_reg <= 1'b1;
				else										cond_reg <= 1'b0;
			end
		end
	end

	/* Greaterthan/Lessthan Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							gl_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							gl_reg <= 1'b0;
			else
			begin
				if (OP == `BCOND)
				begin
					if (BCFUNC == `BGEZ)				gl_reg <= 1'b1;
					else if (BCFUNC == `BLTZ)		gl_reg <= 1'b1;
					else if (BCFUNC == `BGEZL)		gl_reg <= 1'b1;
					else if (BCFUNC == `BLTZL)		gl_reg <= 1'b1;
					else if (BCFUNC == `BLTZAL)	gl_reg <= 1'b1;
					else if (BCFUNC == `BGEZAL)	gl_reg <= 1'b1;
					else if (BCFUNC == `BLTZALL)	gl_reg <= 1'b1;
					else if (BCFUNC == `BGEZALL)	gl_reg <= 1'b1;
					else if (BCFUNC == `TGEI)		gl_reg <= 1'b1;
					else if (BCFUNC == `TGEIU)		gl_reg <= 1'b1;
					else if (BCFUNC == `TLTI)		gl_reg <= 1'b1;
					else if (BCFUNC == `TLTIU)		gl_reg <= 1'b1;
					else 									gl_reg <= 1'b0;
				end
				else if (OP == `SPEC)
				begin 
					if (FUNC == `TGE)					gl_reg <= 1'b1;
					else if (FUNC == `TGEU)			gl_reg <= 1'b1;
					else if (FUNC == `TLT)			gl_reg <= 1'b1;
					else if (FUNC == `TLTU)			gl_reg <= 1'b1;
					else									gl_reg <= 1'b0;
				end
				else if (OP == `BGTZ)				gl_reg <= 1'b1;
				else if (OP == `BLEZ)				gl_reg <= 1'b1;
				else if (OP == `BLEZL)				gl_reg <= 1'b1;
				else if (OP == `BGTZL)				gl_reg <= 1'b1;
				else										gl_reg <= 1'b0;
			end
		end
	end

	/* Lessthan/Greaterthan Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							lng_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							lng_reg <= 1'b0;
			else
			begin
				if (OP == `BCOND)
				begin
					if (BCFUNC == `BLTZ)				lng_reg <= 1'b1;
					else if (BCFUNC == `BLTZL)		lng_reg <= 1'b1;
					else if (BCFUNC == `BLTZAL)	lng_reg <= 1'b1;
					else if (BCFUNC == `BLTZALL)	lng_reg <= 1'b1;
					else if (BCFUNC == `TLTI)		lng_reg <= 1'b1;
					else if (BCFUNC == `TLTIU)		lng_reg <= 1'b1;
					else 									lng_reg <= 1'b0;
				end
				else if (OP == `SPEC)
				begin
					if (FUNC == `TLT)					lng_reg <= 1'b1;
					else if (FUNC == `TLTU)			lng_reg <= 1'b1;
					else 									lng_reg <= 1'b0;
				end
				else if (OP == `BLEZ)				lng_reg <= 1'b1;
				else if (OP == `BLEZL)				lng_reg <= 1'b1;
				else										lng_reg <= 1'b0;
			end
		end
	end

	/* Jump Operation */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						jump_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)						jump_reg <= 1'b0;
			else
			begin
				if (OP == `SPEC)
				begin
					if (FUNC == `JR)				jump_reg <= 1'b1;
					else if (FUNC == `JALR)		jump_reg <= 1'b1;
					else 								jump_reg <= 1'b0;
				end
				else if (OP == `J)				jump_reg <= 1'b1;
				else if (OP == `JAL)				jump_reg <= 1'b1;
				else									jump_reg <= 1'b0;
			end
		end
	end

	/* Jump Registered */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)					jpr_reg <= 1'b0;
		else 
		begin
			if (EN == 1'b0)					jpr_reg <= 1'b0;
			else
			begin
				if (OP == `SPEC)
				begin
					if (FUNC == `JR)			jpr_reg <= 1'b1;
					else if (FUNC == `JALR)	jpr_reg <= 1'b1;
					else							jpr_reg <= 1'b0;
				end
				else								jpr_reg <= 1'b0;
			end
		end
	end

	/* Sign Extend */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)				ext_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)				ext_reg <= 1'b0;
			else
			begin
				if (OP == `ANDI)			ext_reg <= 1'b1;
				else if (OP == `ORI)		ext_reg <= 1'b1;
				else if (OP == `XORI)	ext_reg <= 1'b1;
				else							ext_reg <= 1'b0;
			end
		end
	end

	/* Conditional Move */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)					cmov_reg <= 1'b1;
		else
		begin
			if (EN == 1'b0)					cmov_reg <= 1'b1;
			else
			begin
				if (OP == `SPEC)
				begin
					if (FUNC == `MOVN)		cmov_reg <= 1'b0;
					else if (FUNC == `MOVZ)	cmov_reg <= 1'b0;
					else 							cmov_reg <= 1'b1;
				end
				else								cmov_reg <= 1'b1;
			end
		end
	end

	/* Linked Branch */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							link_reg <= 1'b1;
		else
		begin
			if (EN == 1'b0)							link_reg <= 1'b1;
			else
			begin
				if (OP == `SPEC)
				begin 
					if (FUNC == `JALR)				link_reg <= 1'b0;
					else									link_reg <= 1'b1;
				end
				else if (OP == `BCOND)
				begin
					if (BCFUNC == `BLTZAL)			link_reg <= 1'b0;
					else if (BCFUNC == `BGEZAL)	link_reg <= 1'b0;
					else if (BCFUNC == `BLTZALL)	link_reg <= 1'b0;
					else if (BCFUNC == `BGEZALL)	link_reg <= 1'b0;
					else									link_reg <= 1'b1;
				end
				else if (OP == `JAL)					link_reg <= 1'b0;
				else										link_reg <= 1'b1;
			end
		end
	end

	/* Load Upper Immediate */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)		lui_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)		lui_reg <= 1'b0;
			else
			begin
				if (OP == `LUI)	lui_reg <= 1'b1;
				else					lui_reg <= 1'b0;
			end
		end
	end

	/* Byte Load/Store */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			bls_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)			bls_reg <= 1'b0;
			else
			begin
				if (OP == `LB)			bls_reg <= 1'b1;
				else if (OP == `LBU)	bls_reg <= 1'b1;
				else if (OP == `LWL)	bls_reg <= 1'b1;
				else if (OP == `LWR)	bls_reg <= 1'b1;
				else if (OP == `SB)	bls_reg <= 1'b1;
				else if (OP == `SWL)	bls_reg <= 1'b1;
				else if (OP == `SWR)	bls_reg <= 1'b1;
				else 						bls_reg <= 1'b0;
			end
		end
	end

	/* Halfword Load/Store */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)				hls_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)				hls_reg <= 1'b0;
			else
			begin
				if (OP == `LH)				hls_reg <= 1'b1;
				else if (OP == `LHU)		hls_reg <= 1'b1;
				else if (OP == `LWL)		hls_reg <= 1'b1;
				else if (OP == `LWR)		hls_reg <= 1'b1;
				else if (OP == `SH)		hls_reg <= 1'b1;
				else if (OP == `SWL)		hls_reg <= 1'b1;
				else if (OP == `SWR)		hls_reg <= 1'b1;
				else							hls_reg <= 1'b0;
			end
		end
	end

	/* Sign Extend Memory Data */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			mext_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)			mext_reg <= 1'b0;
			else
			begin
				if (OP == `LBU)		mext_reg <= 1'b1;
				else if (OP == `LHU)	mext_reg <= 1'b1;
				else						mext_reg <= 1'b0;
			end
		end
	end

	/* Right/Left Unaligned Load/Store */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)				rnl_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)				rnl_reg <= 1'b0;
			else 
			begin
				if (OP == `LWR)			rnl_reg <= 1'b1;
				else if (OP == `SWR)		rnl_reg <= 1'b1;
				else 							rnl_reg <= 1'b0;
			end
		end
	end

	/* Exception */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)											exc_reg <= 1'b0;
		else
		begin
			if (RI_PA == 1'b1)										exc_reg <= 1'b1;
			else
			begin
				if (EN == 1'b0)										exc_reg <= 1'b0;
				else
				begin
					if (OP == `SPEC)
					begin 
						if (FUNC == `SYSCALL)						exc_reg <= 1'b1;
						else if (FUNC == `BREAK)					exc_reg <= 1'b1;
						else												exc_reg <= 1'b0;
					end
					else if ((~CP[0] & CP0 & KU) == 1'b1)		exc_reg <= 1'b1;
					else if ((~CP[1] & CP1 & KU) == 1'b1)		exc_reg <= 1'b1;
					else if ((~CP[2] & CP2 & KU) == 1'b1)		exc_reg <= 1'b1;
					else if ((~CP[3] & CP3 & KU) == 1'b1)		exc_reg <= 1'b1;
					else 													exc_reg <= 1'b0;
				end
			end
		end
	end

	/* Compare with Zero */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							comprz_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							comprz_reg <= 1'b0;
			else
			begin
				if (OP == `BCOND)
				begin
					if (BCFUNC == `BLTZ)				comprz_reg <= 1'b1;		
					else if (BCFUNC == `BGEZ)		comprz_reg <= 1'b1;
					else if (BCFUNC == `BLTZL)		comprz_reg <= 1'b1;
					else if (BCFUNC == `BGEZL)		comprz_reg <= 1'b1;
					else if (BCFUNC == `BLTZAL)	comprz_reg <= 1'b1;
					else if (BCFUNC == `BGEZAL)	comprz_reg <= 1'b1;
					else if (BCFUNC == `BLTZALL)	comprz_reg <= 1'b1;
					else if (BCFUNC == `BGEZALL)	comprz_reg <= 1'b1;
					else									comprz_reg <= 1'b0;
				end
				else if (OP == `BLEZ)				comprz_reg <= 1'b1;
				else if (OP == `BGTZ)				comprz_reg <= 1'b1;
				else if (OP == `BLEZL)				comprz_reg <= 1'b1;
				else if (OP == `BGTZL)				comprz_reg <= 1'b1;
				else										comprz_reg <= 1'b0;
			end
		end
	end

	/* Greaterthan/Lessthan Unsigned */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)						glus_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)						glus_reg <= 1'b0;
			else
			begin
				if (OP == `SPEC)
				begin
					if (FUNC == `TGEU)			glus_reg <= 1'b1;
					else if (FUNC == `TLTU)		glus_reg <= 1'b1;
					else 								glus_reg <= 1'b0;
				end
				else if (OP == `BCOND)
				begin
					if (BCFUNC == `TGEIU)		glus_reg <= 1'b1;
					else if (BCFUNC == `TLTIU)	glus_reg <= 1'b1;
					else 								glus_reg <= 1'b0;
				end
				else									glus_reg <= 1'b0;
			end
		end
	end

	/* Conditional Exception */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	cexc_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)	cexc_reg <= 1'b0;
			else
			begin
				if (OP == `SPEC)
				begin
					if (FUNC == `TGE)				cexc_reg <= 1'b1;
					else if (FUNC == `TGEU)		cexc_reg <= 1'b1;
					else if (FUNC == `TLT)		cexc_reg <= 1'b1;
					else if (FUNC == `TLTU)		cexc_reg <= 1'b1;
					else if (FUNC == `TEQ)		cexc_reg <= 1'b1;
					else if (FUNC == `TNE)		cexc_reg <= 1'b1;
					else 								cexc_reg <= 1'b0;
				end
				else if (OP == `BCOND)
				begin
					if (BCFUNC == `TGEI)			cexc_reg <= 1'b1;
					else if (BCFUNC == `TGEIU)	cexc_reg <= 1'b1;
					else if (BCFUNC == `TLTI)	cexc_reg <= 1'b1;
					else if (BCFUNC == `TLTIU)	cexc_reg <= 1'b1;
					else if (BCFUNC == `TEQI)	cexc_reg <= 1'b1;
					else if (BCFUNC == `TNEI)	cexc_reg <= 1'b1;
					else								cexc_reg <= 1'b0;
				end
				else									cexc_reg <= 1'b0;
			end
		end
	end

	/* Load/Store Conditional */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			lc_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)			lc_reg <= 1'b0;
			else
			begin
				if (OP == `LL)			lc_reg <= 1'b1;
				else if (OP == `SC)	lc_reg <= 1'b1;
				else						lc_reg <= 1'b0;
			end
		end
	end

	/* Return From Exception */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)							rfe_reg <= 1'b0;
		else
		begin
			if (EN == 1'b0)							rfe_reg <= 1'b0;
			else
			begin
				if (OP == `COP0) 
				begin 
					if (CPFUNC == `C0)
					begin
						if (FUNC == {1'b0,`RFE})	rfe_reg <= 1'b1;
						else								rfe_reg <= 1'b0;
					end
					else									rfe_reg <= 1'b0;
				end
				else										rfe_reg <= 1'b0;
			end
		end
	end

endmodule
