/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	execute															*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"	/* Constant Definitiions */

module execute(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [5:0]		ALUOP,			/* ALU Opcode */
	input 			ALUSRC,			/* ALU Operand B Switch */
	input 			CLK, 				/* System Clock 50 - 100 MHZ */
	input 			CMOV, 			/* Conditional Move */
	input 			COND, 			/* Conditional Operation */
	input 			CP0,				/* Coprocessor 0 Operation */
	input [31:0]	CP0RGDATA,		/* Data from CP0 Register Read Port 1 */
	input 			CPREGWRITE,		/* Coprocessor Register Write Enable */
	input [1:0]		FORWARD_A,		/* Execute Data Forward Switch Operand A */
	input [1:0]		FORWARD_B,		/* Execute Data Forward Switch Operand B */
	input [31:0]	IMM,				/* Data Immediate */
	input 			LINK,				/* Linked Branch */
	input 			LUI,				/* Load Upper Immediate */
	input [31:0]	PC4,				/* Currect PC plus 4*/
	input 			PCLK, 			/* Pipeline Clock */
	input [4:0] 	RD,				/* Destination Register */
	input [31:0] 	RDDATA_MA,		/* Data forwarded from Memory Access Phase */
	input [31:0] 	RDDATA_WB,		/* Data forwarded from Writeback Phase */
	input [31:0] 	REG1DATA,		/* Data from Register Read Port 1 */
	input [31:0] 	REG2DATA,		/* Data from Register Read Port 2 */
	input 			REGDST, 			/* Destination Register Switch */
	input 			REGWRITE, 		/* Register Write */
	input 			RESET, 			/* System Reset */
	input [4:0]		RT,				/* TISA Operand Register Number 2 */
	/* OUTPUT PORTS */
	output [31:0]	ALU_B,			/* ALU Operand B */
	output [31:0]	ALURESULT,		/* ALU Result */
	output 			CP0REGWRITE,	/* Coprocessor 0 Register Write Enable */
	output 			CREGWRITE,		/* Conditional Register Write */
	output 			EXC,				/* Exception */
	output [6:0] 	EXCCODE,			/* Exception Code */
	output 			MULT_RDY,		/* Multiplication Unit Ready */
	output [4:0] 	WRREG,			/* TISA Destination Register Number */
	output 			ZERO,				/* ALU Result is Zero */
	/* DEBUG PORTS */
	output [31:0] 	HI,				/* Multiplication Unit Hi Register */
	output [31:0] 	LO					/* Multiplication Unit Lo Register */
	);
	
/*****Signals****************************************************************/
	
	wire [31:0]	ALU_A;			/* ALU Operand A */
	wire [31:0]	ALU_A_0;			/* Conditional Move ALU Operand A */
	wire [31:0]	ALU_B_0;			/* Immediate ALU Operand B */
	wire [31:0]	ALU_B_1;			/* ALU Operand B Asynchronous */
	wire [31:0]	ALURESULT_0;	/* ALU Result */
	wire [31:0]	ALURESULT_1;	/* Selected Result */
	wire [31:0]	ALURESULT_2;	/* Selected Result or Upper Immediate */
	wire [1:0]	ALUSRC_A;		/* ALU Operand A Switch */
	wire [1:0]	ALUSRC_B;		/* ALU Operand B Switch */
	wire 			CP0REGWRITE_0;	/* CP0 Regwrite Asynchronous */
	wire 			CREGWRITE_0;	/* Conditional Regwrite Asynchronous */
	wire 			DEC;				/* Conditional Test */
	wire [1:0]	DSEL;				/* Destination Register Selection Switch */
	wire [31:0]	IMMUI;			/* Upper Immediate */
	wire [1:0]	MSEL;				/* Result Selection Switch */
	wire 			MULT_RDY_0;		/* Multipication Unit Ready Asynchronous */
	wire 			OVF;				/* Integer Overflow 1 */
	wire 			OVF2;				/* Integer Overflow 2 */
	wire 			RW_EN;			/* Register Write Enable */
	wire [4:0]	WRREG_0;			/* Write Register Number Asynchronous */
	wire 			ZERO_0;			/* ALU Result is Zero Asynchronous */

/*****Registers****************************************************************/
	
	reg [31:0]	alu_a_reg;			/* ALU Operand A */
	reg [31:0]	alu_a_0_reg;		/* Conditional Move ALU Operand A */
	reg [31:0]	alu_b_reg;			/* ALU Operand B */
	reg [31:0]	alu_b_0_reg;		/* Immediate ALU Operand B */
	reg [5:0]	aluop_reg;			/* ALU Opcode */
	reg [31:0]	aluresult_reg;		/* ALU Result */
	reg [31:0]	aluresult_0_reg;	/* ALU Result */
	reg [31:0]	aluresult_1_reg;	/* Selected Result */
	reg 			alusrc_reg;			/* ALU Operand B Switch */
	reg 			cmov_reg;			/* Conditional Move */
	reg 			cond_reg;			/* Conditional Operation */
	reg 			cp0_reg;				/* Coprocessor 0 Operation */
	reg 			cp0regwrite_reg;	/* Coprocessor 0 Register Write Enable */
	reg 			cpregwrite_reg;	/* Coprocessor Register Write Enable */
	reg 			cregwrite_reg;		/* Conditional Register Write */
	reg [31:0]	cp0rgdata_reg;		/* Data from CP0 Register Read Port 1 */
	reg [1:0]	forward_a_reg;		/* Execute Data Forward Switch Operand A */
	reg [1:0]	forward_b_reg;		/* Execute Data Forward Switch Operand B */
	reg [31:0]	imm_reg;				/* Data Immediate */
	reg [31:0]	immui_reg;			/* Upper Immediate */
	reg 			link_reg;			/* Linked Branch */
	reg			lui_reg;				/* Load Upper Immediate */
	reg			mult_rdy_reg;		/* Multiplication Unit Ready */
	reg [31:0]	pc4_reg;				/* Currect PC plus 4*/
	reg [4:0]	rd_reg;				/* Destination Register */
	reg [31:0]	rddata_ma_reg;		/* Data forwarded from Memory Access Phase */
	reg [31:0]	rddata_wb_reg;		/* Data forwarded from Writeback Phase */
	reg [31:0]	reg1data_reg;		/* Data from Register Read Port 1 */
	reg [31:0]	reg2data_reg;		/* Data from Register Read Port 2 */
	reg 			regdst_reg;			/* Destination Register Switch */
	reg 			regwrite_reg;		/* Register Write */
	reg [4:0] 	rt_reg;				/* TISA Operand Register Number 2 */
	reg [4:0] 	wrreg_reg;			/* TISA Destination Register Number */
	reg 			zero_reg;			/* ALU Result is Zero */

/*****Initialization****************************************************************/

	initial
	begin
		alu_a_reg = 32'b0;
		alu_a_0_reg = 32'b0;
		alu_b_reg = 32'b0;
		alu_b_0_reg = 32'b0;
		aluresult_reg = 32'b0;
		aluresult_0_reg = 32'b0;
		aluresult_1_reg = 32'b0;
		aluop_reg = 6'b0;
		alusrc_reg = 1'b0;
		cmov_reg = 1'b1;
		cp0_reg = 1'b0;
		cp0rgdata_reg = 32'b0;
		cp0regwrite_reg = 1'b0;
		cpregwrite_reg = 1'b0;
		cond_reg = 1'b0;
		cregwrite_reg = 1'b0;
		forward_a_reg = 2'b0;
		forward_b_reg = 2'b0;
		imm_reg = 32'b0;
		immui_reg = 32'b0;
		link_reg = 1'b1;
		lui_reg = 1'b0;
		mult_rdy_reg = 1'b0;
		pc4_reg = 32'b0;
		rd_reg = 5'b0;
		rddata_ma_reg = 32'b0;
		rddata_wb_reg = 32'b0;
		reg1data_reg = 32'b0;
		reg2data_reg = 32'b0;
		regdst_reg = 1'b0;
		regwrite_reg = 1'b0;
		rt_reg = 5'b0;
		wrreg_reg = 5'b0;
		zero_reg = 1'b0;
	end

/*********************************************************************/
	
	assign ALU_B			=	alu_b_reg;
	assign ALURESULT		=	aluresult_reg;
	assign CP0REGWRITE	=	cp0regwrite_reg;
	assign CREGWRITE		=	cregwrite_reg;
	assign MULT_RDY		=	mult_rdy_reg;
	assign WRREG			=	wrreg_reg;
	assign ZERO				=	zero_reg;

/*****Exception Encoding****************************************************************/

	excpencoder ex0(
		.CLK(CLK),
		.EXC(EXC),
		.EXCCODE(EXCCODE),
		.OVF2(OVF2),
		.RESET(RESET)
		);

/*****Operand Selection****************************************************************/

	/* Forward A or Link */
	or2_2	or2(
		.A({~link_reg,~link_reg}),
		.B(forward_a_reg),
		.C(ALUSRC_A)
		);

	/* Forward B or Link */
	or2_2 or3(
		.A({~link_reg,~link_reg}),
		.B(forward_b_reg),
		.C(ALUSRC_B)
		);

	/* Forward A or Link */
	MUX32_4to1	mux0(
		.a0(reg1data_reg),
		.a1(rddata_wb_reg),
		.a2(rddata_ma_reg),	
		.a3(pc4_reg),
		.sel(ALUSRC_A),
		.out(ALU_A)
		);

	/* Forward B or Link */
	MUX32_4to1	mux1(
		.a0(reg2data_reg),
		.a1(rddata_wb_reg),
		.a2(rddata_ma_reg),	
		.a3(32'h4),
		.sel(ALUSRC_B),
		.out(ALU_B_1)
		);

	/* Conditional Move A */
	MUX32_2to1 mux3(
		.a0(32'b0),
		.a1(alu_a_reg),
		.sel(cmov_reg),	
		.out(ALU_A_0)
		);

	/* Immediate Operation B */
	MUX32_2to1	mux4(
		.a0(alu_b_reg),
		.a1(imm_reg),
		.sel(alusrc_reg),	
		.out(ALU_B_0)
		);

/*****Arithmetic Logic Unit Full****************************************************************/

	alu_full alu0(
		.A(alu_a_0_reg),
		.B(alu_b_0_reg),
		.CLK(CLK),
		.CTR(aluop_reg),
		.OVERFLOW(OVF),
		.OVERFLOW2(OVF2),
		.MULT_RDY(MULT_RDY_0),
		.PCLK(PCLK),
		.RESET(RESET),
		.RESULT(ALURESULT_0),
		.SHAMT(imm_reg[10:6]),
		.ZERO(ZERO_0),
		//Debug Ports
		.HI(HI),
		.LO(LO)
		);

/*****Output Selection****************************************************************/

	/* Result Encoder */
	resultdecoder rdec(
		.CMOV(cmov_reg),
		.CP0(cp0_reg),
		.CPREGWRITE(cpregwrite_reg),
		.MSEL(MSEL)
		);

	/* Shift immediate to Upper Halfword */
	shift16 sh0(
		.A(imm_reg),
		.B(IMMUI),
		.C_IN(16'b0)
		);

	/* Result Selection */
	MUX32_4to1	mux2(
		.a0(aluresult_0_reg),
		.a1(alu_a_reg),
		.a2(alu_b_reg),		
		.a3(cp0rgdata_reg),
		.sel(MSEL),
		.out(ALURESULT_1)
		);

	/* Load Upper Immediate */
	MUX32_2to1	mux7(
		.a0(aluresult_1_reg),
		.a1(immui_reg),
		.sel(lui_reg),		
		.out(ALURESULT_2)
		);

/*****Destination Register Selection****************************************************************/

	/* Destination Encoder */
	destdecoder ddec(
		.DSEL(DSEL),
		.LINK(link_reg),
		.REGDST(regdst_reg)
		);

	/* Destination Selection */
	MUX5_4to1 mux5(
		.a0(rt_reg),
		.a1(rd_reg),
		.a2(5'b11111),
		.a3(5'b0),
		.sel(DSEL),
		.out(WRREG_0)
		);

/*****Conditional Test****************************************************************/

	/* Zero or Not Zero */
	MUX1_2to1 mux6(
		.a0(ZERO),
		.a1(~ZERO),
		.sel(cond_reg),
		.out(DEC)
		);

	/* Conditional Move Test or Regwrite*/
	or2_1	or0(
		.A(DEC),
		.B(cmov_reg),
		.C(RW_EN)
		);

	/* Regwrite And Passed Conditional Regwrite */
	and2_1	and0(
		.A(RW_EN),
		.B(regwrite_reg),
		.C(CREGWRITE_0)
		);

/*****CP0 Operation****************************************************************/

	/* CP0 Regwrite */
	and2_1	and1(
		.A(cp0_reg),
		.B(cpregwrite_reg),
		.C(CP0REGWRITE_0)
		);

/*********************************************************************/
	
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			alu_a_reg <= 32'b0;
			alu_a_0_reg <= 32'b0;
			alu_b_reg <= 32'b0;
			alu_b_0_reg <= 32'b0;
			aluresult_reg <= 32'b0;
			aluresult_0_reg <= 32'b0;
			aluresult_1_reg <= 32'b0;
			aluop_reg <= 6'b0;
			alusrc_reg <= 1'b0;
			cmov_reg <= 1'b1;
			cond_reg <= 1'b0;
			cp0_reg <= 1'b0;
			cp0regwrite_reg <= 1'b0;
			cp0rgdata_reg <= 32'b0;
			cpregwrite_reg <= 1'b0;
			cregwrite_reg <= 1'b0;
			forward_a_reg <= 2'b0;
			forward_b_reg <= 2'b0;
			imm_reg <= 32'b0;
			immui_reg <= 32'b0;
			link_reg <= 1'b1;
			lui_reg <= 1'b0;
			mult_rdy_reg <= 1'b0;
			pc4_reg <= 32'b0;
			rd_reg <= 5'b0;
			rddata_ma_reg <= 32'b0;
			rddata_wb_reg <= 32'b0;
			reg1data_reg <= 32'b0;
			reg2data_reg <= 32'b0;
			regdst_reg <= 1'b0;
			regwrite_reg <= 1'b0;
			rt_reg <= 5'b0;
			wrreg_reg <= 5'b0;
			zero_reg <= 1'b0;
		end
		else
		begin
			/* Latch Signals */
			alu_a_reg <= ALU_A;
			alu_a_0_reg <= ALU_A_0;
			alu_b_reg <= ALU_B_1;
			alu_b_0_reg <= ALU_B_0;
			aluresult_reg <= ALURESULT_2;
			aluresult_0_reg <= ALURESULT_0;
			aluresult_1_reg <= ALURESULT_1;
			aluop_reg <= ALUOP;
			alusrc_reg <= ALUSRC;
			cond_reg <= COND;
			cmov_reg <= CMOV;
			cp0_reg <= CP0;
			cp0regwrite_reg <= CP0REGWRITE_0;
			cp0rgdata_reg <= CP0RGDATA;
			cpregwrite_reg <= CPREGWRITE;
			cregwrite_reg <= CREGWRITE_0;
			forward_a_reg <= FORWARD_A;
			forward_b_reg <= FORWARD_B;
			imm_reg <= IMM;
			immui_reg <= IMMUI;
			link_reg <= LINK;
			lui_reg <= LUI;
			mult_rdy_reg <= MULT_RDY_0;
			pc4_reg <= PC4;
			rd_reg <= RD;
			rddata_ma_reg <= RDDATA_MA;
			rddata_wb_reg <= RDDATA_WB;
			reg1data_reg <= REG1DATA;
			reg2data_reg <= REG2DATA;
			regdst_reg <= REGDST;
			regwrite_reg <= REGWRITE;
			rt_reg <= RT;
			wrreg_reg <= WRREG_0;
			zero_reg <= ZERO_0;
		end
	end
endmodule

/*****Destination Register Encoder****************************************************************/

module destdecoder(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				REGDST,	/* Destination Register Switch */
	input				LINK,		/* Linked Branch */
	/* OUTPUT PORTS */
	output [1:0]	DSEL		/* Destination Register Selection Switch */
	);

/*********************************************************************/
	
	assign DSEL	=	((REGDST == 1'b0) && (LINK == 1'b0))?	2'b10:
						(REGDST == 1'b1)?								2'b01:
																			2'b00;
																			
endmodule

/*****Output Encoder****************************************************************/

module resultdecoder(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CMOV,			/* Conditional Move */
	input 			CP0,			/* Coprocessor 0 Operation */
	input 			CPREGWRITE,	/* Coprocessor Register Write Enable */
	/* OUTPUT PORTS */
	output [1:0]	MSEL			/* Result Selection Switch */
	);

/*********************************************************************/
	
	assign MSEL		=	(CMOV == 1'b0)?									2'b01:
							((CP0 == 1'b1) && (CPREGWRITE == 1'b1))?	2'b10:
							((CP0 == 1'b1) && (CPREGWRITE == 1'b0))?	2'b11:
																					2'b00;
																					
endmodule

/*****Exception Encoding****************************************************************/

module excpencoder(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input				RESET,		/* System Reset */
	input				OVF2,			/* Integer Overflow 2 */
	/* OUTPUT PORTS */
	output			EXC,			/* Exception */
	output [6:0]	EXCCODE		/* Exception Code */
	);
	
/*****Registers****************************************************************/

	reg			exc_reg;			/* Exception */
	reg [6:0]	exccode_reg;	/* Exception Code */

/*****Initialization****************************************************************/
	
	initial
	begin
		exc_reg = 1'b0;
		exccode_reg = 7'b0;
	end
	
/*********************************************************************/

	assign	EXC		=	exc_reg;
	assign	EXCCODE	=	exccode_reg;

/*********************************************************************/

	/* Exception */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	exc_reg <= 1'b0;
		else
		begin
			if (OVF2 == 1'b1)	exc_reg <= 1'b0;//1'b1;
			else 					exc_reg <= 1'b0;
		end
	end

	/* Exception Code */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	exccode_reg <= 7'b0;
		else
		begin
			if (OVF2 == 1'b1)	exccode_reg <= `Ovfc;
			else 					exccode_reg <= 7'b0;
		end
	end
	
endmodule
