/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	extension_controller											*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	Extension Peripherals, Bug Fixes, see Manual		*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module extension_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input			CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */ 
	input			OE,			/* Output Enable */
	input			PER0_DNE,		/* */
	input			PER0_PRESENT,	/* */
	input			PER0_PRIVILEGE,	/* */
	input			PER0_WANTS_INTR,	/* */
	input			PER1_DNE,		/* */
	input			PER1_PRESENT,	/* */
	input			PER1_PRIVILEGE,	/* */
	input			PER1_WANTS_INTR,	/* */
	input			RESET,		/* System Reset */
	input			SRT,			/* Start Transaction */
	input			WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output [31:0]	DATA_OUT,		/* Data Bus to Data Path Memory Interface */
	output		DNE,			/* Transaction Done */
	output 		IRQ,			/* */
	output		PER0_BAT_EN,	/* */
	output		PER1_BAT_EN,	/* */
	output [31:0]	STATUS		/* Status Register */
	);

/*****Signals****************************************************************/

	wire [31:0]	BASE_ADDR;		/* */
	wire [3:0]	BM;			/* Byte Selects */
	wire [31:0]	DATA_IN_C;		/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;		/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;		/* Endian Flipped Outgoing Data */
	wire [31:0]	MAX_ADDR;		/* */
	wire		MEN;			/* Transaction Enabled for Extension Controller Configuration Space */
	wire [31:0]	REG_M;		/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;		/* Selected Configuration Register Data */
	wire [31:0]	WMASK_IN;		/* Configuration Register Data Write Mask */
	wire [31:0]	WMASK_REG;		/* Incoming Data Write Mask */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;		/* BASETAG Register */
	reg 		batto;		/* */
	reg [31:0]	control;		/* */
	reg [31:0]	data_in_m_reg;	/* Masked Incoming Data */
	reg [3:0]	dnecnt;		/* Transaction Progress Counter */
	reg		dner;			/* Configuration Space Transaction Done */
	reg		en_reg;		/* */
	reg		irq_reg;		/* */
	reg		men_reg;		/* Transaction Enabled for PWRCTR Configuration Space */
	reg		per0_present_reg;	/* */
	reg		per1_present_reg;	/* */
	reg [31:0]	reg_m_0_reg;	/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;		/* Masked Configuration Register Data */
	reg [1:0]	state;		/* */
	reg [7:0]	timeout;		/* */

/*****Variables****************************************************************/

	integer i;

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter BASETAGREG =		16'h0000;	/* BASETAG Register Relative Address */
	parameter CONTROLREG = 		16'h0004;	/* */

	/* BASETAG Register */
	parameter	BASEl =		31;			/* BASE Bit Range Left */
	parameter	BASEr =		16;			/* BASE Bit Range Right */
	parameter	TAGl = 		15;			/* TAG Bit Range Left */
	parameter	TAGr = 		0;			/* TAG Bit Range Right */
	parameter	TAG =			17;			/* TAG Value */

	/* */
	parameter	ECC_SLOTNOl = 	25;
	parameter	ECC_SLOTNOr =	24;
	parameter	ECC_PER_PRIV =	23;
	parameter	ECC_PER_INTR =	22;
	parameter	ECC_INT_UNLOAD =	21;
	parameter	ECC_INT_LOAD =	20;
	parameter	ECC_IRQ_EN =	18;
	parameter	ECC_IRQ =		17;
	parameter	ECC_RESET =		16;
	parameter	ECC_SIZEl =		15;
	parameter	ECC_SIZEr = 	0;
	parameter	DEFAULTSIZE =	16'h0020;
	parameter	SLOT0 = 		2'b00;
	parameter	SLOT1 =		2'b01;
	parameter	SLOT2 =		2'b10;
	parameter	SLOT3 =		2'b11;

/*****Initialization****************************************************************/

	initial 
	begin
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		batto = 1'b0;
		control = 32'b0;
		control[ECC_SIZEl:ECC_SIZEr] = DEFAULTSIZE;
		dnecnt = 4'b1111;
		dner = 1'b1;
		en_reg = 1'b0;
		irq_reg = 1'b0;
		men_reg = 1'b0;
		per0_present_reg = 1'b0;
		per1_present_reg = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
		state = 2'b0;
		timeout = 10'b0;
	end

/*********************************************************************/

	assign BASETAG = basetagr;
	assign DNE = dner;
	assign IRQ = control[ECC_IRQ_EN] & control[ECC_IRQ];

/*****Address Decoding****************************************************************/

	assign BASE_ADDR = {CADDR[BASEl:BASEr],16'h0008};
	assign MAX_ADDR = {CADDR[BASEl:BASEr],(BASETAGREG + DEFAULTSIZE)};

	assign MEN = ((CADDR[BASEl:BASEr] == ADDR_IN[BASEl:BASEr]) & (ADDR_IN[BASEr-1:0] < BASE_ADDR[BASEr-1:0])) | batto;
	assign EN = ((ADDR_IN >= BASE_ADDR) & (ADDR_IN <= MAX_ADDR)) & ~batto;

	assign PER0_BAT_EN = en_reg & (control[ECC_SLOTNOl:ECC_SLOTNOr] == SLOT0);
	assign PER1_BAT_EN = en_reg & (control[ECC_SLOTNOl:ECC_SLOTNOr] == SLOT1);

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/

	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == CONTROLREG)?control:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BASETAGREG)?basetagr:32'bz;
	assign REG_M_0 = (batto)?32'b0:32'bz;
	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			en_reg = 1'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			en_reg = EN;
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,1'b0,dnecnt[3],dner})
			4'b0xxx	:	begin
							/* Reset */
							dner = 1'b1;
							basetagr = 32'b0;
							basetagr[TAGl:TAGr] = TAG;
							batto = 1'b0;
							control = 32'b0;
							control[ECC_SIZEl:ECC_SIZEr] = DEFAULTSIZE;
							irq_reg = 1'b0;
							per0_present_reg = 1'b0;
							per1_present_reg = 1'b0;
							state = 2'b0;
							timeout = 10'b0;
						end
			4'b11xx	:	begin
							/* Soft Reset */
							dner = 1'b1;
							basetagr = 32'b0;
							basetagr[TAGl:TAGr] = TAG;
							batto = 1'b0;
							control = 32'b0;
							control[ECC_SIZEl:ECC_SIZEr] = DEFAULTSIZE;
							irq_reg = 1'b0;
							per0_present_reg = 1'b0;
							per1_present_reg = 1'b0;
							state = 2'b0;
							timeout = 10'b0;
						end
			4'b1000	:	begin
							/* Update Configuration Registers */
							if ((men_reg | en_reg) & OE)
							begin
								casex(ADDR_IN[BASEr-1:0])
									BASETAGREG	:	begin
													dner = 1'b1;
													//$display("READ BASETAG %x %x %x", basetagr, reg_m_0_reg, DATA_OUT);
													//$stop;
												end
									CONTROLREG	:	begin
													dner = 1'b1;
													//$display("READ CONTROL %x %x %x", control, reg_m_0_reg, DATA_OUT);
													//$stop;
												end
									default	:	begin
													casex(state)
														2'b00		:	begin
																		timeout = timeout + 1;
																		if (~PER0_DNE)
																		begin
																			state = 2'b01;
																			//$display("PER0 Responded");
																			//$stop;
																		end
																		if (~PER1_DNE)
																		begin
																			state = 2'b11;
																			//$display("PER1 Responded");
																			//$stop;
																		end
																		if (batto)
																		begin
																			dner = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																		if (timeout == 0)
																		begin
																			batto = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																	end
														2'b01		:	begin
																		timeout = timeout + 1;
																		if (PER0_DNE)
																		begin
																			dner = 1'b1;
																			//$display("PER0 Done %x", DATA_OUT);
																			//$stop;
																		end
																		if (batto)
																		begin
																			dner = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																		if (timeout == 0)
																		begin
																			batto = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																	end
														2'b11		:	begin
																		timeout = timeout + 1;
																		if (PER1_DNE)
																		begin
																			dner = 1'b1;
																			//$display("PER1 Done %x", DATA_OUT);
																			//$stop;
																		end
																		if (batto)
																		begin
																			dner = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																		if (timeout == 0)
																		begin
																			batto = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																	end
													endcase
												end
								endcase
							end
							if ((men_reg | en_reg) & WE)
							begin
								casex(ADDR_IN[BASEr-1:0])
									BASETAGREG	:	begin
													/* Update Basetag Register */
													basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
													dner = 1'b1;
												end
									CONTROLREG	:	begin
													control = {(data_in_m_reg[31:24] | reg_m_reg[31:24]),control[23:20],(data_in_m_reg[19:16] | reg_m_reg[19:16]),control[15:0]};
													if (control[ECC_IRQ] & WMASK_IN[ECC_IRQ] & irq_reg) irq_reg = 1'b0;
													dner = 1'b1;
												end
									default	:	begin
													casex(state)
														2'b00		:	begin
																		timeout = timeout + 1;
																		if (~PER0_DNE)
																		begin
																			state = 2'b01;
																			//$display("PER0 Responded");
																			//$stop;
																		end
																		if (~PER1_DNE)
																		begin
																			state = 2'b11;
																			//$display("PER1 Responded");
																			//$stop;
																		end
																		if (batto)
																		begin
																			dner = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																		if (timeout == 0)
																		begin
																			batto = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																	end
														2'b01		:	begin
																		timeout = timeout + 1;
																		if (PER0_DNE)
																		begin
																			dner = 1'b1;
																			//$display("PER0 Done %x", DATA_OUT);
																			//$stop;
																		end
																		if (batto)
																		begin
																			dner = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																		if (timeout == 0)
																		begin
																			batto = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																	end
														2'b11		:	begin
																		timeout = timeout + 1;
																		if (PER1_DNE)
																		begin
																			dner = 1'b1;
																			//$display("PER1 Done %x", DATA_OUT);
																			//$stop;
																		end
																		if (batto)
																		begin
																			dner = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																		if (timeout == 0)
																		begin
																			batto = 1'b1;
																			//$display("BAT TIMEOUT %x %x %x", REG_M_0, reg_m_0_reg, DATA_OUT);
																			//$stop;
																		end
																	end
													endcase
												end
								endcase
							end
						end
			4'b10x1	:	begin
							batto = 1'b0;
							state = 0;
							timeout = 10'b0;
							/* Wait for incoming request */
							if (SRT & (men_reg | en_reg))
							begin
								/* Request Recognized */
								dner = 1'b0;
							end
							if (~irq_reg & control[ECC_IRQ])
							begin
								control[ECC_INT_LOAD] = 1'b0;
								control[ECC_INT_UNLOAD] = 1'b0;
								control[ECC_IRQ] = 1'b0;
							end
							if (irq_reg & ~control[ECC_IRQ])
							begin
								control[ECC_IRQ] = 1'b1;
							end
							if (control[ECC_SLOTNOl:ECC_SLOTNOr] == SLOT0)
							begin
								control[ECC_PER_INTR] = PER0_WANTS_INTR;
								control[ECC_PER_PRIV] = PER0_PRIVILEGE;
								if (PER0_PRESENT & ~per0_present_reg)
								begin
									control[ECC_INT_LOAD] = 1'b1;
								end
								if (~PER0_PRESENT & per0_present_reg)
								begin
									control[ECC_INT_UNLOAD] = 1'b1;
								end
							end
							else if (control[ECC_SLOTNOl:ECC_SLOTNOr] == SLOT1)
							begin
								control[ECC_PER_INTR] = PER1_WANTS_INTR;
								control[ECC_PER_PRIV] = PER1_PRIVILEGE;
								if (PER1_PRESENT & ~per1_present_reg)
								begin
									control[ECC_INT_LOAD] = 1'b1;
								end
								if (~PER1_PRESENT & per1_present_reg)
								begin
									control[ECC_INT_UNLOAD] = 1'b1;
								end
							end
							else
							begin
								control[ECC_PER_INTR] = 1'b0;
								control[ECC_PER_PRIV] = 1'b0;
								control[ECC_INT_LOAD] = 1'b0;
								control[ECC_INT_UNLOAD] = 1'b0;
							end
							if (PER0_PRESENT & ~per0_present_reg)
							begin
								per0_present_reg = 1'b1;
								control[ECC_IRQ] = 1'b1;
								irq_reg = 1'b1;
							end
							if (~PER0_PRESENT & per0_present_reg)
							begin
								per0_present_reg = 1'b0;
								control[ECC_IRQ] = 1'b1;
								irq_reg = 1'b1;
							end
							if (PER1_PRESENT &  ~per1_present_reg)
							begin
								per1_present_reg = 1'b1;
								control[ECC_IRQ] = 1'b1;
								irq_reg = 1'b1;
							end
							if (~PER1_PRESENT & per1_present_reg)
							begin
								per1_present_reg = 1'b0;
								control[ECC_IRQ] = 1'b1;
								irq_reg = 1'b1;
							end
						end
			default	:	begin
						end
		endcase
	end

endmodule
