/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	flash_bridge													*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/* This module interfaces to a Micron Q-Flash Memory						*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module flash_bridge(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [25:0]	ADDR_IN,		/* Address bus from Flash Controller */
	input [2:0]		BUS,			/* Bus Width */
	input [31:0]	DATA_IN,		/* Data bus from Flash Controller */
	input [3:0]		DELAY,		/* Flash Transaction Delay */
	input [31:0]	DQ_OUT,		/* Data bus from Flash IC */
	input				FLASHCLK,	/* Flash Clock 50 - 100 MHZ */
	input				OE,			/* Output Enable from Flash Controller */
	input				RESET,		/* System Reset */
	input				RSTPIN,		/* Hardware Reset Control of Flash IC */
	input				SRT,			/* Start Transaction */
	input				WE,			/* Write Enable from Flash Controller */
	/* OUTPUT PORTS */
	output [23:0]	ADDR_OUT,	/* Address bus to Flash IC */
	output			BYTE,			/* Byte Mode */
	output			BYTESEL,		/* Byte Select */
	output			CE0,			/* Chipselect 0 */
	output			CE2,			/* Chipselect 2 */
	output [31:0]	DATA_OUT,	/* Data bus to Flash Controller */
	output			DNE,			/* Transaction Done */
	output [31:0]	DQ_IN,		/* Data bus to Flash IC */
	output [31:0]	DQ_TR,		/* Data bus to Flash IC Tristate Enable */
	output			NCE1,			/* Chipselect 1 */
	output			NOE,			/* Output Enable to Flash IC */
	output			NRP,			/* Hardware Reset of Flash IC */
	output			NWE			/* Write Enable to Flash IC */
	);

/*****Signals****************************************************************/
	
	wire DIR;	/* Data Bus Direction */
	wire BUSY;	/* Interface Busy */
	wire END;	/* Interface Enabled */

/*****Registers****************************************************************/

	reg bsy;		/* Interface Busy */
	reg bytes;	/* Byte Mode */
	reg dne_r;	/* Transaction Done */
	reg een;		/* Interface Enable to Flash Interface */
	reg eend;	/* Interface Enabled */
	reg oen;		/* Output Enable to Flash Interface */
	reg rsten;	/* Reset enable */
	reg wen;		/* Write Enable to Flash Interface */

/*********************************************************************/

	assign DNE = dne_r;
	assign DQ_TR = {32{DIR}};

/*****Initialization****************************************************************/

	initial
	begin
		bsy = 1'b0;
		bytes = 1'b1;
		dne_r = 1'b1;
		een = 1'b0;
		eend = 1'b0;
		oen = 1'b0;
		rsten = 1'b1;
		wen = 1'b0;
	end
	
/*****Flash Interface****************************************************************/

	flash_interface fi(
		.ADDR_IN(ADDR_IN),
		.ADDR_OUT(ADDR_OUT),
		.BUSY(BUSY),
		.BYTE(BYTE),
		.BYTES(bytes),
		.BYTESEL(BYTESEL),
		.CE0(CE0),
		.CE2(CE2),
		.CLK(FLASHCLK),
		.DATA_IN(DATA_IN),
		.DATA_OUT(DATA_OUT),
		.DELAY(DELAY),
		.DIR(DIR),
		.DQ_IN(DQ_OUT),
		.DQ_OUT(DQ_IN),
		.EN(een),
		.END(END),
		.NCE1(NCE1),
		.NOE(NOE),
		.NRP(NRP),
		.NWE(NWE),
		.OE(oen),
		.RESET(RESET),
		.RSTPIN(rsten),
		.WE(wen)
		);

/*********************************************************************/

	always@(posedge FLASHCLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			een = 1'b0;
			wen = 1'b0;
			oen = 1'b0;
			rsten = 1'b1;
			bytes = 1'b1;
			dne_r = 1'b1;
			bsy = 1'b0;
			eend = 1'b0;
		end
		else
		begin
			if (~RSTPIN)
			begin
				if (~END)
				begin
					/* Flash Hardware Reset */
					rsten = 1'b0;
				end
			end
			else
			begin
				rsten = 1'b1;
			end
			if (~END)
			begin
				/* Configure Bus Wideth */
				if (BUS[0]) bytes = 1'b0;
				if (BUS[1]) bytes = 1'b1;
				if (BUS[2]) bytes = 1'b1;
			end
			if (~rsten)
			begin
				if (SRT && dne_r)
				begin
					dne_r = 1'b0;
				end
				else
				begin
					dne_r = 1'b1;
				end
			end
			else
			begin
				if (SRT && dne_r && ~END)
				begin
					/* Start Transaction */
					dne_r = 1'b0;
					een = 1'b1;
				end
				else if (END & ~BUSY & ~dne_r & ~eend)
				begin
					eend = 1'b1;
					if (WE) wen = 1'b1;
					if (OE) oen = 1'b1;
				end
				else if (END & BUSY & ~dne_r & eend & ~bsy)
				begin
					bsy = 1'b1;
				end
				else if (END & BUSY & ~dne_r & eend & bsy)
				begin
					een = 1'b0;
					wen = 1'b0;
					oen = 1'b0;
				end
				else if (~END & ~BUSY & ~dne_r & bsy & eend)
				begin
					dne_r = 1'b1;
					bsy = 1'b0;
					eend = 1'b0;
				end
			end
		end	
	end

endmodule
