/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	flash_interface												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/* This module interfaces to a Micron Q-Flash Memory						*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module flash_interface(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [25:0]	ADDR_IN,		/* Address bus from Flash Bridge */
	input				BYTES,		/* Byte Mode */
	input				CLK,			/* Flash Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data bus from Flash Bridge */
	input [3:0]		DELAY,		/* Flash Transaction Delay */
	input [31:0]	DQ_IN,		/* Data bus from Flash IC */
	input				EN,			/* Enable */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				RSTPIN,		/* Hardware Flash Reset */
	input				WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output [23:0]	ADDR_OUT,	/* Address bus to Flash IC */
	output			BUSY,			/* Interface Busy */
	output			BYTE,			/* Byte Mode */
	output			BYTESEL,		/* Byte Select */
	output			CE0,			/* Chipselect 0 */
	output			CE2,			/* Chipselect 2 */
	output [31:0]	DATA_OUT,	/* Data bus to Flash Bridge */
	output			DIR,			/* Data Bus Direction */
	output [31:0]	DQ_OUT,		/* Data bus to Flash IC */
	output			END,			/* Interface Enabled */
	output			NCE1,			/* Chipselect 1 */
	output			NOE,			/* Output Enable to Flash IC */
	output			NRP,			/* Hardware Reset of Flash IC */
	output			NWE			/* Write Enable to Flash IC */
	);
	
/*****Registers****************************************************************/

	reg [25:0]	addr_r;			/* Address */
	reg			byte_r;			/* Byte Mode */
	reg			bytesel_r;		/* Byte Select */
	reg			ce0_r;			/* Chipselect 0 */
	reg			ce1_r;			/* Chipselect 1 */
	reg			ce2_r;			/* Chipselect 2 */
	reg [31:0]	data_in_r;		/* Data to Flash */
	reg [31:0]	data_out_r;		/* Data to Controller */
	reg [3:0]	delay_r;			/* Transaction Delay */
	reg [3:0]	delay_count;	/* Transaction Delay Counter */
	reg			noe_r;			/* Output Enable to Flash IC */
	reg			nrp_r;			/* Hardware Reset of Flash IC */
	reg			nwe_r;			/* Write Enable to Flash IC */

/*********************************************************************/

	assign ADDR_OUT = 	addr_r[25:2];
	assign BUSY = 			~(nwe_r & noe_r);
	assign BYTE = 			byte_r;
	assign BYTESEL =		bytesel_r;
	assign CE0 = 			ce0_r;
	assign CE2 = 			ce2_r;
	assign DATA_OUT = 	data_out_r;
	assign DIR = 			nwe_r;
	assign DQ_OUT = 		data_in_r;
	assign END = 			(ce0_r & ~ce1_r & ce2_r);
	assign NCE1 = 			ce1_r;
	assign NOE = 			noe_r;
	assign NRP = 			nrp_r;
	assign NWE = 			nwe_r;

/*****Initialization****************************************************************/

	initial
	begin
		addr_r = 26'b0;
		byte_r = 1'b0;
		bytesel_r = 1'b0;
		ce0_r = 1'b0;
		ce1_r = 1'b1;
		ce2_r = 1'b0;
		data_in_r = 32'b0;
		data_out_r = 32'b0;	
		delay_r = 4'b0;
		delay_count = 4'b0;
		noe_r = 1'b1;
		nrp_r = 1'b1;
		nwe_r = 1'b1;
	end

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			addr_r = 26'b0;
			byte_r = 1'b0;
			bytesel_r = 1'b0;
			ce0_r = 1'b0;
			ce1_r = 1'b1;
			ce2_r = 1'b0;
			data_in_r = 32'b0;
			data_out_r = 32'b0;	
			delay_r = 4'b0;
			delay_count = 4'b0;
			noe_r = 1'b1;
			nwe_r = 1'b1;
			nrp_r = 1'b1;
		end
		else
		begin
			if (EN & ~(ce0_r & ~ce1_r & ce2_r))
			begin
				/* Enable */
				ce0_r = 1'b1;
				ce1_r = 1'b0;
				ce2_r = 1'b1;
				delay_count = 4'b0;
			end
			if ((ce0_r & ~ce1_r & ce2_r) & WE & nwe_r)
			begin
				/* Write Enable */
				nwe_r = 1'b0;
				delay_count = 4'b0;
			end
			if ((ce0_r & ~ce1_r & ce2_r) & OE & noe_r)
			begin
				/* Output Enable */
				noe_r = 1'b0;
				delay_count = 4'b0;
			end
			if (ce0_r & ~ce1_r & ce2_r)
			begin
				if (delay_count >= delay_r)
				begin
					/* wait for delay */
					if (nwe_r & noe_r)
					begin
						ce0_r = 1'b0;
						ce1_r = 1'b1;
						ce2_r = 1'b0;
					end
					nwe_r = 1'b1;
					noe_r = 1'b1;
					data_out_r  = DQ_IN;
				end
				else
				begin
					delay_count = delay_count + 1;
				end
			end
			nrp_r = RSTPIN;
			byte_r = BYTES;
			delay_r = DELAY;
			addr_r = ADDR_IN;
			bytesel_r = ~byte_r & (ADDR_IN[0] == 1'b1);
			data_in_r = DATA_IN;
		end
	end

endmodule
