/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	giano_controller												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual		*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`define Request		0
`define Address		1
`define Value		2
`define DebugPort		3

`define OP_INT_IRQ	32'h00000002
`define OP_READ32		32'h00000000
`define OP_WRITE32	32'h00000100
`define OP_WRITE16	32'h00000300
`define OP_WRITE8		32'h00000500
`define OP_ICLOCK		32'h01000000
`define OP_DONE		32'h40000000

`define START		32'h0
`define STOP		32'h4c

module giano_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input			CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */ 
	input			OE,			/* Output Enable */
	input			RESET,		/* System Reset */
	input			SRT,			/* Start Transaction */
	input			WE,			/* Write Enable */

	input			BRAM_ENABLE,
	input [31:0]	BRAM_CONFIG_ADDR,
	input [31:0]	BRAM_CONFIG_SIZE,
	input [31:0]	BRAM_MEM_ADDR,
	input [31:0]	BRAM_MEM_SIZE,

	input			SRAM_ENABLE,
	input [31:0]	SRAM_CONFIG_ADDR,
	input [31:0]	SRAM_CONFIG_SIZE,
	input [31:0]	SRAM_MEM_ADDR,
	input [31:0]	SRAM_MEM_SIZE,

	input			FLASH_ENABLE,
	input [31:0]	FLASH_CONFIG_ADDR,
	input [31:0]	FLASH_CONFIG_SIZE,
	input [31:0]	FLASH_MEM_ADDR,
	input [31:0]	FLASH_MEM_SIZE,

	input			DDRAM_ENABLE,
	input [31:0]	DDRAM_CONFIG_ADDR,
	input [31:0]	DDRAM_CONFIG_SIZE,
	input [31:0]	DDRAM_MEM_ADDR,
	input [31:0]	DDRAM_MEM_SIZE,

	input			INTCTR_ENABLE,
	input [31:0]	INTCTR_ADDR,
	input [31:0]	INTCTR_SIZE,

	input 		USART_ENABLE,
	input [31:0]	USART_ADDR,
	input [31:0]	USART_SIZE,

	input			TIMER_ENABLE,
	input [31:0]	TIMER_ADDR,
	input [31:0]	TIMER_SIZE,

	input			WATCHDOG_ENABLE,
	input [31:0]	WATCHDOG_ADDR,
	input [31:0]	WATCHDOG_SIZE,

	input			GPIO_ENABLE,
	input [31:0]	GPIO_ADDR,
	input [31:0]	GPIO_SIZE,

	input			IDE_ENABLE,
	input [31:0]	IDE_ADDR,
	input [31:0]	IDE_SIZE,

	input			LCD_ENABLE,
	input [31:0]	LCD_ADDR,
	input [31:0]	LCD_SIZE,

	input			PS2_ENABLE,
	input [31:0]	PS2_ADDR,
	input [31:0]	PS2_SIZE,

	input			VGA_ENABLE,
	input [31:0]	VGA_ADDR,
	input [31:0]	VGA_SIZE,

	input 		ETHERNET_ENABLE,
	input [31:0]	ETHERNET_ADDR,
	input [31:0]	ETHERNET_SIZE,

	input			PWRMG_ENABLE,
	input [31:0]	PWRMG_ADDR,
	input [31:0]	PWRMG_SIZE,

	input			AC97_ENABLE,
	input [31:0]	AC97_ADDR,
	input [31:0]	AC97_SIZE,

	input			EC_ENABLE,
	input [31:0]	EC_ADDR,
	input [31:0]	EC_SIZE,

	input 		TRAKKIES_ENABLE,
	input	[31:0]	TRAKKIES_ADDR,
	input [31:0]	TRAKKIES_SIZE,

	/* OUTPUT PORTS */
	output [31:0]	DATA_OUT,		/* Data Bus to Data Path Memory Interface */
	output		DNE,			/* Transaction Done */
	output		IRQ			/* */
	);

/*****Signals****************************************************************/

	wire		MEN;			/* Transaction Enabled for PWRCTR Configuration Space */
	wire		SRAM_MEM_EN;
	wire		SRAM_CONFIG_EN;
	wire		FLASH_MEM_EN;
	wire		FLASH_CONFIG_EN;
	wire		BRAM_MEM_EN;
	wire		BRAM_CONFIG_EN;
	wire		DDRAM_MEM_EN;
	wire		DDRAM_CONFIG_EN;
	wire		INTCTR_EN;
	wire		USART_EN;
	wire		TIMER_EN;
	wire		WATCHDOG_EN;
	wire		GPIO_EN;
	wire		IDE_EN;
	wire		LCD_EN;
	wire		PS2_EN;
	wire		VGA_EN;
	wire		ETHERNET_EN;
	wire		PWRMG_EN;
	wire		AC97_EN;
	wire		EC_EN;
	wire 		TRAKKIES_EN;

/*****Registers****************************************************************/

	reg [31:0]	data_in_m_reg;		/* Masked Incoming Data */
	reg [3:0]	dnecnt;			/* Transaction Progress Counter */
	reg		dner;				/* Configuration Space Transaction Done */
	reg		men_reg;			/* Transaction Enabled for PWRCTR Configuration Space */
	reg [31:0]	reg_m_0_reg;		/* Selected Configuration Register Data */
	reg [1:0]	state;
	reg 		irq_reg;

	reg [31:0] mem[3:0];
	reg 		giano_start;
	reg [7:0]	count;

/*****Parameters****************************************************************/

	parameter	BASEl =	31;			/* BASE Bit Range Left */
	parameter	BASEr =	16;			/* BASE Bit Range Right */
	parameter	TAGl = 	15;			/* TAG Bit Range Left */
	parameter	TAGr = 	0;			/* TAG Bit Range Right */

/*****Initialization****************************************************************/

	initial 
	begin
		dnecnt = 4'b1111;
		dner = 1'b1;
		men_reg = 1'b0;
		reg_m_0_reg = 32'b0;
		state = 2'b11;
		irq_reg = 1'b0;

		giano_start = 1'b0;
		count = 8'b0;
		mem[`Request] = `OP_DONE;
		mem[`Address] = `START;
		mem[`Value] = 32'b0;
		mem[`DebugPort] = 32'b0;
		$PassMemHandle(mem);
	end

/*********************************************************************/

	assign DNE = dner;
	assign IRQ = irq_reg;

/*****Address Decoding****************************************************************/

	assign SRAM_MEM_EN	=	(SRAM_ENABLE && (ADDR_IN >= SRAM_MEM_ADDR) && (ADDR_IN < (SRAM_MEM_ADDR + SRAM_MEM_SIZE)));
	assign SRAM_CONFIG_EN	=	(SRAM_ENABLE && (ADDR_IN >= SRAM_CONFIG_ADDR) && (ADDR_IN < (SRAM_CONFIG_ADDR + SRAM_CONFIG_SIZE)));
	assign FLASH_MEM_EN	=	(FLASH_ENABLE && (ADDR_IN >= FLASH_MEM_ADDR) && (ADDR_IN < (FLASH_MEM_ADDR + FLASH_MEM_SIZE)));
	assign FLASH_CONFIG_EN	=	(FLASH_ENABLE && (ADDR_IN >= FLASH_CONFIG_ADDR) && (ADDR_IN < (FLASH_CONFIG_ADDR + FLASH_CONFIG_SIZE)));
	assign BRAM_MEM_EN	=	(BRAM_ENABLE && (ADDR_IN >= BRAM_MEM_ADDR) && (ADDR_IN < (BRAM_MEM_ADDR + BRAM_MEM_SIZE)));
	assign BRAM_CONFIG_EN	=	(BRAM_ENABLE && (ADDR_IN >= BRAM_CONFIG_ADDR) && (ADDR_IN < (BRAM_CONFIG_ADDR + BRAM_CONFIG_SIZE)));
	assign DDRAM_MEM_EN	=	(DDRAM_ENABLE && (ADDR_IN >= DDRAM_MEM_ADDR) && (ADDR_IN < (DDRAM_MEM_ADDR + DDRAM_MEM_SIZE)));
	assign DDRAM_CONFIG_EN	=	(DDRAM_ENABLE && (ADDR_IN >= DDRAM_CONFIG_ADDR) && (ADDR_IN < (DDRAM_CONFIG_ADDR + DDRAM_CONFIG_SIZE)));
	assign INTCTR_EN		=	(INTCTR_ENABLE && (ADDR_IN >= INTCTR_ADDR) && (ADDR_IN < (INTCTR_ADDR + INTCTR_SIZE)));
	assign USART_EN		=	(USART_ENABLE && (ADDR_IN >= USART_ADDR) && (ADDR_IN < (USART_ADDR + USART_SIZE)));
	assign TIMER_EN		=	(TIMER_ENABLE && (ADDR_IN >= TIMER_ADDR) && (ADDR_IN < (TIMER_ADDR + TIMER_SIZE)));
	assign WATCHDOG_EN	=	(WATCHDOG_ENABLE && (ADDR_IN >= WATCHDOG_ADDR) && (ADDR_IN < (WATCHDOG_ADDR + WATCHDOG_SIZE)));
	assign GPIO_EN		=	(GPIO_ENABLE && (ADDR_IN >= GPIO_ADDR) && (ADDR_IN < (GPIO_ADDR + GPIO_SIZE)));
	assign IDE_EN		=	(IDE_ENABLE && (ADDR_IN >= IDE_ADDR) && (ADDR_IN < (IDE_ADDR + IDE_SIZE)));
	assign LCD_EN		=	(LCD_ENABLE && (ADDR_IN >= LCD_ADDR) && (ADDR_IN < (LCD_ADDR + LCD_SIZE)));
	assign PS2_EN		=	(PS2_ENABLE && (ADDR_IN >= PS2_ADDR) && (ADDR_IN < (PS2_ADDR + PS2_SIZE)));
	assign VGA_EN		=	(VGA_ENABLE && (ADDR_IN >= VGA_ADDR) && (ADDR_IN < (VGA_ADDR + VGA_SIZE)));
	assign ETHERNET_EN	=	(ETHERNET_ENABLE && (ADDR_IN >= ETHERNET_ADDR) && (ADDR_IN < (ETHERNET_ADDR + ETHERNET_SIZE)));
	assign PWRMG_EN		=	(PWRMG_ENABLE && (ADDR_IN >= PWRMG_ADDR) && (ADDR_IN < (PWRMG_ADDR + PWRMG_SIZE)));
	assign AC97_EN		=	(AC97_ENABLE && (ADDR_IN >= AC97_ADDR) && (ADDR_IN < (AC97_ADDR + AC97_SIZE)));
	assign EC_EN		=	(EC_ENABLE && (ADDR_IN >= EC_ADDR) && (ADDR_IN < (EC_ADDR + EC_SIZE)));
	assign TRAKKIES_EN	=	(TRAKKIES_ENABLE && OE && WE);

	assign MEN =	SRAM_MEM_EN | 
				SRAM_CONFIG_EN |
				FLASH_MEM_EN |
				FLASH_CONFIG_EN |
				BRAM_MEM_EN |
				BRAM_CONFIG_EN |
				DDRAM_MEM_EN |
				DDRAM_CONFIG_EN |
				INTCTR_EN |
				USART_EN |
				TIMER_EN |
				WATCHDOG_EN |
				GPIO_EN |
				IDE_EN |
				LCD_EN |
				PS2_EN |
				VGA_EN |
				ETHERNET_EN |
				PWRMG_EN |
				AC97_EN |
				EC_EN |
				TRAKKIES_EN;

/*****Outgoing Data Handling****************************************************************/

	assign DATA_OUT = (OE & men_reg)?reg_m_0_reg:32'bz;

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg <= 32'b0;
			men_reg <= 1'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg <= DATA_IN;
			dnecnt <= {dnecnt[2:0],dner};
			men_reg <= MEN;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,1'b0,dnecnt[3],dner})
			4'b0xxx	:	begin
							/* Reset */
							dner <= 1'b1;
							state <= 2'b11;
							irq_reg <= 1'b0;
							reg_m_0_reg <= 32'b0;
							giano_start <= 1'b0;
							count <= 8'b0;
							mem[`Request] <= `OP_DONE;
							mem[`Address] <= `START;
							mem[`Value] <= 32'b0;
							mem[`DebugPort] <= 32'b0;
						end
			4'b11xx	:	begin
							/* Soft Reset */
							dner <= 1'b1;
							state <= 2'b11;
							irq_reg <= 1'b0;
							reg_m_0_reg <= 32'b0;
							giano_start <= 1'b0;
							count <= 8'b0;
							mem[`Request] <= `OP_DONE;
							mem[`Address] <= `START;
							mem[`Value] <= 32'b0;
							mem[`DebugPort] <= 32'b0;
						end
			4'b1000	:	begin
							/* Update Configuration Registers */
							//dner = 1'b1;
							mem[`DebugPort] <= state;
							case(state)
								2'b00		:	begin
												if (OE)
												begin
													if (TRAKKIES_EN)
													begin
														mem[`Request] <= `OP_ICLOCK | `OP_READ32;	
														//$stop;
													end
													else
													begin
														mem[`Request] <= `OP_READ32;
													end
												end
												else if (WE)
												begin
													case(BYTES)
														2'b00		:	begin
																		mem[`Request] <= `OP_WRITE32;
																	end
														2'b01		:	begin
																		mem[`Request] <= `OP_WRITE8;
																	end
														2'b10		:	begin
																		mem[`Request] <= `OP_WRITE16;
																	end
														2'b11		:	begin
																		//$stop;
																	end
														default	:	begin
																	end
													endcase
													mem[`Value] <= data_in_m_reg;	
												end
												state <= 2'b01;
											end
								2'b01		:	begin
												count <= count + 1;
												if ((mem[`Request] & `OP_DONE) > 0)
												begin
													if ((mem[`Request] & `OP_INT_IRQ) > 0)
													begin
														irq_reg <= 1'b1;
													end
													else
													begin
														irq_reg <= 1'b0;
													end
													state <= 2'b10;
												end
												if (~giano_start && TRAKKIES_EN && (ADDR_IN[31:28] == 4'h8) && (count == 8'hff))
												begin
													giano_start <= 1'b1;
													$stop;
												end
											end
								2'b10		:	begin
												if (OE)
												begin
													if (TRAKKIES_EN)
													begin
														reg_m_0_reg <= 32'hffffffff;
														//$stop;
													end
													else
													begin
														reg_m_0_reg <= mem[`Value];
													end
												end
												dner <= 1'b1;
												state <= 2'b11;
											end
								2'b11		:	begin
												mem[`Address] <= ADDR_IN;
												state <= 2'b00;
											end
								default	:	begin
											end
							endcase
						end
			4'b10x1	:	begin
							/* Wait for incoming request */
							if (SRT & men_reg)
							begin
								/* Request Recognized */
								dner <= 1'b0;
								count <= 8'b0;
							end
						end
			default	:	begin
						end
		endcase
	end

endmodule
