/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	gpio_controller												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual		*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module gpio_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,			/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,			/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,			/* Configuration Base Address */
	input				CLK,				/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,			/* Data Bus from Data Path Memory Interface */ 
	input [31:0]	GPIO_OUT,		/* Bus from Pins */
	input				OE,				/* Output Enable */
	input				RESET,			/* System Reset */
	input				SRT,				/* Start Transaction */
	input				WE,				/* Write Enable */
	/* OUTPUT PORTS */
	output [31:0]	BASETAG,			/* BASETAG Register */
	output [31:0]	DATA_OUT,		/* Data Bus to Data Path Memory Interface */
	output			DNE,				/* Transaction Done */
	output [31:0]	GPIO_IN,			/* Bus to Pins */
	output			GPIO_IRQ,		/* GPIO Interrupt */
	output [31:0]	GPIO_TR,			/* Bus to Pins Tristate Enable */
	output [31:0]	STATUS			/* Status Register */
	);

/*****Signals****************************************************************/
	
	wire [3:0]	BM;				/* Byte Selects */
	wire [31:0]	DATA_IN_C;		/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;		/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;		/* Endian Flipped Outgoing Data */
	wire [31:0]	GPIO_OUT_S;		/* Bus from Pins */
	wire [31:0]	GPIO_INT_S;		/* GPIO Interrupt Logic */
	wire			MEN;				/* Transaction Enabled for GPIO */
	wire [31:0]	REG_M;			/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;			/* Selected Configuration Register Data */
	wire [31:0]	WMASK_REG;		/* Incoming Data Write Mask */
	wire [31:0]	WMASK_IN;		/* Configuration Register Data Write Mask */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;			/* BASETAG Register */
	reg [31:0]	data_in_m_reg;		/* Masked Incoming Data */
	reg [3:0]	dnecnt;				/* Transaction Progress Counter */
	reg			dner;					/* GPIO Space Transaction Done */
	reg [31:0]	gpio_clear;			/* GPIO Clear Data Register */
	reg [31:0]	gpio_data;			/* GPIO Pin Data Register */
	reg [31:0]	gpio_dir;			/* GPIO Direction Register */
	reg [31:0]	gpio_disable;		/* GPIO Disable Register */
	reg [31:0]	gpio_enable;		/* GPIO Enable Register */
	reg [31:0]	gpio_enabled;		/* GPIO Enabled Register */
	reg [31:0]	gpio_in_dir;		/* GPIO Out Disable Register */
	reg [31:0]	gpio_int;			/* GPIO Interrupt Enabled Register */
	reg [31:0]	gpio_int_dis;		/*	GPIO Interrupt Disable Register */
	reg [31:0]	gpio_int_en;		/* GPIO Interrupt	Enable Register */
	reg [31:0]	gpio_int_lvl;		/* GPIO Interrupt Trigger High/Low Register*/
	reg [31:0]	gpio_int_raw;		/* GPIO Interrupt Detection */
	reg [31:0]	gpio_int_tr;		/* GPIO Interrupt Trigger Edge/Level Register*/
	reg [31:0]	gpio_intclear;		/* GPIO Interrupt Status Clear Register */
	reg [31:0]	gpio_intstatus;	/* GPIO Interrupt Status Register */
	reg [31:0]	gpio_out_dir;		/* GPIO Out Enable Register */
	reg [31:0]	gpio_set;			/* GPIO Set Data Register */
	reg [31:0]	gpio_status;		/* GPIO Pin status Register */
	reg			men_reg;				/* Transaction Enabled for GPIO Space */
	reg [31:0]	reg_m_0_reg;		/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;			/* Masked Configuration Register Data */

/*****Variables****************************************************************/

	integer i;

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter	BASETAGREG =				16'h0000;	/* RW	BASETAG Register Relative Address */
	parameter	GPIO_ENABLED =				16'h0004;	/* R	GPIO Enabled Register Relative Address */
	parameter	GPIO_ENABLE =				16'h0004;	/* W	GPIO Enable Register Relative Address */
	parameter	GPIO_DISABLE =				16'h0008;	/* W	GPIO Disable Register Relative Address */
	parameter	GPIO_DIRECTION =			16'h000c;	/* R	GPIO Direction Register Relative Address */
	parameter	GPIO_OUTENABLE =			16'h000c;	/* W	GPIO Out Enable Register Relative Address */
	parameter	GPIO_OUTDISABLE =			16'h0010;	/* W	GPIO Out Disable Register Relative Address */
	parameter	GPIO_PINDATA =				16'h0014;	/* R	GPIO Pin Data Register Relative Address */
	parameter	GPIO_SETDATA =				16'h0014;	/* W	GPIO Set Data Register Relative Address */
	parameter	GPIO_CLEARDATA =			16'h0018;	/* W	GPIO Clear Data Register Relative Address */
	parameter	GPIO_PINSTATUS =			16'h001c;	/* R	GPIO Pin status Register Relative Address */
	parameter	GPIO_INTRSTATUS =			16'h0020;	/* R	GPIO Interrupt Status Register Relative Address */
	parameter	GPIO_INTRSTATUSCLEAR =	16'h0020;	/* W	GPIO Interrupt Status Clear Register Relative Address */
	parameter	GPIO_INTRENABLED =		16'h0024;	/* R	GPIO Interrupt Enabled Register Relative Address */
	parameter	GPIO_INTRENABLE =			16'h0024;	/* W	GPIO Interrupt	Enable Register Relative Address */
	parameter	GPIO_INTRDISABLE =		16'h0028;	/*	W	GPIO Interrupt Disable Register Relative Address */
	parameter	GPIO_INTRTRIGGER =		16'h002c;	/* RW	GPIO Interrupt Trigger Edge/Level Relative Address*/
	parameter	GPIO_INTRNOTLEVEL =		16'h0030;	/* RW GPIO Interrupt Trigger High/Low Relative Address*/

	/* BASETAG Register */
	parameter	BASEl =						31;			/* BASE Bit Range Left */
	parameter	BASEr =						16;			/* BASE Bit Range Right */
	parameter	TAGl = 						15;			/* TAG Bit Range Left */
	parameter	TAGr = 						0;				/* TAG Bit Range Right */
	parameter	TAG =							9;				/* TAG Value */

/*****Initialization****************************************************************/

	initial
	begin
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		dnecnt = 4'b1111;
		dner = 1'b1;
		gpio_clear = 32'b0;
		gpio_data = 32'b0;
		gpio_enable = 32'b0;
		gpio_enabled = 32'b0;
		gpio_dir = 32'b0;
		gpio_disable = 32'b0;
		gpio_in_dir = 32'b0;
		gpio_int = 32'b0;
		gpio_int_dis = 32'b0;
		gpio_int_en = 32'b0;
		gpio_int_lvl = 32'b0;
		gpio_int_raw = 32'b0;
		gpio_int_tr = 32'b0;
		gpio_intclear = 32'b0;
		gpio_intstatus = 32'b0;
		gpio_out_dir = 32'b0;
		gpio_set = 32'b0;
		gpio_status = 32'b0;
		men_reg = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
	end

/*********************************************************************/

	assign STATUS = gpio_status ;
	assign BASETAG = basetagr;
	assign DNE = dner;
	assign GPIO_IN =	gpio_data;
	assign GPIO_OUT_S = GPIO_OUT;
	assign GPIO_TR =	~(gpio_enabled & gpio_dir);
	assign MEN = (CADDR[BASEl:BASEr] == ADDR_IN[BASEl:BASEr]);

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/

	assign REG_M_0 =	(ADDR_IN[BASEr-1:0] == BASETAGREG)?							basetagr:			32'bz;
	assign REG_M_0 =	(OE & (ADDR_IN[BASEr-1:0] == GPIO_ENABLED))?				gpio_enabled:		32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_ENABLE))?				gpio_enable:		32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_DISABLE))?				gpio_disable:		32'bz;
	assign REG_M_0 =	(OE & (ADDR_IN[BASEr-1:0] == GPIO_DIRECTION))?			gpio_dir:			32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_OUTENABLE))?			gpio_out_dir:		32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_OUTDISABLE))?			gpio_in_dir:		32'bz;
	assign REG_M_0 =	(OE & (ADDR_IN[BASEr-1:0] == GPIO_PINDATA))?				gpio_data:			32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_SETDATA))?				gpio_set:			32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_CLEARDATA))?			gpio_clear:			32'bz;
	assign REG_M_0 =	(OE & (ADDR_IN[BASEr-1:0] == GPIO_PINSTATUS))?			gpio_status:		32'bz;
	assign REG_M_0 =	(OE & (ADDR_IN[BASEr-1:0] == GPIO_INTRSTATUS))?			gpio_intstatus:	32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_INTRSTATUSCLEAR))?	gpio_intclear:		32'bz;
	assign REG_M_0 =	(OE & (ADDR_IN[BASEr-1:0] == GPIO_INTRENABLED))?		gpio_int:			32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_INTRENABLE))?			gpio_int_en:		32'bz;
	assign REG_M_0 =	(WE & (ADDR_IN[BASEr-1:0] == GPIO_INTRDISABLE))?		gpio_int_dis:		32'bz;
	assign REG_M_0 =	(ADDR_IN[BASEr-1:0] == GPIO_INTRTRIGGER)?					gpio_int_tr:		32'bz;
	assign REG_M_0 =	(ADDR_IN[BASEr-1:0] == GPIO_INTRNOTLEVEL)?				gpio_int_lvl:		32'bz;
	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);
		
/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*****GPIO Interrupt Monitor****************************************************************/
	
	io_int ioint(
		.GPIO_ENABLED(gpio_enabled),
		.GPIO_INT(GPIO_INT_S),
		.GPIO_INTEN(gpio_int),
		.GPIO_INTSTATUS(gpio_intstatus),
		.GPIO_INTTR(gpio_int_lvl),
		.GPIO_IRQ(GPIO_IRQ),
		.GPIO_NOTLVL(gpio_int_lvl),
		.GPIO_OUT(GPIO_OUT_S),
		.GPIO_STATUS(gpio_status)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
		end
	end
		
	always@(posedge CLK)
	begin
		casex({RESET,1'b0,dnecnt[3],dner})
			4'b0xxx	:	begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								dner = 1'b1;
								gpio_clear = 32'b0;
								gpio_data = 32'b0;
								gpio_dir = 32'b0;
								gpio_disable = 32'b0;
								gpio_enable = 32'b0;
								gpio_enabled = 32'b0;
								gpio_in_dir = 32'b0;
								gpio_int = 32'b0;
								gpio_int_dis = 32'b0;
								gpio_int_en = 32'b0;
								gpio_int_lvl = 32'b0;
								gpio_int_raw = 32'b0;
								gpio_int_tr = 32'b0;
								gpio_intclear = 32'b0;
								gpio_intstatus = 32'b0;
								gpio_out_dir = 32'b0;
								gpio_set = 32'b0;
								gpio_status = 32'b0;
							end
			4'b11xx	:	begin
								/* Software Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								dner = 1'b1;
								gpio_clear = 32'b0;
								gpio_data = 32'b0;
								gpio_dir = 32'b0;
								gpio_disable = 32'b0;
								gpio_enable = 32'b0;
								gpio_enabled = 32'b0;
								gpio_in_dir = 32'b0;
								gpio_int = 32'b0;
								gpio_int_dis = 32'b0;
								gpio_int_en = 32'b0;
								gpio_int_lvl = 32'b0;
								gpio_int_raw = 32'b0;
								gpio_int_tr = 32'b0;
								gpio_intclear = 32'b0;
								gpio_intstatus = 32'b0;
								gpio_out_dir = 32'b0;
								gpio_set = 32'b0;
								gpio_status = 32'b0;
							end
			4'b1000	:	begin
								/* Update Registers */
								dner = 1'b1;
								if (men_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										BASETAGREG				:	begin
																			/* Update Basetag Register */
																			basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
																		end
										GPIO_ENABLE 			:	begin
																			/* Update GPIO Enable Register */
																			gpio_enable = data_in_m_reg | reg_m_reg;
																		end
										GPIO_DISABLE			:	begin
																			/* Update GPIO Disable Register */
																			gpio_disable = data_in_m_reg | reg_m_reg;
																		end
										GPIO_OUTENABLE 		:	begin
																			/* Update GPIO Out Enable Register */
																			gpio_out_dir = data_in_m_reg | reg_m_reg;
																		end
										GPIO_OUTDISABLE		:	begin
																			/* Update GPIO Out Disable Register */
																			gpio_in_dir = data_in_m_reg | reg_m_reg;
																		end
										GPIO_SETDATA 			:	begin
																			/* Update GPIO Set Data Register */
																			gpio_set = data_in_m_reg | reg_m_reg;
																		end
										GPIO_CLEARDATA			:	begin
																			/* Update GPIO Clear Data Register */
																			gpio_clear = data_in_m_reg | reg_m_reg;
																		end
										GPIO_INTRSTATUSCLEAR	:	begin
																			/* Update GPIO Interrupt Status Clear Register */
																			gpio_intclear = data_in_m_reg | reg_m_reg;
																		end
										GPIO_INTRENABLE		:	begin
																			/* Update GPIO Interrupt Enable Register */
																			gpio_int_en = data_in_m_reg | reg_m_reg;
																		end
										GPIO_INTRDISABLE 		:	begin
																			/* Update GPIO Interrupt Disable Register */
																			gpio_int_dis = data_in_m_reg | reg_m_reg;
																		end
										GPIO_INTRTRIGGER 		:	begin
																			/* Update GPIO Interrupt Trigger Register */
																			gpio_int_tr = data_in_m_reg | reg_m_reg;
																		end
										GPIO_INTRNOTLEVEL 	:	begin
																			/* Update GPIO Interrupt Level Register */
																			gpio_int_lvl = data_in_m_reg | reg_m_reg;
																		end
										default					:	begin
																			/* Do Nothing */
																			/* Address not Recognized */
																		end
									endcase
								end
							end
			4'b10x1	:	begin
								/* Wait for incoming request */
								if (SRT & men_reg)
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
								/* Update GPIO Enabled Register */
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_enable[i]) gpio_enabled[i] = 1'b1;
								end
								gpio_enable = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_disable[i]) gpio_enabled[i] = 1'b0;
								end
								gpio_disable= 32'b0;
								/*  Update GPIO Direction Register */
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_out_dir[i]) gpio_dir[i] = 1'b1;
								end
								gpio_out_dir = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_in_dir[i]) gpio_dir[i] = 1'b0;
								end
								gpio_in_dir = 32'b0;
								/* Update GPIO Data Register */
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_set[i]) gpio_data[i] = 1'b1;
								end
								gpio_set = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_clear[i]) gpio_data[i] = 1'b0;
								end
								gpio_clear = 32'b0;
								/* Update GPIO Interrupt Status Register */
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_intclear[i]) gpio_intstatus[i] = 1'b0;
								end
								gpio_intclear = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_int_raw[i]) gpio_intstatus[i] = 1'b1;
								end
								gpio_int_raw = 32'b0;
								/* Update GPIO Interrupt Enabled */
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_int_en[i]) gpio_int[i] = 1'b1;
								end
								gpio_int_en = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_int_dis[i]) gpio_int[i] = 1'b0;
								end
								gpio_int_dis = 32'b0;
								/* Update GPIO Pin Status Register */
								for (i=0; i<32; i=i+1)
								begin
									if (gpio_enabled[i]) gpio_status[i] = GPIO_OUT_S[i];
								end
							end
			default	:	begin
								/* Do Nothing */
							end
		endcase
		/* Detect GPIO Interrupt */
		for (i=0; i<32; i=i+1)
		begin
			if (GPIO_INT_S[i]) gpio_int_raw[i] = 1'b1;
		end
	end

endmodule

/*****GPIO Interrupt Monitor****************************************************************/

module io_int(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	GPIO_ENABLED,		/* GPIO Enabled Register */
	input [31:0]	GPIO_INTEN,			/* GPIO Interrupt Enabled */
	input [31:0]	GPIO_INTSTATUS,	/* GPIO Interrupt Status Register */
	input [31:0]	GPIO_INTTR,			/* GPIO Interrupt Trigger Register */
	input [31:0]	GPIO_NOTLVL,		/* GPIO Interrupt Level Register */
	input [31:0]	GPIO_OUT,			/* Bus from Pins */
	input [31:0]	GPIO_STATUS,		/* Update GPIO Pin Status Register */
	/* OUTPUT PORTS */
	output [31:0]	GPIO_INT,			/* GPIO Interrupt Detection*/
	output			GPIO_IRQ				/* GPIO Interrupt */
	);

/*****Signals****************************************************************/

	wire [31:0] GPIO_IRQ32;

/*********************************************************************/
	
	/* GPIO Interrupt */
	assign GPIO_IRQ32 = (GPIO_INTSTATUS & GPIO_ENABLED & GPIO_INTEN);
	assign GPIO_IRQ = (GPIO_IRQ32 > 0);

/*********************************************************************/
	
	/* GPIO Interrupt Detection*/
	
	assign GPIO_INT[0] =	(~GPIO_ENABLED[0])?																							1'b0:
								(({GPIO_INTTR[0],GPIO_NOTLVL[0]} == 2'b00) & (GPIO_STATUS[0]))?								1'b1:
								(({GPIO_INTTR[0],GPIO_NOTLVL[0]} == 2'b01) & (~GPIO_STATUS[0]))?								1'b1:
								(({GPIO_INTTR[0],GPIO_NOTLVL[0]} == 2'b10) & ({GPIO_STATUS[0],GPIO_OUT[0]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[0],GPIO_NOTLVL[0]} == 2'b11) & ({GPIO_STATUS[0],GPIO_OUT[0]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[1] =	(~GPIO_ENABLED[1])?																							1'b0:
								(({GPIO_INTTR[1],GPIO_NOTLVL[1]} == 2'b00) & (GPIO_STATUS[1]))?								1'b1:
								(({GPIO_INTTR[1],GPIO_NOTLVL[1]} == 2'b01) & (~GPIO_STATUS[1]))?								1'b1:
								(({GPIO_INTTR[1],GPIO_NOTLVL[1]} == 2'b10) & ({GPIO_STATUS[1],GPIO_OUT[1]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[1],GPIO_NOTLVL[1]} == 2'b11) & ({GPIO_STATUS[1],GPIO_OUT[1]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[2] =	(~GPIO_ENABLED[2])?																							1'b0:
								(({GPIO_INTTR[2],GPIO_NOTLVL[2]} == 2'b00) & (GPIO_STATUS[2]))?								1'b1:
								(({GPIO_INTTR[2],GPIO_NOTLVL[2]} == 2'b01) & (~GPIO_STATUS[2]))?								1'b1:
								(({GPIO_INTTR[2],GPIO_NOTLVL[2]} == 2'b10) & ({GPIO_STATUS[2],GPIO_OUT[2]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[2],GPIO_NOTLVL[2]} == 2'b11) & ({GPIO_STATUS[2],GPIO_OUT[2]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[3] =	(~GPIO_ENABLED[3])?																							1'b0:
								(({GPIO_INTTR[3],GPIO_NOTLVL[3]} == 2'b00) & (GPIO_STATUS[3]))?								1'b1:
								(({GPIO_INTTR[3],GPIO_NOTLVL[3]} == 2'b01) & (~GPIO_STATUS[3]))?								1'b1:
								(({GPIO_INTTR[3],GPIO_NOTLVL[3]} == 2'b10) & ({GPIO_STATUS[3],GPIO_OUT[3]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[3],GPIO_NOTLVL[3]} == 2'b11) & ({GPIO_STATUS[3],GPIO_OUT[3]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[4] =	(~GPIO_ENABLED[4])?																							1'b0:
								(({GPIO_INTTR[4],GPIO_NOTLVL[4]} == 2'b00) & (GPIO_STATUS[4]))?								1'b1:
								(({GPIO_INTTR[4],GPIO_NOTLVL[4]} == 2'b01) & (~GPIO_STATUS[4]))?								1'b1:
								(({GPIO_INTTR[4],GPIO_NOTLVL[4]} == 2'b10) & ({GPIO_STATUS[4],GPIO_OUT[4]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[4],GPIO_NOTLVL[4]} == 2'b11) & ({GPIO_STATUS[4],GPIO_OUT[4]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[5] =	(~GPIO_ENABLED[5])?																							1'b0:
								(({GPIO_INTTR[5],GPIO_NOTLVL[5]} == 2'b00) & (GPIO_STATUS[5]))?								1'b1:
								(({GPIO_INTTR[5],GPIO_NOTLVL[5]} == 2'b01) & (~GPIO_STATUS[5]))?								1'b1:
								(({GPIO_INTTR[5],GPIO_NOTLVL[5]} == 2'b10) & ({GPIO_STATUS[5],GPIO_OUT[5]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[5],GPIO_NOTLVL[5]} == 2'b11) & ({GPIO_STATUS[5],GPIO_OUT[5]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[6] =	(~GPIO_ENABLED[6])?																							1'b0:
								(({GPIO_INTTR[6],GPIO_NOTLVL[6]} == 2'b00) & (GPIO_STATUS[6]))?								1'b1:
								(({GPIO_INTTR[6],GPIO_NOTLVL[6]} == 2'b01) & (~GPIO_STATUS[6]))?								1'b1:
								(({GPIO_INTTR[6],GPIO_NOTLVL[6]} == 2'b10) & ({GPIO_STATUS[6],GPIO_OUT[6]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[6],GPIO_NOTLVL[6]} == 2'b11) & ({GPIO_STATUS[6],GPIO_OUT[6]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[7] =	(~GPIO_ENABLED[7])?																							1'b0:
								(({GPIO_INTTR[7],GPIO_NOTLVL[7]} == 2'b00) & (GPIO_STATUS[7]))?								1'b1:
								(({GPIO_INTTR[7],GPIO_NOTLVL[7]} == 2'b01) & (~GPIO_STATUS[7]))?								1'b1:
								(({GPIO_INTTR[7],GPIO_NOTLVL[7]} == 2'b10) & ({GPIO_STATUS[7],GPIO_OUT[7]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[7],GPIO_NOTLVL[7]} == 2'b11) & ({GPIO_STATUS[7],GPIO_OUT[7]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[8] =	(~GPIO_ENABLED[8])?																							1'b0:
								(({GPIO_INTTR[8],GPIO_NOTLVL[8]} == 2'b00) & (GPIO_STATUS[8]))?								1'b1:
								(({GPIO_INTTR[8],GPIO_NOTLVL[8]} == 2'b01) & (~GPIO_STATUS[8]))?								1'b1:
								(({GPIO_INTTR[8],GPIO_NOTLVL[8]} == 2'b10) & ({GPIO_STATUS[8],GPIO_OUT[8]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[8],GPIO_NOTLVL[8]} == 2'b11) & ({GPIO_STATUS[8],GPIO_OUT[8]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[9] =	(~GPIO_ENABLED[9])?																							1'b0:
								(({GPIO_INTTR[9],GPIO_NOTLVL[9]} == 2'b00) & (GPIO_STATUS[9]))?								1'b1:
								(({GPIO_INTTR[9],GPIO_NOTLVL[9]} == 2'b01) & (~GPIO_STATUS[9]))?								1'b1:
								(({GPIO_INTTR[9],GPIO_NOTLVL[9]} == 2'b10) & ({GPIO_STATUS[9],GPIO_OUT[9]} == 2'b01))?	1'b1:
								(({GPIO_INTTR[9],GPIO_NOTLVL[9]} == 2'b11) & ({GPIO_STATUS[9],GPIO_OUT[9]} == 2'b10))?	1'b1:
																																					1'b0;

	assign GPIO_INT[10] =	(~GPIO_ENABLED[10])?																									1'b0:
									(({GPIO_INTTR[10],GPIO_NOTLVL[10]} == 2'b00) & (GPIO_STATUS[10]))?									1'b1:
									(({GPIO_INTTR[10],GPIO_NOTLVL[10]} == 2'b01) & (~GPIO_STATUS[10]))?									1'b1:
									(({GPIO_INTTR[10],GPIO_NOTLVL[10]} == 2'b10) & ({GPIO_STATUS[10],GPIO_OUT[10]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[10],GPIO_NOTLVL[10]} == 2'b11) & ({GPIO_STATUS[10],GPIO_OUT[10]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[11] =	(~GPIO_ENABLED[11])?																									1'b0:
									(({GPIO_INTTR[11],GPIO_NOTLVL[11]} == 2'b00) & (GPIO_STATUS[11]))?									1'b1:
									(({GPIO_INTTR[11],GPIO_NOTLVL[11]} == 2'b01) & (~GPIO_STATUS[11]))?									1'b1:
									(({GPIO_INTTR[11],GPIO_NOTLVL[11]} == 2'b10) & ({GPIO_STATUS[11],GPIO_OUT[11]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[11],GPIO_NOTLVL[11]} == 2'b11) & ({GPIO_STATUS[11],GPIO_OUT[11]} == 2'b10))?	1'b1:
																																				1'b0;

	assign GPIO_INT[12] =	(~GPIO_ENABLED[12])?																									1'b0:
									(({GPIO_INTTR[12],GPIO_NOTLVL[12]} == 2'b00) & (GPIO_STATUS[12]))?									1'b1:
									(({GPIO_INTTR[12],GPIO_NOTLVL[12]} == 2'b01) & (~GPIO_STATUS[12]))?									1'b1:
									(({GPIO_INTTR[12],GPIO_NOTLVL[12]} == 2'b10) & ({GPIO_STATUS[12],GPIO_OUT[12]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[12],GPIO_NOTLVL[12]} == 2'b11) & ({GPIO_STATUS[12],GPIO_OUT[12]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[13] =	(~GPIO_ENABLED[13])?																									1'b0:
									(({GPIO_INTTR[13],GPIO_NOTLVL[13]} == 2'b00) & (GPIO_STATUS[13]))?									1'b1:
									(({GPIO_INTTR[13],GPIO_NOTLVL[13]} == 2'b01) & (~GPIO_STATUS[13]))?									1'b1:
									(({GPIO_INTTR[13],GPIO_NOTLVL[13]} == 2'b10) & ({GPIO_STATUS[13],GPIO_OUT[13]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[13],GPIO_NOTLVL[13]} == 2'b11) & ({GPIO_STATUS[13],GPIO_OUT[13]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[14] =	(~GPIO_ENABLED[14])?																									1'b0:
									(({GPIO_INTTR[14],GPIO_NOTLVL[14]} == 2'b00) & (GPIO_STATUS[14]))?									1'b1:
									(({GPIO_INTTR[14],GPIO_NOTLVL[14]} == 2'b01) & (~GPIO_STATUS[14]))?									1'b1:
									(({GPIO_INTTR[14],GPIO_NOTLVL[14]} == 2'b10) & ({GPIO_STATUS[14],GPIO_OUT[14]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[14],GPIO_NOTLVL[14]} == 2'b11) & ({GPIO_STATUS[14],GPIO_OUT[14]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[15] =	(~GPIO_ENABLED[15])?																									1'b0:
									(({GPIO_INTTR[15],GPIO_NOTLVL[15]} == 2'b00) & (GPIO_STATUS[15]))?									1'b1:
									(({GPIO_INTTR[15],GPIO_NOTLVL[15]} == 2'b01) & (~GPIO_STATUS[15]))?									1'b1:
									(({GPIO_INTTR[15],GPIO_NOTLVL[15]} == 2'b10) & ({GPIO_STATUS[15],GPIO_OUT[15]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[15],GPIO_NOTLVL[15]} == 2'b11) & ({GPIO_STATUS[15],GPIO_OUT[15]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[16] =	(~GPIO_ENABLED[16])?																									1'b0:
									(({GPIO_INTTR[16],GPIO_NOTLVL[16]} == 2'b00) & (GPIO_STATUS[16]))?									1'b1:
									(({GPIO_INTTR[16],GPIO_NOTLVL[16]} == 2'b01) & (~GPIO_STATUS[16]))?									1'b1:
									(({GPIO_INTTR[16],GPIO_NOTLVL[16]} == 2'b10) & ({GPIO_STATUS[16],GPIO_OUT[16]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[16],GPIO_NOTLVL[16]} == 2'b11) & ({GPIO_STATUS[16],GPIO_OUT[16]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[17] =	(~GPIO_ENABLED[17])?																									1'b0:
									(({GPIO_INTTR[17],GPIO_NOTLVL[17]} == 2'b00) & (GPIO_STATUS[17]))?									1'b1:
									(({GPIO_INTTR[17],GPIO_NOTLVL[17]} == 2'b01) & (~GPIO_STATUS[17]))?									1'b1:
									(({GPIO_INTTR[17],GPIO_NOTLVL[17]} == 2'b10) & ({GPIO_STATUS[17],GPIO_OUT[17]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[17],GPIO_NOTLVL[17]} == 2'b11) & ({GPIO_STATUS[17],GPIO_OUT[17]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[18] =	(~GPIO_ENABLED[18])?																									1'b0:
									(({GPIO_INTTR[18],GPIO_NOTLVL[18]} == 2'b00) & (GPIO_STATUS[18]))?									1'b1:
									(({GPIO_INTTR[18],GPIO_NOTLVL[18]} == 2'b01) & (~GPIO_STATUS[18]))?									1'b1:
									(({GPIO_INTTR[18],GPIO_NOTLVL[18]} == 2'b10) & ({GPIO_STATUS[18],GPIO_OUT[18]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[18],GPIO_NOTLVL[18]} == 2'b11) & ({GPIO_STATUS[18],GPIO_OUT[18]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[19] =	(~GPIO_ENABLED[19])?																									1'b0:
									(({GPIO_INTTR[19],GPIO_NOTLVL[19]} == 2'b00) & (GPIO_STATUS[19]))?									1'b1:
									(({GPIO_INTTR[19],GPIO_NOTLVL[19]} == 2'b01) & (~GPIO_STATUS[19]))?									1'b1:
									(({GPIO_INTTR[19],GPIO_NOTLVL[19]} == 2'b10) & ({GPIO_STATUS[19],GPIO_OUT[19]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[19],GPIO_NOTLVL[19]} == 2'b11) & ({GPIO_STATUS[19],GPIO_OUT[19]} == 2'b10))?	1'b1:
																																				1'b0;

	assign GPIO_INT[20] =	(~GPIO_ENABLED[20])?																									1'b0:
									(({GPIO_INTTR[20],GPIO_NOTLVL[20]} == 2'b00) & (GPIO_STATUS[20]))?									1'b1:
									(({GPIO_INTTR[20],GPIO_NOTLVL[20]} == 2'b01) & (~GPIO_STATUS[20]))?									1'b1:
									(({GPIO_INTTR[20],GPIO_NOTLVL[20]} == 2'b10) & ({GPIO_STATUS[20],GPIO_OUT[20]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[20],GPIO_NOTLVL[20]} == 2'b11) & ({GPIO_STATUS[20],GPIO_OUT[20]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[21] =	(~GPIO_ENABLED[21])?																									1'b0:
									(({GPIO_INTTR[21],GPIO_NOTLVL[21]} == 2'b00) & (GPIO_STATUS[21]))?									1'b1:
									(({GPIO_INTTR[21],GPIO_NOTLVL[21]} == 2'b01) & (~GPIO_STATUS[21]))?									1'b1:
									(({GPIO_INTTR[21],GPIO_NOTLVL[21]} == 2'b10) & ({GPIO_STATUS[21],GPIO_OUT[21]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[21],GPIO_NOTLVL[21]} == 2'b11) & ({GPIO_STATUS[21],GPIO_OUT[21]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[22] =	(~GPIO_ENABLED[22])?																									1'b0:
									(({GPIO_INTTR[22],GPIO_NOTLVL[22]} == 2'b00) & (GPIO_STATUS[22]))?									1'b1:
									(({GPIO_INTTR[22],GPIO_NOTLVL[22]} == 2'b01) & (~GPIO_STATUS[22]))?									1'b1:
									(({GPIO_INTTR[22],GPIO_NOTLVL[22]} == 2'b10) & ({GPIO_STATUS[22],GPIO_OUT[22]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[22],GPIO_NOTLVL[22]} == 2'b11) & ({GPIO_STATUS[22],GPIO_OUT[22]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[23] =	(~GPIO_ENABLED[23])?																									1'b0:
									(({GPIO_INTTR[23],GPIO_NOTLVL[23]} == 2'b00) & (GPIO_STATUS[23]))?									1'b1:
									(({GPIO_INTTR[23],GPIO_NOTLVL[23]} == 2'b01) & (~GPIO_STATUS[23]))?									1'b1:
									(({GPIO_INTTR[23],GPIO_NOTLVL[23]} == 2'b10) & ({GPIO_STATUS[23],GPIO_OUT[23]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[23],GPIO_NOTLVL[23]} == 2'b11) & ({GPIO_STATUS[23],GPIO_OUT[23]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[24] =	(~GPIO_ENABLED[24])?																									1'b0:
									(({GPIO_INTTR[24],GPIO_NOTLVL[24]} == 2'b00) & (GPIO_STATUS[24]))?									1'b1:
									(({GPIO_INTTR[24],GPIO_NOTLVL[24]} == 2'b01) & (~GPIO_STATUS[24]))?									1'b1:
									(({GPIO_INTTR[24],GPIO_NOTLVL[24]} == 2'b10) & ({GPIO_STATUS[24],GPIO_OUT[24]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[24],GPIO_NOTLVL[24]} == 2'b11) & ({GPIO_STATUS[24],GPIO_OUT[24]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[25] =	(~GPIO_ENABLED[25])?																									1'b0:
									(({GPIO_INTTR[25],GPIO_NOTLVL[25]} == 2'b00) & (GPIO_STATUS[25]))?									1'b1:
									(({GPIO_INTTR[25],GPIO_NOTLVL[25]} == 2'b01) & (~GPIO_STATUS[25]))?									1'b1:
									(({GPIO_INTTR[25],GPIO_NOTLVL[25]} == 2'b10) & ({GPIO_STATUS[25],GPIO_OUT[25]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[25],GPIO_NOTLVL[25]} == 2'b11) & ({GPIO_STATUS[25],GPIO_OUT[25]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[26] =	(~GPIO_ENABLED[26])?																									1'b0:
									(({GPIO_INTTR[26],GPIO_NOTLVL[26]} == 2'b00) & (GPIO_STATUS[26]))?									1'b1:
									(({GPIO_INTTR[26],GPIO_NOTLVL[26]} == 2'b01) & (~GPIO_STATUS[26]))?									1'b1:
									(({GPIO_INTTR[26],GPIO_NOTLVL[26]} == 2'b10) & ({GPIO_STATUS[26],GPIO_OUT[26]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[26],GPIO_NOTLVL[26]} == 2'b11) & ({GPIO_STATUS[26],GPIO_OUT[26]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[27] =	(~GPIO_ENABLED[27])?																									1'b0:
									(({GPIO_INTTR[27],GPIO_NOTLVL[27]} == 2'b00) & (GPIO_STATUS[27]))?									1'b1:
									(({GPIO_INTTR[27],GPIO_NOTLVL[27]} == 2'b01) & (~GPIO_STATUS[27]))?									1'b1:
									(({GPIO_INTTR[27],GPIO_NOTLVL[27]} == 2'b10) & ({GPIO_STATUS[27],GPIO_OUT[27]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[27],GPIO_NOTLVL[27]} == 2'b11) & ({GPIO_STATUS[27],GPIO_OUT[27]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[28] =	(~GPIO_ENABLED[28])?																									1'b0:
									(({GPIO_INTTR[28],GPIO_NOTLVL[28]} == 2'b00) & (GPIO_STATUS[28]))?									1'b1:
									(({GPIO_INTTR[28],GPIO_NOTLVL[28]} == 2'b01) & (~GPIO_STATUS[28]))?									1'b1:
									(({GPIO_INTTR[28],GPIO_NOTLVL[28]} == 2'b10) & ({GPIO_STATUS[28],GPIO_OUT[28]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[28],GPIO_NOTLVL[28]} == 2'b11) & ({GPIO_STATUS[28],GPIO_OUT[28]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[29] =	(~GPIO_ENABLED[29])?																									1'b0:
									(({GPIO_INTTR[29],GPIO_NOTLVL[29]} == 2'b00) & (GPIO_STATUS[29]))?									1'b1:
									(({GPIO_INTTR[29],GPIO_NOTLVL[29]} == 2'b01) & (~GPIO_STATUS[29]))?									1'b1:
									(({GPIO_INTTR[29],GPIO_NOTLVL[29]} == 2'b10) & ({GPIO_STATUS[29],GPIO_OUT[29]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[29],GPIO_NOTLVL[29]} == 2'b11) & ({GPIO_STATUS[29],GPIO_OUT[29]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[30] =	(~GPIO_ENABLED[30])?																									1'b0:
									(({GPIO_INTTR[30],GPIO_NOTLVL[30]} == 2'b00) & (GPIO_STATUS[30]))?									1'b1:
									(({GPIO_INTTR[30],GPIO_NOTLVL[30]} == 2'b01) & (~GPIO_STATUS[30]))?									1'b1:
									(({GPIO_INTTR[30],GPIO_NOTLVL[30]} == 2'b10) & ({GPIO_STATUS[30],GPIO_OUT[30]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[30],GPIO_NOTLVL[30]} == 2'b11) & ({GPIO_STATUS[30],GPIO_OUT[30]} == 2'b10))?	1'b1:
																																								1'b0;

	assign GPIO_INT[31] =	(~GPIO_ENABLED[31])?																									1'b0:
									(({GPIO_INTTR[31],GPIO_NOTLVL[31]} == 2'b00) & (GPIO_STATUS[31]))?									1'b1:
									(({GPIO_INTTR[31],GPIO_NOTLVL[31]} == 2'b01) & (~GPIO_STATUS[31]))?									1'b1:
									(({GPIO_INTTR[31],GPIO_NOTLVL[31]} == 2'b10) & ({GPIO_STATUS[31],GPIO_OUT[31]} == 2'b01))?	1'b1:
									(({GPIO_INTTR[31],GPIO_NOTLVL[31]} == 2'b11) & ({GPIO_STATUS[31],GPIO_OUT[31]} == 2'b10))?	1'b1:
																																								1'b0;
endmodule
