/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	hazard															*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual		*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module hazard(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input			BRANCH_ID,			/* Branch instruction in Instruction Decode */
	input			CLK,					/* System Clock 50 - 100 MHZ */
	input			CP0REGWRITE_EX,	/* CP0 Register Write Enable */
	input			EXTSTALL,			/* Stall for Extension */
	input			IRQ0,					/* Interrupt */
	input			MEMREAD_EX,			/* Memory Read in Execute Phase */
	input			MEMREAD_MA,			/* Memory Read in Memory Access */
	input			MULT_ID,				/* Multiply Instruction in Instruction Decode */
	input			MULT_RDY,			/* Multiply Unit Ready */
	input			RESET,				/* System Reset */
	input [4:0]	RS_ID,				/* Operand Register 1 in Instruction Decode */
	input [4:0]	RT_ID,				/* Operand Register 2 in Instruction Decode */
	input [4:0]	RTD_EX,				/* Destination Register in Execute Phase */
	input [4:0]	RTD_MA,				/* Destination Register in Memory Access */
	input 		LC_EX,	/* */
	/* OUTPUT PORTS */
	output STALL						/* Pipeline Stall */
	);

/*****Signals****************************************************************/
	
	wire MHZ;				/* Multiply Hazard */
	wire NBSTALL;			/* CP0 & Interrupt Harzard */
	wire RSBHZ_EX;			/* Branch Hazard RS EX */
	wire RSBHZ_MA;			/* Branch Hazard RS MA */
	wire RSBHZ0_MA;		/* Branch Load Hazard RS MA */
	wire RSEQ_EX;			/* RS = RTD_EX */
	wire RSEQ_MA;			/* RS = RTD_MA */
	wire RSEQ0;				/* RS = 0 */
	wire RSEQD_EX;			/* Data Hazard RS EX */
	wire RSEQD_MA;			/* Data Hazard RS MA */
	wire RSLHZ;				/* Load Hazard RS */
	wire RTBHZ_EX;			/* Branch Hazard RT EX */
	wire RTBHZ_MA;			/* Branch Hazard RT MA */
	wire RTBHZ0_MA;		/* Branch Load Hazard RT MA */
	wire RTEQ_EX;			/* RT = RTD_EX */
	wire RTEQ_MA;			/* RT = RTD_MA */
	wire RTEQ0;				/* RT = 0 */
	wire RTEQD_EX;			/* Data Hazard RT EX */
	wire RTEQD_MA;			/* Data Hazard RT MA */
	wire RTLHZ;				/* Load Hazard RT */
	wire RTLCHZ;		/* */
	wire RSLCHZ;		/* */

/*****Registers****************************************************************/

	reg			branch_id_reg;			/* Branch instruction in Instruction Decode */
	reg			cp0regwrite_ex_reg;	/* CP0 Register Write Enable */
	reg			extstall_reg;			/* Stall for Extension */
	reg			irq0_reg;				/* Interrupt */
	reg			memread_ex_reg;		/* Memory Read in Execute Phase */
	reg			memread_ma_reg;		/* Memory Read in Memory Access */
	reg			mult_id_reg;			/* Multiply Instruction in Instruction Decode */
	reg			mult_rdy_reg;			/* Multiply Unit Ready */
	reg [4:0]	rs_id_reg;				/* Operand Register 1 in Instruction Decode */
	reg [4:0]	rt_id_reg;				/* Operand Register 2 in Instruction Decode */
	reg [4:0]	rtd_ex_reg;				/* Destination Register in Execute Phase */
	reg [4:0]	rtd_ma_reg;				/* Destination Register in Memory Access */
	reg			stall_reg;				/* Pipeline Stall */
	reg 			lc_ex_reg;

/*****Initialization****************************************************************/

	initial
	begin
		branch_id_reg = 1'b0;
		cp0regwrite_ex_reg = 1'b0;
		extstall_reg = 1'b0;
		irq0_reg = 1'b0;
		memread_ex_reg = 1'b0;
		memread_ma_reg = 1'b0;
		mult_id_reg = 1'b0;
		mult_rdy_reg = 1'b0;
		rs_id_reg = 1'b0;
		rt_id_reg = 1'b0;
		rtd_ex_reg = 1'b0;
		rtd_ma_reg = 1'b0;
		stall_reg = 1'b0;
		lc_ex_reg = 1'b0;
	end

/*********************************************************************/

	assign STALL	=	stall_reg;

/*****Hazard Detection Logic****************************************************************/

	and2_1	and13(	.A(cp0regwrite_ex_reg),	.B(irq0_reg),			.C(NBSTALL)		);
	and2_1	and0(		.A(mult_id_reg),			.B(~mult_rdy_reg),	.C(MHZ)			);
	and2_1	and9(		.A(RSEQD_EX),				.B(~RSEQ0),				.C(RSEQ_EX)		);
	and2_1	and1(		.A(RSEQ_EX),				.B(memread_ex_reg),	.C(RSLHZ)		);
	and2_1 	and2(		.A(RSEQ_EX),				.B(branch_id_reg),	.C(RSBHZ_EX)	);
	and2_1 	and10(	.A(RSEQD_MA),				.B(~RSEQ0),				.C(RSEQ_MA)		);
	and2_1 	and3(		.A(RSEQ_MA),				.B(memread_ma_reg),	.C(RSBHZ0_MA)	);
	and2_1	and4(		.A(RSBHZ0_MA),				.B(branch_id_reg),	.C(RSBHZ_MA)	);
	and2_1	and11(	.A(RTEQD_EX),				.B(~RTEQ0),				.C(RTEQ_EX)		);
	and2_1	and5(		.A(RTEQ_EX),				.B(memread_ex_reg),	.C(RTLHZ)		);
	and2_1 	and6(		.A(RTEQ_EX),				.B(branch_id_reg),	.C(RTBHZ_EX)	);
	and2_1	and12(	.A(RTEQD_MA),				.B(~RTEQ0),				.C(RTEQ_MA)		);
	and2_1 	and7(		.A(RTEQ_MA),				.B(memread_ma_reg),	.C(RTBHZ0_MA)	);
	and2_1	and8(		.A(RTBHZ0_MA),				.B(branch_id_reg),	.C(RTBHZ_MA)	);

	and2_1	and14(	.A(RTEQ_EX),				.B(lc_ex_reg),		.C(RTLCHZ)			);
	and2_1	and15(	.A(RSEQ_EX),				.B(lc_ex_reg),		.C(RSLCHZ)			);

	equals2_5	eq5(	.IN1(rs_id_reg),	.IN2(5'b0),			.EQ(RSEQ0)		);	
	equals2_5 	eq6(	.IN1(rt_id_reg),	.IN2(5'b0),			.EQ(RTEQ0)		);	
	equals2_5 	eq0(	.IN1(rs_id_reg),	.IN2(rtd_ex_reg),	.EQ(RSEQD_EX)	);
	equals2_5 	eq1(	.IN1(rs_id_reg),	.IN2(rtd_ma_reg),	.EQ(RSEQD_MA)	);
	equals2_5	eq3(	.IN1(rt_id_reg),	.IN2(rtd_ex_reg),	.EQ(RTEQD_EX)	);
	equals2_5	eq4(	.IN1(rt_id_reg),	.IN2(rtd_ma_reg),	.EQ(RTEQD_MA)	);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			branch_id_reg <= 1'b0;
			cp0regwrite_ex_reg <= 1'b0;
			extstall_reg <= 1'b0;
			irq0_reg <= 1'b0;
			memread_ex_reg <= 1'b0;
			memread_ma_reg <= 1'b0;
			mult_id_reg <= 1'b0;
			mult_rdy_reg <= 1'b0;
			rs_id_reg <= 5'b0;
			rt_id_reg <= 5'b0;
			rtd_ex_reg <= 5'b0;
			rtd_ma_reg <= 5'b0;
			lc_ex_reg <= 1'b0;
		end
		else
		begin
			/* Latch Signals */
			branch_id_reg <= BRANCH_ID;
			cp0regwrite_ex_reg <= CP0REGWRITE_EX;
			irq0_reg <= IRQ0;
			extstall_reg <= EXTSTALL;
			memread_ex_reg <= MEMREAD_EX;
			memread_ma_reg <= MEMREAD_MA;
			mult_id_reg <= MULT_ID;
			mult_rdy_reg <= MULT_RDY;
			rs_id_reg <= RS_ID;
			rt_id_reg <= RT_ID;
			rtd_ex_reg <= RTD_EX;
			rtd_ma_reg <= RTD_MA;
			lc_ex_reg <= LC_EX;
		end
	end

	/* Stall Pipeline */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)			stall_reg <= 1'b1;
		else
		begin
			if (MHZ)				stall_reg <= 1'b0;
			else if (RSLCHZ)			stall_reg <= 1'b0;
			else if (RTLCHZ)			stall_reg <= 1'b0;
			else if (RSLHZ)			stall_reg <= 1'b0;
			else if (RSBHZ_EX)		stall_reg <= 1'b0;
			else if (RSBHZ_MA)		stall_reg <= 1'b0;
			else if (RTLHZ)			stall_reg <= 1'b0;
			else if (RTBHZ_EX)		stall_reg <= 1'b0;
			else if (RTBHZ_MA)		stall_reg <= 1'b0;
			else if (NBSTALL)			stall_reg <= 1'b0;
			else if (extstall_reg)	stall_reg <= 1'b0;
			else							stall_reg <= 1'b1;
		end
	end

endmodule
