/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	instruction_decode											*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual		*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"	/* Constant Definitiions */

module instruction_decode(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CLK,			/* System Clock 50 - 100 MHZ */
	input [3:0]		CP,			/* Enabled Coprocessors */
	input 			EN,			/* TISA Enable */
	input 			EN_EXT,		/* Extension Enable */
	input [1:0]		FORWARD_A,	/* Instruction Decode Data Forward Switch Operand A */
	input [1:0]		FORWARD_B,	/* Instruction Decode Data Forward Switch Operand B */
	input [31:0]	INSTR,		/* Instruction Decode Instruction */
	input 			KU,			/* Kernel/User Mode */
	input [31:0]	PC,			/* Instruction Decode Current PC */
	input [31:0]	PC4,			/* Instruction Decode Current PC plus 4*/
	input [31:0]	RDDATA_MA,	/* Data forwarded from Memory Access Phase */
	input [31:0]	RDDATA_WB,	/* Data forwarded from Writeback Phase */
	input [31:0]	REG1DATA,	/* Data from Register Read Port 1 */
	input [31:0]	REG2DATA,	/* Data from Register Read Port 2 */
	input 			RESET,		/* System Reset */
	input 			RI_PA,		/* Pipeline Arbiter Reserved/Recognized Instruction */
	input 			STALL,		/* Pipeline Stall */
	/* OUTPUT PORTS */
	output [31:0]	ALU_A,		/* ALU Operand A */
	output [31:0]	ALU_B,		/* ALU Operand B */
	output [5:0]	ALUOP,		/* ALU Opcode */
	output 			ALUSRC,		/* ALU Operand B Switch */
	output 			BLS,			/* Byte Load/Store */
	output 			BRANCH,		/* Instruction Decode Branch */
	output 			CMOV,			/* Conditional Move */
	output 			COND,			/* Conditional Operation */
	output 			CP0,			/* Coprocessor 0 Operation */
	output 			CPREGWRITE,	/* Coprocessor Register Write Enable */
	output 			EXC,			/* Instruction Decode Exception */
	output [6:0]	EXCCODE,		/* Instruction Decode Exception Code */
	output [31:0]	EXCPC,		/* Instruction Decode Exception PC */
	output 			EXTNOP,		/* Instruction Decode Extension NOP */
	output 			FLUSH,		/* Instruction Decode Flush */
	output 			HLS,			/* Halfword Load/Store */
	output [31:0]	IMM,			/* Data Immediate */
	output [31:0]	JMPADD,		/* Branch Jump Address */
	output 			LC,			/* Load/Store Conditional */
	output 			LINK,			/* Linked Branch */
	output 			LUI,			/* Load Upper Immediate */
	output 			MEMREAD,		/* Memory Read */
	output 			MEMTOREG,	/* Memory to Regiser */
	output 			MEMWRITE,	/* Memory Write */
	output 			MEXT,			/* Sign Extend Memory Data */
	output 			MULT,			/* Instruction Decode Multiplication Unit Operation */
	output 			PCSRC,		/* Branch Address Selection Switch */
	output [4:0]	RD,			/* Destination Register Number */
	output 			REGDST,		/* Destination Register Switch */
	output 			REGWRITE,	/* Register Write */
	output 			RFE,			/* Return From Exception */
	output 			RI,			/* TISA Reserved/Recognized Instruction */
	output 			RNL,			/* Right/Left Unaligned Load/Store */
	output [4:0]	RS,			/* Instruction Decode TISA Operand Register Number 1 */
	output [4:0]	RS0,			/* Raw Operand Register Number 1 */
	output [4:0]	RT,			/* Instruction Decode TISA Operand Register Number 2 */
	output [4:0]	RT0			/* Raw Operand Register Number 2 */
	);
	
/*****Signals****************************************************************/
	
	wire [31:0]	ALU_Aeq;		/* Operand A Data, after Data Forward */
	wire [31:0]	ALU_Beq;		/* Operand B Data, after Data Forward */
	wire [4:0]	BCFUNC;		/* Bi-Conditional Function Code */
	wire			BPCSRC;		/* PCSRC from Branch */
	wire			CDEXC;		/* Condition Exception Decision */
	wire [1:0]	CDSEL;		/* Operand Data Selection Switch */
	wire			CEXC;			/* Conditional Exception */
	wire [31:0]	CMPDATA;		/* Data to Compare with Operand A */
	wire			COMPRZ;		/* Compare with Zero */
	wire			CP1;			/* Coprocessor 1 Operation */
	wire			CP2;			/* Coprocessor 2 Operation */
	wire			CP3;			/* Coprocessor 3 Operation */
	wire [4:0]	CPFUNC;		/* Coprocessor Function Code */
	wire			DEC;			/* Decision, Pass/Fail */
	wire			DEC_0;		/* Decision, Equal, Greaterthan-Equal, Lessthan-Equal */
	wire			DEC_1;		/* Decision, Not-Equal, Greaterthan, Lessthan */
	wire			EQ;			/* Equals */
	wire			EXC_0;		/* Instruction Decode Exception */
	wire			EXC_1;		/* Passed Conditional Exception or Instruction Decode Exception */
	wire			EXC_2;		/* Mask Exception when Pipeline Stalled */
	wire			EXTNOP_0;	/* Mask Extension Flag when Pipeline Stalled */
	wire			EXT;			/* Sign Extend */
	wire			FLUSH_0;		/* Conditional Flush */
	wire			FLUSH_1;		/* Extension Flush or Passed Conditional Flush */
	wire			FLUSH_2;		/* Mask Flush when Pipeline Stalled */
	wire			FLUSH_EN;	/* Flush Enabled Operation */
	wire			FLUSH_EXT;	/* Flush for Extension */
	wire [5:0]	FUNC;			/* Function Code */
	wire			GTT;			/* Greaterthan Test */			
	wire			GL;			/* Greaterthan/Lessthan Operation */
	wire			GLUS;			/* Greaterthan/Lessthan Unsigned */
	wire [31:0]	IMM_0;		/* Sign Extended Data Immediate */
	wire [31:0]	IMMS2; 		/* Data Immediate Shifted to the Left by 2 */
	wire [31:0]	JMPADD_0;	/* Branch Jump Address Asynchronous */
	wire			JPR;			/* Jump Registered */
	wire			JUMP;			/* Jump Operation */
	wire [1:0]	JSEL;			/* Jump Address Selection Swicth */
	wire			LNG;			/* Lessthan/Greaterthan Operation */
	wire			LT;			/* Lessthan Test */
	wire [5:0]	OP;			/* Opcode */
	wire [31:0]	PCRADD;		/* PC4 plus offset */
	wire			PCSRC_0;		/* Branch Address Selection Switch Asynchronous */

/*****Registers****************************************************************/

	reg [31:0]	alu_a_reg;		/* ALU Operand A */
	reg [31:0]	alu_b_reg;		/* ALU Operand B */
	reg [31:0]	cmpdata_reg;	/* Data to Compare with Operand A */
	reg [3:0]	cp_reg;			/* Enabled Coprocessors */
	reg 			en_ext_reg;		/* Extension Enable */
	reg 			en_reg;			/* TISA Enable */
	reg 			exc_reg;			/* Instruction Decode Exception */
	reg 			extnop_reg;		/* Instruction Decode Extension NOP */
	reg 			flush_reg;		/* Instruction Decode Flush */
	reg [1:0]	forward_a_reg;	/* Instruction Decode Data Forward Switch Operand A */
	reg [1:0]	forward_b_reg;	/* Instruction Decode Data Forward Switch Operand B */
	reg [31:0]	imm_reg;			/* Data Immediate */
	reg [31:0]	imms2_reg;		/* Data Immediate Shifted to the Left by 2 */
	reg [31:0]	instr_reg;		/* Instruction Decode Instruction */
	reg [31:0]	jmpadd_reg;		/* Branch Jump Address */
	reg 			ku_reg;			/* Kernel/User Mode */
	reg [31:0]	pc_reg;			/* Instruction Decode Current PC */
	reg [31:0]	pc4_reg;			/* Instruction Decode Current PC plus 4 */
	reg [31:0]	pcradd_reg;		/* PC4 plus offset */
	reg 			pcsrc_reg;		/* Branch Address Selection Switch */
	reg [31:0]	rddata_ma_reg;	/* Data forwarded from Memory Access Phase */
	reg [31:0]	rddata_wb_reg;	/* Data forwarded from Writeback Phase */
	reg [31:0]	reg1data_reg;	/* Data from Register Read Port 1 */
	reg [31:0]	reg2data_reg;	/* Data from Register Read Port 2 */
	reg 			ri_pa_reg;		/* Pipeline Arbiter Reserved/Recognized Instruction */
	reg 			stall_reg;		/* Pipeline Stall */
	
/*****Initialization****************************************************************/

	initial
	begin
		alu_a_reg = 32'b0;
		alu_b_reg = 32'b0;
		cp_reg = 4'b0;
		cmpdata_reg = 32'b0;
		en_ext_reg = 1'b0;
		en_reg = 1'b0;
		exc_reg = 1'b0;
		extnop_reg = 1'b0;
		flush_reg = 1'b0;
		forward_a_reg = 2'b0;
		forward_b_reg = 2'b0;
		instr_reg = 32'b0;
		imm_reg = 32'b0;
		imms2_reg = 32'b0;
		jmpadd_reg = 32'b0;
		ku_reg = 1'b0;
		pc_reg = 32'b0;
		pc4_reg = 32'b0;
		pcradd_reg = 32'b0;
		pcsrc_reg = 1'b0;
		rddata_ma_reg = 32'b0;
		rddata_wb_reg = 32'b0;
		reg1data_reg = 32'b0;
		reg2data_reg = 32'b0;
		ri_pa_reg = 1'b0;
		stall_reg = 1'b1;
	end

/*********************************************************************/

	assign ALU_A	=	alu_a_reg;
	assign ALU_B	=	alu_b_reg;
	assign BCFUNC	=	instr_reg[20:16];
	assign CPFUNC	=	instr_reg[25:21];
	assign EXC		=	exc_reg;
	assign EXCPC	=	pc_reg;
	assign EXTNOP	=	extnop_reg;
	assign FLUSH	=	flush_reg;
	assign IMM		=	imm_reg;
	assign JMPADD	=	jmpadd_reg;
	assign FUNC		=	instr_reg[5:0];
	assign OP		=	instr_reg[31:26];
	assign PCSRC	=	pcsrc_reg;
	assign RD		=	(en_reg == 1'b0)?	`BLANK5:	instr_reg[15:11];
	assign RS		=	(en_reg == 1'b0)?	`BLANK5:	RS0;
	assign RS0		=	instr_reg[25:21];
	assign RT		=	(en_reg == 1'b0)?	`BLANK5:	RT0;
	assign RT0		=	instr_reg[20:16];

/*****Calculate PC4 plus offset****************************************************************/

	/* Sign Extend data immediate */
	signextend16_32 ext16_32(
		.IN(instr_reg[15:0]),
		.SE_OP(EXT),
		.OUT(IMM_0)
		);
		
	/* Shift data immediate Left by 2*/
	shift2 shift0(
		.A(imm_reg),
		.B(IMMS2),
		.C_IN(2'b0)
		);

	/* Calculate PC4 plus offset */
	adder32 add0(
		.A(pc4_reg),
		.B(imms2_reg),
		.C(PCRADD),
		.C_IN(1'b0)
		);

/*****Select Jump Address****************************************************************/
	
	/* Jump Address Selection Encoder */
	jmpadddecoder jdec(
		.JUMP(JUMP),
		.JPR(JPR),
		.JSEL(JSEL)
		);

	MUX32_4to1 mux32_4_3(
		.a0(pcradd_reg),
		.a1({pc4_reg[31:28],instr_reg[25:0],2'b0}),	
		.a2(alu_a_reg),
		.a3(32'b0),
		.sel(JSEL),
		.out(JMPADD_0)
		);

/*****Select Operand Data****************************************************************/

	/* Data Forward Operand A */
	MUX32_4to1 mux32_4_0(
		.a0(reg1data_reg),
		.a1(rddata_wb_reg),
		.a2(rddata_ma_reg),
		.a3(32'b0),
		.sel(forward_a_reg),
		.out(ALU_Aeq)
		);

	/* Data Forward Operand B */
	MUX32_4to1 mux32_4_1(
		.a0(reg2data_reg),
		.a1(rddata_wb_reg),
		.a2(rddata_ma_reg),
		.a3(32'b0),
		.sel(forward_b_reg),
		.out(ALU_Beq)
		);

	/* Operand Data Selection Encoder */
	compdatadecoder cddec(
		.COMPRZ(COMPRZ),
		.ALUSRC(ALUSRC),
		.CDSEL(CDSEL)
		);

	MUX32_4to1 	mux32_4_4(
		.a0(alu_b_reg),
		.a1(32'b0),												
		.a2(imm_reg),
		.a3(32'b0),
		.sel(CDSEL),
		.out(CMPDATA)
		);

/*****Perform Branch/Exception Test****************************************************************/

	/* Test Operands for Equality */
	equals2_32	eq0(
		.IN1(alu_a_reg),
		.IN2(cmpdata_reg),
		.EQ(EQ)
		);

	/* Test Operands for Greater Than */
	greater2_32 gt0(
		.IN1(alu_a_reg),
		.IN2(cmpdata_reg),
		.US(GLUS),
		.GTT(GTT)
		);

	/* Test Operands for Less Than */
	less2_32 lt0(
		.IN1(alu_a_reg),
		.IN2(cmpdata_reg),
		.US(GLUS),
		.LT(LT)
		);

/*****Evaluate Branch/Exception Test****************************************************************/

	/* Equal or Not Equal Test */
	MUX1_2to1 mux1_2_0(
		.a0(DEC_0),
		.a1(DEC_1),
		.sel(COND),
		.out(DEC)
		);

	/* Equal, Greaterthan-Equal, or Lessthan-Equal */
	MUX1_4to1 mux1_4_0(
		.a0(EQ),
		.a1(EQ),
		.a2((GTT | EQ)),
		.a3((LT | EQ)),	
		.sel({GL,LNG}),
		.out(DEC_0)	
		);

	/* Not-Equal, Greaterthan, or Lessthan */
	MUX1_4to1 mux1_4_1(
		.a0(~EQ),
		.a1(~EQ),
		.a2(GTT),
		.a3(LT),				
		.sel({GL,LNG}),
		.out(DEC_1)
		);

	/* Conditional Branch */
	and2_1	and0(
		.A(BRANCH),
		.B(DEC),
		.C(BPCSRC)
		);

	/* Update PC with Jump Address if Branch or Jump */
	or2_1		or0(
		.A(BPCSRC),
		.B(JUMP),
		.C(PCSRC_0)
		);

	/* Conditional Exception */
	and2_1	and2(
		.A(CEXC),
		.B(DEC),
		.C(CDEXC)
		);

	/* Passed Conditional Exception or Instruction Decode Exception */
	or2_1		or1(
		.A(CDEXC),
		.B(EXC_0),
		.C(EXC_1)
		);

	/* Mask Exception when Pipeline Stalled */
	and2_1	and3(
		.A(EXC_1),
		.B(stall_reg),
		.C(EXC_2)
		);

	/* Conditional Flush */
	and2_1 	and1(
		.A(FLUSH_EN),
		.B(~DEC),
		.C(FLUSH_0)
		);

	/* Extension Flush or Passed Conditional Flush */
	or2_1 	or2(
		.A(FLUSH_EXT),
		.B(FLUSH_0),
		.C(FLUSH_1)	
		);

	/* Mask Flush when Pipeline Stalled */
	and2_1 	and4(
		.A(FLUSH_1),
		.B(stall_reg),
		.C(FLUSH_2)
		);

	/* Mask Extension Flag when Pipeline Stalled */
	and2_1 	and5(
		.A(FLUSH_EXT),
		.B(stall_reg),
		.C(EXTNOP_0)
		);

/*****Data Path Control****************************************************************/

	datapathcontrol dpctr(
		.ALUOP(ALUOP),
		.ALUSRC(ALUSRC),
		.BCFUNC(BCFUNC),
		.BLS(BLS),
		.BRANCH(BRANCH),
		.CEXC(CEXC),
		.CLK(CLK),
		.CMOV(CMOV),
		.COMPRZ(COMPRZ),
		.CP(cp_reg),
		.CP1(CP1),
		.CP2(CP2),
		.CP3(CP3),
		.CP0(CP0),
		.CPFUNC(CPFUNC),
		.CPREGWRITE(CPREGWRITE),
		.COND(COND),
		.EN(en_reg),
		.EN_EXT(en_ext_reg),
		.EXC(EXC_0),
		.EXT(EXT),
		.FLUSH(FLUSH_EN),
		.FLUSH_EXT(FLUSH_EXT),
		.FUNC(FUNC),
		.GL(GL),
		.GLUS(GLUS),
		.HLS(HLS),
		.JPR(JPR),
		.JUMP(JUMP),
		.LC(LC),
		.LINK(LINK),
		.LNG(LNG),
		.LUI(LUI),
		.KU(ku_reg),
		.MEMREAD(MEMREAD),
		.MEMTOREG(MEMTOREG),
		.MEMWRITE(MEMWRITE),
		.MEXT(MEXT),
		.MULT(MULT),
		.OP(OP),
		.REGDST(REGDST),
		.REGWRITE(REGWRITE),
		.RESET(RESET),
		.RFE(RFE),
		.RI(RI),
		.RI_PA(RI_PA),
		.RNL(RNL)
		);

/*****Exception Encoder****************************************************************/

	exceptionencoder exen(
		.BCFUNC(BCFUNC),
		.CLK(CLK),
		.CP(cp_reg),
		.CP0(CP0),
		.CP1(CP1),
		.CP2(CP2),
		.CP3(CP3),
		.EXC(exc_reg),
		.EXCCODE(EXCCODE),
		.FUNC(FUNC),
		.KU(ku_reg),
		.OP(OP),
		.RESET(RESET),
		.RI(RI_PA)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			alu_a_reg <= 32'b0;
			alu_b_reg <= 32'b0;
			cp_reg <= 4'b0;
			cmpdata_reg <= 32'b0;
			en_ext_reg <= 1'b0;
			en_reg <= 1'b0;
			exc_reg <= 1'b0;
			extnop_reg <= 1'b0;
			flush_reg <= 1'b0;
			forward_a_reg <= 2'b0;
			forward_b_reg <= 2'b0;
			imm_reg <= 32'b0;
			imms2_reg <= 32'b0;
			instr_reg <= 32'b0;
			jmpadd_reg <= 32'b0;
			ku_reg <= 1'b0;
			pc_reg <= 32'b0;
			pc4_reg <= 32'b0;
			pcradd_reg <= 32'b0;
			pcsrc_reg <= 1'b0;
			rddata_ma_reg <= 32'b0;
			rddata_wb_reg <= 32'b0;
			reg1data_reg <= 32'b0;
			reg2data_reg <= 32'b0;
			ri_pa_reg <= 1'b0;
			stall_reg <= 1'b1;
		end
		else
		begin
			/* Latch Signals */
			alu_a_reg <= ALU_Aeq;
			alu_b_reg <= ALU_Beq;
			cmpdata_reg <= CMPDATA;
			cp_reg <= CP;
			en_ext_reg <= EN_EXT;
			en_reg <= EN;
			exc_reg <= EXC_2;
			extnop_reg <= EXTNOP_0;
			flush_reg <= FLUSH_2;
			forward_a_reg <= FORWARD_A;
			forward_b_reg <= FORWARD_B;
			imm_reg <= IMM_0;
			imms2_reg <= IMMS2;
			instr_reg <= INSTR;
			jmpadd_reg <= JMPADD_0;
			ku_reg <= KU;
			pc_reg <= PC;
			pc4_reg <= PC4;
			pcradd_reg <= PCRADD;
			pcsrc_reg <= PCSRC_0;
			rddata_ma_reg <= RDDATA_MA;
			rddata_wb_reg <= RDDATA_WB;
			reg1data_reg <= REG1DATA;
			reg2data_reg <= REG2DATA;
			ri_pa_reg <= RI_PA;
			stall_reg <= STALL;
		end
	end
	
endmodule

/*****Jump Address Selection Encoder****************************************************************/

module jmpadddecoder(
	input 			JPR,	/* Jump Registered */
	input 			JUMP,	/* Jump Operation */
	output [1:0]	JSEL	/* Jump Address Selection Swicth */
	);

/*********************************************************************/

	assign JSEL	=	((JUMP == 1'b1) && (JPR == 1'b1))?	2'b10:
						(JUMP == 1'b1)?							2'b01:
																		2'b00;
endmodule

/*****Exception Encoder****************************************************************/

module exceptionencoder(
	input [4:0]		BCFUNC,	/* Bi-Conditional Function Code */
	input 			CLK,		/* System Clock 50 - 100 MHZ */ 
	input [3:0]		CP,		/* Enabled Coprocessors */
	input 			CP0,		/* Coprocessor 0 Operation */
	input 			CP1,		/* Coprocessor 1 Operation */
	input 			CP2,		/* Coprocessor 2 Operation */
	input 			CP3,		/* Coprocessor 3 Operation */
	input 			EXC,		/* Instruction Decode Exception */
	input [5:0]		FUNC,		/* Function Code */
	input 			KU,		/* Kernel/User Mode */
	input [5:0]		OP,		/* Opcode */
	input 			RESET,	/* System Reset */
	input 			RI,		/* Reserved/Recognized Instruction */
	output [6:0]	EXCCODE	/* Instruction Decode Exception Code */
	);

/*****Registers****************************************************************/
	
	reg [6:0] exccode_reg;

/*****Initialization****************************************************************/

	initial
	begin
		exccode_reg = 7'b0;
	end
	
/*********************************************************************/

	assign EXCCODE	=	exccode_reg;

/*********************************************************************/
											
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			exccode_reg <= 7'b0;
		end
		else
		begin
			if (EXC == 1'b0)									exccode_reg <= 7'b0;
			else
			begin
				if (RI == 1'b1)								exccode_reg <= `RIc;		/* Reserved Instruction Exception */
				else if (OP == `SPEC)
				begin
					if (FUNC == `SYSCALL)					exccode_reg <= `SYSc;		/* System Call Exception */
					else if (FUNC == `BREAK)				exccode_reg <= `Bpc;		/* Break Point Exception */
					else if (FUNC == `TGE)					exccode_reg <= `Trc;		/* Trap Exception, Greaterthan-Equal */
					else if (FUNC == `TGEU)					exccode_reg <= `Trc;		/* Trap Exception, Greaterthan-Equal Unsigned */
					else if (FUNC == `TLT)					exccode_reg <= `Trc;		/* Trap Exception, Lessthan */
					else if (FUNC == `TLTU)					exccode_reg <= `Trc;		/* Trap Exception, Lessthan Unsigned*/
					else if (FUNC == `TEQ)					exccode_reg <= `Trc;		/* Trap Exception, Equal */
					else if (FUNC == `TNE)					exccode_reg <= `Trc;		/* Trap Exception, Not-Equal */
					else											exccode_reg <= 7'b0;	
				end
				else if (OP == `BCOND)
				begin
					if (BCFUNC == `TGEI)						exccode_reg <= `Trc;		/* Trap Exception, Greaterthan-Equal Immediate */
					else if (BCFUNC == `TGEIU)				exccode_reg <= `Trc;		/* Trap Exception, Greaterthan-Equal Immediate Unsigned */
					else if (BCFUNC == `TLTI)				exccode_reg <= `Trc;		/* Trap Exception, Lessthan Immediate */
					else if (BCFUNC == `TLTIU)				exccode_reg <= `Trc;		/* Trap Exception, Lessthan Immediate Unsigned */
					else if (BCFUNC == `TEQI)				exccode_reg <= `Trc;		/* Trap Exception, Equal Immediate */
					else if (BCFUNC == `TNEI)				exccode_reg <= `Trc;		/* Trap Exception, Not-Equal Immediate */
					else											exccode_reg <= 7'b0;
				end
				else if ((~CP[0] & CP0 & KU) == 1'b1)	exccode_reg <= `CpU0c;	/* Coprocess 0 Exception */
				else if ((~CP[1] & CP1 & KU) == 1'b1)	exccode_reg <= `CpU1c;	/* Coprocess 1 Exception */
				else if ((~CP[2] & CP2 & KU) == 1'b1)	exccode_reg <= `CpU2c;	/* Coprocess 2 Exception */
				else if ((~CP[3] & CP3 & KU) == 1'b1)	exccode_reg <= `CpU3c;	/* Coprocess 3 Exception */
				else												exccode_reg <= 7'b0;
			end
		end
	end
				
endmodule

/*****Operand Selection Encoder****************************************************************/

module compdatadecoder(
	input				ALUSRC,	/* ALU Operand B Switch */
	input				COMPRZ,	/* Compare with Zero */
	output [1:0]	CDSEL		/* Operand Data Selection Switch */
	);

/*********************************************************************/
	
	assign CDSEL	=	(COMPRZ == 1'b1)?	2'b01:
							(ALUSRC == 1'b1)?	2'b10:
													2'b00;

endmodule
