/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	instruction_fetch												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual		*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"	/* Constant Definitiions */

module instruction_fetch(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CLK,				/* System Clock 50 - 100 MHZ */
	/*input 			EDF,				/* Reverse Endian */
	input 			EN_ID,			/* TISA Enable */
	input				EN_EXT,			/* Extension Enable */
	input [31:0]	EXCADDR,			/* Exception Jump Address */
	input 			EXCE,				/* Exception Flush */
	input [31:0]	EXTADD_EXT,		/* Extension Jump Address */
	input 			GR_EXT,			/* Extension Grant Pipeline Resources */
	input [31:0]	INSTR_EDF,		/* Data from Memory */
	input [31:0]	JMPADD,			/* Branch Jump Address */
	input 			KU,				/* Kernel/User Mode */
	input 			PCLK,				/* Pipeline Clock */
	input 			PCNEXT_EXT,		/* Extension Conditional PC Update */
	input 			PCSRC,			/* Branch Address Selection Switch */
	input 			RESET,			/* Reset Instruction Fetch */
	input 			STALL,			/* Pipeline Stall */
	/* OUTPUT PORTS */
	output 			EXC,				/* Exception */
	output [6:0]	EXCCODE,			/* Exception Code */
	output [31:0]	INSTR,			/* Fetched Instruction */
	output [31:0]	PC,				/* Current PC */
	output [31:0]	PC4,				/* Currect PC plus 4 */
	output [31:0]	PPC				/* Current PC to Memory */
	);

/*****Signals****************************************************************/
	
	wire [1:0]	ADDSEL;		/* Next PC Selection Switch */
	wire 			ADEL;			/* Address Error Exception: Unaligned Address Load */
	/*wire [31:0]	INSTR_0;		/* Instruction Data Asynchronous */
	wire [31:0]	PC_NEXT;		/* Next PC */
	wire [31:0]	PC4_0;		/* Current PC plus 4 Asynchronous */
	wire			PCWRITE;		/* PC Write Enable */
	/*wire [31:0]	PPC_0;		/* Current PC to Memory Asynchronous */

/*****Registers****************************************************************/

	/*reg 			edf_reg;				/* Reverse Endian */
	reg 			en_ext_reg;			/* Extension Enable */
	reg 			en_id_reg;			/* TISA Enable */
	reg [31:0]	excaddr_reg;		/* Exception Jump Address */
	reg 			exce_reg;			/* Exception Flush */
	reg [31:0]	extadd_reg;			/* Extension Jump Address */
	reg 			gr_ext_reg;			/* Extension Grant Pipeline Resources */
	reg [31:0]	instr_edf_reg;		/* Data from Memory */
	/*reg [31:0]	instr_reg;			/* Fetched Instruction */
	reg [31:0]	jmpadd_reg;			/* Branch Jump Address */
	reg 			ku_reg;				/* Kernel/User Mode */
	reg [31:0]	pc_reg;				/* Current PC */
	reg [31:0]	pc4_reg;				/* Currect PC plus 4 */
	reg [1:0]	pclkcnt;				/* Pipeline Clock edge detection */
	reg 			pcnext_ext_reg;	/* Extension Conditional PC Update */
	reg 			pcsrc_reg;			/* Branch Address Selection Switch */
	/*reg [31:0]	ppc_reg;				/* Current PC to Memory */
	reg 			stall_reg;			/* Pipeline Stall */

/*****Initialization****************************************************************/

	initial
	begin
		/*edf_reg = 1'b0;*/
		en_ext_reg = 1'b0;
		en_id_reg = 1'b0;
		excaddr_reg = `BLANK32;
		exce_reg = 1'b0;
		extadd_reg = `BLANK32;
		gr_ext_reg = 1'b0;
		/*instr_reg = `BLANK32;*/
		instr_edf_reg = `BLANK32;
		jmpadd_reg = `BLANK32;
		ku_reg = 1'b0;
		pc_reg = `BLANK32;
		pc4_reg = `BLANK32;
		pclkcnt = 2'b0;
		pcnext_ext_reg = 1'b0;
		pcsrc_reg = 1'b0;
		/*ppc_reg = `BLANK32;*/
		stall_reg = 1'b1;
	end

/*********************************************************************/

	assign ADEL			=	(~(pc_reg[1:0] == 2'b0))?							1'b1:		
								((ku_reg == 1'b1) && (pc_reg[31] == 1'b1))?	1'b1:
																							1'b0;
	assign INSTR 		=	instr_edf_reg;
	assign PC 			=	pc_reg;
	assign PC4 			=	pc4_reg;
	assign PCWRITE 	=	stall_reg | exce_reg | pcnext_ext_reg;
	assign PPC 			=	pc_reg;

/*****Exception Encoder****************************************************************/

	excencoder exc(
		.ADEL(ADEL),
		.CLK(CLK),
		.EXCCODE(EXCCODE),
		.EXCE(EXC),
		.RESET(RESET)
		);

/*****Address & Data Translation****************************************************************/

	/* Needs to be moved to the Memory Interface and replaced with TLB */
	/*address_translation at(
		.BLS(1'b0),
		.HLS(1'b0),
		.MEMREAD(1'b1),
		.PADDR(PPC_0),
		.VADDR(pc_reg)
		);*/
	//assign PPC_0 = pc_reg;

	/* Needs to be moved to the Memory Interface */
	/*endianflip32 edfint(
		.IN(instr_edf_reg),
		.EN(edf_reg),
		.OUT(INSTR_0)
		);*/
	//assign INSTR_0 = instr_edf_reg;

/*****Calculate PC plus 4****************************************************************/

	adder32 add0(
		.A(pc_reg),
		.B(32'h00000004),
		.C(PC4_0),
		.C_IN(1'b0)
		);

/*****Next PC Selection ****************************************************************/

	addencoder add(
		.ADDSEL(ADDSEL),
		.EN_EXT(en_ext_reg),
		.EN_ID(en_id_reg),
		.EXCE(exce_reg),
		.GR_EXT(gr_ext_reg),
		.PCNEXT_EXT(pcnext_ext_reg),
		.PCSRC(pcsrc_reg)
		);

	MUX32_4to1 mux0(
		.a0(pc4_reg),
		.a1(jmpadd_reg),
		.a2(excaddr_reg),
		.a3(extadd_reg),
		.sel(ADDSEL),
		.out(PC_NEXT)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			/*edf_reg <= 1'b0;*/
			en_ext_reg <= 1'b0;
			en_id_reg <= 1'b0;
			excaddr_reg <= `BLANK32;
			exce_reg <= 1'b0;
			extadd_reg <= `BLANK32;
			gr_ext_reg <= 1'b0;
			/*instr_reg <= `BLANK32;*/
			instr_edf_reg <= `BLANK32;
			jmpadd_reg <= `BLANK32;
			ku_reg <= 1'b0;
			pc4_reg <= `BLANK32;
			pcnext_ext_reg <= 1'b0;
			pcsrc_reg <= 1'b0;
			/*ppc_reg <= `BLANK32;*/
			stall_reg <= 1'b1;
		end
		else
		begin
			/* Latch Signals */
			/*edf_reg <= EDF;*/
			en_ext_reg <= EN_EXT;
			en_id_reg <= EN_ID;
			excaddr_reg <= EXCADDR;
			exce_reg <= EXCE;
			extadd_reg <= EXTADD_EXT;
			gr_ext_reg <= GR_EXT;
			/*instr_reg <= INSTR_0;*/
			instr_edf_reg <= INSTR_EDF;
			jmpadd_reg <= JMPADD;
			ku_reg <= KU;
			pc4_reg <= PC4_0;
			pcnext_ext_reg <= PCNEXT_EXT;
			pcsrc_reg <= PCSRC;
			/*ppc_reg <= PPC_0;*/
			stall_reg <= STALL;
		end
	end
	
	always@(posedge CLK)
	begin
		casex({pclkcnt,RESET,PCWRITE})
			4'bxx0x	:	begin
								/* Reset */
								pc_reg <= `BLANK32;
							end
			4'b0111	:	begin
								/* if PC Write Enabled, */
								/* Update PC on Posedge of Pipeline Clock */
								pc_reg <= PC_NEXT;
							end
			default	:	begin
								/* NOP */
							end
		endcase
	end

endmodule

/*****PC Selection Encoder****************************************************************/

module addencoder(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			EN_EXT,			/* Extension Enable */
	input 			EN_ID,			/* TISA Enable */
	input 			EXCE,				/* Exception Flush */
	input 			GR_EXT,			/* Extension Grant Pipeline Resources */
	input 			PCNEXT_EXT,		/* Extension Conditional PC Update */
	input 			PCSRC,			/* Branch Address Selection Switch */
	/* OUTPUT PORTS */
	output [1:0]	ADDSEL			/* Next PC Selection Switch */
	);

/*********************************************************************/
	
	assign ADDSEL	=	(EXCE)?						2'b10:
							(GR_EXT & PCNEXT_EXT)?	2'b11:
							(~EN_ID & EN_EXT)?		2'b11:
							(EN_ID & PCSRC)?			2'b01:
															2'b00;

endmodule

/*****Exception Encoder****************************************************************/

module excencoder(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			ADEL,		/* Address Error Exception: Unaligned Address Load */
	input 			CLK,		/* System Clock 50 - 100 MHZ */
	input 			RESET,	/* Reset Instruction Fetch */
	/* OUTPUT PORTS */
	output [6:0]	EXCCODE,	/* Exception Code */
	output 			EXCE		/* Exception */
	);

/*****Registers****************************************************************/
	
	reg [6:0]	exccode_reg;	/* Exception Code */
	reg 			exce_reg;		/* Exception */

/*****Initialization****************************************************************/

	initial
	begin
		exccode_reg = 7'b0;
		exce_reg = 1'b0;
	end

/*********************************************************************/

	assign EXCCODE	=	exccode_reg;
	assign EXCE 	=	exce_reg;

/*********************************************************************/
	
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			exce_reg <= 1'b0;
			exccode_reg <= 7'b0;
		end
		else
		begin
			if (ADEL)
			begin
				/* Address Error Exception: Unaligned Address Load */
				exce_reg <= 1'b1;
				exccode_reg <= `AdELc;
			end
			else
			begin
				/* No Exception */
				exce_reg <= 1'b0;
				exccode_reg <= 7'b0;
			end
		end
	end
	
endmodule
