/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	interrupt_controller											*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module interrupt_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */ 
	input [31:0]	IRQ,			/* Interrupt Bus */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				SRT,			/* Start Transaction */
	input				WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output			AIRQ,			/* Hardware Interrupt */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output [31:0]	DATA_OUT,	/* Data Bus to Data Path Memory Interface */
	output			DNE,			/* Transaction Done */
	output [31:0]	STATUS		/* Status Register */
	);

/*****Signals****************************************************************/
	
	wire [3:0]	BM;			/* Byte Selects */
	wire [31:0]	DATA_IN_C;	/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;	/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;	/* Endian Flipped Outgoing Data */
	wire			MEN;			/* Transaction Enabled for IRQ Configuration Space */
	wire [31:0]	REG_M;		/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;		/* Selected Configuration Register Data */
	wire [31:0]	WMASK_REG;	/* Incoming Data Write Mask */
	wire [31:0]	WMASK_IN;	/* Configuration Register Data Write Mask */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;			/* BASETAG Register */
	reg [31:0]	data_in_m_reg;		/* Masked Incoming Data */
	reg [3:0]	dnecnt;				/* Transaction Progress Counter */
	reg			dner;					/* Configuration Space Transaction Done */
	reg [31:0]	irqenable;			/* IRQ Enabled Register */
	reg [31:0]	irqenableclear;	/* IRQ Enable Clear Register */
	reg [31:0]	irqenableset;		/* IRQ Enable Set Register */
	reg [31:0]	irqrawstatus;		/* IRQ Raw Status Register */
	reg [31:0]	irqsoft;				/* IRQ Soft Register */
	reg [31:0]	irqstatus;			/* IRQ Status Register */
	reg			men_reg;				/* Transaction Enabled for IRQ Configuration Space */
	reg [31:0]	reg_m_0_reg;		/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;			/* Masked Configuration Register Data */

/*****Variables****************************************************************/

	integer i;

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter	BASETAGREG =				16'h0000;	/* BASETAG Register Relative Address */
	parameter	IRQSTATUSREG =				16'h0004;	/* IRQ Status Register Relative Address */
	parameter	IRQRAWSTATUSREG =			16'h0008;	/* IRQ Raw Status Register Relative Address */
	parameter	IRQENABLESETREG = 		16'h000C;	/* IRQ Enable Set Register Relative Address */
	parameter	IRQENABLECLEARREG =		16'h0010;	/* IRQ Enable Clear Register Relative Address */
	parameter	IRQSOFTREG =				16'h0014;	/* IRQ Soft Register Relative Address */

	/* BASETAG Register */
	parameter	BASEl =						31;			/* BASE Bit Range Left */
	parameter	BASEr =						16;			/* BASE Bit Range Right */
	parameter	TAGl = 						15;			/* TAG Bit Range Left */
	parameter	TAGr = 						0;				/* TAG Bit Range Right */
	parameter	TAG =							5;				/* TAG Value */

	/* IRQ Soft Register */
	parameter	irqprog =			 		1;				/* Software Interrupt */
	parameter	irqrst = 					0;				/* Soft Reset */

/*****Initialization****************************************************************/

	initial
	begin
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		dnecnt = 4'b1111;
		dner = 1'b1;
		irqenable = 32'b0;
		irqenableclear = 32'b0;
		irqenableset = 32'b0;
		irqrawstatus = 32'b0;
		irqstatus = 32'b0;
		irqsoft = 32'b0;
		men_reg = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
	end

/*********************************************************************/

	assign BASETAG = basetagr;
	assign DNE = dner;
	assign STATUS = irqrawstatus;
	assign AIRQ = ((~RESET)&&(irqsoft[irqrst]))?1'b0:(| irqstatus);
	assign MEN = (CADDR[BASEl:BASEr] == ADDR_IN[BASEl:BASEr]);

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/

	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BASETAGREG)?basetagr:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == IRQSTATUSREG)?irqstatus:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == IRQRAWSTATUSREG)?irqrawstatus:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == IRQENABLESETREG))?irqenableset:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == IRQENABLESETREG))?irqenable:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == IRQENABLECLEARREG))?irqenableclear:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == IRQENABLECLEARREG))?irqenable:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == IRQSOFTREG)?irqsoft:32'bz;
	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;

/*****Byte Selects****************************************************************/
	
	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,irqsoft[irqrst],dnecnt[3],dner})
			4'b0xxx	:	begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								dner = 1'b1;
								irqenable = 32'b0;
								irqenableclear = 32'b0;
								irqenableset = 32'b0;
								irqrawstatus = 32'b0;
								irqstatus = 32'b0;
								irqsoft = 32'b0;
							end
			4'b11xx	:	begin
								/* Software Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								dner = 1'b1;
								irqenable = 32'b0;
								irqenableclear = 32'b0;
								irqenableset = 32'b0;
								irqrawstatus = 32'b0;
								irqstatus = 32'b0;
								irqsoft = 32'b0;
							end
			4'b1000	:	begin
								/* Update Configuration Registers */
								dner = 1'b1;
								if (men_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										BASETAGREG			:	begin
																		/* Update Basetag Register */
																		basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
																	end
										IRQENABLESETREG	:	begin
																		/* Update IRQ Enable Set */
																		irqenableset = data_in_m_reg | reg_m_reg;
																	end
										IRQENABLECLEARREG	:	begin
																		/* Update IRQ Enable Clear */
																		irqenableclear = data_in_m_reg | reg_m_reg;
																	end
										IRQSOFTREG			:	begin
																		/* Update IRQ Soft */
																		irqsoft = data_in_m_reg | reg_m_reg;
																	end
										default				:	begin
																		/* Do Nothing */
																		/* Address not Recognized */
																	end	
									endcase
								end
							end
			4'b10x1	:	begin
								/* Wait for incoming request */
								if (SRT & men_reg)
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
								/* Update IRQ Enabled */
								for (i=0; i<32; i=i+1)
								begin
									if (irqenableset[i]) irqenable[i] = 1'b1;
								end
								irqenableset = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (irqenableclear[i]) irqenable[i] = 1'b0;
								end
								irqenableclear = 32'b0;
								/* Update IRQ RAW Status */
								for (i=0; i<32; i=i+1)
								begin
									if (IRQ[i]) irqrawstatus[i] = 1'b1;
									else irqrawstatus[i] = 1'b0;
								end
								if (irqsoft[irqprog]) irqrawstatus[irqprog] = 1'b1;
								else irqrawstatus[irqprog] = 1'b0;
								/* Update IRQ Status */
								irqstatus = (irqrawstatus & irqenable);
							end
			default	:	begin
								/* Do Nothing */
							end
		endcase
	end

endmodule
