 /* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	memory access													*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"		/* Constant Definitiions */

module memory_access(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BLS,					/* Byte Load/Store */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				EDF,					/* Reverse Endian */
	input				EXCE,					/* Exception Flush */
	input				HLS,					/* Halfword Load/Store */
	input				KU,					/* Kernel/User Mode */
	input				LC,					/* Load/Store Conditional */
	input [31:0]	MEMDATAIN,			/* Data in to Memory */
	input [31:0]	MEMDATAOUT_EDF,	/* Data in form Memory */
	input				MEMREAD, 			/* Memory Read */
	input				MEMWRITE,			/* Memory Write */
	input				MEXT,					/* Sign Extend Memory Data */
	input				PCLK,					/* Pipeline Clock */
	input				RESET,				/* System Reset */
	input				RNL,					/* Right/Left Unaligned Load/Store */
	input [31:0]	VADDR,				/* Virtual Address */
	input				ALS,
	/* OUTPUT PORTS */
	output			BHLS,					/* Byte/Halfword Load/Store */
	output			CMEMWRITE,			/* Conditional Memory Write */
	output			EXC,					/* Memory Access Exception */
	output [6:0]	EXCCODE,				/* Exception Code */
	output [31:0]	MEMDATAIN_EDF,		/* Memory Access Memory Data In to Memory */	
	output [31:0]	MEMDATAOUT,			/* Memory Data Out */
	output [31:0]	PADDR					/* Physical Address */
	);
	
/*****Signals****************************************************************/
	
	wire			BHLS_0;				/* Byte/Halfword Load/Store Asynchronous */
	wire [7:0]	BYTE_IN;				/* Byte from Memory */
	wire [31:0]	BYTE_OUT;			/* Byte to Memory */
	wire [31:0]	BYTEIN_EXT;			/* Byte from Memory Sign Extended */
	wire [1:0]	BYTESEL;				/* Byte Select */
	wire [1:0]	BYTESEL_0;			/* Byte Select Endian Flip */
	wire			CMEMWRITE_0;		/* Conditional Memory Write Asynchronous */
	wire [31:0]	DATA_IN;				/* Data from Memory */
	wire [31:0]	DATA_OUT;			/* Data to Memory */
	wire [15:0]	HALF_IN;				/* Halfword from Memory */
	wire [31:0]	HALF_OUT;			/* Halfword to Memory */
	wire [31:0]	HALFIN_EXT;			/* Halfword from Memory Sign Extended */
	wire [3:0]	LSEL;					/* Load/Store Data Switch */
	wire [31:0]	MEMDATAIN_0;		/* Data to Memory before Endian Flip */
	/*wire [31:0]	MEMDATAIN_EDF_0;	/* Memory Access Memory Data In to Memory */	
	/*wire [31:0]	MEMDATAOUT_0;		/* Data from Memory Endian Flipped */
	wire [31:0]	MEMDATAOUT_1;		/* Selected Data for Writeback */
	/*wire [31:0]	PADDR_0;				/* Physical Address Asynchronous */
	wire [31:0]	UAL_IN;				/* Unaligned Left In */
	wire [31:0]	UAL_OUT;				/* Unaligned Left Out */
	wire [31:0]	UAR_IN;				/* Unaligned Right In */
	wire [31:0]	UAR_OUT;				/* Unaligned Right Out */

/*****Registers****************************************************************/

	reg 			bhls_reg;
	reg 			bls_reg;
	reg [7:0]	byte_in_reg;
	reg 			cmemwrite_reg;
	reg [31:0] 	data_in_reg;
	reg [31:0] 	data_out_reg;
	reg 			edf_reg;
	reg 			exce_reg;
	reg [15:0] 	half_in_reg;
	reg 			hls_reg;
	reg 			ku_reg;
	reg 			lc_reg;
	reg [31:0] 	memdatain_0_reg;
	/*reg [31:0] 	memdatain_edf_reg;*/
	reg [31:0] 	memdatain_reg;
	/*reg [31:0] 	memdataout_0_reg;*/
	reg [31:0] 	memdataout_edf_reg;
	reg [31:0] 	memdataout_reg;
	reg 			memread_reg;
	reg 			memwrite_reg;
	reg 			mext_reg;
	/*reg [31:0] 	paddr_reg;*/
	reg [1:0] 	pclkcnt;
	reg [31:0] 	reserved;
	reg 			rnl_reg;
	reg [31:0] 	vaddr_reg;
	reg 			als_reg;

/*****Initialization****************************************************************/

	initial
	begin
		bhls_reg = 1'b0;
		bls_reg = 1'b0;
		byte_in_reg = 8'b0;
		cmemwrite_reg = 1'b0;
		data_in_reg = 32'b0;
		data_out_reg = 32'b0;
		half_in_reg = 16'b0;
		hls_reg = 1'b0;
		edf_reg = 1'b0;
		exce_reg = 1'b0;
		ku_reg = 1'b0;
		lc_reg = 1'b0;
		memdatain_0_reg = 32'b0;
		/*memdatain_edf_reg = 32'b0;*/
		memdatain_reg = 32'b0;
		/*memdataout_0_reg = 32'b0;*/
		memdataout_edf_reg = 32'b0;
		memdataout_reg = 32'b0;
		memread_reg = 1'b0;
		memwrite_reg = 1'b0;
		mext_reg = 1'b0;
		/*paddr_reg = 32'b0;*/
		pclkcnt = 2'b0;
		reserved = 32'b1;
		rnl_reg = 1'b0;
		vaddr_reg = 32'b0;
		als_reg = 1'b0;
	end

/*********************************************************************/

	assign ADEL	=				(memread_reg == 1'b0)?														1'b0:
									(({bls_reg,hls_reg} == 2'b00) && ~(vaddr_reg[1:0] == 2'b0))?	1'b1:
									(({bls_reg,hls_reg} == 2'b01) && (vaddr_reg[0] == 1'b1))?		1'b1: 
									((ku_reg == 1'b1) && (vaddr_reg[31] == 1'b1))?						1'b1:
																														1'b0;
	assign ADES	=				(memwrite_reg == 1'b0)?														1'b0:
									(({bls_reg,hls_reg} == 2'b00) && ~(vaddr_reg[1:0] == 2'b0))?	1'b1:
									(({bls_reg,hls_reg} == 2'b01) && (vaddr_reg[0] == 1'b1))?		1'b1: 
									((ku_reg == 1'b1) && (vaddr_reg[31] == 1'b1))?						1'b1:
																														1'b0;
	assign BHLS				=	bhls_reg;
	assign CMEMWRITE		=	cmemwrite_reg;
	assign CMEMWRITE_0	=	/*((lc_reg == 1'b1) && (reserved[0] == 1'b1))?	1'b0:*/	memwrite_reg;
	assign MEMDATAIN_EDF	=	memdatain_0_reg;
	assign MEMDATAOUT		=	memdataout_reg;
	assign PADDR			=	vaddr_reg;

/*****Exception Encoding****************************************************************/

	exccencoder exc0(
		.CLK(CLK),
		.RESET(RESET),
		.ADEL(ADEL),
		.ADES(ADES),
		.EXC(EXC),
		.EXCCODE(EXCCODE)
		);

/*****Address Translation****************************************************************/

	/* Needs to be moved to the Memory Interface and replaced with TLB */
	/*address_translation at(
		.MEMREAD(memread_reg),
		.BLS(bls_reg),
		.HLS(hls_reg),
		.VADDR(vaddr_reg),
		.PADDR(PADDR_0)
		);*/
	/*assign PADDR_0 = vaddr_reg;*/

/*********************************************************************/

	and2_1 and0(
		.A(bls_reg),
		.B(hls_reg),
		.C(BHLS_0)
		);

	xor2_2 xor0(
		.A({edf_reg,edf_reg}),
		.B(vaddr_reg[1:0]),
		.C(BYTESEL)
		);

	xor2_2 xor1(
		.A(BYTESEL),
		.B(2'b11),
		.C(BYTESEL_0)
		);
	
	/* Load/Store Selection Encoder */ 	
	loadecoder ldec(
		.BLS(bls_reg),
		.HLS(hls_reg),
		.RNL(rnl_reg),
		.LC(lc_reg),
		.MEMWRITE(memwrite_reg),
		.LSEL(LSEL)
		);
		
/*****Incoming Data from Memory****************************************************************/

	/* Needs to be moved to the Memory Interface */
	/* Data Translation */
	/*endianflip32 edfout(
			.IN(memdataout_edf_reg),
			.EN(edf_reg),
			.OUT(MEMDATAOUT_0)
			);*/
	//assign MEMDATAOUT_0 = memdataout_edf_reg;
	
	/* Byte Load */
	MUX8_4to1 mux1(
		.a0(memdataout_edf_reg[7:0]), 
		.a1(memdataout_edf_reg[15:8]), 
		.a2(memdataout_edf_reg[23:16]),
		.a3(memdataout_edf_reg[31:24]),
		.sel(BYTESEL),
		.out(BYTE_IN)
		);

	/* Sign Extended Byte Load */
	signextend8_32 ext0(
		.IN(byte_in_reg),
		.SE_OP(mext_reg),
		.OUT(BYTEIN_EXT)
		);

	/* Halfword Load */
	MUX16_2to1 mux2(
		.a0(memdataout_edf_reg[15:0]), 
		.a1(memdataout_edf_reg[31:16]), 
		.sel(BYTESEL[1]), 
		.out(HALF_IN)
		);

	/* Sign Extended Halfword Load */
	signextend16_32 ext1(
		.IN(half_in_reg),
		.SE_OP(mext_reg),
		.OUT(HALFIN_EXT)
		);

	/* Unaligned Left Load */
	MUX32_1to1LL mux3(
		.a0(memdataout_edf_reg),
		.sel(BYTESEL),
		.out(UAL_IN)
		);

	/* Unaligned Right Load */
	MUX32_1to1RL mux4(
		.a0(memdataout_edf_reg), 
		.sel(BYTESEL), 
		.out(UAR_IN)
		);

	/* Load Selection */
	MUX32_4to1 mux5(
		.a0(BYTEIN_EXT),
		.a1(HALFIN_EXT),
		.a2(UAL_IN),
		.a3(UAR_IN),
		.sel(LSEL[1:0]),
		.out(DATA_IN)
		);

	MUX32_4to1 mux6(
		.a0(memdataout_edf_reg), 
		.a1(data_in_reg),
		.a2({31'b0,als_reg}),
		.a3(32'b0),
		.sel(LSEL[3:2]),
		.out(MEMDATAOUT_1)
		);

/*****Data going out to Memory****************************************************************/

	/* Byte Store */
	assign BYTE_OUT = (edf_reg)?	{memdatain_reg[7:0],24'b0}:
											{24'b0,memdatain_reg[7:0]};
	/* Halfword Store */
	assign HALF_OUT = (edf_reg)?	{memdatain_reg[15:8],memdatain_reg[7:0],16'b0}:
											{16'b0,memdatain_reg[15:8],memdatain_reg[7:0]};
	/* Unaligned Left Store */
	/* Not Yet Supported by Memory Interface */
	MUX32_1to1LL mux7(
		.a0(memdatain_reg),
		.sel(BYTESEL_0),
		.out(UAR_OUT)
		);

	/* Unaligned Right Store */
	/* Not Yet Supported by Memory Interface */
	MUX32_1to1RL mux8(
		.a0(memdatain_reg), 
		.sel(BYTESEL_0), 
		.out(UAL_OUT)
		);

	/* Store Slection */
	MUX32_4to1 mux9(
		.a0(BYTE_OUT),
		.a1(HALF_OUT),
		.a2(UAL_OUT),
		.a3(UAR_OUT),
		.sel(LSEL[1:0]),
		.out(DATA_OUT)
		);

	MUX32_2to1 mux10(
		.a0(memdatain_reg), 
		.a1(data_out_reg),
		.sel(LSEL[2]), 
		.out(MEMDATAIN_0)
		);

	/* Needs to be moved to the Memory Interface */
	/* Data Translation */
	/*endianflip32 edfin(
			.IN(memdatain_0_reg),
			.EN(edf_reg),
			.OUT(MEMDATAIN_EDF_0)
			);*/
	//assign MEMDATAIN_EDF_0 = memdatain_0_reg;

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			bls_reg <= 1'b0;
			bhls_reg <= 1'b0;
			byte_in_reg <= 8'b0;
			cmemwrite_reg <= 1'b0;
			data_in_reg <= 32'b0;
			data_out_reg <= 32'b0;
			edf_reg <= 1'b0;
			exce_reg <= 1'b0;
			half_in_reg <= 16'b0;
			hls_reg <= 1'b0;
			ku_reg <= 1'b0;
			lc_reg <= 1'b0;
			memdatain_0_reg <= 32'b0;
			/*memdatain_edf_reg <= 32'b0;*/
			memdatain_reg <= 32'b0;
			/*memdataout_0_reg <= 32'b0;*/
			memdataout_edf_reg <= 32'b0;
			memdataout_reg <= 32'b0;
			memread_reg <= 1'b0;
			memwrite_reg <= 1'b0;
			mext_reg <= 1'b0;
			/*paddr_reg <= 32'b0;*/
			rnl_reg <= 1'b0;
			vaddr_reg <= 32'b0;
			als_reg <= 1'b0;
		end
		else
		begin
			/* Latch Signals */
			bhls_reg <= BHLS_0;
			bls_reg <= BLS;
			byte_in_reg <= BYTE_IN;
			cmemwrite_reg <= CMEMWRITE_0;
			data_in_reg <= DATA_IN;
			data_out_reg <= DATA_OUT;
			edf_reg <= EDF;
			exce_reg <= EXCE;
			half_in_reg <= HALF_IN;
			hls_reg <= HLS;
			ku_reg <= KU;
			lc_reg <= LC;
			memdatain_0_reg <= MEMDATAIN_0;
			memdatain_reg <= MEMDATAIN;
			/*memdatain_edf_reg <= MEMDATAIN_EDF_0;*/
			/*memdataout_0_reg <= MEMDATAOUT_0;*/
			memdataout_edf_reg <= MEMDATAOUT_EDF;
			memdataout_reg <= MEMDATAOUT_1;
			memread_reg <= MEMREAD;
			memwrite_reg <= MEMWRITE;
			mext_reg <= MEXT;
			/*paddr_reg <= PADDR_0;*/
			rnl_reg <= RNL;
			vaddr_reg <= VADDR;
			als_reg <= ALS;
		end
	end

/*********************************************************************/

	/*always@(posedge CLK)
	begin
		case(RESET)
			1'b0		:	begin
								reserved <= 32'b1;
								pclkcnt <= 2'b0;
							end
			default	:	begin
							casex({pclkcnt,exce_reg,lc_reg,memread_reg,memwrite_reg})
								6'b100110	:	begin
														reserved <= vaddr_reg;
													end
								6'b100001	:	begin
														if (reserved == vaddr_reg)
														begin
															reserved[0] <= 1'b1;
														end
													end
								6'b100101	:	begin
														if (reserved == vaddr_reg)
														begin
														end
														else
														begin
															reserved[0] <= 1'b1;
														end
													end
								6'b101xxx	:	begin
														reserved[0] <= 1'b1;
													end
							endcase
						end
		endcase
		pclkcnt <= {pclkcnt[0],PCLK};
	end*/

endmodule

/*********************************************************************/

module loadecoder(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input BLS,
	input HLS,
	input LC,
	input MEMWRITE,
	input RNL,
	/* OUTPUT PORTS */
	output [3:0] LSEL
	);
	
/*********************************************************************/

	assign LSEL[1:0]	= 	((BLS == 1'b1) && (HLS == 1'b0))?						2'b00:
								((BLS == 1'b0) && (HLS == 1'b1))?						2'b01:
								((BLS == 1'b1) && (HLS == 1'b1) && (RNL == 1'b0))?	2'b10:
								((BLS == 1'b1) && (HLS == 1'b1) && (RNL == 1'b1))?	2'b11:
																									2'b00;

	assign LSEL[3:2]	=	((LC == 1'b1) && (MEMWRITE == 1'b1))?	2'b10:
								((BLS == 1'b1) || (HLS == 1'b1))?		2'b01:
																					2'b00;

endmodule

/*****Exception Encoding****************************************************************/

module exccencoder(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input CLK,				/* System Clock 50 - 100 MHZ */
	input RESET,			/* System Reset */
	input ADEL,				/* Address Error Exception: Unaligned Address Load */
	input ADES,				/* Address Error Exception: Unaligned Address Store */
	/* OUTPUT PORTS */
	output EXC,				/* Exception */
	output [6:0] EXCCODE	/* Exception Code */
	);

/*****Registers****************************************************************/

	reg			exce_reg;		/* Exception */
	reg [6:0]	exccode_reg;	/* Exception Code */

/*****Initialization****************************************************************/
	
	initial
	begin
		exce_reg = 1'b0;
		exccode_reg = 7'b0;
	end

/*********************************************************************/

	assign EXC		=	exce_reg;
	assign EXCCODE	=	exccode_reg;

/*********************************************************************/
													
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			exce_reg <= 1'b0;
			exccode_reg <= 7'b0;
		end
		else
		begin
			if (ADEL)
			begin
				/* Address Error Exception: Unaligned Address Load */
				exce_reg <= 1'b1;
				exccode_reg <= `AdELc;
			end
			else if (ADES)
			begin
				/* Address Error Exception: Unaligned Address Store */
				exce_reg <= 1'b1;
				exccode_reg <= `AdESc;
			end
			else
			begin
				/* No Exception */
				exce_reg <= 1'b0;
				exccode_reg <= 7'b0;
			end
		end
	end
	
endmodule
