/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	memory_arbiter													*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"

module memory_arbiter(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BLS,					/* Byte Load/Store Data Memory */
	input				BLS_EXT_IN,			/* Byte Load/Store Extension */ 
	input [31:0]	DATA,					/* Data from Memory Bus */
	input [31:0]	DM_ADDR,				/* Data Memory Address */
	input [31:0]	DM_DATA_IN,			/* Data Memroy Data In */
	input				DM_OE,				/* Data Memory Output Enable */
	input				DM_WE,				/* Data Memory Write Enable */
	input				DRWDNE,				/* Data Read/Write Done */
	input				EXC_EXT_MEM_IN,	/* Extension Exception */
	input				EXC_IF_MEM,			/* Instruction Fetch Exception */
	input				EXC_MA_MEM,			/* Memory Access Exception */
	input				EXTRWDNE_IN,		/* Extension Read/Write Done */
	input				FLUSH,				/* Instruction Fetch Flush */
	input				HLS,					/* Halfword Load/Store Data Memory */
	input				HLS_EXT_IN,			/* Halfword Load/Store Extension */
	input [31:0]	IM_ADDR,				/* Instruction Memory Address */
	input				IM_OE,				/* Instruction Memory Output Enable */
	input				IREADDNE,			/* Instruction Read Done */
	input [31:0]	MADDR_EXT_IN,		/* Extension Address */
	input [31:0]	MDATA_EXT_IN,		/* Extension Data In */
	input				MEMCLK,				/* Memory Clock 50 - 100 MHZ */
	input				MOE_EXT_IN,			/* Extension Output Enable */
	input				MWE_EXT_IN,			/* Extension Write Enable */
	input				PCLK,					/* Pipeline Clock */
	input				RESET,				/* System Reset */
	input				RNL,					/* Right/Left Unaligned Load/Store Data Memory */
	input				RNL_EXT_IN,			/* Right/Left Unaligned Load/Store Extension */
	input				STALL,				/* Pipeline Stall */
	input				ICLKDNE,
	input				BUBBLE_WB,
	input [31:0]		PC_WB,
	input				LC,
	input [5:0]		CONTROL,
	/* OUTPUT PORTS */
	output [31:0]	DM_DATA_OUT,		/* Data Memory Data Out to Data Path */
	output [31:0]	DMADDR,				/* Data Memory Address */
	output [31:0]	DMDATA,				/* Data Memroy Data */
	output			DNE,					/* All Transactions complete */
	output [5:0]	DRWCTR,				/* Data Read/Write Control */
	output			DRWSRT,				/* Data Read/Write Start */
	output [31:0]	EXTMADDR_OUT,		/* Extension Address */
	output [31:0]	EXTMDATA_OUT,		/* Extension Data */
	output [5:0]	EXTRWCTR_OUT,		/* Extension Read/Write Control */
	output			EXTRWSRT_OUT,		/* Extension Read/Write Start */
	output [31:0]	IM_DATA_OUT,		/* Instruction Memory Data Out to Data Path*/
	output [31:0]	IMADDR,				/* Instruction Memory Address */ 
	output [5:0]	IREADCTR,			/* Instruction Read Control */
	output			IREADSRT,			/* Instruction Read Start */
	output [31:0]	MDATA_EXT_OUT,		/* Extension Data Out to Data Path */
	output			MDATAVLD_EXT_OUT,	/* Extension Data Valid */
	output			ICLKSRT,
	output [5:0]	ICLKCTR,
	output [31:0]	ICLKPC,
	output 		ALS
	);
	
/*****Registers****************************************************************/

	reg [31:0]	dmaddr_reg;		/* Data Memory Address */
	reg [31:0]	dmdata_reg;		/* Data Memroy Data */
	reg [5:0]	drwctr_reg;		/* Data Read/Write Control */
	reg			drwsrt_reg;		/* Data Read/Write Start */
	reg [5:0]	extrwctr_reg;	/* Extension Read/Write Control */
	reg [31:0]	extmaddr_reg;	/* Extension Address */
	reg [31:0]	extmdata_reg;	/* Extension Data */
	reg			extrwsrt_reg;	/* Extension Read/Write Start */
	reg [31:0]	imaddr_reg;		/* Instruction Memory Address */ 
	reg [31:0]	imdata;			/* Instruction Memory Data */
	reg [5:0]	ireadctr_reg;	/* Instruction Read Control */ 
	reg			ireadsrt_reg;	/* Instruction Read Start */
	reg [1:0]	pclkcnt;			/* Pipeline Clock Edge Detection */
	reg			transrt;			/* Transaction Started */
	reg 			iclksrt_reg;
	reg [5:0]	iclkctr_reg;
	reg [31:0]	iclkpc_reg;

/*****Registers****************************************************************/

	assign DM_DATA_OUT =			dmdata_reg;
	assign DMADDR =				dmaddr_reg;
	assign DMDATA =				dmdata_reg;
	assign DNE = 					~drwsrt_reg & ~ireadsrt_reg & ~extrwsrt_reg & ~iclksrt_reg;
	assign DRWCTR =				drwctr_reg;
	assign DRWSRT =				drwsrt_reg;
	assign EXTMADDR_OUT =		extmaddr_reg;
	assign EXTMDATA_OUT =		extmdata_reg;
	assign EXTRWCTR_OUT =		extrwctr_reg;
	assign EXTRWSRT_OUT =		extrwsrt_reg;
	assign IM_DATA_OUT =			imdata;
	assign IREADCTR =				ireadctr_reg;
	assign IREADSRT =				ireadsrt_reg;
	assign IMADDR =				imaddr_reg;
	assign MDATA_EXT_OUT =		extmdata_reg;
	assign MDATAVLD_EXT_OUT =	~extrwsrt_reg;
	assign ICLKSRT =			iclksrt_reg;
	assign ICLKCTR =			iclkctr_reg;
	assign ICLKPC =			iclkpc_reg;
	assign ALS = drwctr_reg[5];

/*****Initialization****************************************************************/

	initial
	begin
		dmaddr_reg = 32'b0;
		dmdata_reg = 32'b0;
		drwctr_reg = 6'b0;
		drwsrt_reg = 1'b0;
		extmaddr_reg = 32'b0;
		extmdata_reg = 32'b0;
		extrwctr_reg = 6'b0;
		extrwsrt_reg = 1'b0;
		imaddr_reg = 32'b0;
		imdata = 32'b0;
		ireadctr_reg = 6'h00;
		ireadsrt_reg = 1'b0;
		pclkcnt = 2'b00;
		transrt = 1'b0;
		iclksrt_reg = 1'b0;
		iclkctr_reg = 6'h10;
		iclkpc_reg = 32'b0;
	end

/*********************************************************************/

	/* Pipeline Clock Edge Detection */
	always@(posedge MEMCLK)
	begin
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge MEMCLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			dmaddr_reg <= 32'b0;
			dmdata_reg <= 32'b0;
			drwctr_reg <= 6'b0;
			drwsrt_reg <= 1'b0;
			extrwctr_reg <= 6'b0;
			extrwsrt_reg <= 1'b0;
			extmaddr_reg <= 32'b0;
			extmdata_reg <= 32'b0;
			imaddr_reg <= 32'b0;
			imdata <= 32'b0;
			ireadctr_reg <= 6'h00;
			ireadsrt_reg <= 1'b0;
			transrt <= 1'b0;
			iclksrt_reg <= 1'b0;
			iclkctr_reg <= 6'h10;
			iclkpc_reg <= 32'b0;
		end
		else
		begin
			case(pclkcnt)
				/* Sample the memory request on the falling edge of Pipeline Clock */
				2'b10		:	begin
									/* Sampling the instruction read request */
									if (IM_OE & STALL & ~FLUSH & ~EXC_IF_MEM)
									begin
										imaddr_reg <= IM_ADDR;
										ireadctr_reg <= 6'h00;
										ireadsrt_reg <= 1'b1;
									end
									else
									begin
										ireadsrt_reg <= 1'b0;
									end
									/* Sampling the Data read/write request */
									if (~EXC_MA_MEM & (DM_OE ^ DM_WE))
									begin
										dmaddr_reg <= DM_ADDR;
										if (DM_OE)
										begin 
											drwctr_reg <= {LC,1'b0,1'b0,BLS,HLS,RNL};
										end
										if (DM_WE)
										begin
											dmdata_reg <= DM_DATA_IN;
											drwctr_reg <= {LC,1'b0,1'b1,BLS,HLS,RNL};
										end
										drwsrt_reg <= 1'b1;
									end
									else
									begin
										if (DM_OE & DM_WE)
										begin
										end
										drwsrt_reg <= 1'b0;
									end
									/* Sampling the extension read/write request */
									if (~EXC_EXT_MEM_IN & (MOE_EXT_IN ^ MWE_EXT_IN))
									begin
										extmaddr_reg <= MADDR_EXT_IN;
										if (MOE_EXT_IN)
										begin
											extrwctr_reg <= {1'b0,1'b0,1'b0,BLS_EXT_IN,HLS_EXT_IN,RNL_EXT_IN};
										end
										if (MWE_EXT_IN)
										begin
											extmdata_reg <= MDATA_EXT_IN;
											extrwctr_reg <= {1'b0,1'b0,1'b1,BLS_EXT_IN,HLS_EXT_IN,RNL_EXT_IN};
										end
										extrwsrt_reg <= 1'b1;
									end
									else
									begin
										if (MOE_EXT_IN & MWE_EXT_IN)
										begin
										end
										extrwsrt_reg <= 1'b0;
									end
									/* Sampling the committing instructions */
									if (~BUBBLE_WB)
									begin
										iclkpc_reg <= PC_WB;
										iclkctr_reg <= 5'h10;
										iclksrt_reg <= 1'b1;
										//$stop;
									end
								end
				/* Look for the memory request to complete while Pipeline Clock low */
				2'b00		:	begin
									/* If a memory transaction has been initiated already, then check if it has completed */
									if (transrt)
									begin
										/* First priority to instruction read transaction */
										if (ireadsrt_reg & IREADDNE)
										begin
											imdata <= DATA;
											ireadsrt_reg <= 1'b0;
											ireadctr_reg <= CONTROL;
											transrt <= 1'b0;
										end
										/* Next priority to data read/write transaction */
										else if (~ireadsrt_reg & drwsrt_reg & DRWDNE)
										begin
											if (~drwctr_reg[3])
											begin
												/* If read transaction, then pick up the data */
												dmdata_reg <= DATA;
											end
											drwsrt_reg <= 1'b0;
											drwctr_reg <= CONTROL;
											transrt <= 1'b0;
										end
										/* Extension has last priority */
										else if (~ireadsrt_reg & ~drwsrt_reg & extrwsrt_reg & EXTRWDNE_IN)
										begin
											if (~extrwctr_reg[3])
											begin
												/* If it was an extension memory read request, then pick up the data */
												extmdata_reg <= DATA;
											end
											extrwsrt_reg <= 1'b0;
											extrwctr_reg <= CONTROL;
											transrt <= 1'b0;
										end
										else if (~ireadsrt_reg & ~drwsrt_reg & ~extrwsrt_reg & iclksrt_reg & ICLKDNE)
										begin
											iclksrt_reg <= 1'b0;
											iclkctr_reg <= CONTROL;
											transrt <= 1'b0;
											//$stop;
										end
									end
									/* If the transaction has not been started, see if a transaction needs to be started */
									else
									begin
										/*  Instruction read has highest priority - If there is a pending Instr. read request, start that */
										if (ireadsrt_reg & ~IREADDNE)
										begin
											transrt <= 1'b1;
										end
										/* else check if there is a pending data read/write request */
										else if (drwsrt_reg & ~DRWDNE)
										begin
											transrt <= 1'b1;
										end
										/* else check to see if there is a pending data request from the extension */
										else if (extrwsrt_reg & ~EXTRWDNE_IN)
										begin
											transrt <= 1'b1;
										end
										else if (iclksrt_reg & ~ICLKDNE)
										begin
											transrt <= 1'b1;
											//$stop;
										end
									end
								end
				default	:	begin
									/* Do Nothing */
								end
			endcase
		end
	end

endmodule
