/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	memory_bus_front												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"

module memory_bus_front(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	DMADDR,					/* Data Memory Address */
	input [31:0]	DMDATA,					/* Data Memory Data */
	input [5:0]		DRWCTR,					/* Data Read/Write Control */
	input				DRWSRT,					/* Data Read/Write Start */
	input [31:0]	EXTMADDR_IN,			/* Extension Address */
	input [31:0]	EXTMDATA_IN,			/* Extension Data */
	input [5:0]		EXTRWCTR_IN,			/* Extension Read/Write Control */
	input				EXTRWSRT_IN,			/* Extension Read/Write Start */
	input [31:0]	IMADDR,					/* Instruction Address */ 
	input [5:0]		IREADCTR,				/* Instruction Read Control */
	input				IREADSRT,				/* Instruction Read Start */
	input [31:0]	MDATA_OUT,				/* Data from Memory Controller */
	input				MDNE,						/* Memory Controller Done */
	input				MEMCLK,					/* Memory Clock 50 - 100 MHZ */
	input				RESET,					/* System Reset */
	input				TDNE,
	input [31:0]	PADDR,
	input 			ICLKSRT,
	input [5:0]		ICLKCTR,
	input [31:0]	ICLKPC,
	input				EXC_MA_MEM,
	/* OUTPUT PORTS */
	output [31:0]	ADDR,						/* Address to Memory Controller */
	output [31:0]	DATA,						/* Data to Data Path */
	output [1:0]	BYTES,					/* Bytes to Memory Controller */
	output			DRWDNE,					/* Data Read/Write Done */
	output			EXTRWDNE_OUT,			/* Extension Read/Write Done */
	output			IREADDNE,				/* Instruction Read Done */
	output [31:0]	MADDR_SNOOP_OUT,		/* Address Monitor */
	output [31:0]	MDATA_IN,				/* Data to Memory Controller */
	output [31:0]	MDATA_SNOOP_OUT,		/* Data Monitor */
	output			MDATAVLD_SNOOP_OUT,	/* Data Valid Monitor */
	output			MOE_SNOOP_OUT,			/* Output Enable Monitor */
	output			MWE_SNOOP_OUT,			/* Write Enable Monitor */
	output			OE,						/* Output Enable to Memory Controller */
	output			SRT,						/* Start to Memory Controller */
	output			WE,						/* Write Enable */
	output			TBLS,
	output			THLS,
	output			TSRT,
	output			TMEMREAD,
	output [31:0]	VADDR,
	output			ICLKDNE,
	output			INSTRCLK,
	output [5:0]	CONTROL
	);
	
/*****Registers****************************************************************/

	reg [31:0]	addr_reg;			/* Transaction Address */
	reg [1:0]	bytes_reg;			/* Transaction Bytes */
	reg [5:0]	control;				/* Transaction Control */
	reg [31:0]	count;				/* Debug Counter */
	reg [31:0]	data_reg;			/* Transaction Data */
	reg			drwdne_reg;			/* Data Read/Write Done */
	reg			dwait;				/* Data Memory Wait */
	reg			extrwdne_reg;		/* Extension Read/Write Done */
	reg			extwait_reg;		/* Extension Wait */
	reg			ireaddne_reg;		/* Instruction Read Done */
	reg			iwait;				/* Instruction Wait */
	reg			mdatavld_snoop;	/* Data Valid Monitor */
	/*reg			moe_snoop;			/* Output Enable Monitor */ 
	/*reg			mwe_snoop;			/* Write Enable Monitor */
	reg			oe_reg;				/* Output Enable to Memory Controller */
	reg			srt_reg;				/* Start to Memory Controller */
	reg [2:0]	state;				/* State Variable */
	reg			we_reg;				/* Write Enable to Memory Controller */
	reg 			tsrt_reg;
	reg [31:0]	paddr_reg;
	reg 			iclkdne_reg;
	reg			iclkwait;
	reg [31:0]	reserved;

/*****Parameters****************************************************************/

	parameter TRAKKIES_EN	=	0;

/*****Initialization****************************************************************/

	initial
	begin
		addr_reg = 32'b0;
		bytes_reg = 2'b0;
		control = 6'b0;
		count = 32'b0;
		data_reg = 32'b0;
		drwdne_reg = 1'b1;
		dwait = 1'b0;
		extrwdne_reg = 1'b1;
		extwait_reg = 1'b0;
		ireaddne_reg = 1'b1;
		iwait = 1'b0;
		oe_reg = 1'b0;
		srt_reg = 1'b0;
		state = 3'b111;
		we_reg = 1'b0;
		paddr_reg = 32'b0;
		tsrt_reg = 1'b0;
		iclkdne_reg = 1'b1;
		iclkwait = 1'b0;
		reserved = 32'b1;
	end

/*********************************************************************/

	assign ADDR = 						paddr_reg;
	assign BYTES = 					bytes_reg;
	assign DATA = 						data_reg;
	assign DRWDNE = 					drwdne_reg;
	assign EXTRWDNE_OUT = 			extrwdne_reg;
	assign IREADDNE = 				ireaddne_reg;
	assign MADDR_SNOOP_OUT = 		addr_reg;
	assign MDATA_IN = 				data_reg;
	assign MDATA_SNOOP_OUT = 		data_reg;
	assign MDATAVLD_SNOOP_OUT = 	mdatavld_snoop;
	assign MOE_SNOOP_OUT = 			oe_reg;
	assign MWE_SNOOP_OUT = 			we_reg;
	assign OE = 						oe_reg;
	assign SRT = 						srt_reg;
	assign WE = 						we_reg;
	assign TBLS =						control[2];
	assign THLS =						control[1];
	assign TMEMREAD =					~control[3];
	assign TSRT =						tsrt_reg;
	assign VADDR = 					addr_reg;
	assign INSTRCLK =			control[4];
	assign ICLKDNE = iclkdne_reg;
	assign CONTROL = control;
	
/*********************************************************************/

	always@(posedge MEMCLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			addr_reg <= 32'b0;
			bytes_reg <= 2'b0;
			control <= 5'b0;
			data_reg <= 32'b0;
			drwdne_reg <= 1'b1;
			dwait <= 1'b0;
			extrwdne_reg <= 1'b1;
			extwait_reg <= 1'b0;
			ireaddne_reg <= 1'b1;
			iwait <= 1'b0;
			mdatavld_snoop <= 1'b0;
			/*moe_snoop <= 1'b0;*/
			/*mwe_snoop <= 1'b0;*/
			oe_reg <= 1'b0;
			srt_reg <= 1'b0;
			state <= 3'b111;
			we_reg <= 1'b0;
			paddr_reg <= 32'b0;
			tsrt_reg <= 1'b0;
			iclkdne_reg <= 1'b1;
			iclkwait <= 1'b0;
			reserved <= 32'b1;
		end
		else
		begin
			/*moe_snoop <= 1'b0;*/
			/*mwe_snoop <= 1'b0;*/
			if (EXC_MA_MEM & ~reserved[0])
			begin
				/* Exception:  Canceling Reservation */
				reserved[0] <= 1'b1;
			end
			case(state)
				3'b000		:	begin
									tsrt_reg <= 1'b0;
									if (TDNE)
									begin
										paddr_reg <= PADDR;
										state <= 3'b001;
										//$stop;
									end
								end
				3'b001		:	begin
									if (control[4] & ~TRAKKIES_EN)
									begin
										state <= 3'b110;
									end
									else
									begin
										state <= 3'b010;
									end
								end
				3'b010		:	begin
									casex({control[5],control[3]})
										2'b10		:	begin
														/* Reserving Address */
														reserved <= paddr_reg;
														state <= 3'b011;
														//$display("Reserving Address %x, %x, %x", reserved, paddr_reg, addr_reg);
														//$display("Control %x", control);
														//$stop;
													end
										2'b01		:	begin
														if (reserved == paddr_reg)
														begin
															/* Store: Cancel Reservation */
															reserved[0] <= 1'b1;
															state <= 3'b011;
															//$display("Store Cancels Reservation %x, %x, %x", reserved, paddr_reg, addr_reg);
															//$display("Control %x", control);
															//$stop;
														end
														else
														begin
															state <= 3'b011;
															//$stop;
														end
													end
										2'b11		:	begin
														if (reserved == paddr_reg)
														begin
															/* Completing Atomic Read Modify Write */
															state <= 3'b011;
															//$display("Complete Atomic Read Modify Write %x, %x, %x", reserved, paddr_reg, addr_reg);
															//$display("Data %x, %x", data_reg, DATA);
															//$display("Control %x", control);
															//$stop;
														end
														else
														begin
															/* SC: Canceling Reservation */
															reserved[0] <= 1'b1;
															control[5] <= 1'b0;
															state <= 3'b110;
															//$display("SC Cancels Reservation %x, %x, %x", reserved, paddr_reg, addr_reg);
															//$display("Control %x", control);
															//$stop;
														end
													end
										default	:	begin
														state <= 3'b011;
														//$stop;
													end
									endcase
									
								end
				3'b011		:	begin
									state <= 3'b100;
								end
				3'b100		:	begin
									casex(control[4:0])
										5'b1xxxx	:	begin
															/* Transitioning the Clock */
															bytes_reg <= 2'b0;
															oe_reg <= 1'b1;
															we_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*moe_snoop <= 1'b1;*/
															//$stop;
														end
										5'b0000x	:	begin
															/* Loading Word */
															bytes_reg <= 2'b0;
															oe_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*moe_snoop <= 1'b1;*/	
														end
										5'b0001x	:	begin
															/* Loading Word for Half Word */
															bytes_reg <= 2'b0;
															oe_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*moe_snoop <= 1'b1;*/
														end
										5'b0010x	:	begin
															/* Loading Word for Byte */
															bytes_reg <= 2'b0;
															oe_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*moe_snoop <= 1'b1;*/
														end
										5'b00110	:	begin
															/* Loading Word for Left Unaligned */
															bytes_reg <= 2'b0;
															oe_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*moe_snoop <= 1'b1;*/
														end
										5'b00111	:	begin
															/* Loading Word for Right Unaligned */
															bytes_reg <= 2'b0;
															oe_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*moe_snoop <= 1'b1;*/
														end
										5'b0100x	:	begin
															/* Writing Word */
															bytes_reg <= 2'b0;
															we_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*mwe_snoop <= 1'b1;*/
														end
										5'b0101x	:	begin
															/* Writing Half Word */
															bytes_reg <= 2'b10;
															we_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*mwe_snoop <= 1'b1;*/
														end
										5'b0110x	:	begin
															/* Writing Byte */
															bytes_reg <= 2'b01;
															we_reg <= 1'b1;
															srt_reg <= 1'b1;
															/*mwe_snoop <= 1'b1;*/
														end
										5'b01110	:	begin
															/* Write Left Unaligned Word */
															/* Currently not supported */
														end
										5'b01111	:	begin
															/* Write Right Unaligned Word */
															/* Currently not supported */
														end
										default	:	begin
															/* Do Nothing */
														end
									endcase
									count <= 32'b0;
									state <= 3'b101;
								end
				3'b101		:	begin
									if (~MDNE)
									begin
										/* If memory controller is working on the request, remove the start signal */
										srt_reg <= 1'b0;
									end
									if ((MDNE) && (~srt_reg))
									begin	
										/* If memory controller is done, reset enables, */
										we_reg <= 1'b0;
										oe_reg <= 1'b0;	
										if (~control[3])
										begin
											/* pass data up one level (if memory read) and move to next state. */
											data_reg <= MDATA_OUT;
										end
										state <= 3'b110;
									end
									else
									begin
										if (count == 0)
										begin
										end
										count <= count + 1;
									end
								end
				3'b110		:	begin	
									/* This state indicates that the memory trasnaction had completed. Assert done signals */
									count <= 32'b0;
									/* Notify the extension that a memory transaction has completed */
									mdatavld_snoop <= 1'b1;		
									if (IREADSRT && ~ireaddne_reg)
									begin
										ireaddne_reg <= 1'b1;
										iwait <= 1'b1;
									end
									else if (DRWSRT && ~drwdne_reg)
									begin
										drwdne_reg <= 1'b1;
										dwait <= 1'b1;
									end
									else if (EXTRWSRT_IN && ~extrwdne_reg)
									begin
										extrwdne_reg <= 1'b1;
										extwait_reg <= 1'b1;
									end
									else if (ICLKSRT && ~iclkdne_reg)
									begin
										iclkdne_reg <= 1'b1;
										iclkwait <= 1'b1;
										//$stop;
									end
									else
									begin
									end
									if ((iwait & ~IREADSRT) || (dwait & ~DRWSRT) || (extwait_reg & ~EXTRWSRT_IN) || (iclkwait & ~ICLKSRT))
									begin
										state <= 3'b111;
										iwait <= 1'b0;
										dwait <= 1'b0;
										extwait_reg <= 1'b0;
										iclkwait <= 1'b0;
									end
								end
				3'b111		:	begin
									if (count == 0)
									begin
									end
									count <= count + 1;
									/* Wait till the memory request has been identified by the uppper state machine (on pclkcnt) */
									/* Then, create appropriate control signals and jump to state 00 */
									if (IREADSRT || DRWSRT || EXTRWSRT_IN || ICLKSRT)
									begin
										mdatavld_snoop <= 1'b0;
										if (IREADSRT)
										begin
											addr_reg <= IMADDR;
											control <= IREADCTR;
											ireaddne_reg <= 1'b0;
										end
										else if (DRWSRT)
										begin
											addr_reg <= DMADDR;
											if (DRWCTR[3])
											begin
												data_reg <= DMDATA;
											end
											control <= DRWCTR;
											drwdne_reg <= 1'b0;
										end
										else if (EXTRWSRT_IN)
										begin
											addr_reg <= EXTMADDR_IN;
											if (EXTRWCTR_IN[3])
											begin
												data_reg <= EXTMDATA_IN;
											end
											control <= EXTRWCTR_IN;
											extrwdne_reg <= 1'b0;
										end
										else if (ICLKSRT)
										begin
											addr_reg <= ICLKPC;
											control <= ICLKCTR;
											iclkdne_reg <= 1'b0;
											//$stop;
										end
										else
										begin
										end
										tsrt_reg <= 1'b1;
										if (~TDNE)
										begin
											state <= 3'b0;
											//$stop;
										end
									end
								end
				default	:	begin
								end
			endcase
		end
	end

endmodule
