/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	memory_controller												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	MEMORY_CONTROLLER combines all the on-chip peripherals and			*/
/*	peripheral interfaces together on a single bus.  Each of the		*/
/*	peripherals on the bus listen to the address bus for address		*/
/*	ranges they are assigned.  When the peripheral recognize an			*/
/*	incoming address it latches the control, address and data and		*/
/*	acknowledges the request by lowering its done signal.  When the	*/
/*	memory transaction is complete the peripheral raises the done		*/
/*	signal.  In the case of a memory read, the data is valid with the	*/
/*	done is raised.																	*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Extension Peripherals, Bug Fixes, see Manual		*/
/*																							*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

/*****Peripheral Constants****************************************************************/

/* Peripheral Tags */
`define BRAM_TAG			0						/* BlockRam */
`define PMT_TAG			1						/* Peripheral Mapping Table */
`define SRAM_TAG			2						/* SRAM */
`define DDRAM_TAG			3						/* DDRAM */
`define FLASH_TAG			4						/* Flash */
`define INTCTR_TAG		5						/* Interrupt Controller */
`define USART_TAG			6						/* USART */
`define TIMER_TAG			7						/* Timers */
`define WATCHDOG_TAG		8						/* Watchdog Timer */
`define GPIO_TAG			9						/* General Purpose I/O */
`define SYSACE_TAG		10						/* System ACE Compact Flash */
`define LCD_TAG			11						/* LCD */
`define PS2_TAG			12						/* PS2 */
`define VGA_TAG			13						/* VGA */
`define ETHERNET_TAG		14						/* Ethernet */
`define AC97_TAG			15						/* Audio Codec */
`define PWRMG_TAG			16						/* Power Management Controller */

/* Peripheral Base Address */
`define SRAM_ADDR			32'hfffd0000		/* SRAM */
`define DDRAM_ADDR		32'hfffc0000		/* DDRAM */
`define FLASH_ADDR		32'hfffb0000		/* Flash */
`define INTCTR_ADDR		32'hfffa0000		/* Interrupt Controller */
`define USART_ADDR		32'hfff90000		/* USART */
`define TIMER_ADDR		32'hfff80000		/* Timers */
`define WATCHDOG_ADDR	32'hfff70000		/* Watchdog Timer */
`define GPIO_ADDR			32'hfff60000		/* General Purpose I/O */
`define IDE_ADDR			32'hfff50000		/* System ACE Compact Flash */
`define LCD_ADDR			32'hfff40000		/* LCD */
`define PS2_ADDR			32'hfff30000		/* PS2 */
`define VGA_ADDR			32'hfff20000		/* VGA */
`define ETHERNET_ADDR	32'hfff10000		/* Ethernet */
`define PWRMG_ADDR		32'hfff00000		/* Power Management Controller */
`define AC97_ADDR			32'hffef0000		/* Audio Codec */
`define BRAM_ADDR			32'hffdf0000		/* BlockRam */
`define EC_ADDR			32'hffee0000		/* Extension Controller */

/* Peripheral Interrupts */
`define TIMER_INT			0						/* Timers */
`define GPIO_INT			2						/* General Purpose I/O */
`define WATCHDOG_INT		3						/* Watchdog Timer */
`define SYSACE_INT		4						/* System ACE Compact Flash */
`define ETHERNET_INT		5						/* Ethernet */
`define PS2_INT			6						/* PS2 */
`define AC97_INT			7						/* Audio Codec */
`define SRAM_INT			8						/* SRAM */
`define USART_INT			9						/* USART */
`define EC_INT				10						/* Extension Controller */
`define SLOT1_INT			30						/* Extension1 */
`define SLOT0_INT			31						/* Extension0 */

module memory_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,					/* Transaction Address */
	input				BRAMCLK,					/* Blockram Clock 50 - 100 MHZ */
	input				BRDY,						/* System ACE Buffer Ready */
	input [1:0]		BYTES,					/* Transaction Bytes */
	input				CLK_IN_PER,				/* System ACE Clock */
	input [31:0]	DATA_IN,					/* Transaction Data */
	input [31:0]	DATA_MEM_OUT,			/* Data from Memory Bus */
	input [3:0]		DATA_PAR_OUT,			/* Data Parity from Memory Bus */
	input [15:0]	DATA_PER_OUT,			/* Data from Peripheral Bus */
	input				FLASHCLK,				/* Flash Clock 50 - 100 MHZ */
	input [31:0]	GPIO_OUT,				/* Data from GPIO bus */
	input				INT_EXT0_IN,			/* */
	input				INT_EXT1_IN,			/* */
	input				IRQ_SYSACE,				/* System ACE IRQ */
	input	[31:0]	MDATA_BUS_IN,			/* */
	input				MEMCLK,					/* Memory Clock 50 - 100 MHZ */
	input				OE,						/* Output Enable */
	input				PER0_DNE_IN,			/* */
	input				PER0_PRESENT_IN,		/* */
	input				PER0_PRIVILEGE_IN,	/* */
	input				PER0_WANTS_INTR_IN,	/* */
	input				PER1_DNE_IN,			/* */
	input				PER1_PRESENT_IN,		/* */
	input				PER1_PRIVILEGE_IN,	/* */
	input				PER1_WANTS_INTR_IN,	/* */
	input				RESET,					/* System Reset */
	input				RX,						/* Serial Recieve */
	input				SRAMCLK,					/* SRAM Clock 50 - 100 MHZ */
	input				SRT,						/* Transaction Start */ 
	input				TMRCLK,					/* Timer Clock 10 MHZ */
	input				USARTCLK,				/* USART Clock 100 MHZ */
	input				WE,						/* Write Enable */
	input				RE,
	input				PER_EXT0,
	input 			PER_EXT1,
	/* OUTPUT PORTS */
	output [23:0]	ADDR_MEM,				/* Address Memory Bus */
	output [6:0]	ADDR_PER,				/* Address Peripheral Bus */
	output			AIRQ,						/* Hardware Interrupt */
	output			BYTE,						/* Flash Byte Mode */
	output			BYTESEL,					/* Flash Byte Select */
	output			CE_FLASH,				/* Flash Chipselect */
	output [31:0]	DATA_MEM_IN,			/* Data to Memory Bus */
	output [31:0]	DATA_MEM_TR,			/* Memory Bus Tristate Enable */ 
	output [31:0]	DATA_OUT,				/* Data to Memory Interface */
	output [3:0]	DATA_PAR_IN,			/* Data Parity to Memory Bus */
	output [3:0]	DATA_PAR_TR,			/* Memory Bus Parity Tristate Enable */
	output [15:0]	DATA_PER_IN,			/* Data to Peripheral Bus */
	output [15:0]	DATA_PER_TR,			/* Peripheral Tristate Enable */
	output			DNE,						/* Transaction Done */
	output [31:0]	GPIO_IN,					/* Data to GPIO bus */
	output [31:0]	GPIO_TR,					/* GPIO bus Tristate Enable */
	output [31:0]	MADDR_BUS,				/* */
	output [1:0]	MBYTES_BUS,				/* */
	output [31:0]	MDATA_BUS_OUT,			/* */
	output			MODE,						/* SRAM Mode */
	output			MOE_BUS,					/* */
	output			MSRT_BUS,	/* */
	output			MWE_BUS,					/* */
	output			NADVLD,					/* SRAM Advanced Load */
	output [3:0]	NBW,						/* SRAM Byte Enables */
	output			NCE_SRAM,				/* SRAM Chipselects */
	output			NCE_SYSACE,				/* System ACE Chipselects */
	output			NCEN,						/* SRAM Clock Enable */
	output			NOE_MEM,					/* Output Enable Memory Bus */
	output			NOE_PER,					/* Output Enable Peripheral Bus */
	output			NRP,						/* Flash Hardware Reset */
	output			NWE_MEM,					/* Write Enable Memroy Bus */
	output			NWE_PER,					/* Write Enable Peripheral Bus */
	output			PER0_BAT_EN_OUT,		/* */
	output			PER1_BAT_EN_OUT,		/* */
	output			SCLK, 					/* USART Baud Clock */
	output			TX,						/* Serial Transmit */
	output			ZZ,						/* SRAM Sleep */
	/* IO PORTS */
	inout				CLK_EXT					/* Timer External Clock */
	);

/*****Signals****************************************************************/

	wire [6:0]	ADDR_BRAM;			/* Blockram Address bus */
	wire [23:0]	ADDR_FLASH;			/* Flash Address bus */
	wire [23:0]	ADDR_SRAM;			/* SRAM Address bus */
	wire [6:0]	ADDR_SYSACE;		/* System ACE Address bus */
	wire			CE0_FLASH;			/* Flash Chipselect 0 */
	wire			CE2_FLASH;			/* Flash Chipselect 2 */
	wire			CE2_SRAM;			/* SRAM Chipselect 2 */
	wire			CLK_OUT_BRAM;		/* Blockram Clock 50 - 100 MHZ */
	wire [31:0]	DATA_AL;				/* Data Addressed Aligned */
	wire [31:0]	DATA_BRAM_OUT;		/* Blockram Data Out */
	wire [31:0]	DATA_BRAM_IN;		/* Blockram Data In */
	wire [31:0]	DATA_OUT_0;			/* Data Out to Memory Interface Asynchronous */
	wire			DNE_BRAM;			/* Blockram Done */
	wire			DNE_FLASH;			/* Flash Done */
	wire			DNE_GPIO;			/* GPIO Done */
	wire			DNE_IC;				/* Interrupt Controller Done */
	wire			DNE_PMC;				/* Power Management Controller Done */
	wire			DNE_SRAM;			/* SRAM Done */
	wire			DNE_SYSACE;			/* System ACE Done */
	wire			DNE_TIMER;			/* Timer Done */
	wire			DNE_USART;			/* USART Done */
	wire [31:0]	FLASHDQ_IN;			/* Flash Data bus to buffers */
	wire [31:0]	FLASHDQ_OUT;		/* Flash Data bus from buffers */
	wire [31:0]	FLASHDQ_TR;			/* Flash Data bus Tristate Enable */
	wire			GPIO_IRQ;			/* GPIO Interrupt */
	wire [31:0]	IRQ;					/* Interrupt Bus */
	wire			MDNE;					/* Memory Controller Done */
	wire			MWE;					/* Blockram Write Enable */
	wire			NCE1_FLASH;			/* Flash Chipselect 1 */
	wire			NCE1_SRAM;			/* SRAM Chipselect 1 */
	wire			NCE3_SRAM;			/* SRAM Chipselect 3 */
	wire			NOE_FLASH;			/* Flash Output Enable */
	wire			NOE_SRAM;			/* SRAM Output Enable */
	wire			NOE_SYSACE;			/* System ACE Output Enable */
	wire			NWE_FLASH;			/* Flash Write Enable */
	wire			NWE_SRAM;			/* SRAM Write Enable */
	wire			NWE_SYSACE;			/* System ACE Write Enable */
	wire			PR_IRQ;				/* SRAM Parity Error Interrupt */
	wire [15:0]	SACDQ_IN;			/* System ACE Data bus to buffers */
	wire [15:0]	SACDQ_OUT;			/* System ACE Data bus from buffers */
	wire [15:0]	SACDQ_TR;			/* System ACE Data bus Tristate Enable */
	wire [31:0]	SRAMDQ_IN;			/* SRAM Data bus to buffers */
	wire [31:0]	SRAMDQ_OUT;			/* SRAM Data bus from buffers */
	wire [31:0]	SRAMDQ_TR;			/* SRAM Data bus Tristate Enable */
	wire [3:0]	SRAMDQP_IN;			/* SRAM Data Parity bus from buffers */
	wire [3:0]	SRAMDQP_OUT;		/* SRAM Data Parity bus to buffers */
	wire [3:0]	SRAMDQP_TR;			/* SRAM Data Parity bus Tristate Enable */
	wire			SYSACE_IRQ;			/* System ACE Interrupt */
	wire			USART_IRQ;			/* USART Interrupt */
	wire			TDIRQ;				/* Down Counter Interrupt */
	wire			TFIRQ;				/* Free Counter Interrupt */
	wire			W_DNE_EC;			/* */
	wire			W_DNE_GC;			/* */
	wire			W_EC_IRQ;			/* */
	wire [31:0]	DATA_IN_EDF;
	wire 			AIRQ_IC;
	wire			AIRQ_GC;

/*****Registers****************************************************************/
	
	reg [31:0]	data_out_r;	/* Data Out to Memory Interface */
	reg			dne_r;		/* Memory Controller Done */ 
	reg			srt_r;		/* Start Transaction */
	reg [1:0]	state;		/* state variable */
	reg [9:0]	timeout;		/* Timeout Counter */
	reg [31:0]	irq_reg;

/*****Parameters****************************************************************/

	parameter SRAM_EN			=	1;
	parameter DDRAM_EN		=	0;
	parameter FLASH_EN		=	1;
	parameter INTCTR_EN		=	1;
	parameter USART_EN		=	1;
	parameter TIMER_EN		=	1;
	parameter WATCHDOG_EN	=	0;
	parameter GPIO_EN			=	1;
	parameter IDE_EN			=	1;
	parameter LCD_EN			=	0;
	parameter PS2_EN			=	0;
	parameter VGA_EN			=	0;
	parameter ETHERNET_EN	=	0;
	parameter PWRMG_EN		=	1;
	parameter AC97_EN			=	0;
	parameter BRAM_EN			=	1;
	parameter EC_EN			= 	1;
	parameter GC_EN			=	0;
	parameter TRAKKIES_EN	=	0;

/*****Initialization****************************************************************/
	
	initial
	begin
		data_out_r = 32'b0;
		dne_r = 1'b1;
		srt_r = 1'b0;
		state = 2'b0;
		timeout = 10'b0;
		irq_reg = 32'b0;
	end

/*********************************************************************/

	//assign DATA_OUT	=	data_out_r;
	assign DATA_OUT_0	=	MDATA_BUS_IN;
	assign DNE			=	dne_r;
	assign MDNE			=	(~SRAM_EN | DNE_SRAM) & 
						(~USART_EN | DNE_USART) & 
						(~TIMER_EN | DNE_TIMER) & 
						(~INTCTR_EN | DNE_IC) & 
						(~PWRMG_EN | DNE_PMC) & 
						(~BRAM_EN | DNE_BRAM) & 
						(~FLASH_EN | DNE_FLASH) & 
						(~GPIO_EN | DNE_GPIO) & 
						(~IDE_EN | DNE_SYSACE) &
						(~EC_EN | W_DNE_EC) &
						(~GC_EN | W_DNE_GC) &
						(~PER_EXT0 | PER0_DNE_IN) &
						(~PER_EXT1 | PER1_DNE_IN);

/*****Internal Bus****************************************************************/

	assign MDATA_BUS_OUT	=	DATA_AL;
	assign MADDR_BUS		=	ADDR_IN;
	assign MOE_BUS			=	OE;
	assign MSRT_BUS	=	srt_r;
	assign MWE_BUS			=	WE;
	assign MBYTES_BUS = 	BYTES;

/*****Interrupt Bus****************************************************************/

	assign IRQ[`TIMER_INT] = (TIMER_EN & (TDIRQ | TFIRQ));
	assign IRQ[`USART_INT] = (USART_EN & USART_IRQ);
	assign IRQ[`SRAM_INT] = (SRAM_EN & PR_IRQ);
	assign IRQ[`GPIO_INT] = (GPIO_EN & GPIO_IRQ);
	assign IRQ[`SYSACE_INT] = (IDE_EN & SYSACE_IRQ);
	assign IRQ[`EC_INT] = (EC_EN & W_EC_IRQ);
	assign IRQ[`SLOT0_INT] = INT_EXT0_IN;
	assign IRQ[`SLOT1_INT] = INT_EXT1_IN;
	assign IRQ[29:11] = 22'b0;
	assign IRQ[7:5] = 3'b0;
	assign IRQ[3] = 1'b0;
	assign IRQ[1] = 1'b0;
	assign AIRQ =	(INTCTR_EN)?	AIRQ_IC:
				(GC_EN)?		AIRQ_GC:
							1'b0;

/*****Peripheral Bus****************************************************************/

	/* System is on the Peripheral Bus */

	assign ADDR_PER = ADDR_SYSACE;
	assign DATA_PER_IN = SACDQ_IN;
	assign DATA_PER_TR = SACDQ_TR;
	assign NOE_PER = NOE_SYSACE;
	assign NWE_PER = NWE_SYSACE;
	assign SACDQ_OUT = DATA_PER_OUT;

/*****Memory Bus****************************************************************/

	/* SRAM and Flash Share the Memory Bus */
	
	/* only one of the Flash chipselects is connected to a FPGA pin */
	assign CE_FLASH = CE2_FLASH & ~NCE1_FLASH & CE0_FLASH;

	/* only one of the SRAM chipselects is connected to a FPGA pin */
	assign NCE_SRAM = NCE1_SRAM | ~CE2_SRAM | NCE3_SRAM;

	/* Only SRAM has the Parity bits */
	assign DATA_PAR_IN = SRAMDQP_IN;
	assign DATA_PAR_TR = SRAMDQP_TR;
	assign SRAMDQP_OUT = DATA_PAR_OUT;

	/* Incoming Signals */

	assign FLASHDQ_OUT = DATA_MEM_OUT;
	assign SRAMDQ_OUT = DATA_MEM_OUT;

	/* Outgoing Signals */

	BSMUX1_2to1 bsmux0(
		.a0(NWE_SRAM), 
		.a1(NWE_FLASH), 
		.def(1'b1), 
		.en0(~DNE_SRAM), 
		.en1(~DNE_FLASH), 
		.out(NWE_MEM)
		);
		
	BSMUX1_2to1 bsmux1(
		.a0(NOE_SRAM), 
		.a1(NOE_FLASH), 
		.def(1'b1), 
		.en0(~DNE_SRAM), 
		.en1(~DNE_FLASH), 
		.out(NOE_MEM)
		);
		
	BSMUX24_2to1 bsmux2(
		.a0(ADDR_SRAM), 
		.a1(ADDR_FLASH), 
		.def(24'b0), 
		.en0(~DNE_SRAM), 
		.en1(~DNE_FLASH), 
		.out(ADDR_MEM)
		);
		
	BSMUX32_2to1 bsmux3(
		.a0(SRAMDQ_IN), 
		.a1(FLASHDQ_IN), 
		.def(32'b0), 
		.en0(~DNE_SRAM), 
		.en1(~DNE_FLASH), 
		.out(DATA_MEM_IN)
		);
		
	BSMUX32_2to1 bsmux4(
		.a0(SRAMDQ_TR), 
		.a1(FLASHDQ_TR), 
		.def(32'hffffffff), 
		.en0(~DNE_SRAM), 
		.en1(~DNE_FLASH), 
		.out(DATA_MEM_TR)
		);

/*****Address Align Data****************************************************************/

	endianflip32 edfin(
		.IN(DATA_IN),
		.EN(RE),
		.OUT(DATA_IN_EDF)
		);

	data_align dain(
		.ADDR(ADDR_IN[1:0]),
		.DATA_IN(DATA_IN_EDF),
		.DATA_OUT(DATA_AL)
		);

	endianflip32 edfout(
		.IN(data_out_r),
		.EN(RE),
		.OUT(DATA_OUT)
		);
		
/*****SRAM Controller****************************************************************/

	sram_controller sc(
		.ADDR_IN(ADDR_IN),
		.ADDR_OUT(ADDR_SRAM),
		.BYTES(BYTES),
		.CADDR(`SRAM_ADDR),
		.CE2(CE2_SRAM),
		.CLK(MEMCLK),
		.DATA_IN(DATA_AL),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_SRAM),
		.DQ_IN(SRAMDQ_IN),
		.DQ_OUT(SRAMDQ_OUT),
		.DQ_TR(SRAMDQ_TR),
		.DQP_IN(SRAMDQP_IN),
		.DQP_OUT(SRAMDQP_OUT),
		.DQP_TR(SRAMDQP_TR),
		.MODE(MODE),
		.NADVLD(NADVLD),
		.NBW(NBW),
		.NCE1(NCE1_SRAM),
		.NCE3(NCE3_SRAM),
		.NCEN(NCEN),
		.NOE(NOE_SRAM),
		.NWE(NWE_SRAM),
		.OE(OE),
		.PR_IRQ(PR_IRQ),
		.RESET(RESET & SRAM_EN),
		.SRAMCLK(SRAMCLK),
		.SRT(srt_r),
		.WE(WE),
		.ZZ(ZZ)
		);

/*****Flash Controller****************************************************************/

	flash_controller fc(
		.ADDR_IN(ADDR_IN),
		.ADDR_OUT(ADDR_FLASH),
		.BYTE(BYTE),
		.BYTES(BYTES),
		.BYTESEL(BYTESEL),
		.CADDR(`FLASH_ADDR),
		.CE0(CE0_FLASH),
		.CE2(CE2_FLASH),
		.CLK(MEMCLK),
		.DATA_IN(DATA_AL),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_FLASH),
		.DQ_IN(FLASHDQ_IN),
		.DQ_OUT(FLASHDQ_OUT),
		.DQ_TR(FLASHDQ_TR),
		.FLASHCLK(FLASHCLK),
		.NCE1(NCE1_FLASH),
		.NOE(NOE_FLASH),
		.NRP(NRP),
		.NWE(NWE_FLASH),
		.OE(OE),
		.RESET(RESET & FLASH_EN),
		.SRT(srt_r),
		.WE(WE)
		);

/*****General Purpose I/O****************************************************************/

	gpio_controller gpc(
		.ADDR_IN(ADDR_IN),
		.BYTES(BYTES),
		.CADDR(`GPIO_ADDR),
		.CLK(MEMCLK),
		.DATA_IN(DATA_AL),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_GPIO),
		.GPIO_IN(GPIO_IN),
		.GPIO_IRQ(GPIO_IRQ),
		.GPIO_OUT(GPIO_OUT),
		.GPIO_TR(GPIO_TR),
		.OE(OE),
		.RESET(RESET & GPIO_EN),
		.SRT(srt_r),
		.WE(WE)
		);

/*****System ACE Controller****************************************************************/

	sysace_controller sac(
		.ADDR_IN(ADDR_IN),
		.ADDR_OUT(ADDR_SYSACE),
		.BRDY(BRDY),
		.BYTES(BYTES),
		.CADDR(`IDE_ADDR),
		.CLK(MEMCLK),
		.CLK_IN(CLK_IN_PER),
		.DATA_IN(DATA_AL),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_SYSACE),
		.DQ_IN(SACDQ_IN),
		.DQ_OUT(SACDQ_OUT),
		.DQ_TR(SACDQ_TR),
		.IRQ(IRQ_SYSACE),
		.NCE(NCE_SYSACE),
		.NOE(NOE_SYSACE),
		.NWE(NWE_SYSACE),
		.OE(OE),
		.RESET(RESET & IDE_EN),
		.SA_IRQ(SYSACE_IRQ),
		.SRT(srt_r),
		.WE(WE)
		);

/*****USART Controller****************************************************************/

	usart_controller uc(
		.ADDR_IN(ADDR_IN),
		.BYTES(BYTES),
		.CADDR(`USART_ADDR),
		.CLK(MEMCLK),
		.DATA_IN(DATA_AL),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_USART),
		.OE(OE),
		.RESET(RESET & USART_EN),
		.RX(RX),
		.SCLK(SCLK),
		.SRT(srt_r),
		.TX(TX),
		.USART_IRQ(USART_IRQ),
		.USARTCLK(USARTCLK),
		.WE(WE)
		);

/*****Timer Controller****************************************************************/

	timer_controller tc(
		.ADDR_IN(ADDR_IN),
		.BYTES(BYTES),
		.CADDR(`TIMER_ADDR),
		.CLK(MEMCLK),
		.CLK_EXT(CLK_EXT),
		.DATA_IN(DATA_AL),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_TIMER),
		.OE(OE),
		.RESET(RESET & TIMER_EN),
		.SRT(srt_r),
		.TDIRQ(TDIRQ),
		.TFIRQ(TFIRQ),
		.TMRCLK(TMRCLK),
		.WE(WE)
		);

/*****Interrupt Controller****************************************************************/

	interrupt_controller ic(
		.AIRQ(AIRQ_IC),
		.ADDR_IN(ADDR_IN),
		.BYTES(BYTES),
		.CADDR(`INTCTR_ADDR),
		.CLK(MEMCLK),
		.DATA_IN(DATA_AL),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_IC),
		.IRQ(irq_reg),
		.OE(OE),
		.RESET(RESET & INTCTR_EN),
		.SRT(srt_r),
		.WE(WE)
		);
		
/*****Power Management Controller****************************************************************/

	power_management_controller pmc(
		.ADDR_IN(ADDR_IN),
		.BYTES(BYTES),
		.CADDR(`PWRMG_ADDR),
		.CLK(MEMCLK),
		.DATA_IN(DATA_AL),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_PMC),
		.OE(OE),
		.RESET(RESET & PWRMG_EN),
		.SRT(srt_r),
		.WE(WE)
		);

/*****BlockRAM Controller****************************************************************/

	blockram_controller brc(
		.ADDR_IN(ADDR_IN),
		.ADDR_OUT(ADDR_BRAM),
		.BRAMCLK(BRAMCLK),
		.BYTES(BYTES),
		.CADDR(`BRAM_ADDR),
		.CLK(MEMCLK),
		.CLK_OUT(CLK_OUT_BRAM),
		.DATA_IN(DATA_AL),
		.DATA_MEM_IN(DATA_BRAM_IN),
		.DATA_MEM_OUT(DATA_BRAM_OUT),
		.DATA_OUT(DATA_OUT_0),
		.DNE(DNE_BRAM),
		.MWE(MWE),
		.OE(OE),
		.RESET(RESET & BRAM_EN),
		.SRT(srt_r),
		.WE(WE)
		);

/*****Bootloader/Peripheral Mapping Table BlockRAM****************************************************************/

	blockram_br br(
		.addr(ADDR_BRAM),
		.clk(CLK_OUT_BRAM),
		.din(DATA_BRAM_IN),
		.dout(DATA_BRAM_OUT),
		.we(MWE)
		);

/*****Extension Controller****************************************************************/

	extension_controller ec(
		.ADDR_IN(ADDR_IN),
		.BYTES(BYTES),
		.CADDR(`EC_ADDR),
		.CLK(MEMCLK),
		.DATA_IN(DATA_AL),
		.OE(OE),
		.PER0_DNE(PER0_DNE_IN),
		.PER0_PRESENT(PER0_PRESENT_IN),
		.PER0_PRIVILEGE(PER0_PRIVILEGE_IN),
		.PER0_WANTS_INTR(PER0_WANTS_INTR_IN),
		.PER1_DNE(PER1_DNE_IN),
		.PER1_PRESENT(PER1_PRESENT_IN),
		.PER1_PRIVILEGE(PER1_PRIVILEGE_IN),
		.PER1_WANTS_INTR(PER1_WANTS_INTR_IN),
		.RESET(RESET & EC_EN),
		.SRT(srt_r),
		.WE(WE),
		.DATA_OUT(DATA_OUT_0),
		.DNE(W_DNE_EC),
		.IRQ(W_EC_IRQ),
		.PER0_BAT_EN(PER0_BAT_EN_OUT),
		.PER1_BAT_EN(PER1_BAT_EN_OUT)
		);

/*****Giano Controller****************************************************************/

	/*giano_controller gc(
		.ADDR_IN(ADDR_IN),
		.BYTES(BYTES),
		.CLK(MEMCLK),
		.DATA_IN(DATA_IN_EDF),
		.OE(OE),
		.RESET(RESET & GC_EN),
		.SRT(srt_r),
		.WE(WE),
		.BRAM_ENABLE(~BRAM_EN),
		.BRAM_CONFIG_ADDR(`BRAM_ADDR),
		.BRAM_CONFIG_SIZE(32'h0000ffff),
		.BRAM_MEM_ADDR(32'hffff0000),
		.BRAM_MEM_SIZE(32'h0000ffff),
		.SRAM_ENABLE(~SRAM_EN),
		.SRAM_CONFIG_ADDR(`SRAM_ADDR),
		.SRAM_CONFIG_SIZE(32'h00000008),
		.SRAM_MEM_ADDR(32'h00000000),
		.SRAM_MEM_SIZE(32'h01000000),
		//.SRAM_MEM_SIZE(32'hffffffff),
		.FLASH_ENABLE(~FLASH_EN),
		.FLASH_CONFIG_ADDR(`FLASH_ADDR),
		.FLASH_CONFIG_SIZE(32'h00000008),
		.FLASH_MEM_ADDR(32'hf0000000),
		.FLASH_MEM_SIZE(32'h00800000),
		.DDRAM_ENABLE(~DDRAM_EN),
		.DDRAM_CONFIG_ADDR(`DDRAM_ADDR),
		.DDRAM_CONFIG_SIZE(32'h0000ffff),
		.DDRAM_MEM_ADDR(32'he0000000),
		.DDRAM_MEM_SIZE(32'h0fffffff),
		.INTCTR_ENABLE(~INTCTR_EN),
		.INTCTR_ADDR(`INTCTR_ADDR),
		.INTCTR_SIZE(32'h00000018),
		.USART_ENABLE(~USART_EN),
		.USART_ADDR(`USART_ADDR),
		.USART_SIZE(32'h00000040),
		.TIMER_ENABLE(~TIMER_EN),
		.TIMER_ADDR(`TIMER_ADDR),
		.TIMER_SIZE(32'h00000020),
		.WATCHDOG_ENABLE(~WATCHDOG_EN),
		.WATCHDOG_ADDR(`WATCHDOG_ADDR),
		.WATCHDOG_SIZE(32'h0000ffff),
		.GPIO_ENABLE(~GPIO_EN),
		.GPIO_ADDR(`GPIO_ADDR),
		.GPIO_SIZE(32'h00000040),
		.IDE_ENABLE(~IDE_EN),
		.IDE_ADDR(`IDE_ADDR),
		.IDE_SIZE(32'h0000ffff),
		.LCD_ENABLE(~LCD_EN),
		.LCD_ADDR(`LCD_ADDR),
		.LCD_SIZE(32'h0000ffff),
		.PS2_ENABLE(~PS2_EN),
		.PS2_ADDR(`PS2_ADDR),
		.PS2_SIZE(32'h0000ffff),
		.VGA_ENABLE(~VGA_EN),
		.VGA_ADDR(`VGA_ADDR),
		.VGA_SIZE(32'h0000ffff),
		.ETHERNET_ENABLE(~ETHERNET_EN),
		.ETHERNET_ADDR(`ETHERNET_ADDR),
		.ETHERNET_SIZE(32'h0000ffff),
		.PWRMG_ENABLE(~PWRMG_EN),
		.PWRMG_ADDR(`PWRMG_ADDR),
		.PWRMG_SIZE(32'h0000ffff),
		.AC97_ENABLE(~AC97_EN),
		.AC97_ADDR(`AC97_ADDR),
		.AC97_SIZE(32'h0000ffff),
		.EC_ENABLE(~EC_EN),
		.EC_ADDR(`EC_ADDR),
		.EC_SIZE(32'h00000040),
		.TRAKKIES_ENABLE(TRAKKIES_EN),
		.TRAKKIES_ADDR(32'h80000000),
		.TRAKKIES_SIZE(32'h0fffffff),
		.DATA_OUT(DATA_OUT_0),
		.DNE(W_DNE_GC),
		.IRQ(AIRQ_GC)
		);*/

/*********************************************************************/

	always@(posedge MEMCLK)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			dne_r <= 1'b1;
			srt_r <= 1'b0;
			state <= 2'b0;
			timeout <= 10'b0;
			data_out_r <= 32'b0;
			irq_reg <= 32'b0;
		end
		else
		begin
			irq_reg <= IRQ;
			case(state)
				2'b00		:	begin
									/* Waiting for a request */
									dne_r <= 1'b1;
									srt_r <= 1'b0;
									timeout <= 10'b0;
									if (SRT)
									begin
										/* Recieved Request */
										data_out_r <= 32'hffffffff;
										state <= 2'b01;
										//$stop;
									end
								end
				2'b01		:	begin
									dne_r <= 1'b0;
									srt_r <= 1'b1;
									timeout <= timeout + 1;
									if (~MDNE)
									begin
										/* Request Acknowledged by Peripheral */
										state <= 2'b10;
										//$stop;
									end
									else if (timeout == 10'b1000)
									begin
										/* No Peripheral Recognized Request, Timeout */
										data_out_r <= 32'hffffffff;
										state <= 2'b00;
										//$stop;
									end
								end
				2'b10		:	begin
									srt_r <= 1'b0;
									timeout <= timeout + 1;
									if (MDNE)
									begin
										/* Request Completed */
										if (OE) data_out_r <= DATA_OUT_0;
										else data_out_r <= 32'hffffffff;
										state <= 2'b00;
										//$stop;
									end
									else if (timeout == 10'b0)
									begin
										/* Request Failed to complete before Timout */
										data_out_r <= 32'hffffffff;
										state <= 2'b00;
										//$stop;
									end
								end
			endcase
		end
	end

endmodule
