/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	memory_interface												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	MEMORY_INTERFACE is the top level interface for the memory			*/
/*	subsystem to the eMIPS microprocessor pipeline.  Like in the		*/
/*	standard MIPS, the interface exposes two ports, one for the			*/
/*	instruction fetch, the memory access and the extension.  Since		*/
/*	the system has only a single path to memory the MEMORY_INTERFACE	*/
/* is broken into three parts.													*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Extension Peripherals, Bug Fixes, see Manual		*/
/*																							*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module memory_interface(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BLS,						/* Byte Load/Store Data Memory */
	input				BLS_EXT_IN,				/* Byte Load/Store Extension */ 
	input				BRAMCLK,					/* Blockram Clock 50 - 100 MHZ */
	input				BRDY,						/* System ACE Buffer Ready */
	input				CLK,						/* System Clock 50 - 100 MHZ */
	input				CLK_IN_PER,				/* System ACE Clock */
	input [31:0]	DM_ADDR,					/* Data Memory Address */
	input [31:0]	DM_DATA_IN,				/* Data Memroy Data In */
	input				DM_OE,					/* Data Memory Output Enable */
	input				DM_WE,					/* Data Memory Write Enable */
	input				EXC_EXT_MEM_IN,		/* Extension Exception */
	input				EXC_IF_MEM,				/* Instruction Fetch Exception */
	input				EXC_MA_MEM,				/* Memory Access Exception */
	input				FLASHCLK,				/* Flash Clock 50 - 100 MHZ */
	input				FLUSH,					/* Instruction Fetch Flush */
	input [31:0]	GPIO_OUT,				/* Data from GPIO bus */
	input				HLS,						/* Halfword Load/Store Data Memory */
	input				HLS_EXT_IN,				/* Halfword Load/Store Extension */
	input [31:0]	IM_ADDR,					/* Instruction Memory Address */
	input				IM_OE,					/* Instruction Memory Output Enable */
	input				INT_EXT0_IN,			/* */
	input				INT_EXT1_IN,			/* */
	input				IRQ_SYSACE,				/* System ACE IRQ */
	input [31:0]	MADDR_EXT_IN,			/* Extension Address */
	input [31:0]	MDATA_EXT_IN,			/* Extension Data In */
	input [31:0]	MDATA_OUT,				/* Data from Memory Bus */
	input [3:0]		MDATAP_OUT,				/* Data Parity from Memory Bus */
	input				MEMCLK,					/* Memory Clock 50 - 100 MHZ */
	input				MEMCLK2,					/* Memory Clock 50 - 100 MHZ */
	input				MLOCK_EXT_IN,			/* Memory Interface Lock */
	input				MOE_EXT_IN,				/* Extension Output Enable */
	input				MWE_EXT_IN,				/* Extension Write Enable */
	input [31:0]	PC,						/* Program Counter Writeback, Committed */
	input				PCLK,						/* Pipeline Clock */
	input [15:0]	PDATA_OUT,				/* Data from Peripheral Bus */
	input				PER0_DNE_IN,			/* */
	input				PER0_PRESENT_IN,		/* */
	input				PER0_PRIVILEGE_IN,	/* */
	input				PER0_WANTS_INTR_IN,	/* */
	input				PER1_DNE_IN,			/* */
	input				PER1_PRESENT_IN,		/* */
	input				PER1_PRIVILEGE_IN,	/* */
	input				PER1_WANTS_INTR_IN,	/* */
	input				RESET,					/* System Reset */
	input				RNL,						/* Right/Left Unaligned Load/Store Data Memory */
	input				RNL_EXT_IN,				/* Right/Left Unaligned Load/Store Extension */
	input				RX,						/* Serial Recieve */
	input				SRAMCLK,					/* SRAM Clock 50 - 100 MHZ */
	input				STALL,					/* Pipeline Stall */
	input				TMRCLK,					/* Timer Clock 10 MHZ */
	input				USARTCLK,				/* USART Clock 100 MHZ */
	input 			RE,
	input [31:0]	MDATA_PER_EXT_IN,
	input 			BUBBLE_WB,
	input [31:0]	PC_WB,
	input				PER_EXT0,
	input				PER_EXT1,
	input				LC,
	/* OUTPUT PORTS */
	output			BYTE,						/* Flash Byte Mode */
	output			BYTESEL,					/* Flash Byte Select */
	output			CE_FLASH,				/* Flash Chipselect */
	output [31:0]	DM_DATA_OUT,			/* Data Memory Data Out to Data Path */
	output			DNE,						/* All Transactions complete */
	output [31:0]	GPIO_IN,					/* Data to GPIO bus */
	output [31:0]	GPIO_TR,					/* GPIO bus Tristate Enable */
	output [31:0]	IM_DATA_OUT,			/* Instruction Memory Data Out to Data Path*/
	output			IRQ,						/* Hardware Interrupt */
	output [23:0]	MADDR_OUT,				/* Address Memory Bus */
	output [31:0]	MADDR_PER_OUT,			/* Extension Peripheral Address Bus */
	output [31:0]	MADDR_SNOOP_OUT,		/* Address Monitor */
	output [1:0]	MBYTES_PER_OUT,		/* */
	output [31:0]	MDATA_EXT_OUT,			/* Extension Data Out to Data Path */
	output [31:0]	MDATA_IN,				/* Data to Memory Bus */
	output [31:0]	MDATA_PER_OUT,			/* Extension Peripheral Data Bus */
	output [31:0]	MDATA_SNOOP_OUT,		/* Data Monitor */
	output [31:0]	MDATA_TR,				/* Memory Bus Tristate Enable */ 
	output [3:0]	MDATAP_IN,				/* Data Parity to Memory Bus */
	output [3:0]	MDATAP_TR,				/* Memory Bus Parity Tristate Enable */
	output			MDATAVLD_EXT_OUT,		/* Extension Data Valid */
	output			MDATAVLD_SNOOP_OUT,	/* Data Valid Monitor */
	output			MODE,						/* SRAM Mode */
	output			MOE_PER_OUT,			/* Extension Peripheral Output Enable */
	output			MOE_SNOOP_OUT,			/* Output Enable Monitor */
	output			MSRT_PER_OUT,	/* */
	output			MWE_PER_OUT,			/* Extension Peripheral Write Enable */
	output			MWE_SNOOP_OUT,			/* Write Enable Monitor */
	output			NADVLD,					/* SRAM Advanced Load */
	output [3:0]	NBW,						/* SRAM Byte Enables */
	output			NCE_SRAM,				/* SRAM Chipselects */
	output			NCE_SYSACE,				/* System ACE Chipselects */
	output			NCEN,						/* SRAM Clock Enable */
	output			NRP,						/* Flash Hardware Reset */
	output			NOE_MEM,					/* Output Enable Memory Bus */
	output			NOE_PER,					/* Output Enable Peripheral Bus */
	output			NWE_MEM,					/* Write Enable Memroy Bus */
	output			NWE_PER,					/* Write Enable Peripheral Bus */
	output [6:0]	PADDR_OUT,				/* Address Peripheral Bus */
	output [15:0]	PDATA_IN,				/* Data Parity to Memory Bus */
	output [15:0]	PDATA_TR,				/* Memory Bus Parity Tristate Enable */
	output			PER0_BAT_EN_OUT,		/* */
	output			PER1_BAT_EN_OUT,		/* */
	output			SCLK,						/* USART Baud Clock */
	output			TX,						/* Serial Transmit */
	output			ZZ,						/* SRAM Sleep */
	output			ALS,
	/* IO PORTS */
	inout				CLK_EXT					/* Timer External Clock */
	);

/*****Signals****************************************************************/

	wire [31:0]	ADDR;				/* Address bus to Memory Controller */
	wire [31:0]	DATA;				/* Serialized Data bus */
	wire [1:0]	BYTES;			/* Bytes to Memory Controller */
	wire [5:0]	DRWCTR;			/* Data Read/Write Control */
	wire [31:0]	DMADDR;			/* Data Address bus to Memroy Bus Front */
	wire [31:0]	DMDATA;			/* Data bus to Memory Bus Front */
	wire			DRWDNE;			/* Data Read/Write Done */
	wire			DRWSRT;			/* Data Read/Write Start */
	wire [5:0]	IREADCTR;		/* Instruction Read Control */
	wire [31:0]	IMADDR;			/* Instruction Address bus */
	wire			IREADDNE;		/* Instruction Read Done */
	wire			IREADSRT;		/* Instruction Read Start */
	wire [31:0]	MADDR_BUS;		/* */
	wire [1:0]	MBYTES_BUS;		/* */
	wire [31:0]	MDATA_BUS_OUT;	/* */
	wire			MDNE;				/* Memory Controller Done */
	wire [31:0]	MFBDATA_IN;		/* Data bus to Memory Controller */
	wire [31:0]	MFBDATA_OUT;	/* Data bus from Memory Controller */
	wire			MOE_BUS;			/* */
	wire 			MSRT_BUS;	/* */
	wire			MWE_BUS;			/* */
	wire			OE;				/* Memory Controller Output Enable */
	wire			SRT;				/* Memory Cotnroller Start */
	wire [31:0]	W_EXTMADDR;		/* Extension Address bus */
	wire [31:0]	W_EXTMDATA;		/* Extension Data bus */
	wire [5:0]	W_EXTRWCTR;		/* Extension Read/Write Control */
	wire			W_EXTRWDNE;		/* Extension Read/Write Done */
	wire			W_EXTRWSRT;		/* Extension Read/Write Start */
	wire [31:0]	W_MADDR_PER;	/* Extension Peripheral Address bus */
	wire [31:0]	W_MDATA_PER;	/* Extension Peripheral Data bus */
	wire			W_MOE_PER;		/* Extension Peripheral Output Enable */
	wire			W_MWE_PER;		/* Extension Peripheral Write Enable */
	wire [31:0]	W_PER_DATA;		/* */
	wire			WE;				/* Memory Controller Write Enable */
	wire			TBLS;
	wire			THLS;
	wire			TMEMREAD;
	wire 			TSRT;
	wire			TDNE;
	wire [31:0]	VADDR;
	wire [31:0] PADDR;
	wire 			ICLKDNE;
	wire			ICLKSRT;
	wire [5:0]	ICLKCTR;
	wire [31:0]	ICLKPC;
	wire			INSTRCLK;
	wire [5:0]	CONTROL;
	

/*********************************************************************/

	assign MADDR_PER_OUT	=	MADDR_BUS;
	assign MDATA_PER_OUT	=	MDATA_BUS_OUT;
	assign MOE_PER_OUT	=	MOE_BUS;
	assign MSRT_PER_OUT 	=	MSRT_BUS;
	assign MWE_PER_OUT	=	MWE_BUS;
	assign MBYTES_PER_OUT	=	MBYTES_BUS;

/*****Memory Arbiter****************************************************************/

	memory_arbiter ma(
		.BLS(BLS),
		.BLS_EXT_IN(BLS_EXT_IN),
		.DATA(DATA),
		.DM_ADDR(DM_ADDR),
		.DM_DATA_IN(DM_DATA_IN),
		.DM_DATA_OUT(DM_DATA_OUT),
		.DM_OE(DM_OE),
		.DM_WE(DM_WE),
		.DMADDR(DMADDR),
		.DMDATA(DMDATA),
		.DNE(DNE),
		.DRWCTR(DRWCTR),
		.DRWDNE(DRWDNE),
		.DRWSRT(DRWSRT),
		.EXC_EXT_MEM_IN(EXC_EXT_MEM_IN),
		.EXC_IF_MEM(EXC_IF_MEM),
		.EXC_MA_MEM(EXC_MA_MEM),
		.EXTMADDR_OUT(W_EXTMADDR),
		.EXTMDATA_OUT(W_EXTMDATA),
		.EXTRWCTR_OUT(W_EXTRWCTR),
		.EXTRWDNE_IN(W_EXTRWDNE),
		.EXTRWSRT_OUT(W_EXTRWSRT),
		.FLUSH(FLUSH),
		.HLS(HLS),
		.HLS_EXT_IN(HLS_EXT_IN),
		.IM_ADDR(IM_ADDR),
		.IM_DATA_OUT(IM_DATA_OUT),
		.IM_OE(IM_OE),
		.IMADDR(IMADDR),
		.IREADCTR(IREADCTR),
		.IREADDNE(IREADDNE),
		.IREADSRT(IREADSRT),
		.MADDR_EXT_IN(MADDR_EXT_IN), 
		.MDATA_EXT_IN(MDATA_EXT_IN), 
		.MDATA_EXT_OUT(MDATA_EXT_OUT),
		.MDATAVLD_EXT_OUT(MDATAVLD_EXT_OUT),
		.MEMCLK(MEMCLK2),
		.MOE_EXT_IN(MOE_EXT_IN),
		.MWE_EXT_IN(MWE_EXT_IN),
		.PCLK(PCLK),
		.RESET(RESET),
		.RNL(RNL),
		.RNL_EXT_IN(RNL_EXT_IN),
		.STALL(STALL),
		.BUBBLE_WB(BUBBLE_WB),
		.PC_WB(PC_WB),
		.ICLKDNE(ICLKDNE),
		.ICLKSRT(ICLKSRT),
		.ICLKCTR(ICLKCTR),
		.ICLKPC(ICLKPC),
		.LC(LC),
		.CONTROL(CONTROL),
		.ALS(ALS)
		);

/*****Memory Bus Front****************************************************************/

	memory_bus_front mbf(
		.ADDR(ADDR),
		.BYTES(BYTES),
		.DATA(DATA),
		.DMADDR(DMADDR),
		.DMDATA(DMDATA),
		.DRWCTR(DRWCTR),
		.DRWDNE(DRWDNE),
		.DRWSRT(DRWSRT),
		.EXTMADDR_IN(W_EXTMADDR),
		.EXTMDATA_IN(W_EXTMDATA), 
		.EXTRWCTR_IN(W_EXTRWCTR),
		.EXTRWDNE_OUT(W_EXTRWDNE),
		.EXTRWSRT_IN(W_EXTRWSRT),
		.IMADDR(IMADDR),
		.IREADCTR(IREADCTR),
		.IREADDNE(IREADDNE),
		.IREADSRT(IREADSRT),
		.MADDR_SNOOP_OUT(MADDR_SNOOP_OUT),
		.MDATA_IN(MFBDATA_IN),
		.MDATA_SNOOP_OUT(MDATA_SNOOP_OUT),
		.MDATA_OUT(MFBDATA_OUT),
		.MDATAVLD_SNOOP_OUT(MDATAVLD_SNOOP_OUT),
		.MDNE(MDNE),
		.MEMCLK(MEMCLK2),
		.MOE_SNOOP_OUT(MOE_SNOOP_OUT),
		.MWE_SNOOP_OUT(MWE_SNOOP_OUT), 
		.OE(OE),
		.RESET(RESET),
		.SRT(SRT),
		.WE(WE),
		.TBLS(TBLS),
		.THLS(THLS),
		.TMEMREAD(TMEMREAD),
		.VADDR(VADDR),
		.TDNE(TDNE),
		.PADDR(PADDR),
		.TSRT(TSRT),
		.ICLKSRT(ICLKSRT),
		.ICLKCTR(ICLKCTR),
		.ICLKPC(ICLKPC),
		.ICLKDNE(ICLKDNE),
		.INSTRCLK(INSTRCLK),
		.CONTROL(CONTROL),
		.EXC_MA_MEM(EXC_MA_MEM)
		);

/*****Memory Management Unit****************************************************************/

	memory_management_unit mmu(
		.CLK(MEMCLK2),
		.BLS(TBLS),
		.HLS(THLS),
		.MEMREAD(TMEMREAD),
		.TSRT(TSRT),
		.RESET(RESET),
		.VADDR(VADDR),
		.TDNE(TDNE),
		.PADDR(PADDR),
		.INSTRCLK(INSTRCLK)
		);

/*****Memory Controller****************************************************************/

	memory_controller mc(
		.ADDR_IN(ADDR),
		.ADDR_MEM(MADDR_OUT),
		.ADDR_PER(PADDR_OUT),
		.AIRQ(IRQ),
		.BRAMCLK(BRAMCLK),
		.BRDY(BRDY),
		.BYTE(BYTE),
		.BYTES(BYTES),
		.BYTESEL(BYTESEL),
		.CE_FLASH(CE_FLASH),
		.CLK_EXT(CLK_EXT),
		.CLK_IN_PER(CLK_IN_PER),
		.DATA_IN(MFBDATA_IN),
		.DATA_OUT(MFBDATA_OUT),
		.DATA_MEM_IN(MDATA_IN),
		.DATA_MEM_OUT(MDATA_OUT),
		.DATA_MEM_TR(MDATA_TR),
		.DATA_PAR_IN(MDATAP_IN),
		.DATA_PAR_OUT(MDATAP_OUT),
		.DATA_PAR_TR(MDATAP_TR),
		.DATA_PER_IN(PDATA_IN),
		.DATA_PER_OUT(PDATA_OUT),
		.DATA_PER_TR(PDATA_TR),
		.DNE(MDNE),
		.FLASHCLK(FLASHCLK),
		.GPIO_IN(GPIO_IN),
		.GPIO_OUT(GPIO_OUT),
		.GPIO_TR(GPIO_TR),
		.INT_EXT0_IN(INT_EXT0_IN),
		.INT_EXT1_IN(INT_EXT1_IN),
		.IRQ_SYSACE(IRQ_SYSACE),
		.MADDR_BUS(MADDR_BUS),
		.MBYTES_BUS(MBYTES_BUS),
		.MDATA_BUS_IN(MDATA_PER_EXT_IN),
		.MDATA_BUS_OUT(MDATA_BUS_OUT),
		.MEMCLK(MEMCLK),
		.MODE(MODE),
		.MOE_BUS(MOE_BUS),
		.MSRT_BUS(MSRT_BUS),
		.MWE_BUS(MWE_BUS),
		.NADVLD(NADVLD),
		.NBW(NBW),
		.NCE_SRAM(NCE_SRAM),
		.NCE_SYSACE(NCE_SYSACE),
		.NCEN(NCEN),
		.NOE_MEM(NOE_MEM),
		.NOE_PER(NOE_PER),
		.NRP(NRP),
		.NWE_MEM(NWE_MEM),
		.NWE_PER(NWE_PER),
		.OE(OE),
		.PER0_BAT_EN_OUT(PER0_BAT_EN_OUT),
		.PER0_DNE_IN(PER0_DNE_IN),
		.PER0_PRESENT_IN(PER0_PRESENT_IN),
		.PER0_PRIVILEGE_IN(PER0_PRIVILEGE_IN),
		.PER0_WANTS_INTR_IN(PER0_WANTS_INTR_IN),
		.PER1_BAT_EN_OUT(PER1_BAT_EN_OUT),
		.PER1_DNE_IN(PER1_DNE_IN),
		.PER1_PRESENT_IN(PER1_PRESENT_IN),
		.PER1_PRIVILEGE_IN(PER1_PRIVILEGE_IN),
		.PER1_WANTS_INTR_IN(PER1_WANTS_INTR_IN),
		.RESET(RESET),
		.RX(RX),
		.SCLK(SCLK),
		.SRAMCLK(SRAMCLK),
		.SRT(SRT),
		.TMRCLK(TMRCLK),
		.TX(TX),
		.USARTCLK(USARTCLK),
		.WE(WE),
		.ZZ(ZZ),
		.RE(RE),
		.PER_EXT0(PER_EXT0),
		.PER_EXT1(PER_EXT1)	
		);
		
endmodule
