 /* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	memory_management_unit										*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module memory_management_unit(
	input CLK,
	input BLS,
	input HLS,
	input MEMREAD,
	input TSRT,
	input RESET,
	input [31:0] VADDR,
	input INSTRCLK,
	output TDNE,
	output [31:0] PADDR
	);

	wire [31:0] PADDR_0;

	reg bls_reg, hls_reg, memread_reg, tsrt_reg, tdne_reg;
	reg [31:0] vaddr_reg, paddr_reg;
	reg [1:0] state;

	initial
	begin
		bls_reg = 1'b0;
		hls_reg = 1'b0;
		memread_reg = 1'b0;
		tsrt_reg = 1'b0;
		tdne_reg = 1'b1;
		vaddr_reg = 32'b0;
		paddr_reg = 32'b0;
		state = 2'b00;
	end

	assign TDNE = tdne_reg;
	assign PADDR = paddr_reg;

	address_translation at(
		.BLS(bls_reg),
		.HLS(hls_reg),
		.MEMREAD(memread_reg),
		.PADDR(PADDR_0),
		.VADDR(vaddr_reg)
		);

	always@(posedge CLK)
	begin
		if (RESET == 0)
		begin
			tsrt_reg <= 1'b0;
			bls_reg <= 1'b0;
			hls_reg <= 1'b0;
			memread_reg <= 1'b0;
			vaddr_reg <= 32'b0;
		end
		else
		begin
			tsrt_reg <= TSRT;
			bls_reg <= BLS;
			hls_reg <= HLS;
			memread_reg <= MEMREAD;
			vaddr_reg <= VADDR;
		end
	end

	always@(posedge CLK)
	begin
		if (RESET == 0)
		begin
			tdne_reg <= 1'b1;
			paddr_reg <= 32'b0;
		end
		else
		begin
			case(state)
				2'b00		:	begin
								if (tsrt_reg)
								begin
									tdne_reg <= 1'b0;
									state <= 2'b01;
									//$stop;
								end
							end
				2'b01		:	begin
								if (INSTRCLK)
								begin
									paddr_reg <= vaddr_reg;
									//$stop;
								end
								else
								begin
									paddr_reg <= PADDR_0;
								end
								state <= 2'b10;
								//$stop;
							end
				2'b10		:	begin
								tdne_reg <= 1'b1;
								state <= 2'b11;
								//$stop;
							end
				2'b11		:	begin
								if (~tsrt_reg)
								begin
									state <= 2'b00;
									//$stop;
								end
							end
				default	:	begin
							end
			endcase
		end
	end

endmodule
