/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	multiplier														*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"	/* Constant Definitiions */

module multiplier(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	A,			/* ALU Operand A */
	input [31:0]	B,			/* ALU Operand B */
	input 			CLK,		/* System Clock 50 - 100 MHZ */
	input [5:0]		CTR,		/* ALU Control/Opcode */
	input 			PCLK,		/* Pipeline Clock */
	input 			RESET,	/* System Reset */
	/* OUTPUT PORTS */
	output 			RDY,		/* Ready */
	output [31:0]	RESULT,	/* Result */
	/* DEBUG PORTS */
	output [31:0]	HI,		/* Hi Register */
	output [31:0]	LO			/* Lo Register */
	);

/*****Signals****************************************************************/
	
	wire			DNE_M;		/* Multiplication Done */
	wire			DNE_D;		/* Division Done */
	wire [63:0]	PRODUCT;		/* Multiplication Result */
	wire [31:0]	QUOTIENT;	/* Division Result */
	wire [31:0]	REMAINDER;	/* Division Remainder */

/*****Registers****************************************************************/

	reg [31:0]	Ainreg;		/* ALU Operand A */
	reg [31:0]	Binreg;		/* ALU Operand B */
	reg [5:0]	ctrreg;		/* ALU Control/Opcode */
	reg [1:0]	dnecnt;		/* Done Edge Detection */
	reg			dsrt;			/* Start Divider */
	reg			fin;			/* Results Pending */
	reg			msrt;			/* Start Multiplier */
	reg [1:0]	pclkcnt;		/* Pipeline Clock Edge Detection */
	reg			rdy_r;		/* Ready */
	reg [31:0]	resultHI;	/* Hi Register */
	reg [31:0]	resultLO;	/* Lo Register */
	reg			sign;			/* Signed/Unsigned */

/*****Initialization****************************************************************/
	
	initial
	begin
		Ainreg = 32'b0;
		Binreg = 32'b0;
		ctrreg = 6'b0;
		dnecnt = 2'b00;
		dsrt = 1'b0;
		fin = 1'b0;
		msrt = 1'b0;
		rdy_r = 1'b0;
		resultHI = 32'b0;
		resultLO = 32'b0;
		sign = 1'b0;
	end

/*********************************************************************/
	
	assign HI		=	resultHI;
	assign LO		=	resultLO;
	assign RDY		=	rdy_r;
	assign RESULT	=	(CTR == `MFHI)?	resultHI:
							(CTR == `MFLO)?	resultLO:
													32'b0;

/*****32bit Multiply****************************************************************/
	
	multiply32 m0(
		.CLK(CLK),
		.RESET(RESET),
		.SRT(msrt),
		.DNE(DNE_M),
		.SIGNED(sign),
		.MULTIPLIER(Binreg),
		.MULTIPLICAND(Ainreg),
		.PRODUCT(PRODUCT)
		);

/*****32bit Divide****************************************************************/
		
	divide32 d0(
		.CLK(CLK),
		.RESET(RESET),
		.SRT(dsrt),
		.DNE(DNE_D),
		.SIGNED(sign),
		.DIVIDER(Binreg),
		.DIVIDEND(Ainreg),
		.QUOTIENT(QUOTIENT),
		.REMAINDER(REMAINDER)
		);

/*********************************************************************/

	/* Ready */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			rdy_r <= 1'b0;
		end
		else
		begin
			case(pclkcnt)
				2'b01		:	begin
									/* Raise Ready on a Posedge of the Pipeline Clock */
									rdy_r <= dnecnt[0] & ~fin & ~msrt & ~dsrt;
								end
				default	:	begin
									/* Lower Ready when Operation Starts */
									if (~(dnecnt[0] & ~fin) | (msrt | dsrt)) rdy_r <= dnecnt[0] & ~fin & ~msrt & ~dsrt;
								end
			endcase
		end
	end
	
	always@(posedge CLK)
	begin
		case(RESET)
			1'b0		:	begin
								/* Reset */
								Ainreg <= 32'b0;
								Binreg <= 32'b0;
								ctrreg <= 6'b111111;
								dnecnt <= 2'b11;
								dsrt <= 1'b0;
								fin <= 1'b0;
								msrt <= 1'b0;
								pclkcnt <= 2'b00;
								resultHI <= 32'b0;
								resultLO <= 32'b0;
								sign <= 1'b0;
							end
			default	:	begin
								casex({pclkcnt,RDY,msrt,dsrt,CTR})
									11'b01x1xxxxxxx	:	begin
																	/* Lower Start Multiplier */
																	msrt <= 1'b0;
																end
									11'b01xx1xxxxxx	:	begin
																	/* Lower Start Divider */
																	dsrt <= 1'b0;
																end
									11'bxx000xxxxxx	:	begin
																	/* Wait for Operation to Complete */
																end
									{5'b10100,`MULT}	:	begin
																	/* Start Multiply */
																	ctrreg <= CTR;
																	Ainreg <= A;
																	Binreg <= B;
																	sign <= 1'b1;
																	msrt <= 1'b1;
																end
									{5'b10100,`MULTU}	:	begin
																	/* Start Unsigned Multiply */
																	ctrreg <= CTR;
																	Ainreg <= A;
																	Binreg <= B;
																	sign <= 1'b0;
																	msrt <= 1'b1;
																end
									{5'b10100,`DIV}	:	begin
																	/* Start Divide */
																	ctrreg <= CTR;
																	Ainreg <= A;
																	Binreg <= B;
																	sign <= 1'b1;
																	dsrt <= 1'b1;
																end
									{5'b10100,`DIVU}	:	begin
																	/* Start Unsigned Divide */
																	ctrreg <= CTR;
																	Ainreg <= A;
																	Binreg <= B;
																	sign <= 1'b0;
																	dsrt <= 1'b1;
																end
									{5'b10100,`MTHI}	:	begin
																	/* Set Hi Register */
																	ctrreg <= CTR;
																	resultHI <= A;
																end
									{5'b10100,`MTLO}	:	begin
																	/* Set Lo Register */
																	ctrreg <= CTR;
																	resultLO <= A;
																end
									default				:	begin
																end
							endcase
							if ((pclkcnt == 2'b01) && (fin))
							begin
								fin <= 1'b0;
								case(ctrreg)
									`MULT		:	begin
														/* Finish Multiply */
														resultHI <= PRODUCT[63:32];
														resultLO <= PRODUCT[31:0];
													end
									`MULTU	:	begin
														/* Finish Unsigned Multiply */
														resultHI <= PRODUCT[63:32];
														resultLO <= PRODUCT[31:0];
													end
									`DIV		:	begin
														/* Finish Divide */
														resultHI <= REMAINDER;
														resultLO <= QUOTIENT;
													end
									`DIVU		:	begin
														/* Finish Unsinged Divide */
														resultHI <= REMAINDER;
														resultLO <= QUOTIENT;
													end
								endcase
							end
							/* Pipeline Clock Edge Detection */
							pclkcnt <= {pclkcnt[0],PCLK};
							/* Done Edge Detection */
							dnecnt <= {dnecnt[0],(DNE_M & DNE_D)};
							if (dnecnt == 2'b01)
							begin
								/* Operation Finished */
								fin <= 1'b1;
							end
						end
		endcase
	end
	
endmodule

/*****32bit Multiply****************************************************************/

module multiply32(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,				/* System Clock 50 - 100 MHZ */
	input [31:0]	MULTIPLIER,		/* Multiplier, Operand B */
	input [31:0]	MULTIPLICAND,	/* Multiplicand, Operand A */
	input				RESET,			/* System Reset */
	input				SIGNED,			/* Signed */
	input				SRT,				/* Start */
	/* OUTPUT PORTS */
	output			DNE,				/* Done */
	output [63:0]	PRODUCT			/* Product, {HI,LO} */
	);

/*****Signals****************************************************************/
	
	wire [31:0]	MULTIPLIER_U;		/* Unsigned Multiplier */
	wire [31:0]	MULTIPLICAND_U;	/* Unsigned Multiplicand */
	wire			NO;					/* Negative Output */

/*****Registers****************************************************************/

	reg [4:0]	bitcnt;						/* Bit Counter */
	reg			dne_0_reg;					/* Pipelined Done 0 */
	reg			dne_1_reg;					/* Pipelined Done 1 */
	reg			dne_r;						/* Done */
	reg			dne_reg;						/* Pipelined Done */
	reg [63:0]	multiplicand_r;			/* Multiplicand */
	reg [31:0]	multiplicand_u_0_reg;	/* Pipelined 2-Compliment Converted Multiplicand 0 */
	reg [31:0]	multiplicand_u_1_reg;	/* Pipelined 2-Compliment Converted Multiplicand 1 */
	reg [31:0]	multiplicand_u_reg;		/* Unsigned Multiplicand */
	reg [31:0]	multiplier_r;				/* Multiplier */
	reg [31:0]	multiplier_u_0_reg;		/* Pipelined 2-Compliment Converted Multipler 0 */
	reg [31:0]	multiplier_u_1_reg;		/* Pipelined 2-Compliment Converted Multipler 1 */
	reg [31:0]	multiplier_u_reg;			/* Unsigned Multiplier */
	reg			no_reg;						/* Negative Output */
	reg [63:0]	product_0_reg;				/* Pipelined 2-Compliment Converted Product 0 */
	reg [63:0]	product_1_reg;				/* Pipelined 2-Compliment Converted Product 1 */
	reg [63:0]	product_r;					/* Unsigned Product */
	reg [63:0]	product_reg;				/* Product */
	reg			srt_0_reg;					/* Pipelined Start 0 */
	reg			srt_1_reg;					/* Pipelined Start 1 */
	reg			srt_reg;						/* Pipelined Start */
	reg [1:0]	state;						/* State Variable */

/*****Initialization****************************************************************/

	initial
	begin
		bitcnt = 5'b0;
		dne_0_reg = 1'b1;
		dne_1_reg = 1'b1;
		dne_r = 1'b1;
		dne_reg = 1'b1;
		multiplicand_r = 64'hffffffffffffffff;
		multiplicand_u_0_reg = 32'b0;
		multiplicand_u_1_reg = 32'b0;
		multiplicand_u_reg = 32'b0;
		multiplier_r = 32'hffffffff;
		multiplier_u_0_reg = 32'b0;
		multiplier_u_1_reg = 32'b0;
		multiplier_u_reg = 32'b0;
		no_reg = 1'b0;
		product_0_reg = 64'b0;
		product_1_reg = 64'b0;
		product_r = 64'hffffffffffffffff;
		product_reg = 64'b0;
		srt_0_reg = 1'b0;
		srt_1_reg = 1'b0;
		srt_reg = 1'b0;
		state = 2'b0;
	end
	
/*********************************************************************/

	assign DNE					=	dne_reg;
	assign NO					=	no_reg;
	assign MULTIPLIER_U		= 	multiplier_u_reg;
	assign MULTIPLICAND_U	=	multiplicand_u_reg;
	assign PRODUCT 			=	product_reg;

/*********************************************************************/

	/* Piplined Start & Done Signals */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)					
		begin
			/* Reset */
			dne_0_reg <= 1'b1;
			dne_1_reg <= 1'b1;
			dne_reg <= 1'b1;
			srt_0_reg <= 1'b0;
			srt_1_reg <= 1'b0;
			srt_reg <= 1'b0;
		end
		else						
		begin
			if (SIGNED & (MULTIPLIER[31] | MULTIPLICAND[31]))	srt_reg <= srt_0_reg;
			else 																srt_reg <= SRT;
			srt_0_reg <= 													srt_1_reg;
			srt_1_reg <= 													SRT;
			if (SIGNED & (MULTIPLIER[31] ^ MULTIPLICAND[31]))	dne_reg <= dne_0_reg;
			else 																dne_reg <= dne_r;
			dne_0_reg <= 													dne_1_reg;
			dne_1_reg <= 													dne_r;
		end
	end

	/* Pipelined Data */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)					
		begin
			/* Reset */
			multiplicand_u_0_reg <= 32'b0;
			multiplicand_u_1_reg <= 32'b0;
			multiplicand_u_reg <= 32'b0;
			multiplier_u_0_reg <= 32'b0;
			multiplier_u_1_reg <= 32'b0;
			multiplier_u_reg <= 32'b0;
			no_reg <= 1'b0;
			product_0_reg <= 64'b0;
			product_1_reg <= 64'b0;
			product_reg <= 64'b0;
		end
		else						
		begin
			if (SIGNED & MULTIPLIER[31])	multiplier_u_reg <= multiplier_u_0_reg;
			else 									multiplier_u_reg <= MULTIPLIER;
			multiplier_u_0_reg <= multiplier_u_1_reg + 32'b1;
			multiplier_u_1_reg <= ~MULTIPLIER;
			if (SIGNED & MULTIPLICAND[31])	multiplicand_u_reg <= multiplicand_u_0_reg;
			else 										multiplicand_u_reg <= MULTIPLICAND;
			multiplicand_u_0_reg <= multiplicand_u_1_reg + 32'b1;
			multiplicand_u_1_reg <= ~MULTIPLICAND;
			if (NO)	product_reg <= product_0_reg;
			else		product_reg <= product_r;
			product_0_reg <= product_1_reg + 64'b1;
			product_1_reg <= ~product_r;
			no_reg <= SIGNED & (MULTIPLIER[31] ^ MULTIPLICAND[31]);
		end
	end
	
	/* Multiply by Shift and Add */
	always@(posedge CLK)
	begin
		case(RESET)
			1'b0		:	begin
								/* Reset */
								bitcnt <= 5'b0;
								dne_r <= 1'b1;
								multiplier_r <= 32'hffffffff;
								multiplicand_r <= 64'hffffffffffffffff;
								product_r <= 64'hffffffffffffffff;
								state <= 2'b0;
							end
			default	:	begin
							case(state)
								2'b00	:	begin
												/* Initial State */
												dne_r <= 1'b1;
												bitcnt <= 5'b0;
												if (srt_reg)
												begin
													/* Start */
													state <= 2'b01;
												end
											end
								2'b01	:	begin
												/* Initialization State */
												dne_r <= 1'b0;
												bitcnt <= 5'b0;
												product_r <= 64'b0;
												multiplier_r <= MULTIPLIER_U;
												multiplicand_r <= {32'b0,MULTIPLICAND_U};
												if (~srt_reg)
												begin
													/* State Set */
													state <= 2'b10;
												end
											end
								2'b10	:	begin
												if ((bitcnt == 5'b0) && ((multiplier_r == 32'b0) || (multiplicand_r == 64'b0)))
												begin
													/* Multiplication by Zero */
													state <= 2'b00;
												end
												else
												begin
													if (multiplier_r[0])
													begin
														product_r <= product_r + multiplicand_r;
													end
													multiplier_r <= multiplier_r >> 1;
													multiplicand_r <= multiplicand_r << 1;
													bitcnt <= bitcnt + 1;	
													if (bitcnt == 5'b11111)
													begin
														state <= 2'b00;
													end
												end
											end
							endcase
						end
		endcase
	end
	
endmodule

/*****32bit Divide****************************************************************/

module divide32(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,			/* System Clock */
	input [31:0]	DIVIDEND,	/* Dividend, Operand A */
	input [31:0]	DIVIDER,		/* Divider, Operand B */
	input				RESET,		/* System Reset */
	input				SIGNED,		/* Signed */
	input				SRT,			/* Start */
	/* OUTPUT PORTS */
	output			DNE,			/* Done */
	output [31:0]	QUOTIENT,	/* Quotient, LO */
	output [31:0]	REMAINDER	/* Remainder, HI */
	);

/*****Signals****************************************************************/
	
	wire [63:0]	DIFF;				/* Difference of Divider from Dividend */
	wire [31:0]	DIVIDEND_U;		/* Unsigned Divident */
	wire [31:0]	DIVIDER_U;		/* Unsigned Divider */
	wire			NO;				/* Negative Output */
	
/*****Registers****************************************************************/

	reg [4:0]	bitcnt;				/* Bit Counter */
	reg [63:0]	dividend_r;			/* Dividend */
	reg [31:0]	dividend_u_0_reg;	/* Pipelined 2-Compliment Converted Dividend 0 */
	reg [31:0]	dividend_u_1_reg;	/* Pipelined 2-Compliment Converted Dividend 1 */
	reg [31:0]	dividend_u_reg;	/* Unsigned Dividend */
	reg [63:0]	divider_r;			/* Divider */
	reg [31:0]	divider_u_0_reg;	/* Pipelined 2-Compliment Converted Divider 0 */
	reg [31:0]	divider_u_1_reg;	/* Pipelined 2-Compliment Converted Divider 1 */
	reg [31:0]	divider_u_reg;		/* Unsigned Divider */
	reg 			dne_0_reg;			/* Pipelined Done 0 */
	reg 			dne_1_reg;			/* Pipelined Done 1 */
	reg 			dne_r;				/* Done */
	reg 			dne_reg;				/* Pipelined Done */
	reg 			no_reg;				/* Negative Output */
	reg [31:0]	quotient_0_reg;	/* Pipelined 2-Compliment Converted Quotient 0 */ 
	reg [31:0]	quotient_1_reg;	/* Pipelined 2-Compliment Converted Quotient 1 */
	reg [31:0]	quotient_r;			/* Unsigned Quotient */
	reg [31:0]	quotient_reg;		/* Quotient */
	reg [31:0]	remainder_0_reg;	/* Pipelined 2-Compliment Converted Remainder 0 */
	reg [31:0]	remainder_1_reg;	/* Pipelined 2-Compliment Converted Remainder 1 */
	reg [31:0]	remainder_reg;		/* Remainder */
	reg [1:0]	state;				/* State Variable */
	reg 			srt_0_reg;			/* Pipelined Start 0 */
	reg 			srt_1_reg;			/* Pipelined Start 1 */
	reg 			srt_reg;				/* Pipelined Start */

/*****Initialization****************************************************************/
	
	initial
	begin
		bitcnt = 5'b0;
		dividend_r = 64'hffffffffffffffff;
		dividend_u_0_reg = 32'b0;
		dividend_u_1_reg = 32'b0;
		dividend_u_reg = 32'b0;
		divider_r = 64'hffffffffffffffff;
		divider_u_0_reg = 32'b0;
		divider_u_1_reg = 32'b0;
		divider_u_reg = 32'b0;
		dne_0_reg = 1'b1;
		dne_1_reg = 1'b1;
		dne_r = 1'b1;
		dne_reg = 1'b1;
		no_reg = 1'b0;		
		quotient_0_reg = 32'b0;
		quotient_1_reg = 32'b0;
		quotient_reg = 32'b0;
		remainder_0_reg = 32'b0;
		remainder_1_reg = 32'b0;
		remainder_reg = 32'b0;
		srt_0_reg = 1'b0;
		srt_1_reg = 1'b0;
		srt_reg = 1'b0;
		state = 2'b0;
		quotient_r = 32'hffffffff;
	end

/*********************************************************************/
	
	assign DIFF			=	dividend_r - divider_r;
	assign DIVIDEND_U	= 	dividend_u_reg;
	assign DIVIDER_U	=	divider_u_reg;
	assign DNE			=	dne_reg;
	assign NO			=	no_reg;
	assign QUOTIENT	=	quotient_reg;
	assign REMAINDER	=	remainder_reg;

/*********************************************************************/

	/* Piplined Start & Done Signals */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)					
		begin
			/* Reset */
			dne_0_reg <= 1'b1;
			dne_1_reg <= 1'b1;
			dne_reg <= 1'b1;
			srt_0_reg <= 1'b0;
			srt_1_reg <= 1'b0;
			srt_reg <= 1'b0;
		end
		else						
		begin
			if (SIGNED & (DIVIDER[31] | DIVIDEND[31]))	srt_reg <= srt_0_reg;
			else 														srt_reg <= SRT;
			srt_0_reg <= 											srt_1_reg;
			srt_1_reg <= 											SRT;
			if (SIGNED & (DIVIDER[31] ^ DIVIDEND[31]))	dne_reg <= dne_0_reg;
			else 														dne_reg <= dne_r;
			dne_0_reg <= 											dne_1_reg;
			dne_1_reg <= 											dne_r;
		end
	end

	/* Pipelined Data */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			dividend_u_0_reg <= 32'b0;
			dividend_u_1_reg <= 32'b0;
			dividend_u_reg <= 32'b0;
			divider_u_0_reg <= 32'b0;
			divider_u_1_reg <= 32'b0;
			divider_u_reg <= 32'b0;
			no_reg <= 1'b0;
			quotient_0_reg <= 32'b0;
			quotient_1_reg <= 32'b0;
			quotient_reg <= 32'b0;
			remainder_0_reg <= 32'b0;
			remainder_1_reg <= 32'b0;
			remainder_reg <= 32'b0;
		end
		else
		begin
			if (SIGNED & DIVIDEND[31])	dividend_u_reg <= dividend_u_0_reg;
			else								dividend_u_reg <= DIVIDEND;
			dividend_u_0_reg <= dividend_u_1_reg + 32'b1;
			dividend_u_1_reg <= ~DIVIDEND;
			if (SIGNED & DIVIDER[31])	divider_u_reg <= divider_u_0_reg;
			else								divider_u_reg <= DIVIDER;
			divider_u_0_reg <= divider_u_1_reg + 32'b1;
			divider_u_1_reg <= ~DIVIDER;
			if (NO)	quotient_reg <= quotient_0_reg;
			else 		quotient_reg <= quotient_r;
			quotient_0_reg <= quotient_1_reg + 32'b1;
			quotient_1_reg <= ~quotient_r;
			if (SIGNED & DIVIDEND[31])	remainder_reg <= remainder_0_reg;
			else 								remainder_reg <= dividend_r[31:0];
			remainder_0_reg <= remainder_1_reg + 32'b1;
			remainder_1_reg <= ~dividend_r[31:0];
			no_reg <= SIGNED & (DIVIDER[31] ^ DIVIDEND[31]);
		end
	end

	/* Divide by Shift and Subtract */
	always@(posedge CLK)
	begin
		case(RESET)
			1'b0		:	begin
								/* Reset */
								dne_r <= 1'b1;
								state <= 2'b0;
								bitcnt <= 5'b0;
								quotient_r <= 32'hffffffff;
								divider_r <= 64'hffffffffffffffff;
								dividend_r <= 64'hffffffffffffffff;
							end
			default	:	begin
							case(state)
								2'b00	:	begin
												/* Initial State */
												dne_r <= 1'b1;
												bitcnt <= 5'b0;
												if (srt_reg)
												begin
													/* Start */
													state <= 2'b01;
												end
											end
								2'b01	:	begin
												/* Initialization State */
												dne_r <= 1'b0;
												bitcnt <= 5'b0;
												quotient_r <= 32'b0;
												divider_r <= {1'b0,DIVIDER_U,31'b0};
												dividend_r <= {32'b0,DIVIDEND_U};
												if (~srt_reg)
												begin
													/* State Set */
													state <= 2'b10;
												end
											end
								2'b10	:	begin
												if ((bitcnt == 5'b0) && ((divider_r == 64'b0) || (dividend_r == 64'b0)))
												begin
													/* Divide by Zero */
													quotient_r <= dividend_r[31:0];
													dividend_r <= 64'h0;
													state <= 2'b00;
												end
												else
												begin
													if (DIFF[63]) 
													begin
														quotient_r <= quotient_r << 1;
													end
													else
													begin
														dividend_r <= DIFF;
														quotient_r <= (quotient_r << 1) + 1;
													end
													divider_r <= divider_r >> 1;
													bitcnt <= bitcnt + 1;
													if (bitcnt == 5'b11111)
													begin
														state <= 2'b00;
													end
												end
										end
							endcase
						end
		endcase
	end
	
endmodule
