/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	pipeline_arbiter												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`define TISA_PR	2'b10		/* TISA Priority */

module pipeline_arbiter(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 		ACK_EXT0,	/* Extension 0 Enable Acknowledged */
	input 		ACK_EXT1,	/* Extension 1 Enable Acknowledged */
	input 		CLK,			/* System Clock 50 - 100 MHZ */
	input [7:0]	EXT0CTR,		/* Extension 0 Control Register */
	input [7:0]	EXT1CTR,		/* Extension 1 Control Register */
	input 		PCLK,			/* Pipeline Clock */
	input 		PI_EXT0,		/* Extension 0 Passive/Parrallel Instruction */
	input 		PI_EXT1,		/* Extension 1 Passive/Parrallel Instruction */
	input 		RESET,		/* System Reset */
	input 		RI_ID,		/* TISA Reserved/Recognized Instruction */
	input 		RI_EXT0,		/* Extension 0 Reserved/Recognized Instruction */
	input 		RI_EXT1,		/* Extension 1 Reserved/Recognized Instruction */
	input 		STALL,		/* Pipeline Stall */
	input			EXCEXT,		/* */
	/* OUTPUT PORTS */
	output 		EN_EXT0,		/* Extension 0 Enable */
	output 		EN_EXT1,		/* Extension 1 Enable */
	output 		EN_ID,		/* TISA Enable */
	output 		EXTSTALL,	/* Stall for Extension */
	output		GR_EXT0,		/* Extension 0 Grant Pipeline Resources */
	output 		GR_EXT1,		/* Extension 1 Grant Pipeline Resources */
	output 		RI_PA			/* Pipeline Arbiter Reserved/Recognized Instruction */
	);
	
/*****Signals****************************************************************/

	wire EN_EXT_0;				/* Extension 0 Recognized Instruction */
	wire EN_EXT_1;				/* Extension 1 Recognized Instruction */
	wire EN_EXT0_0;			/* Extension 0 Enable Asynchronous */
	wire EN_EXT1_0;			/* Extension 1 Enable Asynchronous */
	wire EN_ID_0;				/* TISA Enable Asynchronous */
	wire EXT0_D;				/* Extension 0 Arbitration Daisy Chain */
	wire EXT0_W;				/* Extension 0 Wins Arbitration */
	wire EXT1_D;				/* Extension 1 Arbitration Dasiy Chain */
	wire EXT1_W;				/* Extension 1 Wins Arbitration */
	wire EXT0CTR_EN;			/* Extension 0 Execution Enable */
	wire EXT0CTR_LD;			/* Extension 0 Loaded */
	wire [1:0] EXT0CTR_PR;	/* Extension 0 Priority */
	wire EXT0CTR_TR;			/* Extension 0 Trap Enable */
	wire EXT0VTISA;			/* Extension 0 Wins over TISA */
	wire EXT0VEXT1;			/* Extension 0 Wins over EXT1 */
	wire EXT1CTR_EN;			/* Extension 1 Execution Enable */
	wire EXT1CTR_LD;			/* Extension 1 Loaded */
	wire [1:0] EXT1CTR_PR;	/* Extension 1 Priority */
	wire EXT1CTR_TR;			/* Extension 1 Trap Enable */
	wire EXT1VTISA;			/* Extension 1 Wins over TISA */
	wire EXT1VEXT0;			/* Extension 1 Wins over EXT0 */
	wire EXTSTALL_0;			/* Stall for Extension Asynchronous */
	wire RI_EXT_0;				/* Extension 0 Request to Execute Instruction */
	wire RI_EXT_1;				/* Extension 1 Request to Execute Instruction */
	wire RI_PA_0;				/* Pipeline Arbiter Reserved/Recognized Instruction Asynchronous */
	wire TISA_W;				/* TISA Wins Arbitration */
	wire TISAVEXT0;			/* TISA Wins over EXT0 */
	wire TISAVEXT1;			/* TISA Wins over EXT1 */

/*****Registers****************************************************************/

	reg			ack_ext0_reg;	/* Extension 0 Enable Acknowledged */
	reg 			ack_ext1_reg;	/* Extension 1 Enable Acknowledged */
	reg 			en_ext0_reg;	/* Extension 0 Enabled Last Cycle */
	reg 			en_ext0_0_reg;	/* Extension 0 Enable */
	reg 			en_ext1_reg;	/* Extension 1 Enabled Last Cycle */
	reg 			en_ext1_0_reg;	/* Extension 1 Enable */
	reg 			en_id_reg;		/* TISA Enable */
	reg [7:0]	ext0ctr_reg;	/* Extension 0 Control Register */
	reg [7:0]	ext1ctr_reg;	/* Extension 1 Control Register */
	reg 			extstall_reg;	/* Stall for Extension */
	reg 			gr_ext0_reg;	/* Extension 0 Grant Pipeline Resources */
	reg 			gr_ext1_reg;	/* Extension 1 Grant Pipeline Resources */
	reg [1:0]	pclkcnt;			/* Pipeline Clock edge detection */
	reg 			pi_ext0_reg;	/* Extension 0 Passive/Parrallel Instruction */
	reg 			pi_ext1_reg;	/* Extension 1 Passive/Parrallel Instruction */
	reg 			ri_ext0_reg;	/* Extension 0 Reserved/Recognized Instruction */
	reg 			ri_ext1_reg;	/* Extension 1 Reserved/Recognized Instruction */
	reg 			ri_id_reg;		/* TISA Reserved/Recognized Instruction */
	reg 			ri_pa_reg;		/* Pipeline Arbiter Reserved/Recognized Instruction */
	reg 			stall_reg;		/* Pipeline Stall */
	reg			excext_reg;		/* */

/*****Parameters****************************************************************/

	/* Extension Controller Bit Definitions */
	parameter	EXTPRr	=	4;	/* Most Significant Extension Priority Bit */
	parameter	EXTPRl	=	3;	/* Least Significant Extension Priority Bit */
	parameter	EXTTR		=	2;	/* Extension Trap Enable Bit */
	parameter	EXTLD		=	1;	/* Extension Load Bit */
	parameter	EXTEN		=	0;	/* Extnesion Execution Enable */

/*****Initialization****************************************************************/

	initial
	begin
		ack_ext0_reg = 1'b0;
		ack_ext1_reg = 1'b0;
		en_ext0_0_reg = 1'b0;
		en_ext0_reg = 1'b0;
		en_ext1_0_reg = 1'b0;
		en_ext1_reg = 1'b0;
		en_id_reg = 1'b0;
		ext0ctr_reg = 8'b0;
		ext1ctr_reg = 8'b0;
		extstall_reg = 1'b0;
		gr_ext0_reg = 1'b0; 
		gr_ext1_reg = 1'b0;
		pclkcnt = 2'b0;
		pi_ext0_reg = 1'b0;
		pi_ext1_reg = 1'b0;
		ri_ext0_reg = 1'b0;
		ri_ext1_reg = 1'b0;
		ri_id_reg = 1'b0;
		ri_pa_reg = 1'b0;
		stall_reg = 1'b0;
		excext_reg = 1'b0;
	end
	
/*********************************************************************/

	assign EN_EXT_0	=	EXT0CTR_LD & EXT0CTR_EN & ~ri_ext0_reg;
	assign EN_EXT_1	=	EXT1CTR_LD & EXT1CTR_EN & ~ri_ext1_reg;
	assign EN_EXT0 	=	en_ext0_0_reg;
	assign EN_EXT0_0 	=	(~stall_reg)?		1'b0:
								(TISA_W)?			1'b0:
								(EXT0_W)?			1'b1:
														1'b0;
	assign EN_EXT1 	=	en_ext1_0_reg;
	assign EN_EXT1_0	=	(~stall_reg)?		1'b0:
								(TISA_W)?			1'b0:
								(EXT0_W)?			1'b0:
								(EXT1_W)?			1'b1:
														1'b0;
	assign EN_ID 		= 	en_id_reg;
	assign EN_ID_0 	= 	(TISA_W)?								1'b1:
								(en_ext0_0_reg & pi_ext0_reg)?	1'b1:
								(en_ext1_0_reg & pi_ext1_reg)?	1'b1:
																			1'b0;
	assign EXT0_D		=	~TISA_W;
	assign EXT0_W		=	EXT0_D & EN_EXT_0 & (RI_ID | EXT0VTISA) & (~EN_EXT_1 | EXT0VEXT1);
	assign EXT0VTISA	=	EXT0CTR_PR >= `TISA_PR;
	assign EXT0VEXT1	=	EXT0CTR_PR >= EXT1CTR_PR;
	assign EXT0CTR_EN	=	ext0ctr_reg[EXTEN];
	assign EXT0CTR_LD	=	ext0ctr_reg[EXTLD];
	assign EXT0CTR_PR	=	ext0ctr_reg[EXTPRr:EXTPRl];
	assign EXT0CTR_TR	=	ext0ctr_reg[EXTTR];
	assign EXT1_D		=	EXT0_D & ~EXT0_W;
	assign EXT1_W		=	EXT1_D & EN_EXT_1 & (RI_ID | EXT1VTISA) & (~EN_EXT_0 | EXT1VEXT0);
	assign EXT1CTR_EN	=	ext1ctr_reg[EXTEN];
	assign EXT1CTR_LD	=	ext1ctr_reg[EXTLD];
	assign EXT1CTR_PR	=	ext1ctr_reg[EXTPRr:EXTPRl];
	assign EXT1CTR_TR	=	ext1ctr_reg[EXTTR];
	assign EXT1VTISA	=	EXT1CTR_PR >= `TISA_PR;
	assign EXT1VEXT0	=	EXT1CTR_PR >= EXT0CTR_PR;
	assign EXTSTALL 	=	extstall_reg;
	assign EXTSTALL_0	=	(ack_ext0_reg & gr_ext0_reg) | (ack_ext1_reg & gr_ext1_reg);
	assign GR_EXT0 	=	gr_ext0_reg;
	assign GR_EXT1 	=	gr_ext1_reg;
	assign RI_EXT_0	=	~EXT0CTR_LD | (EXT0CTR_LD & ((~EXT0CTR_TR & ~EXT0CTR_EN & ri_ext0_reg) | (EXT0CTR_TR & ~EXT0CTR_EN) | (EXT0CTR_EN & (ri_ext0_reg | pi_ext0_reg))));
	assign RI_EXT_1	=	~EXT1CTR_LD | (EXT1CTR_LD & ((~EXT1CTR_TR & ~EXT1CTR_EN & ri_ext1_reg) | (EXT1CTR_TR & ~EXT1CTR_EN) | (EXT1CTR_EN & (ri_ext1_reg | pi_ext1_reg))));
	assign RI_PA 		=	RI_PA_0;
	assign RI_PA_0 	=	RI_ID & & RI_EXT_0 & RI_EXT_1;
	assign TISA_W		=	~RI_ID & (~EN_EXT_0 | TISAVEXT0) & (~EN_EXT_1 | TISAVEXT1);
	assign TISAVEXT0	=	`TISA_PR >= EXT0CTR_PR;
	assign TISAVEXT1	=	`TISA_PR >= EXT1CTR_PR;

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			ack_ext0_reg <= 1'b0;
			ack_ext1_reg <= 1'b0;
			en_ext0_0_reg <= 1'b0;
			en_ext0_reg <= 1'b0;
			en_ext1_0_reg <= 1'b0;
			en_ext1_reg <= 1'b0;
			en_id_reg <= 1'b0;
			ext0ctr_reg <= 8'b0;
			ext1ctr_reg <= 8'b0;
			extstall_reg <= 1'b0;
			gr_ext0_reg <= 1'b0; 
			gr_ext1_reg <= 1'b0;
			pi_ext0_reg <= 1'b0;
			pi_ext1_reg <= 1'b0;
			ri_ext0_reg <= 1'b0;
			ri_ext1_reg <= 1'b0;
			ri_id_reg <= 1'b0;
			ri_pa_reg <= 1'b0;
			stall_reg <= 1'b0;
			excext_reg <= 1'b0;
		end
		else
		begin
			case({pclkcnt})
				2'b01		:	begin
									/* Synchronize to Posedge of Pipeline Clock */
									if (en_ext0_0_reg) en_ext0_reg <= 1'b1;								/* if Enabled Set Enabled Flag */
									else if (~gr_ext0_reg & ~en_ext0_0_reg) en_ext0_reg <= 1'b0;	/* Clear Enabled Flag */
									if (en_ext1_0_reg) en_ext1_reg <= 1'b1;								/* if Enabled Set Enabled Flag */
									else if (~gr_ext1_reg & ~en_ext1_0_reg) en_ext1_reg <= 1'b0;	/* Clear Enabled Flag */
									if (gr_ext0_reg & ~ack_ext0_reg) gr_ext0_reg <= 1'b0;			/* Clear GR */
									if (gr_ext1_reg & ~ack_ext1_reg) gr_ext1_reg <= 1'b0;			/* Clear GR */
									if (gr_ext0_reg & excext_reg)	gr_ext0_reg <= 1'b0;
									if (gr_ext1_reg & excext_reg)	gr_ext1_reg <= 1'b0;
								end
				default	:	begin
								end
			endcase
			if (en_ext0_reg & ack_ext0_reg) gr_ext0_reg <= 1'b1; 		/* if Enabled and Acknowleged Set GR */
			if (en_ext1_reg & ack_ext1_reg) gr_ext1_reg <= 1'b1;		/* if Enabled and Acknowleged Set GR */
			/* Latch Signals */
			ack_ext0_reg <= ACK_EXT0;
			ack_ext1_reg <= ACK_EXT1;
			en_ext0_0_reg <= EN_EXT0_0;
			en_ext1_0_reg <= EN_EXT1_0;
			en_id_reg <= EN_ID_0;
			ext0ctr_reg <= EXT0CTR;
			ext1ctr_reg <= EXT1CTR;
			extstall_reg <= EXTSTALL_0;
			pi_ext0_reg <= PI_EXT0;
			pi_ext1_reg <= PI_EXT1;
			ri_ext0_reg <= RI_EXT0;
			ri_ext1_reg <= RI_EXT1;
			ri_id_reg <= RI_ID;
			ri_pa_reg <= RI_PA_0;
			stall_reg <= STALL;
			excext_reg <= EXCEXT;
		end
	end
	
endmodule
