/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	power_management_controller								*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module power_management_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */ 
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				SRT,			/* Start Transaction */
	input				WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output [31:0]	DATA_OUT,	/* Data Bus to Data Path Memory Interface */
	output			DNE,			/* Transaction Done */
	output [31:0]	STATUS		/* Status Register */
	);

/*****Signals****************************************************************/

	wire [3:0]	BM;				/* Byte Selects */
	wire [31:0]	DATA_IN_C;		/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;		/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;		/* Endian Flipped Outgoing Data */
	wire			MEN;				/* Transaction Enabled for PWRCTR Configuration Space */
	wire [31:0]	REG_M;			/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;			/* Selected Configuration Register Data */
	wire [31:0]	WMASK_IN;		/* Configuration Register Data Write Mask */
	wire [31:0]	WMASK_REG;		/* Incoming Data Write Mask */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;			/* BASETAG Register */
	reg [31:0]	data_in_m_reg;		/* Masked Incoming Data */
	reg [3:0]	dnecnt;				/* Transaction Progress Counter */
	reg			dner;					/* Configuration Space Transaction Done */
	reg			men_reg;				/* Transaction Enabled for PWRCTR Configuration Space */
	reg [31:0]	perpwrdisable;		/* Peripheral Power Disable */
	reg [31:0]	perpwrenable;		/* Peripheral Power Enable */
	reg [31:0]	perpwrstatus;		/* Peripheral Power Status */
	reg [31:0]	reg_m_0_reg;		/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;			/* Masked Configuration Register Data */
	reg [31:0]	syspwrdisable;		/* System Power Disable */
	reg [31:0]	syspwrenable;		/* System Power Enable */
	reg [31:0]	syspwrstatus;		/* System Power Status */

/*****Variables****************************************************************/

	integer i;

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter BASETAGREG =			16'h0000;	/* BASETAG Register Relative Address */
	parameter SYSPWRENABLEREG =	16'h0004;	/* System Power Relative Enable Address */
	parameter SYSPWRDISABLEREG =	16'h0008;	/* System Power Relative Disable Address */
	parameter PERPWRENABLEREG =	16'h000C;	/* Peripheral Power Enable Relative Address */
	parameter PERPWRDISABLEREG =	16'h0010;	/* Peripheral Power Disable Relative Address */

	/* BASETAG Register */
	parameter	BASEl =				31;			/* BASE Bit Range Left */
	parameter	BASEr =				16;			/* BASE Bit Range Right */
	parameter	TAGl = 				15;			/* TAG Bit Range Left */
	parameter	TAGr = 				0;				/* TAG Bit Range Right */
	parameter	TAG =					16;			/* TAG Value */

/*****Initialization****************************************************************/

	initial 
	begin
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		dnecnt = 4'b1111;
		dner = 1'b1;
		men_reg = 1'b0;
		perpwrdisable = 32'b0;
		perpwrenable = 32'b0;
		perpwrstatus = 32'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
		syspwrdisable = 32'b0;
		syspwrenable = 32'b0;
		syspwrstatus = 32'b0;
	end

/*********************************************************************/

	assign BASETAG = basetagr;
	assign DNE = dner;
	assign STATUS = syspwrstatus;
	assign MEN = (CADDR[BASEl:BASEr] == ADDR_IN[BASEl:BASEr]);

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/

	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BASETAGREG)?basetagr:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == SYSPWRENABLEREG))?syspwrenable:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == SYSPWRENABLEREG))?syspwrstatus:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == SYSPWRDISABLEREG))?syspwrdisable:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == SYSPWRDISABLEREG))?syspwrstatus:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == PERPWRENABLEREG))?perpwrenable:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == PERPWRENABLEREG))?perpwrstatus:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == PERPWRDISABLEREG))?perpwrdisable:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == PERPWRDISABLEREG))?perpwrstatus:32'bz;
	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,1'b0,dnecnt[3],dner})
			4'b0xxx	:	begin
								/* Reset */
								dner = 1'b1;
								basetagr = 32'b0;
								syspwrenable = 32'b0;
								syspwrdisable = 32'b0;
								syspwrstatus = 32'b0;
								perpwrenable = 32'b0;
								perpwrdisable = 32'b0;
								perpwrstatus = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
							end
			4'b11xx	:	begin
								/* Soft Reset */
								dner = 1'b1;
								basetagr = 32'b0;
								syspwrenable = 32'b0;
								syspwrdisable = 32'b0;
								syspwrstatus = 32'b0;
								perpwrenable = 32'b0;
								perpwrdisable = 32'b0;
								perpwrstatus = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
							end
			4'b1000	:	begin
								/* Update Configuration Registers */
								dner = 1'b1;
								if (men_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										BASETAGREG			:	begin
																		/* Update Basetag Register */
																		basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
																	end
										SYSPWRENABLEREG	:	begin
																		/* Update System Power Enable Register */
																		syspwrenable = data_in_m_reg | reg_m_reg;
																	end
										SYSPWRDISABLEREG	:	begin
																		/* Update System Power Disable Register */
																		syspwrdisable = data_in_m_reg | reg_m_reg;
																	end	
										PERPWRENABLEREG	:	begin
																		/* Update Peripheral Power Enable Register */
																		perpwrenable = data_in_m_reg | reg_m_reg;
																	end
										PERPWRDISABLEREG	:	begin
																		/* Update Peripheral Power Disable Register */
																		perpwrdisable = data_in_m_reg | reg_m_reg;
																	end
										default				:	begin
																		/* Do Nothing */
																		/* Address not Recognized */
																	end
									endcase
								end
							end
			4'b10x1	:	begin
								/* Wait for incoming request */
								if (SRT & men_reg)
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
								/* Update System Power Status Register */
								for (i=0; i<32; i=i+1)
								begin
									if (syspwrenable[i]) syspwrstatus[i] = 1'b1;
								end
								syspwrenable = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (syspwrdisable[i]) syspwrstatus[i] = 1'b0;
								end
								syspwrdisable = 32'b0;
								/* Update Peripheral Power Status Register */
								for (i=0; i<32; i=i+1)
								begin
									if (perpwrenable[i]) perpwrstatus[i] = 1'b1;
								end
								perpwrenable = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (perpwrdisable[i]) perpwrstatus[i] = 1'b0;
								end
								perpwrdisable = 32'b0;
							end
			default	:	begin
							end
		endcase
	end

endmodule
