/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	registerfile4													*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	eBug Extension, Bug Fixes, see Manual				*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module registerfile4(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BHLS1,			/* Write Port 1 Byte/Halfword Load/Store */
	input				CLK,				/* System Clock 50 - 100 MHZ */ 
	input [1:0]		DMADD,			/* Least Significant Bits of Data Addresss */
	input				EDF,				/* Endian Flip */
	input				GR,				/* Extension Grant */
	input				PCLK,				/* Pipeline Clock */
	input [4:0]		RD_MA,			/* Destination Register Memory Access */
	input [4:0]		RD_WB,			/* Destination Register Writeback */
	input [31:0]	RDDATA_MA,		/* Write Data Memory Access */
	input [31:0]	RDDATA_WB,		/* Write Data Writeback Access */
	input [4:0]		RDREG1,			/* Read Port 1 Register Number */
	input [4:0]		RDREG2,			/* Read Port 2 Register Number */
	input [4:0]		RDREG3,			/* Read Port 3 Register Number */
	input [4:0]		RDREG4,			/* Read Port 4 Register Number */
	input				REGWRITE_MA,	/* Register Write Enable Memory Access */
	input				REGWRITE_WB,	/* Register Write Enable Write back */
	input				REGWRITE1,		/* Write Port 1 Enable */
	input				REGWRITE2,		/* Write Port 2 Enable */
	input				RESET,			/* System Reset */
	input				RNL1,				/* Right/Left Unaligned Load/Store */
	input [31:0]	WRDATA1,			/* Write Port 1 Data */
	input [31:0]	WRDATA2,			/* Write Port 2 Data */
	input [4:0]		WRREG1,			/* Write Port 1 Register Number */
	input [4:0]		WRREG2,			/* Write Port 2 Register Number */
	input				FSPECIAL,		/* */
	input [31:0]	STATUS_REGISTER,	/* */
	input [31:0]	BADVADDR_REGISTER,	/* */
	input [31:0]	CAUSE_REGISTER,	/* */
	input [31:0]	FPSTATUS_REGISTER,	/* */
	input [31:0]	FPIMP_REGISTER,	/* */
	input [31:0]	MULT_HI,			/* */
	input [31:0]	MULT_LO,			/* */
	/* OUTPUT PORTS */
	output			EMPTY_WRBUF,	/* Write Buffer Empty */
	output			FULL_WRBUF,		/* Write Buffer Full */
	output [31:0]	RDDATA1,			/* Read Port 1 Data */
	output [31:0]	RDDATA2,			/* Read Port 2 Data */
	output [31:0]	RDDATA3,			/* Read Port 3 Data */
	output [31:0]	RDDATA4,			/* Read Port 4 Data */
	output			REGDNE,			/* Register Done */
	output			REGRDY,			/* Register Ready */
	/* DEBUG PORTS */
	output [31:0] regfile00,
	output [31:0] regfile01,
	output [31:0] regfile02,
	output [31:0] regfile03,
	output [31:0] regfile04,
	output [31:0] regfile05,
	output [31:0] regfile06,
	output [31:0] regfile07,
	output [31:0] regfile08,
	output [31:0] regfile09,
	output [31:0] regfile10,
	output [31:0] regfile11,
	output [31:0] regfile12,
	output [31:0] regfile13,
	output [31:0] regfile14,
	output [31:0] regfile15,
	output [31:0] regfile16,
	output [31:0] regfile17,
	output [31:0] regfile18,
	output [31:0] regfile19,
	output [31:0] regfile20,
	output [31:0] regfile21,
	output [31:0] regfile22,
	output [31:0] regfile23,
	output [31:0] regfile24,
	output [31:0] regfile25,
	output [31:0] regfile26,
	output [31:0] regfile27,
	output [31:0] regfile28,
	output [31:0] regfile29,
	output [31:0] regfile30,
	output [31:0] regfile31
	);

/*****Signals****************************************************************/

	wire			BHLS;			/* Current Byte/Halfword Load/Store */
	wire [1:0]	BYTESEL;		/* Current Byte Select */
	wire [1:0]	BYTESEL1;	/* Write Port 1 Byte Select */
	wire			PCLK_NE;		/* Negative Edge of Pipeline Clock */
	wire [31:0]	RDDATA;		/* Read Data */
	wire [31:0]	RDDATA0;		/* Read Data Asynchronous */
	wire [4:0]	RDREG0;		/* Read Register Number */
	wire			REGWRITE;	/* Current Register Write Enable */
	wire			RNL;			/* Right/Left Load/Store */
	wire [31:0]	WRDATA;		/* Current Write Data */
	wire [31:0]	WRDATA0;		/* Current Register Value */
	wire			WREN;			/* Write Enable */
	wire			WRENB;		/* Write Buffer Enable */
	wire [4:0]	WRREG;		/* Write Register Number */

/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;			/* Pipeline Edge Detection */
	reg [31:0]	RDDATA1reg;		/* Read Port 1 Data */
	reg [31:0]	RDDATA2reg;		/* Read Port 2 Data */
	reg [31:0]	RDDATA3reg;		/* Read Port 3 Data */
	reg [31:0]	RDDATA4reg;		/* Read Port 4 Data */
	reg			rden;				/* Read Buffer Enable */
	reg [4:0]	rdreg0_0_reg;	/* Read Data */
	reg [1:0]	rdstate;			/* Read State */
	reg			redata;			/* Read Register */
	reg [31:0]	register;		/* New Register Value */
	reg [41:0]	wrbufs;			/* Write Buffer Output */
	reg			wrdne;			/* Write Done */
	reg			wrpending;		/* Write Pending */
	reg [2:0]	wrstate;			/* Write State */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		RDDATA1reg = 32'b0;
		RDDATA2reg = 32'b0;
		RDDATA3reg = 32'b0;
		RDDATA4reg = 32'b0;
		rden = 1'b0;
		rdreg0_0_reg = 5'b0;
		rdstate = 2'b0;
		redata = 1'b0;
		register = 32'b0;
		wrbufs = 42'b0;
		wrdne = 1'b1;
		wrpending = 1'b0;
		wrstate = 3'b0;
	end

/*********************************************************************/

	assign REGRDY = GR & ~FULL_WRBUF & ~REGWRITE_WB & ~REGWRITE_MA;
	assign REGDNE = wrdne;

/*****Read Logic****************************************************************/

	assign RDDATA1 = RDDATA1reg;
	assign RDDATA2 = RDDATA2reg;
	assign RDDATA3 = RDDATA3reg;
	assign RDDATA4 = RDDATA4reg;

	assign RDDATA0 = 		((FSPECIAL) && (rdreg0_0_reg == 5'h1))?		MULT_HI:
								((FSPECIAL) && (rdreg0_0_reg == 5'h2))?		MULT_LO:
								((FSPECIAL) && (rdreg0_0_reg == 5'h3))?		STATUS_REGISTER:
								((FSPECIAL) && (rdreg0_0_reg == 5'h4))?		BADVADDR_REGISTER:
								((FSPECIAL) && (rdreg0_0_reg == 5'h5))?		CAUSE_REGISTER:
								((FSPECIAL) && (rdreg0_0_reg == 5'h6))?		FPSTATUS_REGISTER:
								((FSPECIAL) && (rdreg0_0_reg == 5'h7))?		FPIMP_REGISTER:
								(FSPECIAL)?												32'b0:
								(rdreg0_0_reg == 5'b0)?								32'b0:
								((REGWRITE_MA) & (RD_MA == rdreg0_0_reg))?	RDDATA_MA:	
								((REGWRITE_WB) & (RD_WB == rdreg0_0_reg))?	RDDATA_WB:
																							RDDATA;	
	/* Time Multiplexed Read Port */
	MUX5_4to1 mux0(
		.a0(RDREG1), 
		.a1(RDREG2), 
		.a2(RDREG3),
		.a3(RDREG4), 
		.sel(rdstate),
		.out(RDREG0)
		);

/*****Write Logic****************************************************************/

	assign PCLK_NE = (pclkcnt == 2'b10);
	assign WREN = ~wrdne & wrbufs[41];
	assign WRENB = REGRDY & (REGWRITE1 | REGWRITE2);

	/* Endian Flip Address Bits */
	xor2_2 xor0(
		.A({EDF,EDF}),
		.B(DMADD),
		.C(BYTESEL1)
		);

/*****Write Buffer****************************************************************/

	fifo wrbuf(
		.din({REGWRITE1,BHLS1,RNL1,BYTESEL1,WRREG1,WRDATA1,REGWRITE2,4'b0,WRREG2,WRDATA2}),
		.rd_clk(CLK),
		.rd_en(rden),
		.rst(~RESET),
		.wr_clk(CLK),
		.wr_en(WRENB),
		.dout({REGWRITE,BHLS,RNL,BYTESEL,WRREG,WRDATA}),
		.empty(EMPTY_WRBUF),
		.full(FULL_WRBUF)
		);

/*****Registers****************************************************************/
	
	registerblock rf(
		.rst(~RESET),
		.addra(RDREG0),
		.addrb(wrbufs[36:32]),
		.clka(CLK),
		.clkb(CLK),
		.dinb(register),
		.douta(RDDATA),
		.doutb(WRDATA0),
		.web(WREN),
		.regfile00(regfile00),
		.regfile01(regfile01),
		.regfile02(regfile02),
		.regfile03(regfile03),
		.regfile04(regfile04),
		.regfile05(regfile05),
		.regfile06(regfile06),
		.regfile07(regfile07),
		.regfile08(regfile08),
		.regfile09(regfile09),
		.regfile10(regfile10),
		.regfile11(regfile11),
		.regfile12(regfile12),
		.regfile13(regfile13),
		.regfile14(regfile14),
		.regfile15(regfile15),
		.regfile16(regfile16),
		.regfile17(regfile17),
		.regfile18(regfile18),
		.regfile19(regfile19),
		.regfile20(regfile20),
		.regfile21(regfile21),
		.regfile22(regfile22),
		.regfile23(regfile23),
		.regfile24(regfile24),
		.regfile25(regfile25),
		.regfile26(regfile26),
		.regfile27(regfile27),
		.regfile28(regfile28),
		.regfile29(regfile29),
		.regfile30(regfile30),
		.regfile31(regfile31)
		);

/*********************************************************************/

	/* Pipeline Clock Edge Detection */
	always@(posedge CLK)
	begin
		pclkcnt <= {pclkcnt[0],PCLK};
	end
	
	/* Write Buffer Detection */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			wrpending <= 1'b0;
		end
		else
		begin
			if (EMPTY_WRBUF == 0) wrpending <= 1'b1;
			else if ((EMPTY_WRBUF == 1) && (wrdne == 1'b0)) wrpending <= 1'b0;
		end
	end

	/* Latch Read Port */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			rdreg0_0_reg <= 5'b0;
		end
		else
		begin
			/* Latch Signal */
			rdreg0_0_reg <= RDREG0;
		end
	end

	/* Time Multiplexed Read Port */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			rdstate <= 2'b0;
			RDDATA1reg <= 32'b0;
			RDDATA2reg <= 32'b0;
			RDDATA3reg <= 32'b0;
			RDDATA4reg <= 32'b0;
		end
		else
		begin
			if (GR)
			begin
				/* Extension Active */
				case(rdstate)
					2'b00		:	begin
										RDDATA4reg <= RDDATA0;
										rdstate <= 2'b01;
									end
					2'b01		:	begin
										RDDATA1reg <= RDDATA0;
										rdstate <= 2'b10;
									end
					2'b10		:	begin
										RDDATA2reg <= RDDATA0;
										rdstate <= 2'b11;
									end
					2'b11		:	begin
										RDDATA3reg <= RDDATA0;
										rdstate <= 2'b00;
									end
					default	:	begin
										rdstate <= 2'b00;
									end
				endcase
			end
			else
			begin
				/* Normal Operation */
				case(rdstate)
					2'b00		:	begin
										RDDATA2reg <= RDDATA0;
										rdstate <= 2'b01;
									end
					2'b01		:	begin
										RDDATA1reg <= RDDATA0;
										rdstate <= 2'b00;
									end
					default	:	begin
										rdstate <= 2'b00;
									end
				endcase
			end
		end
	end

	/* Write Buffer */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			wrdne <= 1'b1;
			rden <= 1'b0;
			redata <= 1'b0;
			register <= 32'b0;
			wrstate <= 3'b000;
			wrbufs <= 42'b0;
		end
		else
		begin
			case(wrstate)
				/* Normal Operation */
				3'b000	:	begin
									if (REGRDY) 
									begin
										/* Change Mode */
										wrstate <= 3'b100;
									end
									else
									begin
										if (PCLK_NE)
										begin
											/* Check for Register Write */
											wrstate <= 3'b001;
										end
									end
								end
				3'b001	:	begin
									if (REGWRITE1 & (WRREG1 != 5'b0))
									begin
										/* Latch Register Write */
										wrbufs <= {REGWRITE1,BHLS1,RNL1,BYTESEL1,WRREG1,WRDATA1};
										wrstate <= 3'b010;
									end
									else 
									begin
										/* Reset State */
										wrbufs <= 42'b0;
										wrstate <= 3'b011;
									end
									wrdne <= 1'b0;
								end
				3'b010	:	begin
									/* Write Register */
									casex(wrbufs[40:39])
										2'b0x		:	begin
															/* Write Word */
															register <= wrbufs[31:0];
															wrstate <= 3'b011;
														end
										2'b10		:	begin
															/* Write Unaligned Word Left */
															casex({redata,wrbufs[38:37]})
																3'b0xx	:	begin
																					/* Read Current Register */
																					redata <= 1'b1;
																				end
																3'b100	:	begin
																					register <= {wrbufs[31:24],WRDATA0[23:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b011;
																				end
																3'b101	:	begin
																					register <= {wrbufs[31:16],WRDATA0[15:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b011;
																				end
																3'b110	:	begin
																					register <= {wrbufs[31:8],WRDATA0[7:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b011;
																				end
																3'b111	:	begin
																					register <= wrbufs[31:0];
																					redata <= 1'b0;
																					wrstate <= 3'b011;
																				end
																default	:	begin
																				end
															endcase
														end
										2'b11		:	begin
															/* Write Unaligned Word Right */
															casex({redata,wrbufs[38:37]})
																3'b0xx	:	begin
																					/* Read Current Register */
																					redata <= 1'b1;
																				end
																3'b100	:	begin
																					register <= wrbufs[31:0];
																					redata <= 1'b0;
																					wrstate <= 3'b011;
																				end
																3'b101	:	begin
																					register <= {WRDATA0[31:24],wrbufs[23:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b011;
																				end
																3'b110	:	begin
																					register <= {WRDATA0[31:16],wrbufs[15:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b011;
																				end
																3'b111	:	begin
																					register <= {WRDATA0[31:8],wrbufs[7:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b011;
																				end
																default	:	begin
																				end
															endcase
														end
										default	:	begin
														end
									endcase
								end
				3'b011	:	begin
									/* Return to inactive state */
									wrdne <= 1'b1;
									wrstate <= 3'b000;
								end
				/* Extension Active */
				3'b100	:	begin
									if (~GR & EMPTY_WRBUF)
									begin
										/* Change Mode */
										wrstate <= 3'b000;
									end
									else
									begin
										if (wrpending)
										begin
											/* Check Buffer for Register Write */
											rden <= 1'b1;
											wrstate <= 3'b101;
										end
									end
								end
				3'b101	:	begin
									if (wrpending & ~rden) 
									begin
										if ((REGWRITE == 1'b1) && (WRREG != 5'b0))
										begin
											/* Latch Register Write */
											wrbufs <= {REGWRITE,BHLS,RNL,BYTESEL,WRREG,WRDATA};
											wrstate <= 3'b110;
										end
										else 
										begin
											/* Reset State */
											wrbufs <= 42'b0;
											wrstate <= 3'b111;
										end
										wrdne <= 1'b0;
									end
									rden <= 1'b0;
								end
				3'b110	:	begin
									/* Write Register */
									casex(wrbufs[40:39])
										2'b0x		:	begin
															/* Write Word */
															register <= wrbufs[31:0];
															wrstate <= 3'b111;
														end
										2'b10		:	begin
															/* Write Unaligned Word Left */
															casex({redata,wrbufs[38:37]})
																3'b0xx	:	begin
																					/* Read Current Register */
																					redata <= 1'b1;
																				end
																3'b100	:	begin
																					register <= {wrbufs[31:24],WRDATA0[23:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b111;
																				end
																3'b101	:	begin
																					register <= {wrbufs[31:16],WRDATA0[15:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b111;
																				end
																3'b110	:	begin
																					register <= {wrbufs[31:8],WRDATA0[7:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b111;
																				end
																3'b111	:	begin
																					register <= wrbufs[31:0];
																					redata <= 1'b0;
																					wrstate <= 3'b111;
																				end
																default	:	begin
																				end
															endcase
														end
										2'b11		:	begin
															/* Write Unaligned Word Right */
															casex({redata,wrbufs[38:37]})
																3'b0xx	:	begin
																					/* Read Current Register */
																					redata <= 1'b1;
																				end
																3'b100	:	begin
																					register <= wrbufs[31:0];
																					redata <= 1'b0;
																					wrstate <= 3'b111;
																				end
																3'b101	:	begin
																					register <= {WRDATA0[31:24],wrbufs[23:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b111;
																				end
																3'b110	:	begin
																					register <= {WRDATA0[31:16],wrbufs[15:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b111;
																				end
																3'b111	:	begin
																					register <= {WRDATA0[31:8],wrbufs[7:0]};
																					redata <= 1'b0;
																					wrstate <= 3'b111;
																				end
																default	:	begin
																				end
															endcase
														end
										default	:	begin
														end
									endcase
								end
				3'b111	:	begin
									/* Reset State */
									wrdne <= 1'b1;
									wrstate <= 3'b100;
								end
				default	:	begin
									wrstate <= 3'b000;
								end
			endcase
		end
	end
	
endmodule
