/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Nathaniel Lee Lynch												*/
/*					Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	shift																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

`include "decode.v"	/* Constant Definitiions */

module shift(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [4:0]		A,			/* Registered Shift Amount */
	input				CLK,		/* System Clock 50 - 100 MHZ */
	input [5:0]		CTR,		/* ALU Control/Opcode */
	input [31:0]	IN,		/* Input */
	input 			RESET,	/* System Reset */
	input [4:0]		SHAMT,	/* Immediate Shift Amount */
	/* OUTPUT PORTS */
	output [31:0]	OUT		/* Output */
	);

/*****Signals****************************************************************/

	wire [31:0] sign[31:0];	/* Variable Shifter Sign Mask */

/*****Registers****************************************************************/
	
	reg [4:0]	shiftamt;	/* Shift Amount */
	reg [31:0]	sign_reg;	/* Shifter Sign Mask */
	reg [31:0]	sllv_reg;	/* Shift Left */
	reg [31:0]	sra_reg;		/* Shift Right Arithmetic */
	reg [31:0]	temp;			/* Shift Right */
	
/*****Initialization****************************************************************/
	
	initial
	begin
		shiftamt = 5'b0;
		sign_reg = 32'b0;
		sllv_reg = 32'b0;
		sra_reg = 32'b0;
		temp = 32'b0;
	end

/*********************************************************************/

	assign OUT	=	(CTR == `SLL)?		sllv_reg:
						(CTR == `SRL)?		temp:
						(CTR == `SRA)?		sra_reg:
						(CTR == `SLLV)?	sllv_reg:
						(CTR == `SRLV)?	temp:
						(CTR == `SRAV)?	sra_reg:
												IN;

/*****Variable Shifter Sign Mask****************************************************************/

	assign sign[0] = {IN[31],31'h00000000};
 	assign sign[1] = {2{IN[31]}}<< 30;
	assign sign[2] = {3{IN[31]}}<< 29;
	assign sign[3] = {4{IN[31]}}<< 28;
	assign sign[4] = {5{IN[31]}}<< 27;
	assign sign[5] = {6{IN[31]}}<< 26;
	assign sign[6] = {7{IN[31]}}<< 25;
	assign sign[7] = {8{IN[31]}}<< 24;
	assign sign[8] = {9{IN[31]}}<< 23;
	assign sign[9] = {10{IN[31]}}<< 22;
	assign sign[10] = {11{IN[31]}}<< 21;
	assign sign[11] = {12{IN[31]}}<< 20;
	assign sign[12] = {13{IN[31]}}<< 19;
	assign sign[13] = {14{IN[31]}}<< 18;
	assign sign[14] = {15{IN[31]}}<< 17;
	assign sign[15] = {16{IN[31]}}<< 16;
	assign sign[16] = {17{IN[31]}}<< 15;
	assign sign[17] = {18{IN[31]}}<< 14;
	assign sign[18] = {19{IN[31]}}<< 13;
	assign sign[19] = {20{IN[31]}}<< 12;
	assign sign[20] = {21{IN[31]}}<< 11;
	assign sign[21] = {22{IN[31]}}<< 10;
	assign sign[22] = {23{IN[31]}}<< 9;
	assign sign[23] = {24{IN[31]}}<< 8;
	assign sign[24] = {25{IN[31]}}<< 7;
	assign sign[25] = {26{IN[31]}}<< 6;
	assign sign[26] = {27{IN[31]}}<< 5;
	assign sign[27] = {28{IN[31]}}<< 4;
	assign sign[28] = {29{IN[31]}}<< 3;
	assign sign[29] = {30{IN[31]}}<< 2;
	assign sign[30] = {31{IN[31]}}<< 1;
	assign sign[31] = {32{IN[31]}};

/*********************************************************************/

	/* Shift Amount */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)		shiftamt <= 5'b0;
		else
		begin
			if (CTR[2] == 1'b0)	shiftamt <= SHAMT;
			else 						shiftamt <= A;
		end
	end

	/* Shift Right */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	temp <= 32'b0;
		else						temp <= (IN>>shiftamt);
	end

	/* Shifter Sign Mask */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	sign_reg <= 32'b0;
		else 						sign_reg <= sign[shiftamt];
	end

	/* Shift Right Arithmetic */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	sra_reg <= 32'b0;
		else 						sra_reg <= sign_reg|temp;
	end

	/* Shift Left */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)	sllv_reg <= 32'b0;
		else						sllv_reg <= IN<<shiftamt;
	end
	
endmodule
