/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	sram_bridge														*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/* This module interfaces to a Cypress CY7C1354B SRAM						*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module sram_bridge(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [24:0]	ADDR_IN,			/* Address bus from SRAM Controller */
	input				BURST_ORDER,	/* Mode Control */
	input [1:0]		BYTES,			/* Bytes to be written */
	input				CLOCK_MASK,		/* Clock Enable Control */
	input [31:0]	DATA_IN,			/* Data bus from SRAM Controller */
	input [31:0]	DQ_OUT,			/* Data bus from SRAM IC */
	input [3:0]		DQP_OUT,			/* Data bus parity from SRAM IC */
	input				OE,				/* Output Enable */
	input				RESET,			/* System Reset */
	input				SLEEP,			/* Sleep Control */
	input				SRAMCLK,			/* SRAM Clock 50 - 100 MHZ */
	input				SRT,				/* Start */
	input				WE,				/* Write Enable */
	/* OUTPUT PORTS */
	output [22:0]	ADDR_OUT,		/* Address bus to SRAM IC */
	output			CE2,				/* Chipselect 2 */
	output			CEN,				/* Clock Enable */
	output [31:0]	DATA_OUT,		/* Data bus to SRAM Controller */
	output			DNE,				/* Done */
	output [31:0]	DQ_IN,			/* Data bus to SRAM IC */
	output [31:0]	DQ_TR,			/* Data bus Trisate Enable */
	output [3:0]	DQP_IN,			/* Data bus parity to SRAM IC */
	output [3:0]	DQP_TR,			/* Data bus parity Tristate Enable */
	output			MODE,				/* Mode */
	output			NADVLD,			/* Advanced Load */
	output [3:0]	NBW,				/* Byte Enable */
	output			NCE1,				/* Chipselect 1 */
	output			NCE3,				/* Chipselect 3 */
	output			NOE,				/* Output Enable to SRAM IC */
	output			NWE,				/* Write Enable to SRAM IC */
	output [3:0]	PR,				/* Parity Error */
	output			ZZ					/* Sleep */
	);

/*****Signals****************************************************************/

	wire			BUSY;		/* SRAM Interface BUSY */
	wire			BUSY0;	/* SRAM Interface BUSY or Enabled */
	wire [3:0]	DIR;		/* Data bus direction */
	wire			END;		/* SRAM Interace Enabled */

/*****Registers****************************************************************/

	reg bo;		/* Burst Order */
	reg bsy;		/* SRAM Interface Busy */
	reg dne_r;	/* Done */
	reg cen_r;	/* Clock Enable */
	reg en;		/* Enable */
	reg oen;		/* Output Enable */
	reg wen;		/* Write Enable */
	reg zz_r;	/* Sleep */

/*****Initialization****************************************************************/

	initial
	begin
		bo = 1'b0;
		bsy = 1'b0;
		cen_r = 1'b0;
		dne_r = 1'b1;
		en = 1'b1;
		oen = 1'b1;
		wen = 1'b1;
		zz_r = 1'b0;
	end

/*********************************************************************/

	assign BUSY = 		BUSY0 || END;
	assign DNE = 		dne_r;
	assign DQ_TR = 	{{8{DIR[3]}},{8{DIR[2]}},{8{DIR[1]}},{8{DIR[0]}}};
	assign DQP_TR = 	DIR;

/*****SRAM Interface****************************************************************/

	sram_interface si(
		.ADDR_IN(ADDR_IN),
		.ADDR_OUT(ADDR_OUT),
		.ADVLD(NADVLD),
		.BUSY(BUSY0),
		.BURST_LATCH(1'b0),
		.BURST_ORDER(bo),
		.BW(NBW),
		.BYTES(BYTES),
		.CE1(NCE1),
		.CE2(CE2),
		.CE3(NCE3),
		.CEN(CEN),
		.CLK(SRAMCLK),
		.CLOCK_MASK(cen_r),
		.DATA_IN(DATA_IN),
		.DATA_IO_IN(DQ_OUT),
		.DATA_IO_OUT(DQ_IN),
		.DATA_OUT(DATA_OUT),
		.DATA_PIO_IN(DQP_OUT),
		.DATA_PIO_OUT(DQP_IN),
		.DIR(DIR),
		.EN(en),
		.END(END),
		.MODE(MODE),
		.OE(NOE),
		.OEN(oen),
		.PARE(PR),
		.RESET(RESET),
		.SLEEP(zz_r),
		.WE(NWE),
		.WEN(wen),
		.ZZ(ZZ)
		);

/*********************************************************************/
			
	always@(posedge SRAMCLK)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			wen = 1'b1;
			oen = 1'b1;
			en = 1'b1;
			zz_r = 1'b0;
			cen_r = 1'b0;
			bo = 1'b0;
			dne_r = 1'b1;
			bsy = 1'b0;
		end
		else
		begin
			if (SLEEP)
			begin
				if (~BUSY)
				begin
					/* Put SRAM IC to sleep */
					zz_r = 1'b1;
				end
			end
			else
			begin
				zz_r = 1'b0;
			end
			if (CLOCK_MASK)
			begin
				if (~BUSY)
				begin
					/* Disable SRAM Clock */
					cen_r = 1'b1;
				end
			end
			else
			begin
				cen_r = 1'b0;
			end
			/* Change Burst Order */
			if (BURST_ORDER)
			begin
				if (~BUSY)
				begin
					bo = 1'b1;
				end
			end
			else
			begin
				if (~BUSY)
				begin
					bo = 1'b0;
				end
			end
			if (zz_r || cen_r)
			begin
				if (SRT && dne_r)
				begin
					dne_r = 1'b0;
				end
				else
				begin
					dne_r = 1'b1;
				end
			end
			else
			begin
				if (SRT && ~BUSY)
				begin	
					/* Recieved Request */
					en = 1'b0;
					dne_r = 1'b0;
				end
				else if (BUSY && END && ~BUSY0)
				begin
					bsy = 1'b1;
					if (WE) wen = 1'b0;
					if (OE) oen = 1'b0;
				end
				else if (BUSY)
				begin
					en = 1'b1;
					wen = 1'b1;
					oen = 1'b1;
				end
				else if (~dne_r && ~BUSY && bsy)
				begin
					/* Interface Done */
					bsy = 1'b0;
					dne_r = 1'b1;
				end
			end
		end
	end

endmodule
