/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	sram_controller												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/* This module interfaces to a Cypress CY7C1354B SRAM						*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module sram_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input	[1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */
	input [31:0]	DQ_OUT,		/* Data Bus from SRAM IC */
	input [3:0]		DQP_OUT,		/* Data Bus Parity from SRAM IC */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				SRAMCLK,		/* SRAM Clock 50 - 100 MHZ */
	input				SRT,			/* Start Transaction */
	input				WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output [23:0]	ADDR_OUT,	/* Address Bus to SRAM IC */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output			CE2,			/* Chipselect 2 */
	output [31:0]	DATA_OUT,	/* Data Bus to Data Path Memory Interface */
	output			DNE,			/* Transaction Done */
	output [31:0]	DQ_IN,		/* Data Bus to SRAM IC */
	output [31:0]	DQ_TR,		/* Data Bus to SRAM IC Trisate Enable */
	output [3:0]	DQP_IN,		/* Data Bus Parity to SRAM IC */
	output [3:0]	DQP_TR,		/* Data Bus Parity to SRAM IC Trisate Enable */
	output			EN,			/* Transaction Enabled for SRAM Address Space */
	output			MODE,			/* SRAM Mode */
	output			NADVLD,		/* Advanced Load */
	output [3:0]	NBW,			/* Byte Enables */
	output			NCE1,			/* Chipselect 1 */
	output			NCE3,			/* Chipselect 3 */
	output			NCEN,			/* Clock Enable */
	output			NOE,			/* Output Enable to SRAM IC */
	output			NWE,			/* Write Enable to SRAM IC */
	output			PR_IRQ,		/* Parity Error Interrupt */
	output [31:0]	STATUS,		/* Control Register */
	output			ZZ				/* Sleep */
	);

/*****Signals****************************************************************/

	wire [22:0]	ADDR_OUT_C;	/* Address Bus to SRAM IC */
	wire [31:0]	BASE_ADDR;	/* SRAM Base Address */
	wire [3:0]	BM;			/* Byte Selects */
	wire [31:0]	DATA_IN_C;	/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;	/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;	/* Endian Flipped Outgoing Data */
	wire [31:0]	DATA_SRAM;	/* Data from SRAM */
	wire			DNE_SRAM;	/* SRAM Bridge Transaction Done */
	wire [31:0]	MAX_ADDR;	/* SRAM Max Address */
	wire			MEN;			/* Transaction Enabled for SRAM Configuration Space */
	wire			NOE0;			/* SRAM Output Enable */
	wire			NWE0;			/* SRAM Write Enable */
	wire [3:0]	PR;			/* Parity Error */
	wire [31:0]	REG_M;		/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;		/* Selected Configuration Register Data */
	wire [31:0]	WMASK_IN;	/* Incoming Data Write Mask */
	wire [31:0]	WMASK_REG;	/* Configuration Register Data Write Mask */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;		/* BASETAG Register */
	reg [31:0]	control;			/* Control Register */
	reg [31:0]	data_in_m_reg;	/* Masked Incoming Data */
	reg [3:0]	dnecnt;			/* Transaction Progress Counter */
	reg			dner;				/* Configuration Space Transaction Done */
	reg			en_reg;			/* Transaction Enabled for SRAM Address Space */
	reg			men_reg;			/* Transaction Enabled for SRAM Configuration Space */
	reg [31:0]	reg_m_0_reg;	/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;		/* Masked Configuration Register Data */
	reg [2:0]	resetcnt;		/* Reset Counter */

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter	BASETAGREG =	16'h0000;	/* BASETAG Register Relative Address */
	parameter	CONTROLREG =	16'h0004;	/* Control Register Relative Address */

	parameter	BASEl =			31;			/* BASE Bit Range Left */
	parameter	BASEr =			16;			/* BASE Bit Range Right */
	parameter	TAGl = 			15;			/* TAG Bit Range Left */
	parameter	TAGr = 			0;				/* TAG Bit Range Right */
	parameter	TAG =				2;				/* TAG Value */
	parameter	DEFAULTBASE =	16'h0000;	/* Default Base Value */

	parameter	SIZEl = 			31;			/* Size Bit Range Left */
	parameter	SIZEr = 			16;			/* Size Bit Range Right */
	parameter	BUS32 =			15;			/* 32-bit wide Bus */
	parameter	BUS16 =			14;			/* 16-bit wide Bus */
	parameter	BUS8 = 			13;			/* 8-bit wide Bus */
	parameter	RST = 			12;			/* Soft Reset */
	parameter	PR3 =				11;			/* Parity Error Byte 3 */
	parameter	PR2 = 			10;			/* Parity Error Byte 2 */
	parameter	PR1 = 			9;				/* Parity Error Byte 1 */
	parameter	PR0 =				8;				/* Parity Error Byte 0 */
	parameter	SLEEP =			7;				/* Sleep */
	parameter	CLOCK_MASK =	6;				/* Disable Clock */
	parameter	BURST_MODE =	5;				/* Single or Burst Action */
	parameter	BURST_ORDER =	4;				/* Mode */
	parameter	CLOCK_Delayl =	3;				/* Delay Clock Bit Range Left */
	parameter	CLOCK_Delayr =	0;				/* Delay Clock Bit Range Right */
	parameter	DEFAULTSIZE =	16'h0100;	/* Default Size Value */

/*****Initialization****************************************************************/

	initial
	begin
		dner = 1'b1;
		basetagr = 32'b0;
		basetagr[BASEl:BASEr] = DEFAULTBASE;
		basetagr[TAGl:TAGr] = TAG;
		control = 32'b0;
		control[SIZEl:SIZEr] = DEFAULTSIZE;
		control[BUS32] = 1'b1;
		control[SLEEP] = 1'b1;
		resetcnt = 3'b0;
		men_reg = 1'b0;
		en_reg = 1'b0;
		dnecnt = 4'b1111;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
	end

/*********************************************************************/

	assign BASETAG = basetagr;
	assign DNE = DNE_SRAM & dner;
	assign PR_IRQ = PR[0] | PR[1] | PR[2] | PR[3]; 
	assign STATUS = control;

/*****Address decoding****************************************************************/

	assign BASE_ADDR = (basetagr[BASEl:BASEr]<<BASEr);
	assign MAX_ADDR = ((basetagr[BASEl:BASEr] | control[SIZEl:SIZEr])<<BASEr);

	assign EN = (ADDR_IN >= BASE_ADDR) & (ADDR_IN < MAX_ADDR);
	assign MEN = (CADDR[BASEl:BASEr] == ADDR_IN[BASEl:BASEr]);

	assign ADDR_OUT = {1'b0,ADDR_OUT_C};
	assign NWE = NWE0;
	assign NOE = NOE0;
	
/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/
	
	assign DATA_OUT = (OE & en_reg)?DATA_SRAM:32'bz;
	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == CONTROLREG)?control:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BASETAGREG)?basetagr:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed1(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*****SRAM Bridge****************************************************************/
	
	sram_bridge sb(
		.ADDR_IN(ADDR_IN[24:0]),
		.ADDR_OUT(ADDR_OUT_C),
		.BURST_ORDER(control[BURST_ORDER]),
		.BYTES(BYTES),
		.CE2(CE2),
		.CEN(NCEN),
		.CLOCK_MASK(control[CLOCK_MASK]),
		.DATA_IN(DATA_IN),
		.DATA_OUT(DATA_SRAM),
		.DNE(DNE_SRAM),
		.DQ_IN(DQ_IN),
		.DQ_OUT(DQ_OUT),
		.DQ_TR(DQ_TR),
		.DQP_IN(DQP_IN),
		.DQP_OUT(DQP_OUT),
		.DQP_TR(DQP_TR),
		.MODE(MODE),
		.NADVLD(NADVLD),
		.NBW(NBW),
		.NCE1(NCE1),
		.NCE3(NCE3),
		.NOE(NOE0),
		.NWE(NWE0),
		.OE(OE & en_reg),
		.PR(PR),
		.RESET(RESET & resetcnt[2]),
		.SLEEP(control[SLEEP]),
		.SRAMCLK(SRAMCLK),
		.SRT(SRT & en_reg),
		.WE(WE & en_reg),
		.ZZ(ZZ)
		);

/*********************************************************************/
		
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			en_reg = 1'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			en_reg = EN;
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,control[RST],dnecnt[3],dner})
			4'b0xxx	:	begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[BASEl:BASEr] = DEFAULTBASE;
								basetagr[TAGl:TAGr] = TAG;
								control = 32'b0;
								control[SIZEl:SIZEr] = DEFAULTSIZE;
								control[BUS32] = 1'b1;
								control[SLEEP] = 1'b1;
								dner = 1'b1;
								resetcnt = 3'b0;
							end
			4'b11xx	:	begin
								/* Soft Reset */
								basetagr = 32'b0;
								basetagr[BASEl:BASEr] = DEFAULTBASE;
								basetagr[TAGl:TAGr] = TAG;
								control = 32'b0;
								control[SIZEl:SIZEr] = DEFAULTSIZE;
								control[BUS32] = 1'b1;
								control[SLEEP] = 1'b1;
								dner = 1'b1;
								resetcnt = 3'b0;
							end
			4'b1000	:	begin
								/* Update Configuration Registers */
								dner = 1'b1;
								if (men_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										CONTROLREG	:	begin
																/* Update Control Register */
																control = data_in_m_reg | reg_m_reg;
															end
										BASETAGREG	:	begin
																/* Update Basetag Register */
																basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
															end
										default		:	begin
																/* Do Nothing */
																/* Address not Recognized */
															end
									endcase
								end
							end
			4'b10x1	:	begin
								/* Wait for incoming request */
								if (SRT & men_reg)
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
								/* Parity Error */
								if (PR[0]) control[PR0] = 1'b1;
								if (PR[1]) control[PR1] = 1'b1;
								if (PR[2]) control[PR2] = 1'b1;
								if (PR[3]) control[PR3] = 1'b1;
								if (~resetcnt[2]) resetcnt = resetcnt + 1;
							end
			default	:	begin
								/* Do Nothing */
							end
		endcase	
	end

endmodule
