/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	sram_interface													*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/* This module interfaces to a Cypress CY7C1354B SRAM						*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module sram_interface(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [24:0]	ADDR_IN,			/* Address bus from SRAM Bridge */
	input				BURST_LATCH,	/* Burst Mode Latch */
	input				BURST_ORDER,	/* Mode */
	input [1:0]		BYTES,			/* Bytes to be Written */
	input				CLK,				/* SRAM Clock */
	input				CLOCK_MASK,		/* Clock Enable */
	input [31:0]	DATA_IN,			/* Data bus from SRAM Bridge */
	input [31:0]	DATA_IO_IN,		/* Data bus from SRAM IC */
	input [3:0]		DATA_PIO_IN,	/* Data bus Parity from SRAM IC */
	input				EN,				/* Enable */
	input				OEN,				/* Output Enable */
	input				RESET,			/* System Reset */
	input				SLEEP,			/* Sleep */
	input				WEN,				/* Write Enable */
	/* OUTPUT PORTS */
	output [22:0]	ADDR_OUT,		/* Address to SRAM IC */
	output			ADVLD,			/* Advandced Load */
	output			BUSY,				/* Interface Busy */
	output [3:0]	BW,				/* Byte Enables */
	output			CE1,				/* Chipselects 1 */
	output			CE2,				/* Chipselects 2 */
	output			CE3,				/* Chipselects 3 */
	output			CEN,				/* Clock Enable */
	output [31:0]	DATA_IO_OUT,	/* Data bus to SRAM IC */
	output [31:0]	DATA_OUT,		/* Data bus to SRAM Bridge */
	output [3:0]	DATA_PIO_OUT,	/* Data bus Parity to SRAM IC */
	output [3:0]	DIR,				/* Data bus Direction */
	output			DRDY,				/* Data Ready */
	output			END,				/* Enabled */
	output			MODE,				/* Mode */
	output			OE,				/* Output Enable to SRAM IC */
	output [3:0]	PARE,				/* Parity Error */
	output			WE,				/* Write Enable to SRAM IC */
	output			ZZ					/* Sleep */
	);

/*****Signals****************************************************************/

	wire [3:0]	BW_IN;	/* Byte Enables Asynchronous */
	wire			OE_P0;	/* Output Enable Pipeline 0 */
	wire			OE_P1;	/* Output Enable Pipeline 1 */
	wire			OE_P2;	/* Output Enable Pipeline 2 */
	wire			OE_P3;	/* Output Enable Pipeline 3 */
	wire			WE_P0;	/* Write Enable Pipeline 0 */
	wire			WE_P1;	/* Write Enable Pipeline 1 */
	wire			WE_P2;	/* Write Enable Pipeline 2 */
	wire			WE_P3;	/* Write Enable Pipeline 3 */

/*****Registers****************************************************************/

	reg [24:0]	addr_p;		/* Address */
	reg [3:0]	bw_r_p;		/* Byte Enables */
	reg			advld_r;		/* Advanced Load */
	reg			ce1_r;		/* Chipselect 1 */
	reg			ce2_r;		/* Chipselect 2 */
	reg			ce3_r;		/* Chipselect 3 */
	reg			cen_r;		/* Clock Enable */
	reg			en_r_p0;		/* Enable Pipeline 0 */
	reg			en_r_p1;		/* Enable Pipeline 1 */
	reg			en_r_p2;		/* Enable Pipeline 2 */
	reg			en_r_p3;		/* Enable Pipeline 3 */
	reg			mode_r;		/* Mode */
	reg			oe_r_p;		/* Output Enable */
	reg [3:0]	oe_r_p0;		/* Output Enable Pipeline 0 */
	reg [3:0]	oe_r_p1;		/* Output Enable Pipeline 1 */
	reg [3:0]	oe_r_p2;		/* Output Enable Pipeline 2 */
	reg [3:0]	oe_r_p3;		/* Output Enable Pipeline 3 */
	reg [3:0]	pare_p;		/* Parity Error */
	reg [31:0]	rddata_p;	/* Read Data Pipelined 3 */
	reg [8:0]	rddata0_p;	/* Read 0 Data Pipelined 2 */
	reg [8:0]	rddata1_p;	/* Read 1 Data Pipelined 2 */
	reg [8:0]	rddata2_p;	/* Read 2 Data Pipelined 2 */
	reg [8:0]	rddata3_p;	/* Read 3 Data Pipelined 2 */
	reg			we_r_p;		/* Write Enable */
	reg [3:0]	we_r_p0;		/* Write Enable Pipeline 0 */
	reg [3:0]	we_r_p1;		/* Write Enable Pipeline 1 */
	reg [3:0]	we_r_p2;		/* Write Enable Pipeline 2 */
	reg [3:0]	we_r_p3;		/* Write Enable Pipeline 3 */
	reg [31:0]	wrdata_p2;	/* Write Data Pipeline 2 */
	reg [7:0]	wrdata0_p0;	/* Write 0 Data Pipeline 0 */
	reg [8:0]	wrdata0_p1;	/* Write 0 Data Pipeline 1 */
	reg [7:0]	wrdata1_p0;	/* Write 1 Data Pipeline 0 */
	reg [8:0]	wrdata1_p1;	/* Write 1 Data Pipeline 1 */
	reg [7:0]	wrdata2_p0;	/* Write 2 Data Pipeline 0 */
	reg [8:0]	wrdata2_p1;	/* Write 2 Data Pipeline 1 */
	reg [7:0]	wrdata3_p0;	/* Write 3 Data Pipeline 0 */
	reg [8:0]	wrdata3_p1;	/* Write 3 Data Pipeline 1 */
	reg [3:0]	wrdatap_p2;	/* Write Data Parity Pipeline 2 */
	reg			zz_r;			/* Sleep */

/*****Initialization****************************************************************/

	initial
	begin
		addr_p <= 25'b0;
		advld_r <= 1'b0;
		bw_r_p <= 4'b1111;
		ce1_r <= 1'b1;
		ce2_r <= 1'b0;
		ce3_r <= 1'b1;
		cen_r <= 1'b0;
		en_r_p3 <= 1'b1;
		en_r_p2 <= 1'b1;
		en_r_p1 <= 1'b1; 
		en_r_p0 <= 1'b1;
		mode_r <= 1'b0;
		oe_r_p <= 1'b1;
		oe_r_p0 <= 4'b1111;
		oe_r_p1 <= 4'b1111;
		oe_r_p2 <= 4'b1111;
		oe_r_p3 <= 4'b1111;
		pare_p <= 4'b0;
		rddata_p <= 32'b0;
		rddata0_p <= 9'b0;
		rddata1_p <= 9'b0;
		rddata2_p <= 9'b0;
		rddata3_p <= 9'b0;
		we_r_p <= 1'b1;
		we_r_p0 <= 4'b1111;
		we_r_p1 <= 4'b1111;
		we_r_p2 <= 4'b1111;
		we_r_p3 <= 4'b1111;
		wrdata_p2 <= 32'b0;
		wrdata0_p0 <= 8'b0;
		wrdata0_p1 <= 9'b0;
		wrdata1_p0 <= 8'b0;
		wrdata1_p1 <= 9'b0;
		wrdata2_p0 <= 8'b0;
		wrdata2_p1 <= 9'b0;
		wrdata3_p0 <= 8'b0;
		wrdata3_p1 <= 9'b0;
		wrdatap_p2 <= 4'b0;
		zz_r <= 1'b0;
	end
		
/*********************************************************************/

	assign ADDR_OUT = addr_p[24:2];
	assign ADVLD = advld_r;
	assign BUSY = OE_P0 || OE_P1 || OE_P2 || OE_P3 || WE_P0 || WE_P1 || WE_P2 || WE_P3;
	assign BW = bw_r_p;
	assign CE1 = ce1_r;
	assign CE2 = ce2_r;
	assign CE3 = ce3_r;
	assign CEN = cen_r;
	assign DATA_OUT = rddata_p;
	assign DATA_IO_OUT = wrdata_p2;
	assign DATA_PIO_OUT = wrdatap_p2;
	assign DIR = we_r_p2;
	assign DRDY = OE_P3;
	assign END = ~en_r_p0 || ~en_r_p1 || ~en_r_p2 || ~en_r_p3;
	assign MODE = mode_r;
	assign OE = oe_r_p;
	assign OE_P0 = ~oe_r_p0[0] || ~oe_r_p0[1] || ~oe_r_p0[2] || ~oe_r_p0[3];
	assign OE_P1 = ~oe_r_p1[0] || ~oe_r_p1[1] || ~oe_r_p1[2] || ~oe_r_p1[3];
	assign OE_P2 = ~oe_r_p2[0] || ~oe_r_p2[1] || ~oe_r_p2[2] || ~oe_r_p2[3];
	assign OE_P3 = ~oe_r_p3[0] || ~oe_r_p3[1] || ~oe_r_p3[2] || ~oe_r_p3[3]; 
	assign PARE = pare_p;
	assign WE = we_r_p;
	assign WE_P0 = ~we_r_p0[0] || ~we_r_p0[1] || ~we_r_p0[2] || ~we_r_p0[3];
	assign WE_P1 = ~we_r_p1[0] || ~we_r_p1[1] || ~we_r_p1[2] || ~we_r_p1[3];
	assign WE_P2 = ~we_r_p2[0] || ~we_r_p2[1] || ~we_r_p2[2] || ~we_r_p2[3];
	assign WE_P3 = ~we_r_p3[0] || ~we_r_p3[1] || ~we_r_p3[2] || ~we_r_p3[3]; 
	assign ZZ = zz_r;

/*****Byte Enables****************************************************************/
	
	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BW_IN)
		);

/*********************************************************************/
		
	always@(posedge CLK)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			advld_r <= 1'b0;
			ce1_r <= 1'b1;
			ce2_r <= 1'b0;
			ce3_r <= 1'b1;
			cen_r <= 1'b0;
			en_r_p3 <= 1'b1;
			en_r_p2 <= 1'b1;
			en_r_p1 <= 1'b1; 
			en_r_p0 <= 1'b1;
			mode_r <= 1'b0;
			oe_r_p <= 1'b1;
			oe_r_p0 <= 4'b1111;
			oe_r_p1 <= 4'b1111;
			oe_r_p2 <= 4'b1111;
			oe_r_p3 <= 4'b1111;
			we_r_p <= 1'b1;
			we_r_p0 <= 4'b1111;
			we_r_p1 <= 4'b1111;
			we_r_p2 <= 4'b1111;
			we_r_p3 <= 4'b1111;
			zz_r <= 1'b0;
		end
		else
		begin
			/* Output Signals */
			advld_r <= BURST_LATCH;
			ce1_r <= EN;
			ce2_r <= ~EN;
			ce3_r <= EN;
			cen_r <= CLOCK_MASK;
			mode_r <= BURST_ORDER;
			zz_r <= SLEEP;
			/* Pipeline Signals */
			en_r_p3 <= en_r_p2;
			en_r_p2 <= en_r_p1;
			en_r_p1 <= en_r_p0; 
			en_r_p0 <= EN;
			we_r_p3 <= we_r_p2;
			oe_r_p3 <= oe_r_p2;
			we_r_p2 <= we_r_p1;
			oe_r_p2 <= oe_r_p1;
			we_r_p1 <= we_r_p0;
			oe_r_p1 <= oe_r_p0;
			we_r_p0 <= {4{WEN}};
			oe_r_p0 <= {4{OEN}};
			we_r_p <= WEN;
			oe_r_p <= OEN;
		end
	end
	
	always@(posedge CLK)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			addr_p <= 25'b0;
			bw_r_p <= 4'b1111;
		end
		else
		begin
			/* Address and Byte Enables */
			addr_p <= ADDR_IN;
			bw_r_p[0] <= BW_IN[0];
			bw_r_p[1] <= BW_IN[1];
			bw_r_p[2] <= BW_IN[2];
			bw_r_p[3] <= BW_IN[3];
		end
	end
	
	always@(posedge CLK)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			wrdata0_p1 <= 9'b0;
			wrdata1_p1 <= 9'b0;
			wrdata2_p1 <= 9'b0;
			wrdata3_p1 <= 9'b0;
			wrdata0_p0 <= 8'b0;
			wrdata1_p0 <= 8'b0;
			wrdata2_p0 <= 8'b0;
			wrdata3_p0 <= 8'b0;
		end
		else
		begin
			/* Pipelined Write Data */
			wrdata_p2 <= {wrdata3_p1[7:0],wrdata2_p1[7:0],wrdata1_p1[7:0],wrdata0_p1[7:0]};
			wrdatap_p2 <= {wrdata3_p1[8],wrdata2_p1[8],wrdata1_p1[8],wrdata0_p1[8]};
			wrdata0_p1[7:0] <= wrdata0_p0;
			wrdata0_p1[8] <= wrdata0_p0[7] ^ wrdata0_p0[6] ^ wrdata0_p0[5] ^ wrdata0_p0[4] ^ wrdata0_p0[3] ^ wrdata0_p0[2] ^ wrdata0_p0[1] ^ wrdata0_p0[0];
			wrdata1_p1[7:0] <= wrdata1_p0;
			wrdata1_p1[8] <= wrdata1_p0[7] ^ wrdata1_p0[6] ^ wrdata1_p0[5] ^ wrdata1_p0[4] ^ wrdata1_p0[3] ^ wrdata1_p0[2] ^ wrdata1_p0[1] ^ wrdata1_p0[0];
			wrdata2_p1[7:0] <= wrdata2_p0;
			wrdata2_p1[8] <= wrdata2_p0[7] ^ wrdata2_p0[6] ^ wrdata2_p0[5] ^ wrdata2_p0[4] ^ wrdata2_p0[3] ^ wrdata2_p0[2] ^ wrdata2_p0[1] ^ wrdata2_p0[0];
			wrdata3_p1[7:0] <= wrdata3_p0;
			wrdata3_p1[8] <= wrdata3_p0[7] ^ wrdata3_p0[6] ^ wrdata3_p0[5] ^ wrdata3_p0[4] ^ wrdata3_p0[3] ^ wrdata3_p0[2] ^ wrdata3_p0[1] ^ wrdata3_p0[0];
			wrdata0_p0 <= DATA_IN[7:0];
			wrdata1_p0 <= DATA_IN[15:8];
			wrdata2_p0 <= DATA_IN[23:16];
			wrdata3_p0 <= DATA_IN[31:24];
		end
	end
	
	always@(posedge CLK)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			rddata_p <= 32'b0;
			pare_p <= 4'b0;
		end
		else
		begin
			if (zz_r || cen_r)
			begin
				rddata_p <= 32'b0;
				pare_p <= 4'b0;
			end
			else
			begin
				/* Pipelined Read Data */
				rddata_p <= {rddata3_p[7:0],rddata2_p[7:0],rddata1_p[7:0],rddata0_p[7:0]};
				pare_p[0] <= rddata0_p[8] ^ rddata0_p[7] ^ rddata0_p[6] ^ rddata0_p[5] ^ rddata0_p[4] ^ rddata0_p[3] ^ rddata0_p[2] ^ rddata0_p[1] ^ rddata0_p[0];
				pare_p[1] <= rddata1_p[8] ^ rddata1_p[7] ^ rddata1_p[6] ^ rddata1_p[5] ^ rddata1_p[4] ^ rddata1_p[3] ^ rddata1_p[2] ^ rddata1_p[1] ^ rddata1_p[0];
				pare_p[2] <= rddata2_p[8] ^ rddata2_p[7] ^ rddata2_p[6] ^ rddata2_p[5] ^ rddata2_p[4] ^ rddata2_p[3] ^ rddata2_p[2] ^ rddata2_p[1] ^ rddata2_p[0];
				pare_p[3] <= rddata3_p[8] ^ rddata3_p[7] ^ rddata3_p[6] ^ rddata3_p[5] ^ rddata3_p[4] ^ rddata3_p[3] ^ rddata3_p[2] ^ rddata3_p[1] ^ rddata3_p[0];
			end
		end
	end
		
	always@(posedge CLK)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			rddata0_p <= 9'b0;
			rddata1_p <= 9'b0;
			rddata2_p <= 9'b0;
			rddata3_p <= 9'b0;
		end
		else
		begin
			/* Read Data */
			if (~oe_r_p2[0]) rddata0_p <= {DATA_PIO_IN[0],DATA_IO_IN[7:0]};
			if (~oe_r_p2[1]) rddata1_p <= {DATA_PIO_IN[1],DATA_IO_IN[15:8]};
			if (~oe_r_p2[2]) rddata2_p <= {DATA_PIO_IN[2],DATA_IO_IN[23:16]};
			if (~oe_r_p2[3]) rddata3_p <= {DATA_PIO_IN[3],DATA_IO_IN[31:24]};
		end
	end
	
endmodule
