/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	sysace_bridge													*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module sysace_bridge(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [6:0]		ADDR_IN,		/* Address bus from Sysace Controller */
	input				BRDY,			/* Buffer Ready Signal from System ACE IC */
	input [1:0]		BUFW,			/* System ACE IC Bus Width */
	input [2:0]		BUS,			/* Logical Bus Width */
	input [1:0]		BYTES,		/* Number of Bytes */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input				CLK_IN,		/* System ACE Clock */
	input [31:0]	DATA_IN,		/* Data Bus from Sysace Controller */
	input				DEBUG,		/* Debug Control */
	input [15:0]	DQ_OUT,		/* Data Bus from System ACE IC */
	input				IRQ,			/* Interrupt from System ACE */
	input				OE,			/* Output Enable from Sysace Controller */
	input				RESET,		/* System Reset */
	input				SRT,			/* Start Transaction */
	input [3:0]		TDELAY,		/* Delay Amount */
	input				WE,			/* Write Enable from Sysace Controller */
	/* OUTPUT PORTS */
	output [6:0]	ADDR_OUT,	/* Address bus to System ACE IC */
	output			BD,			/* Buffer Debug Error */
	output			BRDY_SA,		/* Buffer Ready from Sysace Interface */
	output			DNE,			/* Transaction Done */
	output [31:0]	DATA_OUT,	/* Data bus to Sysace Controller */
	output [15:0]	DQ_IN,		/* Data bus to System ACE IC */
	output [15:0]	DQ_TR,		/* Data bus to System ACE Tristate Enable */
	output			IRQ_SA,		/* Interrupt from Sysace Interface */
	output			NCE,			/* Chipselect to System ACE IC */
	output			NOE,			/* Output Enable to System ACE IC */
	output			NWE			/* Write Enable to System ACE IC */
	);

/*****Signals****************************************************************/

	wire [6:0]	ADDR_CURR;		/* Current Byte/Halfword Address */
	wire [6:0]	BBM;				/* Bytes Pending */
	wire [3:0]	BM;				/* Byte Selects */
	wire			BUSY;				/* Interface Busy */
	wire [7:0]	DATA_IN_8;		/* Data in 8-bit */
	wire [15:0]	DATA_IN_16;		/* Data in 16-bit */
	wire [15:0]	DATA_IN_SEL;	/* Data in Selected */
	wire [15:0]	DATA_OUT_16;	/* Data from System ACE IC */
	wire [31:0]	DATA_OUT_SEL;	/* Data routed to selected bit range */
	wire			DIR;				/* Data bus direction */
	wire			END;				/* Interface Enabled */

/*****Registers****************************************************************/

	reg [6:0]	addr_r;			/* Word Address */
	reg			bd_r;				/* Buffer Error */
	reg [1:0]	bufw_r;			/* System ACE IC Bus Width */ 
	reg [2:0]	bus_r;			/* Logical Bus Width */
	reg [1:0]	bytes_r;			/* Number of Bytes */
	reg [6:0]	bytespd_r;		/* Bytes Pending */
	reg			bytesel;			/* Byte Select */
	reg [31:0]	data_in_r;		/* Data in */
	reg [31:0]	data_out_r;		/* Data out */
	reg [2:0]	datasel;			/* Data location Select */
	reg			debug_r;			/* Debug Enable */
	reg			dne_r;			/* Transaction Done */
	reg			een;				/* Enable */
	reg			oen;				/* Output Enable */
	reg [3:0]	state;			/* State Variable */
	reg [3:0]	tdelay_r;		/* Delay Amount */
	reg			wen;				/* Write Enable */

/*****Initialization****************************************************************/

	initial
	begin
		addr_r = 7'b0;
		bd_r = 1'b0;
		bufw_r = 2'b0;
		bus_r = 3'b0;
		bytes_r = 2'b0;
		bytesel = 1'b0;
		bytespd_r = 7'b0;
		data_in_r = 32'b0;
		data_out_r = 32'b0;
		datasel = 3'b0;
		debug_r = 1'b0;
		dne_r = 1'b1;
		een = 1'b0;
		oen = 1'b0;
		state = 4'b0;
		tdelay_r = 4'b0;
		wen = 1'b0;
	end

/*********************************************************************/

	assign ADDR_CURR = {addr_r[6:2],datasel[1:0]};
	assign BD = bd_r & addr_r[6];
	assign DATA_OUT = data_out_r;
	assign DNE = dne_r;
	assign DQ_TR = {16{DIR}};

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(addr_r[1:0]),
		.BYTES(bytes_r),
		.BW(BM)
		);

	byteselsa bssa(
		.BM(BM),
		.BUFW(bufw_r[1:0]),
		.BUFEN(addr_r[6]),
		.BUS(bus_r),
		.BBM(BBM)
		);

/*****Data Handling****************************************************************/

	/* Data from System ACE */
	dataflip df(
		.CLK(CLK),
		.DATA_IN(DATA_OUT_16),
		.SW(bytespd_r),
		.DATA_OUT(DATA_OUT_SEL)
		);

	/* Data to System ACE */

	MUX8_4to1 mux0(
		.a0(data_in_r[31:24]), 
		.a1(data_in_r[23:16]), 
		.a2(data_in_r[15:8]),
		.a3(data_in_r[7:0]), 
		.sel(datasel[1:0]), 
		.out(DATA_IN_8)
		);

	MUX16_2to1 mux1(
		.a0({data_in_r[23:16],data_in_r[31:24]}),
		.a1({data_in_r[7:0],data_in_r[15:8]}),
		.sel(datasel[1]), 
		.out(DATA_IN_16)
		);

	MUX16_2to1 mux2(
		.a0({8'b0,DATA_IN_8}), 
		.a1(DATA_IN_16), 
		.sel(datasel[2]), 
		.out(DATA_IN_SEL)
		);
		
/*****System ACE Interface****************************************************************/

	sysace_interface si(
		.ADDR_IN(ADDR_CURR),
		.ADDR_OUT(ADDR_OUT),
		.BRDY(BRDY),
		.BRDY_SA(BRDY_SA),
		.BUSY(BUSY),
		.CLK(CLK),
		.CLK_IN(CLK_IN),
		.DATA_IN(DATA_IN_SEL),
		.DATA_OUT(DATA_OUT_16),
		.DEBUG(debug_r & addr_r[6]),
		.DIR(DIR),
		.DQ_IN(DQ_OUT),
		.DQ_OUT(DQ_IN),
		.EN(een),
		.END(END),
		.IRQ(IRQ),
		.IRQ_SA(IRQ_SA),
		.NCE(NCE),
		.NOE(NOE),
		.NWE(NWE),
		.OE(oen),
		.RESET(RESET),
		.TDELAY(tdelay_r),
		.WE(wen)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			addr_r = 7'b0;
			bd_r = 1'b0;
			bufw_r = 2'b0;
			bus_r = 3'b0;
			bytes_r = 2'b0;
			bytesel = 1'b0;
			bytespd_r = 7'b0;
			data_in_r = 32'b0;
			data_out_r = 32'b0;
			datasel = 3'b0;
			debug_r = 1'b0;
			dne_r = 1'b1;
			een = 1'b0;
			oen = 1'b0;
			state = 4'b0;
			tdelay_r = 4'b0;
			wen = 1'b0;
		end
		else
		begin
			case(state)
				4'b0000		:	begin
										bytes_r = BYTES;
										bus_r = BUS;
										bufw_r = BUFW[1:0];
										tdelay_r = TDELAY;
										debug_r = DEBUG;
										if (~BRDY_SA)
										begin
											/* Reset bytesel */
											bytesel = 1'b0;
										end
										if (SRT)
										begin
											/* Request Recieved */
											state = 4'b0001;
										end
									end
				4'b0001		:	begin
										/* Latch Control */
										dne_r = 1'b0;
										bd_r = 1'b0;
										addr_r = ADDR_IN;
										data_in_r = DATA_IN;
										state = 4'b0010;
									end
				4'b0010		:	begin
										/* Reset Data and Encode Byte Order */
										bytespd_r = BBM;
										data_out_r = 32'b0;
										een = 1'b1;
										state = 4'b0100;
									end
				4'b0011		:	begin
										if (BRDY_SA & addr_r[6])
										begin
											/* Oscillate Bytesel */
											bytesel = ~bytesel;
										end
										dne_r = 1'b1;
										data_in_r = 32'b0;
										een = 1'b0;
										if ((BBM[5] & addr_r[6]) & (data_out_r[31:24] == 8'b0)) bd_r = 1'b1;
										if ((BBM[5] & addr_r[6]) & (data_out_r[23:16] == 8'b0)) bd_r = 1'b1;
										if ((BBM[4] & addr_r[6]) & (data_out_r[15:8] == 8'b0)) bd_r = 1'b1;
										if ((BBM[4] & addr_r[6]) & (data_out_r[7:0] == 8'b0)) bd_r = 1'b1;
										if ((BBM[3] & addr_r[6]) & (data_out_r[31:24] == 8'b0)) bd_r = 1'b1;
										if ((BBM[2] & addr_r[6]) & (data_out_r[23:16] == 8'b0)) bd_r = 1'b1;
										if ((BBM[1] & addr_r[6]) & (data_out_r[15:8] == 8'b0)) bd_r = 1'b1;
										if ((BBM[0] & addr_r[6]) & (data_out_r[7:0] == 8'b0)) bd_r = 1'b1;
										state = 4'b0000;
									end
				4'b0100		:	begin
										if (bytespd_r == 7'b0)
										begin
											/* All Bytes Captured */
											state = 4'b0011;
										end
										else if (bytespd_r[6])
										begin
											/* 32-bit Bus not supported */
											bytespd_r[6] = 1'b0;
											state = 4'b0011;
										end
										else if (bytespd_r[5])
										begin
											/* 16-bit Bus Hi */
											datasel = 3'b100;
											state = 4'b0101;
										end
										else if (bytespd_r[4])
										begin
											/* 16-bit Bus Lo */
											datasel = 3'b110;
											state = 4'b0101;
										end
										else if (bytespd_r[3])
										begin
											/* 8-bit Bus Byte 3 */
											if (bytesel & addr_r[6] & bufw_r[0])
											begin
												/* For Reading Buffer */
												datasel = 3'b001;
											end
											else
											begin
												datasel = 3'b000;
											end
											state = 4'b0101;
										end
										else if (bytespd_r[2])
										begin
											/* 8-bit Bus Byte 2 */
											datasel = 3'b001;
											state = 4'b0101;
										end
										else if (bytespd_r[1])
										begin
											/* 8-bit Bus Byte 1 */
											datasel = 3'b010;
											state = 4'b0101;
										end
										else if (bytespd_r[0])
										begin
											/* 8-bit Bus Byte 0 */
											datasel = 3'b011;
											state = 4'b0101;
										end
										else
										begin
											state = 4'b0011;
										end
									end
				4'b0101		:	begin
										if (END)
										begin
											/* Interface Enabled */
											state = 4'b0110;
										end
									end
				4'b0110		:	begin
										/* Operation Signals */
										if (WE) wen = 1'b1;
										if (OE) oen = 1'b1;
										state = 4'b0111;
									end
				4'b0111		:	begin
										/* Operation Started */
										if (BUSY)
										begin
											state = 4'b1000;
										end
									end
				4'b1000		:	begin
										/* Deassert Operation Signals */
										wen = 1'b0;
										oen = 1'b0;
										state = 4'b1001;
									end
				4'b1001		:	begin
										/* Wait for Operation to Finish */
										if (~BUSY) 
										begin
											state = 4'b1010;
										end
									end
				4'b1010		:	begin
										/* Capture Data */
										if (bytespd_r == 7'b0)
										begin
											/* All Bytes Captured */
											state = 4'b0011;
										end
										else if (bytespd_r[6])
										begin
											/* 32-bit Bus not Supported */
											bytespd_r[6] = 1'b0;
											state = 4'b0011;
										end
										else if (bytespd_r[5])
										begin
											/* 16-bit Bus Hi */
											data_out_r[31:24] = DATA_OUT_SEL[31:24];
											data_out_r[23:16] = DATA_OUT_SEL[23:16];
											state = 4'b1011;
										end
										else if (bytespd_r[4])
										begin
											/* 16-bit Bus Lo */
											data_out_r[15:8] = DATA_OUT_SEL[15:8];
											data_out_r[7:0] = DATA_OUT_SEL[7:0];
											state = 4'b1011;
										end
										else if (bytespd_r[3])
										begin
											/* 8-bit Bus Byte 3 */
											data_out_r[31:24] = DATA_OUT_SEL[31:24];
											state = 4'b1011;
										end
										else if (bytespd_r[2])
										begin
											/* 8-bit Bus Byte 2 */
											data_out_r[23:16] = DATA_OUT_SEL[23:16];
											state = 4'b1011;
										end
										else if (bytespd_r[1])
										begin
											/* 8-bit Bus Byte 1 */
											data_out_r[15:8] = DATA_OUT_SEL[15:8];
											state = 4'b1011;
										end
										else if (bytespd_r[0])
										begin
											/* 8-bit Bus Byte 0 */
											data_out_r[7:0] = DATA_OUT_SEL[7:0];
											state = 4'b1011;
										end
										else
										begin
											state = 4'b0011;
										end
									end
				4'b1011		:	begin
										if (bytespd_r == 7'b0)
										begin
											/* All Bytes Captured */
											state = 4'b0011;
										end
										else if (bytespd_r[6])
										begin
											/* 32-bit Bus not Supported */
											bytespd_r[6] = 1'b0;
											state = 4'b0011;
										end
										else if (bytespd_r[5])
										begin
											/* Clear 16-bit Bus Hi */
											bytespd_r[5] = 1'b0;
											state = 4'b0100;
										end
										else if (bytespd_r[4])
										begin
											/* Clear 16-bit Bus Lo */
											bytespd_r[4] = 1'b0;
											state = 4'b0100;
										end
										else if (bytespd_r[3])
										begin
											/* Clear 8-bit Bus Byte 3 */
											bytespd_r[3] = 1'b0;
											state = 4'b0100;
										end
										else if (bytespd_r[2])
										begin
											/* Clear 8-bit Bus Byte 2 */
											bytespd_r[2] = 1'b0;
											state = 4'b0100;
										end
										else if (bytespd_r[1])
										begin
											/* Clear 8-bit Bus Byte 1 */
											bytespd_r[1] = 1'b0;
											state = 4'b0100;
										end
										else if (bytespd_r[0])
										begin
											/* Clear 8-bit Bus Byte 0 */
											bytespd_r[0] = 1'b0;
											state = 4'b0100;
										end
										else
										begin
											state = 4'b0011;
										end
									end
				default		:	begin
										state = 4'b0000;
									end
			endcase
		end
	end

endmodule

/*****Byte Selects****************************************************************/

module byteselsa(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [3:0]		BM,		/* Byte Selects */
	input 			BUFEN,	/* Buffer Enable */
	input [1:0]		BUFW,		/* System ACE Bus Width */
	input [2:0]		BUS,		/* Logic Bus Width */
	/* OUTPUT PORTS */
	output [6:0]	BBM		/* Bytes Pending */
	);

/*****Signals****************************************************************/
	
	wire [3:0] BM1;		/* Bytes Locations Selected */
	
/*********************************************************************/

	assign BM1[0] = 	(BUFEN)?	BM[0] | BUFW[0] | BUFW[1]:
							BM[0];
	assign BM1[1] =	(BUFEN)?	BM[1] | BUFW[0] | BUFW[1]:
							BM[1];
	assign BM1[2] = 	(BUFEN)?	BM[2] | BUFW[0]:
							BM[2];
	assign BM1[3] = 	BM[3];

	assign BBM[0] =	(BUS == 3'b001)?	~BM1[0]:
							1'b0;
	assign BBM[1] =	(BUS == 3'b001)?	~BM1[1]:
							1'b0;
	assign BBM[2] = 	(BUS == 3'b001)?	~BM1[2]:
							1'b0;
	assign BBM[3] = 	(BUS == 3'b001)?	~BM1[3]:
							1'b0;
	assign BBM[4] =	(BUS == 3'b010)?	~BM1[0] | ~BM1[1]:
							1'b0;
	assign BBM[5] =	(BUS == 3'b010)?	~BM1[2] | ~BM1[3]:
							1'b0;
	assign BBM[6] =	(BUS == 3'b100)?	~BM1[0] | ~BM1[1] | ~BM1[2] | ~BM1[3]:
							1'b0;
endmodule

/*****Data Routing****************************************************************/

module dataflip(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,		/* System Clock 50 - 100 MHZ */
	input [15:0]	DATA_IN,	/* Data in */
	input [6:0]		SW,		/* Switch, bytes pending */
	/* OUTPUT PORTS */
	output [31:0]	DATA_OUT	/* Data out */
	);

/*****Signals****************************************************************/
	
	wire [31:0] DATA_OUT0;	/* Data out Asynchronous */

/*****Registers****************************************************************/
	
	reg [15:0] datain;	/* Data in */
	reg [31:0] dataout;	/* Data out */

/*****Initialization****************************************************************/
	
	initial
	begin
		datain = 16'b0;
		dataout = 32'b0;
	end

/*********************************************************************/
	
	assign DATA_OUT = dataout;

/*****Route Data to appropriate Byte****************************************************************/
	
	assign DATA_OUT0[7:0] =		(SW[6])?	8'hff:
										(SW[5])?	8'hff:
										(SW[4])?	datain[15:8]:
										(SW[3])?	8'hff:
										(SW[2])?	8'hff:
										(SW[1])?	8'hff:
										(SW[0])?	datain[7:0]:
													8'hff;
												
	assign DATA_OUT0[15:8] =	(SW[6])?	8'hff:
										(SW[5])?	8'hff:
										(SW[4])?	datain[7:0]:
										(SW[3])?	8'hff:
										(SW[2])?	8'hff:
										(SW[1])?	datain[7:0]:
										(SW[0])?	8'hff:
													8'hff;
												
	assign DATA_OUT0[23:16] =	(SW[6])?	8'hff:
										(SW[5])?	datain[15:8]:
										(SW[4])?	8'hff:
										(SW[3])?	8'hff:
										(SW[2])?	datain[7:0]:
										(SW[1])?	8'hff:
										(SW[0])?	8'hff:
													8'hff;
												
	assign DATA_OUT0[31:24] =	(SW[6])?	8'hff:
										(SW[5])?	datain[7:0]:
										(SW[4])?	8'hff:
										(SW[3])?	datain[7:0]:
										(SW[2])?	8'hff:
										(SW[1])?	8'hff:
										(SW[0])?	8'hff:
													8'hff;

/*********************************************************************/

	always@(posedge CLK)
	begin
		datain = DATA_IN;
		dataout = DATA_OUT0;
	end

endmodule
