/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	sysace_controller												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module sysace_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input				BRDY,			/* Buffer Ready from System ACE IC */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input				CLK_IN,		/* System ACE Clock */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */
	input [15:0]	DQ_OUT,		/* Data Bus from System ACE IC */
	input				IRQ,			/* System ACE IRQ */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				SRT,			/* Start Transaction */
	input				WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output [6:0]	ADDR_OUT,	/* Address Bus to System ACE IC */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output [31:0]	DATA_OUT,	/* Data Bus to Data Path Memory Interface */
	output			DNE,			/* Transaction Done */
	output [15:0]	DQ_IN,		/* Data Bus to System ACE IC */
	output [15:0]	DQ_TR,		/* Data Bus to System ACE IC Trisate Enable */
	output			NCE,			/* Chipselect */
	output			NOE,			/* Output Enable to System ACE IC */
	output			NWE,			/* Write Enable to System ACE IC */
	output			SA_IRQ,		/* System ACE Controller Interrupt */
	output [31:0]	STATUS		/* Control Register */
	);

/*****Signals****************************************************************/

	wire [6:0]	ADDR_OUT_C;	/* Address Bus to System ACE IC */
	wire [31:0]	BASE_ADDR;	/* System ACE Base Address */
	wire			BD;			/* Buffer Debug Error */
	wire [3:0]	BM;			/* Byte Selects */
	wire [31:0]	DATA_IN_C;	/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;	/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;	/* Endian Flipped Outgoing Data */
	wire [31:0]	DATA_SA;		/* Data from System ACE Interface */
	wire			DNE_SA;		/* System ACE Bridge Transaction Done */
	wire			EN;			/* Transaction Enabled for System ACE Address Space */
	wire [31:0]	MAX_ADDR;	/* System ACE Max Address */
	wire			MEN;			/* Transaction Enabled for System ACE Configuration Space */
	wire			NOE0;			/* System ACE Output Enable */
	wire			NWE0;			/* System ACE Write Enable */
	wire [31:0]	REG_M;		/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;		/* Selected Configuration Register Data */
	wire			SABRDY;		/* System ACE Buffer Ready */
	wire			SAIRQ;		/* System ACE IRQ */
	wire [31:0]	WMASK_IN;	/* Incoming Data Write Mask */
	wire [31:0]	WMASK_REG;	/* Configuration Register Data Write Mask */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;		/* BASETAG Register */
	reg [1:0]	bdcnt;			/* Buffer Debug Error */
	reg [31:0]	control;			/* Control Register */
	reg [31:0]	data_in_m_reg;	/* Masked Incoming Data */
	reg [3:0]	dnecnt;			/* Transaction Progress Counter */
	reg			dner;				/* Configuration Space Transaction Done */
	reg			en_reg;			/* Transaction Enabled for System ACE Address Space */
	reg [31:0]	errors;			/* Buffer Error Counter */
	reg			men_reg;			/* Transaction Enabled for System ACE Configuration Space */
	reg [31:0]	reg_m_0_reg;	/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;		/* Masked Configuration Register Data */
	reg [2:0]	resetcnt;		/* Reset Counter */

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter	BASETAGREG =	16'h0000;	/* BASETAG Register Relative Address */
	parameter	CONTROLREG =	16'h0004;	/* Control Register Relative Address */
	parameter 	ERRORSREG =		16'h0008;	/* Error Counter Register Relative Address */

	/* BASETAG Register */
	parameter	BASEl =			31;			/* BASE Bit Range Left */
	parameter	BASEr =			16;			/* BASE Bit Range Right */
	parameter	TAGl = 			15;			/* TAG Bit Range Left */
	parameter	TAGr = 			0;				/* TAG Bit Range Right */
	parameter	TAG =				10;			/* TAG Value */

	/* Control Register */
	parameter	DEBUG = 			31;			/* Debug Enable */
	parameter	BUFW32 =			30;			/* System ACE IC 32-bit Bus */
	parameter	BUFW16 =			29;			/* System ACE IC 16-bit Bus */
	parameter	BUFW8 = 			28;			/* System ACE IC 8-bit Bus */
	parameter	TDELAYl =		27;			/* Delay Amount Bit Range Left */
	parameter	TDELAYr =		24;			/* Delay Amount Bit Range Right */
	parameter	IRQ_MASKl =		23;			/* Interrupt Mask Bit Range Left */
	parameter	IRQ_MASKr =		22;			/* Interrupt Mask Bit Range Right */
	parameter	BRDYI =			21;			/* Buffer Ready Interrupt */
	parameter	IRQI =			20;			/* System ACE IRQ Interrupt */
	parameter	BUS32 =			19;			/* Logical 32-bit Bus */
	parameter	BUS16 =			18;			/* Logical 16-bit Bus */
	parameter	BUS8 = 			17;			/* Logical 8-bit Bus */
	parameter	RST = 			16;			/* Software Reset */
	parameter	SIZEl = 			15;			/* Size Bit Range Left */
	parameter	SIZEr = 			0;				/* Size Bit Range Right */
	parameter	DEFAULTSIZE =	16'h0100;	/* Default Size Value */

/*****Initialization****************************************************************/
	
	initial
	begin
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		bdcnt = 2'b0;
		control = 32'b0;
		control[SIZEl:SIZEr] = DEFAULTSIZE;
		control[BUS8] = 1'b1;
		control[BUFW32] = 1'b1;
		control[TDELAYl:TDELAYr] = 4'b1111;
		dnecnt = 4'b1111;
		dner = 1'b1;
		en_reg = 1'b0;
		errors = 32'b0;
		men_reg = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
		resetcnt = 3'b0;
	end

/*********************************************************************/

	assign BASETAG = basetagr;
	assign DNE = DNE_SA & dner;
	assign STATUS = control;
	assign SA_IRQ = (control[BRDYI] & control[IRQ_MASKl]) | (control[IRQI] & control[IRQ_MASKr]);

/*****Address Decoding****************************************************************/

	assign BASE_ADDR = ((CADDR[BASEl:BASEr]<<BASEr) | (32'b1<<(7)));
	assign MAX_ADDR = ((CADDR[BASEl:BASEr]<<BASEr) | control[SIZEl:SIZEr] | (32'b1<<(7)));

	assign EN = (ADDR_IN >= BASE_ADDR) & (ADDR_IN < MAX_ADDR);
	assign MEN = ~en_reg & (CADDR[BASEl:BASEr] == ADDR_IN[BASEl:BASEr]);

	assign ADDR_OUT = ADDR_OUT_C;
	assign NWE = NWE0;
	assign NOE = NOE0;

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/
	
	assign DATA_OUT = (OE & en_reg)?DATA_SA:32'bz;
	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == CONTROLREG)?control:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BASETAGREG)?basetagr:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == ERRORSREG)?errors:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed1(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*****System ACE Bridge****************************************************************/

	sysace_bridge sb(
		.ADDR_IN(ADDR_IN[6:0]),
		.ADDR_OUT(ADDR_OUT_C),
		.BD(BD),
		.BRDY(BRDY),
		.BRDY_SA(SABRDY),
		.BUFW(control[BUFW16:BUFW8]),
		.BUS(control[BUS32:BUS8]),
		.BYTES(BYTES),
		.CLK(CLK),
		.CLK_IN(CLK_IN),
		.DATA_IN(DATA_IN),
		.DATA_OUT(DATA_SA),
		.DEBUG(control[DEBUG]),
		.DNE(DNE_SA),
		.DQ_IN(DQ_IN),
		.DQ_OUT(DQ_OUT),
		.DQ_TR(DQ_TR),
		.IRQ(IRQ),
		.IRQ_SA(SAIRQ),
		.NCE(NCE),
		.NOE(NOE0),
		.NWE(NWE0),
		.OE(en_reg & OE),
		.RESET(RESET & resetcnt[2]),
		.SRT(en_reg & SRT),
		.TDELAY(control[TDELAYl:TDELAYr]),
		.WE(en_reg & WE)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			en_reg = 1'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			en_reg = EN;
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,control[RST],dnecnt[3],dner})
			4'b0xxx	:	begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								bdcnt = 2'b0;
								control = 32'b0;
								control[SIZEl:SIZEr] = DEFAULTSIZE;
								control[BUS8] = 1'b1;
								control[BUFW32] = 1'b1;
								control[TDELAYl:TDELAYr] = 4'b1111;
								dner = 1'b1;
								errors = 32'b0;
								resetcnt = 3'b0;
							end
			4'b11xx	:	begin
								/* Soft Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								bdcnt = 2'b0;
								control = 32'b0;
								control[SIZEl:SIZEr] = DEFAULTSIZE;
								control[BUS8] = 1'b1;
								control[BUFW32] = 1'b1;
								control[TDELAYl:TDELAYr] = 4'b1111;
								dner = 1'b1;
								errors = 32'b0;
								resetcnt = 3'b0;
							end
			4'b1000	:	begin
								/* Update Configuration Registers */
								dner = 1'b1;
								if (men_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										CONTROLREG	:	begin
																/* Update Control Register */
																control = data_in_m_reg | reg_m_reg;
															end
										BASETAGREG	:	begin
																/* Update Basetag Register */
																basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
															end
										ERRORSREG	:	begin
																/* Update Error Register */
																errors = data_in_m_reg | reg_m_reg;
															end
										default		:	begin
																/* Do Nothing */
																/* Address not Recognized */
															end
									endcase
								end
							end
			4'b10x1	:	begin
								/* Wait for incoming request */
								if (SRT & men_reg)
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
								/* Interrupts */
								control[IRQI] = SAIRQ;
								control[BRDYI] = SABRDY;
								if (~resetcnt[2]) resetcnt = resetcnt + 1;
								/* Debug Errors */
								if (bdcnt == 2'b01) errors = errors+1;
								bdcnt = {bdcnt[0],BD};
							end
			default	:	begin
								/* Do Nothing */
							end
		endcase	
	end
	
endmodule
